/************************************************************************/
/* GL_FileIO.cc: Global utility layer for file IO			*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 11/02/01 */
/************************************************************************/

/* last update: 11/21/01 */

/* modifications: (most recent first)
 *
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/13/01 RD	added support for line wrapping
 * 11/08/01 RD	completed implementation of class GL_IO
 * 11/02/01 RD	initial version (extract from Global.cc)
 */

#include "Global.h"
#include "GL_FileIO.h"

#include <unistd.h>
#include <sys/stat.h>
#include <stdarg.h>


/*** constants and macros ***********************************************/


#ifdef SOLARIS
#define READ_ACCESS_CODE	"r"
#define WRITE_ACCESS_CODE	"w"
#define RW_ACCESS_CODE		"r+"
#endif /* SOLARIS */

#ifdef SUNOS4
#define READ_ACCESS_CODE	"r"
#define WRITE_ACCESS_CODE	"w"
#define RW_ACCESS_CODE		"r+"
#endif /* SUNOS4 */

#ifdef NETBSD
#define READ_ACCESS_CODE	"r"
#define WRITE_ACCESS_CODE	"w"
#define RW_ACCESS_CODE		"r+"
#endif /* NETBSD */

#ifdef LINUX
#define READ_ACCESS_CODE	"r"
#define WRITE_ACCESS_CODE	"w"
#define RW_ACCESS_CODE		"r+"
#endif /* LINUX */

#ifdef GNUWIN32
#define READ_ACCESS_CODE	"rb"
#define WRITE_ACCESS_CODE	"wb"
#define RW_ACCESS_CODE		"r+"
#endif /* GNUWIN32 */

#ifndef READ_ACCESS_CODE
#error "Unknown system! READ_ACCESS_CODE definition missing!"
#endif /* READ_ACCESS_CODE */
#ifndef WRITE_ACCESS_CODE
#error "Unknown system! WRITE_ACCESS_CODE definition missing!"
#endif /* WRITE_ACCESS_CODE */
#ifndef RW_ACCESS_CODE
#error "Unknown system! RW_ACCESS_CODE definition missing!"
#endif /* RW_ACCESS_CODE */


/*** internal variables *************************************************/


/* (none) */


/*** exported variables *************************************************/


int GL_IO::SystemTabStep = GL_TAB_STEP_SYSTEM;	/* '\t' tabulator stepping */


/************************************************************************/
/*** implementation of exported functions			      ***/
/************************************************************************/


	/***************/
	/* class GL_IO */
	/***************/


GL_IO::GL_IO(			/* constructor #1 */
	GL_IO_MODE	Mode,		/* (for file I/O) */
	FILE		*FileHandle)
{

assert(FileHandle != NULL);

GL_IO::Mode		= Mode;
GL_IO::FileHandle	= FileHandle;
GL_IO::StringHandle	= NULL;
GL_IO::LineNo		= 1;
GL_IO::ColNo		= 0;
GL_IO::Pos		= 0;
GL_IO::Tab		= 0;
GL_IO::TabStep		= GL_TAB_STEP_DEFAULT;
GL_IO::LineWrap		= GL_LINE_WRAP_DEFAULT;

} /* end of GL_IO::GL_IO #1 */


GL_IO::GL_IO(			/* constructor #2 */
	GL_IO_MODE	Mode,		/* (for file I/O) */
	String		*StringHandle)
{

assert(StringHandle != NULL);

GL_IO::Mode		= Mode;
GL_IO::FileHandle	= NULL;
GL_IO::StringHandle	= StringHandle;
GL_IO::LineNo		= 1;
GL_IO::ColNo		= 0;
GL_IO::Pos		= 0;
GL_IO::Tab		= 0;
GL_IO::TabStep		= GL_TAB_STEP_DEFAULT;
GL_IO::LineWrap		= GL_LINE_WRAP_DEFAULT;

} /* end of GL_IO::GL_IO #1 */


GL_IO::~GL_IO(void)		/* destructor */
{

/* nothing to do */

} /* end of GL_IO::~GL_IO */


GL_HANDLE *GL_IO::OpenFile(	/* open a file */
	const char	*Filename,	/* NULL for tmp. file, "-" for stdio */
	GL_IO_MODE	Mode /* = GL_IO_WRITE */)
{
FILE	*File;

if (Filename)
   { switch(Mode)
	{ case GL_IO_READ:
	     { if (!(File = GL_OpenFileForReading(Filename)))
		  { return(NULL);
		   } /* fi */
	       return(new GL_IO(Mode, File));
	      }
	  case GL_IO_WRITE:
	     { if (!(File = GL_OpenFileForWriting(Filename)))
		  { return(NULL);
		   } /* fi */
	       return(new GL_IO(Mode, File));
	      }
	  case GL_IO_RW:
	     { if (!(File = GL_OpenFile(Filename)))
		  { return(NULL);
		   } /* fi */
	       return(new GL_IO(Mode, File));
	      }
	  default:
	     { GL_Error = GL_ERROR_ILLEGAL_ARGUMENTS;
	       return(NULL);
	      }
	 } /* hctiws */
    } /* fi */
else
   { if (Mode == GL_IO_RW)
	{ if (!(File = GL_OpenTmpFile()))
	     { return(NULL);
	      } /* fi */
	  return(new GL_IO(Mode, File));
	 } /* fi */
     else
	{ GL_Error = GL_ERROR_ILLEGAL_ARGUMENTS;
	  return(NULL);
	 } /* esle */
    } /* esle */

} /* end of GL_IO::OpenFile */


GL_HANDLE *GL_IO::OpenString(	/* open a string */
	String		*String,	/* caller-supplied string */
	GL_IO_MODE	Mode /* = GL_IO_WRITE */)
{

assert(String != NULL);

return(new GL_IO(Mode, String));

} /* end of GL_IO::OpenString */


ERROR GL_IO::Check(void)	/* check for any I/O errors */
{

if (FileHandle)
   { if (ferror(FileHandle))
	{ GL_ErrorMessage = "File I/O error";
	  GL_ErrorMessage += GL_SystemErrorMessage();
	  GL_Error = GL_ERROR_FILE_IO_ERROR;
	 } /* fi */
    } /* fi */

return(GL_Error);

} /* end of GL_IO::Check */


ERROR GL_IO::ReadEOF(void)	/* read (and check) EOF condition */
{

if (  (Mode != GL_IO_READ)
    &&(Mode != GL_IO_RW))
   { return(GL_ERROR_RW_MODE_VIOLATION);
    } /* fi */

if (FileHandle)
   { return(GL_ReadEOF(FileHandle));
    } /* fi */
else
   { assert(StringHandle != NULL);
     if (StringHandle->length() != Pos)
	{ return(GL_ERROR_NOT_AT_END_OF_STRING);
	 } /* fi */
    } /* esle */

return(GL_ERROR_NO_ERROR);

} /* end of GL_IO::ReadEOF */


ERROR GL_IO::Rewind(void)	/* rewind an RW file/string */
{

if (Mode != GL_IO_RW)
   { return(GL_ERROR_RW_MODE_VIOLATION);
    } /* fi */

if (FileHandle)
   { return(GL_RewindTmpFile(FileHandle));
    } /* fi */
else
   { assert(StringHandle != NULL);
     Pos = 0;
     LineNo = 1;
     ColNo = 0;
    } /* esle */

return(GL_ERROR_NO_ERROR);

} /* end of GL_IO::Rewind */


void GL_IO::Close(void)		/* close the file/string I/O */
{

if (FileHandle)
   { GL_CloseFile(FileHandle);
    } /* fi */

delete(this);	/* suicide */

return;

} /* end of GL_IO::Close */


ERROR GL_IO::CloseWithCheck(	/* close the file/string I/O, check error */
	void)
{

if (FileHandle)
   { GL_Error = GL_CloseFileWithCheck(FileHandle);
    } /* fi */

delete(this);	/* suicide */

return(GL_Error);

} /* end of GL_IO::CloseWithCheck */


unsigned int GL_IO::GetLineNo(	/* get current line number */
	void)
{

return(LineNo);

} /* end of GL_IO::GetLineNo */


unsigned int GL_IO::GetColNo(	/* get current column number */
	void)
{

return(ColNo);

} /* end of GL_IO::GetColNo */


unsigned int GL_IO::GetPos(	/* get current byte position */
	void)
{

return(Pos);

} /* end of GL_IO::GetPos */


int GL_IO::GetTab(		/* get current tabulator setting */
	void)
{

return(Tab);

} /* end of GL_IO::GetTab */


int GL_IO::GetTabStep(		/* get tabulator stepping */
	void)
{

return(TabStep);

} /* end of GL_IO::GetTabStep */


int GL_IO::GetSystemTabStep(	/* get system tabulator stepping */
	void)
{

return(SystemTabStep);

} /* end of GL_IO::GetSystemTabStep */


unsigned int GL_IO::GetLineWrap( /* get column of line wrapping */
	void)
{

return(LineWrap);

} /* end of GL_IO::GetLineWrap */


void GL_IO::SetTab(		/* set new tabulator setting */
	int		Tab)
{

if (Tab >= 0)
   { this->Tab = Tab;
    } /* fi */
else
   { GL_Error = GL_ERROR_ILLEGAL_ARGUMENTS;
    } /* esle */

} /* end of GL_IO::SetTab */


void GL_IO::SetTabStep(		/* set new tabulator stepping */
	int		TabStep)
{

if (TabStep >= 0)
   { this->TabStep = TabStep;
    } /* fi */
else
   { GL_Error = GL_ERROR_ILLEGAL_ARGUMENTS;
    } /* esle */

} /* end of GL_IO::SetTabStep */


void GL_IO::SetSystemTabStep(	/* set new system tabulator stepping */
	int		SystemTabStep)
{

if (SystemTabStep >= 0)
   { GL_IO::SystemTabStep = SystemTabStep;
    } /* fi */
else
   { GL_Error = GL_ERROR_ILLEGAL_ARGUMENTS;
    } /* esle */

} /* end of GL_IO::SetSystemTabStep */


void GL_IO::SetLineWrap(	/* set column of line wrapping */
	unsigned int	LineWrap)
{

/* no invalid value possible */

this->LineWrap = LineWrap;

} /* end of GL_IO::SetLineWrap */


void GL_IO::TabStepUp(void)	/* increment tabulator by one step */
{

Tab += TabStep;
assert(Tab >= 0);	/* must not become negative */

} /* end of GL_IO::TabStepUp */


void GL_IO::TabStepDown(void)	/* decrement tabulator by one step */
{

Tab -= TabStep;
assert(Tab >= 0);	/* must not become negative */

} /* end of GL_IO::TabStepDown */


void GL_IO::Add2Tab(		/* increase/decrease tabulator setting */
	int		Chars)
{

Tab += Chars;
assert(Tab >= 0);	/* must not become negative */

} /* end of GL_IO::Add2Tab */


ERROR GL_IO::Read(		/* read (binary) data from file/string */
	void		*Bytes,
	size_t		NumBytes)
{

assert(Bytes != NULL);
assert(NumBytes >= 0);

if (  (Mode != GL_IO_READ)
    &&(Mode != GL_IO_RW))
   { return(GL_ERROR_RW_MODE_VIOLATION);
    } /* fi */

if (FileHandle)
   { Pos += NumBytes;
     // binary I/O ignores LineNo, ColNo
     return(GL_ReadBytes(FileHandle, Bytes, NumBytes));
    } /* fi */
else
   { assert(StringHandle != NULL);
     if (Pos + NumBytes <= StringHandle->length())
	{ memcpy(Bytes, &StringHandle->chars()[Pos], NumBytes);
	  Pos += NumBytes;
	  // binary I/O ignores LineNo, ColNo
	 } /* fi */
     else
	{ GL_ErrorMessage.form("Reading %d bytes from string failed", NumBytes);
	  return(GL_ERROR_READ_BYTES_FAILED);
	 } /* esle */
    } /* esle */

return(GL_ERROR_NO_ERROR);

} /* end of GL_IO::Read */


char GL_IO::GetC(void)		/* getc() replacement */
{
char	Char;

if (  (Mode != GL_IO_READ)
    &&(Mode != GL_IO_RW))
   { GL_Error = GL_ERROR_RW_MODE_VIOLATION;
     return(0);
    } /* fi */

if (FileHandle)
   { Char = fgetc(FileHandle);
     if (feof(FileHandle))
	{ return(0);		/* return zero at end of file */
	 } /* fi */
    } /* fi */
else
   { assert(StringHandle != NULL);
     if (Pos == StringHandle->length())
	{ return(0);		/* return zero at end of string */
	 } /* fi */
     assert(Pos < StringHandle->length());
     Char = StringHandle->chars()[Pos];
    } /* esle */

Pos++;
ColNo++;
if (Char == '\n')
   { LineNo++;
     ColNo = 0;
    } /* fi */

return(Char);

} /* end of GL_IO::GetC */


char *GL_IO::GetS(		/* gets() replacement (#1) */
	char		*String,
	int		MaxLen)
{
int	l;

assert(String != NULL);
assert(MaxLen > 0);

if (  (Mode != GL_IO_READ)
    &&(Mode != GL_IO_RW))
   { GL_Error = GL_ERROR_RW_MODE_VIOLATION;
     return(NULL);
    } /* fi */

if (FileHandle)
   { if (!(fgets(String, MaxLen, FileHandle)))
	{ GL_ErrorMessage = "File I/O error";
	  GL_ErrorMessage += GL_SystemErrorMessage();
	  GL_Error = GL_ERROR_FILE_IO_ERROR;
	  return(NULL);
	 } /* fi */
     l = strlen(String);
    } /* fi */
else
   { assert(StringHandle != NULL);
     for(l=0; l<MaxLen-1; l++)
	{ if (Pos + l < StringHandle->length())
	     { String[l] = StringHandle->chars()[Pos + l];
	       if (String[l] == '\n')
		  { break;
		   } /* fi */
	      } /* fi */
	  else
	     { break;
	      } /* esle */
	 } /* rof */
     String[++l] = 0;
    } /* esle */

Pos += l;
ColNo += l;
if (String[l-1] == '\n')
   { LineNo++;
     ColNo = 0;
    } /* fi */

return(String);

} /* end of GL_IO::GetS (#1) */


String *GL_IO::GetS(		/* gets() replacement (#2) */
	String		*String)
{
char	Char;
int	l;

assert(String != NULL);

if (  (Mode != GL_IO_READ)
    &&(Mode != GL_IO_RW))
   { GL_Error = GL_ERROR_RW_MODE_VIOLATION;
     return(NULL);
    } /* fi */

if (FileHandle)
   { *String = "";
     while (!(feof(FileHandle)))
	{ Char = fgetc(FileHandle);
	  *String += Char;
	  if (Char == '\n')
	     { break;
	      } /* fi */
	 } /* elihw */
     l = String->length();
    } /* fi */
else
   { assert(StringHandle != NULL);
     for(l=0; ; l++)
	{ if (Pos + l < StringHandle->length())
	     { if (StringHandle->chars()[Pos+l] == '\n')
		  { l++;
		    break;
		   } /* fi */
	      } /* fi */
	  else
	     { break;
	      } /* esle */
	 } /* rof */
     *String = StringHandle->at((int)Pos, l);
    } /* esle */

Pos += l;
ColNo += l;
if (*String[l-1] == '\n')
   { LineNo++;
     ColNo = 0;
    } /* fi */

return(String);

} /* end of GL_IO::GetS (#2) */


ERROR GL_IO::Write(		/* write (binary) data to file/string */
	const void	*Bytes,
	size_t		NumBytes)
{

assert(Bytes != NULL);
assert(NumBytes >= 0);

if (  (Mode != GL_IO_WRITE)
    &&(Mode != GL_IO_RW))
   { return(GL_ERROR_RW_MODE_VIOLATION);
    } /* fi */

if (FileHandle)
   { Pos += NumBytes;
     // binary I/O ignores LineNo, ColNo
     return(GL_WriteBytes(FileHandle, Bytes, NumBytes));
    } /* fi */
else
   { assert(StringHandle != NULL);
     if (Pos + NumBytes <= StringHandle->length())
	{ StringHandle->at((int)Pos, NumBytes) = String((const char*)Bytes,
								NumBytes);
	  Pos += NumBytes;
	  // binary I/O ignores LineNo, ColNo
	 } /* fi */
     else
	{ if (Pos == StringHandle->length())
	     { *StringHandle += String((const char*)Bytes, NumBytes);
	       Pos += NumBytes;
	       // binary I/O ignores LineNo, ColNo
	      } /* fi */
	  else
	     { GL_ErrorMessage.form("Writing %d bytes to string failed",
								NumBytes);
	       return(GL_ERROR_READ_BYTES_FAILED);
	      } /* esle */
	 } /* esle */
    } /* esle */

return(GL_ERROR_NO_ERROR);

} /* end of GL_IO::Write */


void GL_IO::PrintF(		/* printf() replacement */
	const char	*Format,
	...)
{
va_list		ArgPtr;
String		TmpString;

assert(Format != NULL);

if (  (Mode != GL_IO_WRITE)
    &&(Mode != GL_IO_RW))
   { GL_Error = GL_ERROR_RW_MODE_VIOLATION;
     return;
    } /* fi */

va_start(ArgPtr, Format);
TmpString.vform(Format, ArgPtr);
va_end(ArgPtr);

if (FileHandle)
   { fputs(TmpString.chars(), FileHandle);
    } /* fi */
else
   { assert(StringHandle != NULL);
     if (Pos + TmpString.length() <= StringHandle->length())
	{ StringHandle->at((int)Pos, TmpString.length()) = TmpString;
	 } /* fi */
     else
	{ if (Pos == StringHandle->length())
	     { *StringHandle += TmpString;
	      } /* fi */
	  else
	     { GL_Error = GL_ERROR_ILLEGAL_ARGUMENTS;
	       return;
	      } /* esle */
	 } /* esle */
    } /* esle */

Pos += TmpString.length();
ColNo += TmpString.length();
/* note: for better performance, no support for '\n' inside the string */
if (  (TmpString.length() > 0)
    &&(TmpString[TmpString.length()-1] == '\n'))
   { LineNo++;
     ColNo = 0;
    } /* fi */

} /* end of GL_IO::PrintF */


void GL_IO::PutS(		/* puts() replacement (#1) */
	const char	*String)
{
int	l;

if (  (Mode != GL_IO_WRITE)
    &&(Mode != GL_IO_RW))
   { GL_Error = GL_ERROR_RW_MODE_VIOLATION;
     return;
    } /* fi */

l = strlen(String);

if (FileHandle)
   { fputs(String, FileHandle);
    } /* fi */
else
   { assert(StringHandle != NULL);
     if (Pos + l <= StringHandle->length())
	{ StringHandle->at((int)Pos, l) = String;
	 } /* fi */
     else
	{ if (Pos == StringHandle->length())
	     { *StringHandle += String;
	      } /* fi */
	  else
	     { GL_Error = GL_ERROR_ILLEGAL_ARGUMENTS;
	       return;
	      } /* esle */
	 } /* esle */
    } /* esle */

Pos += l;
ColNo += l;
/* note: for better performance, no support for '\n' inside the string */
if (  (l > 0)
    &&(String[l-1] == '\n'))
   { LineNo++;
     ColNo = 0;
    } /* fi */

} /* end of GL_IO::PutS (#1) */


void GL_IO::PutS(		/* puts() replacement (#2) */
	const String	String)
{

if (  (Mode != GL_IO_WRITE)
    &&(Mode != GL_IO_RW))
   { GL_Error = GL_ERROR_RW_MODE_VIOLATION;
     return;
    } /* fi */

if (FileHandle)
   { fputs(String.chars(), FileHandle);
    } /* fi */
else
   { assert(StringHandle != NULL);
     if (Pos + String.length() <= StringHandle->length())
	{ StringHandle->at((int)Pos, String.length()) = String;
	 } /* fi */
     else
	{ if (Pos == StringHandle->length())
	     { *StringHandle += String;
	      } /* fi */
	  else
	     { GL_Error = GL_ERROR_ILLEGAL_ARGUMENTS;
	       return;
	      } /* esle */
	 } /* esle */
    } /* esle */

Pos += String.length();
ColNo += String.length();
/* note: for better performance, no support for '\n' inside the string */
if (  (String.length() > 0)
    &&(String[String.length()-1] == '\n'))
   { LineNo++;
     ColNo = 0;
    } /* fi */

} /* end of GL_IO::PutS (#2) */


void GL_IO::PutC(		/* putc() replacement */
	const char	Char)
{

if (  (Mode != GL_IO_WRITE)
    &&(Mode != GL_IO_RW))
   { GL_Error = GL_ERROR_RW_MODE_VIOLATION;
     return;
    } /* fi */

if (FileHandle)
   { fputc(Char, FileHandle);
    } /* fi */
else
   { assert(StringHandle != NULL);
     if (Pos + 1 <= StringHandle->length())
	{ StringHandle->at((int)Pos, 1) = Char;
	 } /* fi */
     else
	{ if (Pos == StringHandle->length())
	     { *StringHandle += Char;
	      } /* fi */
	  else
	     { GL_Error = GL_ERROR_ILLEGAL_ARGUMENTS;
	       return;
	      } /* esle */
	 } /* esle */
    } /* esle */

Pos += 1;
ColNo += 1;
if (Char == '\n')
   { LineNo++;
     ColNo = 0;
    } /* fi */

} /* end of GL_IO::PutC */


void GL_IO::PutNL(void)		/* start a new line */
{

if (  (Mode != GL_IO_WRITE)
    &&(Mode != GL_IO_RW))
   { GL_Error = GL_ERROR_RW_MODE_VIOLATION;
     return;
    } /* fi */

if (FileHandle)
   { fputc('\n', FileHandle);
    } /* fi */
else
   { assert(StringHandle != NULL);
     if (Pos + 1 <= StringHandle->length())
	{ StringHandle->at((int)Pos, 1) = '\n';
	 } /* fi */
     else
	{ if (Pos == StringHandle->length())
	     { *StringHandle += '\n';
	      } /* fi */
	  else
	     { GL_Error = GL_ERROR_ILLEGAL_ARGUMENTS;
	       return;
	      } /* esle */
	 } /* esle */
    } /* esle */

Pos += 1;
LineNo++;
ColNo = 0;

} /* end of GL_IO::PutNL */


void GL_IO::PutTAB(void)	/* put a tabulator */
{
int		n,
		t;
String		TmpString;

if (  (Mode != GL_IO_WRITE)
    &&(Mode != GL_IO_RW))
   { GL_Error = GL_ERROR_RW_MODE_VIOLATION;
     return;
    } /* fi */

if (  (StringHandle)
    &&(Pos != StringHandle->length()))
   { GL_Error = GL_ERROR_ILLEGAL_ARGUMENTS;	/* TAB insert not supported */
     return;
    } /* fi */

t = Tab - ColNo;
if (t <= 0)
   { return;	/* don't go backwards */
    } /* fi */

TmpString = "";
if (SystemTabStep > 0)
   { t += ColNo % SystemTabStep;
     for(n=0; n <= t-SystemTabStep; n += SystemTabStep)
	{ TmpString += '\t';
	 } /* rof */
     for( ; n<t; n++)
	{ TmpString += ' ';
	 } /* rof */
    } /* fi */
else
   { for(n=0; n<t; n++)
	{ TmpString += ' ';
	 } /* rof */
    } /* esle */

if (FileHandle)
   { fputs(TmpString.chars(), FileHandle);
    } /* fi */
else
   { assert(StringHandle != NULL);
     *StringHandle += TmpString;
    } /* esle */

Pos += TmpString.length();
ColNo = Tab;

} /* end of GL_IO::PutTAB */


void GL_IO::PutNLTAB(void)	/* start a new line and put a tabulator */
{
int		n;
String		TmpString;

if (  (Mode != GL_IO_WRITE)
    &&(Mode != GL_IO_RW))
   { GL_Error = GL_ERROR_RW_MODE_VIOLATION;
     return;
    } /* fi */

if (  (StringHandle)
    &&(Pos != StringHandle->length()))
   { GL_Error = GL_ERROR_ILLEGAL_ARGUMENTS;	/* NLTAB insert not supported */
     return;
    } /* fi */

TmpString = "\n";
if (SystemTabStep > 0)
   { for(n=0; n <= Tab-SystemTabStep; n += SystemTabStep)
	{ TmpString += '\t';
	 } /* rof */
     for( ; n<Tab; n++)
	{ TmpString += ' ';
	 } /* rof */
    } /* fi */
else
   { for(n=0; n<Tab; n++)
	{ TmpString += ' ';
	 } /* rof */
    } /* esle */

if (FileHandle)
   { fputs(TmpString.chars(), FileHandle);
    } /* fi */
else
   { assert(StringHandle != NULL);
     *StringHandle += TmpString;
    } /* esle */

Pos += TmpString.length();
LineNo++;
ColNo = Tab;

} /* end of GL_IO::PutNLTAB */


void GL_IO::WrapLine(void)	/* wrap line around if too long */
{

if (  (LineWrap > 0)
    &&(ColNo >= LineWrap))
   { PutNL();
    } /* fi */

} /* end of GL_IO::WrapLine */


void GL_IO::WrapLineTAB(void)	/* wrap line around and put TAB if too long */
{

if (  (LineWrap > 0)
    &&(ColNo >= LineWrap)
    &&(ColNo > (unsigned int)Tab))
   { PutNLTAB();
    } /* fi */

} /* end of GL_IO::WrapLineTAB */


	/*** support functions for file I/O ***/


FILE *GL_OpenFileForReading(	/* open a file for read access */
	const char *Filename)	/* (supports "-" for stdin) */
{
FILE	*FileHandle;

assert(Filename != NULL);

if (0 == strcmp(Filename, "-"))
   { return(stdin);
    } /* fi */

if ((GL_Error = GL_FileIsReadable(Filename)))
   { return(NULL);
    } /* fi */

if (!(FileHandle = fopen(Filename, READ_ACCESS_CODE)))
   { GL_ErrorMessage.form("Cannot open file \"%s\" for read access",
			Filename);
     GL_ErrorMessage += GL_SystemErrorMessage();
     GL_Error = GL_ERROR_FILE_R_OPEN_FAILED;
     return(NULL);
    } /* fi */

return(FileHandle);

} /* end of GL_OpenFileForReading */


FILE *GL_OpenFileForWriting(	/* open a (new) file for write access */
	const char *Filename)	/* (supports "-" for stdout) */
{
FILE	*FileHandle;

if (0 == strcmp(Filename, "-"))
   { return(stdout);
    } /* fi */

if ((GL_Error = GL_FileIsCreatable(Filename)))
   { return(NULL);
    } /* fi */

if (!(FileHandle = fopen(Filename, WRITE_ACCESS_CODE)))
   { GL_ErrorMessage.form("Cannot open file \"%s\" for write access",
			Filename);
     GL_ErrorMessage += GL_SystemErrorMessage();
     GL_Error = GL_ERROR_FILE_W_OPEN_FAILED;
     return(NULL);
    } /* fi */

return(FileHandle);

} /* end of GL_OpenFileForWriting */


FILE *GL_OpenFile(		/* open a file for read/write access */
	const char *Filename)
{
FILE	*FileHandle;

if (!(FileHandle = fopen(Filename, RW_ACCESS_CODE)))
   { GL_ErrorMessage.form("Cannot open file \"%s\" for read/write access",
			Filename);
     GL_ErrorMessage += GL_SystemErrorMessage();
     GL_Error = GL_ERROR_FILE_RW_OPEN_FAILED;
     return(NULL);
    } /* fi */

return(FileHandle);

} /* end of GL_OpenFile */


FILE *GL_OpenTmpFile(		/* open a new temp. file for writing */
	void)			/* (temp. file is deleted on close) */
{

FILE	*FileHandle;

if (!(FileHandle = tmpfile()))
   { GL_ErrorMessage = "Temporary file creation failed";
     GL_ErrorMessage += GL_SystemErrorMessage();
     GL_Error = GL_ERROR_TMP_FILE_CREATE_FAILED;
     return(NULL);
    } /* fi */

return(FileHandle);

} /* end of GL_OpenTmpFile */


ERROR GL_RewindTmpFile(		/* rewind the temp. file to beginning */
	FILE	*FileHandle)	/* (allow read or write from start) */
{

if (fseek(FileHandle, 0, SEEK_SET))
   { GL_ErrorMessage = "Temporary file rewind failed";
     GL_ErrorMessage += GL_SystemErrorMessage();
     return(GL_Error = GL_ERROR_TMP_FILE_REWIND_FAILED);
    } /* fi */

return(GL_ERROR_NO_ERROR);

} /* end of GL_RewindTmpFile */


void GL_CloseFile(		/* close a file */
	FILE *FileHandle)
{

assert(FileHandle != NULL);

if (  (FileHandle != stdin)
    &&(FileHandle != stdout))
   { fclose(FileHandle);
    } /* fi */

} /* end of GL_CloseFile */


ERROR GL_CloseFileWithCheck(	/* close a file and check for errors */
	FILE *FileHandle)
{

assert(FileHandle != NULL);

if (ferror(FileHandle))
   { if (  (FileHandle != stdin)
	 &&(FileHandle != stdout))
	{ fclose(FileHandle);
	 } /* fi */
     return(GL_ERROR_PROBLEM_WHILE_CLOSING_FILE);
    } /* fi */

if (  (FileHandle != stdin)
    &&(FileHandle != stdout))
   { if (fclose(FileHandle))
	{ return(GL_ERROR_PROBLEM_WHILE_CLOSING_FILE);
	 } /* fi */
    } /* fi */

return(GL_ERROR_NO_ERROR);

} /* end of GL_CloseFileWithCheck */


ERROR GL_DeleteFile(		/* delete a file */
	const char	*Filename)
{
if (unlink(Filename))
   { GL_ErrorMessage.form("Cannot delete file \"%s\"",
			Filename);
     GL_ErrorMessage += GL_SystemErrorMessage();
     return(GL_ERROR_FILE_DELETE_FAILED);
    } /* fi */

return(GL_ERROR_NO_ERROR);

} /* end of GL_DeleteFile */


ERROR GL_FileIsReadable(	/* check if file is read-accessable */
	const char	*Filename)	/* (supports "-" for stdin) */
{
struct stat	StatBuf;

assert(Filename != NULL);

if (0 == strcmp(Filename, "-"))
   { return(GL_ERROR_NO_ERROR);
    } /* fi */

if (0 != access(Filename, R_OK))
   { GL_ErrorMessage.form("File \"%s\" not readable%s",
			Filename, GL_SystemErrorMessage());
     return(GL_ERROR_FILE_NOT_READABLE);
    } /* fi */
if (0 != stat(Filename, &StatBuf))
   { GL_ErrorMessage.form("File \"%s\" not readable%s",
			Filename, GL_SystemErrorMessage());
     return(GL_ERROR_FILE_NOT_READABLE);
    } /* fi */
if ((StatBuf.st_mode & S_IFMT) != S_IFREG)
   { GL_ErrorMessage.form("File \"%s\" not readable;" GL_ERROR_MSG_NEWLINE
			"not an ordinary file",
			Filename);
     return(GL_ERROR_FILE_NOT_READABLE);
    } /* fi */

return(GL_ERROR_NO_ERROR);

} /* end of GL_FileIsReadable */


ERROR GL_FileIsCreatable(	/* check if file is write-accessable */
	const char	*Filename)	/* (supports "-" for stdout) */
{

assert(Filename != NULL);

if (0 == strcmp(Filename, "-"))
   { return(GL_ERROR_NO_ERROR);
    } /* fi */

if (0 == access(Filename, F_OK))	/* existing? */
   { if (0 == unlink(Filename))
	{ return(GL_ERROR_NO_ERROR);	/* removed! */
	 } /* fi */
     else
	{ GL_ErrorMessage.form("Unlinking existing file \"%s\" failed%s",
			Filename, GL_SystemErrorMessage());
	  return(GL_ERROR_UNLINK_FAILED);
	 } /* fi */
    } /* fi */

return(GL_ERROR_NO_ERROR);

} /* end of GL_FileIsCreatable */


ERROR GL_ReadBytes(		/* read (binary) data from file */
	FILE		*FileHandle,
	void		*Bytes,
	size_t		NumBytes)
{

assert(NumBytes > 0);

if (NumBytes != fread(Bytes, 1, NumBytes, FileHandle))
   { GL_ErrorMessage.form("Reading %d bytes from file failed%s",
			NumBytes, GL_SystemErrorMessage());
     return(GL_Error = GL_ERROR_READ_BYTES_FAILED);
    } /* fi */

return(GL_ERROR_NO_ERROR);

} /* end of GL_ReadBytes */


ERROR GL_WriteBytes(		/* write (binary) data to file */
	FILE		*FileHandle,
	const void	*Bytes,
	size_t		NumBytes)
{

assert(NumBytes > 0);

if (NumBytes != fwrite(Bytes, 1, NumBytes, FileHandle))
   { GL_ErrorMessage.form("Writing %d bytes to file failed%s",
			NumBytes, GL_SystemErrorMessage());
     return(GL_Error = GL_ERROR_WRITE_BYTES_FAILED);
    } /* fi */

return(GL_ERROR_NO_ERROR);

} /* end of GL_WriteBytes */


ERROR GL_ReadEOF(		/* read (and check) EOF from file */
	FILE		*FileHandle)
{
char	Trash;

fread(&Trash, 1, 1, FileHandle);	/* try to read one more byte */
if (!(feof(FileHandle)))
   { clearerr(FileHandle);
     return(GL_ERROR_NOT_AT_END_OF_FILE);
    } /* fi */

clearerr(FileHandle);

return(GL_ERROR_NO_ERROR);

} /* end of GL_ReadEOF */


const char *GL_FindFileInDirList(	/* find a file for read access */
	const char	*Filename,	/* (applying DirList and Suffix) */
	const char	*Suffix /* = NULL */,	/* (NULL if not found) */
	GL_STRINGLIST	*DirList /* = NULL */)
{
GL_STRINGLIST	*Dir;
static string	NewFilename;	/* return buffer! */

assert(Filename != NULL);

if (Filename[0] == '/')		/* absolute filename? */
   { NewFilename = Filename;
     if (Suffix)
	{ NewFilename += Suffix;
	 } /* fi */
     if ((GL_ERROR_NO_ERROR == GL_FileIsReadable(NewFilename.chars())))
	{ return(NewFilename.chars());
	 } /* fi */
     return(NULL);	/* absolute file not found */
    } /* fi */

Dir = DirList;		/* apply the list of directories */
while(Dir)
   { NewFilename = Dir->String;
     NewFilename += "/";
     NewFilename += Filename;
     if (Suffix)
	{ NewFilename += Suffix;
	 } /* fi */
     if ((GL_ERROR_NO_ERROR == GL_FileIsReadable(NewFilename.chars())))
	{ return(NewFilename.chars());
	 } /* fi */
     Dir = Dir->Succ;
    } /* elihw */

return(NULL);	/* no such file found in directory list */

} /* end of GL_FindFileInDirList */


/* EOF GL_FileIO.cc */
