/************************************************************************/
/* thread.h: SpecC run-time simulation library, thread management	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 03/26/01 */
/************************************************************************/

/* last update: 12/10/01 */

/* modifications: (most recent first)
 *
 * 12/10/01 RD	added case for Win32 thread usage
 * 07/31/01 RD	dynamically switch used native thread package
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/16/01 RD	resolved name space competition with user by moving more
 *		definitions into _specc name space and adding prefixes
 * 05/16/01 RD	separated common thread code (in thread.[h|cc]) from
 *		platform-specific thread code (in PosixThread.[h|cc])
 * 05/16/01 RD	tiny fix for older gcc's
 * 05/14/01 RD	clean up messy exception data structures
 * 05/11/01 RD	added support for exception handling
 * 04/16/01 RD	renamed "SIM_Time" to "sim_time"
 * 04/03/01 RD	added support for 'notifyone'
 * 04/02/01 RD	added support for 'wait' and 'notify'
 * 04/02/01 RD	replaced event_queue and ready_queue with generic queue
 * 03/30/01 RD	added generic classes _specc::queue and _specc::queue_elem
 * 03/30/01 RD	added event_queue
 * 03/29/01 RD	added ready_queue
 * 03/28/01 RD	refinement
 * 03/26/01 RD	initial version
 */

#ifndef THREAD_H
#define THREAD_H

#include "specc.h"

#ifdef THREAD_NAME_Posix
#include "PosixThread.h"	/* use implementation based on posix threads */
#endif /* THREAD_NAME_Posix */
#ifdef THREAD_NAME_Win32
#include "Win32Thread.h"	/* use implementation based on win32 threads */
#endif /* THREAD_NAME_Win32 */


/*** constants and macros ***********************************************/


	/* settable options */

#define SIM_MINIMIZE_CONTEXT_SWITCHES	/* enable/disable optimization */


	/* execution order in ready queue (choose one!) */

#define ADD(t)		Append(t)	/* ready queue is FIFO */
//#define ADD(t)	Prepend(t)	/* ready queue is LIFO */


	/* ordering in event queue (among equal time stamps) */

//#define SIM_CMP_TIME(a,b)	(a>=b)	/* order is FIFO */
#define SIM_CMP_TIME(a,b)	(a>b)	/* order is LIFO */


	/* selection of 'notifyone' (choose one!) */

#define SIM_NOTIFYONE_SELECT_FIRST_WAITING_THREAD
//#define SIM_NOTIFYONE_SELECT_LAST_WAITING_THREAD


	/* return value in deadlock situation */

#define RESULT_DEADLOCK		0	// OK
//#define RESULT_DEADLOCK	10	// ERROR


	/* return value in fatal error situation */

//#define RESULT_FATAL		0	// OK
#define RESULT_FATAL		10	// ERROR


/*** enumeration types **************************************************/


enum SIM_State			/* the set of thread states */
{
SIM_STATE_CREATED,		// initial state
SIM_STATE_RUNNING,		// currently running (only one at a time)
SIM_STATE_READY,		// ready to run (threads in the ready queue)
SIM_STATE_NOTIFIED,		// notified (just awakened by 'notify')
SIM_STATE_WAITFOR,		// waiting for time increase (at 'waitfor')
SIM_STATE_WAIT,			// waiting for events (at 'wait')
SIM_STATE_JOIN,			// father waiting for children to complete
SIM_STATE_COMPLETE,		// terminating child
SIM_STATE_TRYING,		// sensitive to exceptions (at 'try')
SIM_STATE_ABORTING,		// running abort-type exception handler
SIM_STATE_INTERRUPTED,		// running interrupt-type exception handler
SIM_STATE_SUSPENDED,		// suspended from execution due to exception
SIM_STATE_SUSPENDED2		// multiple times suspended
};


/*** type definitions ***************************************************/


typedef enum SIM_State	sim_state;	/* thread status type */


/*** class declarations *************************************************/


	/*********************/
	/*** _specc::queue ***/
	/*********************/


class _specc::queue		/* generic queue of threads */
{
public:
thread		*First,		/* the first thread in the queue */
		*Last;		/* the last thread in the queue */
unsigned int	Length;		/* length of the queue */


queue(void);			/* constructor #1 */

queue(				/* constructor #2 */
	thread	*First);

~queue(void);			/* destructor */


bool IsEmpty(void);		/* check whether this queue is empty */

void Append(			/* append the thread to this queue */
	thread	*Thread);

void Prepend(			/* prepend the thread to this queue */
	thread	*Thread);

void Insert(			/* insert thread into the queue    */
	thread		*Thread);	/* according to time stamp */

void Remove(			/* remove the thread from this queue */
	thread	*Thread);
};


	/**************************/
	/*** _specc::queue_elem ***/
	/**************************/


class _specc::queue_elem	/* thread element in queue */
{
public:
thread		*Next,		/* next thread in the queue (or NULL) */
		*Prev;		/* previous thread in the queue (or NULL) */
queue		*Queue,		/* link to the header of the queue */
		*QueueBackup;	/* backup for queue info during suspension */
sim_time	TimeStamp;	/* TimeStamp (used for waitfor queue only) */


queue_elem(void);		/* constructor #1 */

~queue_elem(void);		/* destructor */


bool IsInQueue(void);		/* check if this thread is in any queue */

bool IsInQueue(			/* check if this thread is in that queue */
	queue	*Queue);
};


	/**********************/
	/*** _specc::thread ***/
	/**********************/


class _specc::thread :		/* class for threads */
	public _specc::thread_base,	/* each thread is a specific thread */
	public _specc::queue_elem	/* and also a queue element         */
{
public:
sim_state	Status,		/* current status (see above) */
		StatusBackup;	/* backup for status info during suspension */
behavior	*Behavior;	/* the behavior of this thread (NULL if root) */
thread		*Parent,	/* parent thread (NULL if root) */
		*NextSibling,	/* next sibling (NULL if none) */
		*FirstChild,	/* first child (NULL) if none) */
		*LastChild;	/* last child (NULL) if none) */
unsigned int	NumChildren,	/* number of children */
		NumCompleted;	/* number of children that have completed */
sim_time	CreationTime;	/* time stamp of thread creation time */
event_ptr	*EventList;	/* list of events to 'wait' for (or NULL) */
exception	*Exceptions;	/* list of exceptions for 'try' (or NULL) */
thread		*ExceptionChild;/* link to current exception child (or NULL) */


thread(				/* constructor #1 */
	behavior	*Behavior,
	thread		*Creator,
	thread		*Parent = NULL);

~thread(void);			/* destructor */


static void Start(void);	/* initialize thread usage */

static void End(void);		/* clean up after thread usage */

static thread *Create(		/* create a new child thread */
	behavior	*Behavior,	/* (NULL for root thread) */
	thread		*Parent);	/* (NULL for root thread) */

void Delete(void);		/* delete this (completed) thread */

void DeleteChildren(void);	/* delete all child threads */

void Join(			/* wait for a child thread to complete */
	thread		*Thread);	/* (NULL joins will all children) */

void Suspend(void);		/* suspend this thread tree (for a while) */

void Resume(			/* resume this thread tree (from suspension) */
	sim_time	Delay);	/* (taking delay into account) */

void Abort(void);		/* abort this thread tree (force completion) */

void Run(			/* transfer control to a specific thread */
	thread		*Thread);

void Schedule(void);		/* transfer control to another thread */

void HandleException(		/* handle an exception */
	exception	*Exception);
};


/*** exported variables *************************************************/


	/* (none) */


/************************************************************************/
/*** prototype definitions for exported functions		      ***/
/************************************************************************/


	/* (none) */


#endif /* __THREAD_H */

/* EOF thread.h */
