/************************************************************************/
/* specc.h: SpecC run-time simulation library, compiler-level API	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 03/21/01 */
/************************************************************************/

/* last update: 11/19/01 */

/* modifications: (most recent first)
 *
 * 11/19/01 RD	bug fix: made channel destructor virtual
 * 09/07/01 RD	bug fix: modified 'piped' template to use static memory
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/16/01 RD	resolved name space competition with user by moving more
 *		definitions into _specc name space and adding prefixes
 * 05/16/01 RD	separated common thread code (in thread.[h|cc]) from
 *		platform-specific thread code (in PosixThread.[h|cc])
 * 05/15/01 RD	cleaned up the source code
 * 05/14/01 RD	added _specc::exception to separate data on heap and stack
 * 05/09/01 RD	added support for exception handling
 * 04/25/01 RD	added copy-assignment to piped template (bug fix)
 * 04/16/01 RD	renamed "SIM_Time" to "sim_time"
 * 04/05/01 RD	added support for new 'pipe' syntax with termination
 * 04/04/01 RD	added support for 'pipe', 'piped'
 * 04/02/01 RD	added _specc::event_list
 * 04/02/01 RD	added _specc::event_ptr
 * 04/02/01 RD	replaced event_queue and ready_queue with generic queue
 * 03/30/01 RD	added generic classes _specc::queue and _specc::queue_elem
 * 03/30/01 RD	added _specc::event_queue
 * 03/29/01 RD	added _specc::ready_queue
 * 03/28/01 RD	added inclusion of "piped.h"
 * 03/21/01 RD	initial version
 */


#ifndef __SPECC_H
#define __SPECC_H

#include "sim.h"


/*** constants and macros ***********************************************/


#ifndef NULL
#define NULL	0
#endif


/*** class declarations *************************************************/


	/**************/
	/*** _specc ***/
	/**************/


class _specc			/* wrapper class serving as name space */
{
public:

	/* wrapped sub-classes */

class event;			/* class for event types */
class event_ptr;		/* class for a list of events */
class event_list;		/* class for a list of event lists */
class behavior;			/* base class for behaviors */
class channel;			/* base class for channels */
class fork;			/* class for parallel behaviors */
class try_block;		/* class for exception-enabled behaviors */
class exception_block;		/* class for exception handler behaviors */
class exception;		/* class for exception handlers (dynamic) */
class queue;			/* class for queues of threads */
class queue_elem;		/* class for elements in thread queues */
class thread_base;		/* base class for threads (native threads) */
class thread;			/* class for threads */
class piped_base;		/* base class for 'piped' variables */


	/* wrapped variables */

static sim_time CurrentTime;	/* current simulation time */

static thread	*RootThread;	/* the root thread */

static thread	*CurrentThread;	/* the current thread */

static queue	ReadyQueue;	/* the ready queue */

static queue	WaitforQueue;	/* the event queue for 'waitfor' */

static queue	WaitQueue;	/* the queue for 'wait'ing threads */

static queue	NotifiedQueue;	/* the queue for 'notify'ed threads */

static queue	TryQueue;	/* the queue of 'try'ing threads */

static queue	SuspendQueue;	/* the queue of suspended threads */

static event	*NotifiedEvents;/* list of notified events */

static event_list *Notify1List;	/* list of 'notifyone'ed events */


	/* wrapped functions */

static void start(void);	/* initialize the simulation engine */

static void end(void);		/* clean up the simulation engine */

static void abort(		/* cleanly abort with a message */
	const char	*Format,	/* (arguments as for printf) */
	...		);

static void par(		/* SpecC 'par' replacement */
	fork		*First,		/* (NULL-terminated list of forks) */
	...		);

static void pipe(		/* SpecC 'pipe' replacement (infinite, old) */
	fork		*First,		/* (NULL-terminated list of forks, */
	...		);		/* NULL-term. list of piped vars.) */

static void pipe(		/* SpecC 'pipe' replacement (finite, new) */
	unsigned int	NumStages,	/* total number of pipeline stages */
	unsigned int	FirstStage,	/* first active stage */
	unsigned int	LastStage,	/* last active stage */
	...		);		/* (list of forks (length NumStages), */
					/* NULL-termin. list of piped vars.)  */

static void tryTrapInterrupt(	/* SpecC 'try-trap-interrupt' replacement */
	try_block	*TryBlock,
	exception_block	*First,
	...		);

static void waitfor(		/* SpecC 'waitfor' replacement */
	sim_time	Delay);

static void wait(		/* SpecC 'wait' replacement */
	event		*First,		/* (NULL-terminated list of events) */
	...		);

static void notify(		/* SpecC 'notify' replacement */
	event		*First,		/* (NULL-terminated list of events) */
	...		);

static void notifyone(		/* SpecC 'notifyone' replacement */
	event		*First,		/* (NULL-terminated list of events) */
	...		);

static sim_time getCurrentTime(	/* obtain current simulation time */
	void);

static void fatalError(		/* fatal error, abort simulation */
	const char	*Msg,
	int		ErrorNo);
};


	/*********************/
	/*** _specc::event ***/
	/*********************/


class _specc::event		/* class for event types */
{
public:
bool		Notified;	/* flag whether this event has been notified */
bool		Triggered;	/* flag whether this event has been triggered */
event		*Next;		/* pointer to next notified event (or NULL) */


event(void);			/* constructor #1 */

~event(void);			/* destructor */


void Reset(void);		/* reset the list of events */
};


	/*************************/
	/*** _specc::event_ptr ***/
	/*************************/


class _specc::event_ptr		/* class for a list of events */
{
public:
event_ptr	*Next;		/* pointer to the next element (or NULL) */
event		*Event;		/* pointer to the event */


event_ptr(			/* constructor #1 */
	event	*Event);

~event_ptr(void);		/* destructor */
};


	/**************************/
	/*** _specc::event_list ***/
	/**************************/


class _specc::event_list	/* class for a list of event lists */
{
public:
event_list	*Next;		/* pointer to next event list (or NULL) */
event_ptr	*EventList;	/* pointer to the first element of this list */


event_list(			/* constructor #1 */
	event_ptr	*EventList);

~event_list(void);		/* destructor */


bool Triggers(			/* check whether any event matches this list */
	event_ptr	*EventList2);
};


	/************************/
	/*** _specc::behavior ***/
	/************************/


class _specc::behavior		/* base class for behaviors */
{
public:
behavior(void);			/* constructor #1 */

virtual ~behavior(void);	/* destructor */

virtual void main(void) = 0;	/* mandatory main method */
};


	/***********************/
	/*** _specc::channel ***/
	/***********************/


class _specc::channel		/* base class for channels */
{
public:
channel(void);			/* constructor #1 */

virtual ~channel(void);		/* destructor */
};


	/********************/
	/*** _specc::fork ***/
	/********************/


class _specc::fork		/* class for parallel behaviors */
{
public:
behavior	*Behavior;	/* the concurrent behavior */


fork(				/* constructor #1 */
	behavior	*Behavior);

~fork(void);			/* destructor */
};


	/*************************/
	/*** _specc::try_block ***/
	/*************************/


class _specc::try_block		/* class for exception-enabled behaviors */
{
public:
behavior	*Behavior;	/* the sensitive behavior (NULL if empty) */


try_block(			/* constructor #1 */
	behavior	*Behavior);

~try_block(void);		/* destructor */
};


	/*******************************/
	/*** _specc::exception_block ***/
	/*******************************/


class _specc::exception_block	/* class for exception handler behaviors */
{
public:
bool		IsTrap;		/* flag for trap (true) or interrupt (false) */
behavior	*Behavior;	/* the exception handler (NULL if empty) */
event_ptr	*EventList;	/* list of triggering events */


exception_block(		/* constructor #1 */
	bool		IsTrap,
	behavior	*Behavior,
	event		*First,
	...);

~exception_block(void);		/* destructor */
};


	/*************************/
	/*** _specc::exception ***/
	/*************************/


class _specc::exception		/* class for exception handlers (dynamic) */
{
public:
exception	*Next;		/* next exception in list (or NULL) */
bool		IsTrap;		/* flag for trap (true) or interrupt (false) */
behavior	*Behavior;	/* the exception handler (NULL if empty) */
event_ptr	*EventList;	/* list of triggering events */


exception(			/* constructor #1 */
	exception_block	*ExceptionBlock);

~exception(void);		/* destructor */


exception *FirstMatch(		/* obtain first match with this event list */
	event_ptr	*EventList2);	/* (or NULL if not found) */

exception *FirstMatch(		/* obtain first match with notified events */
	void);				/* (or NULL if not found) */
};


	/**************************/
	/*** _specc::piped_base ***/
	/**************************/


class _specc::piped_base	/* base class for 'piped' variables */
{
public:
piped_base(void);		/* constructor #1 */

virtual ~piped_base(void);	/* destructor */

virtual void update(void) = 0;	/* mandatory update method */
};


	/**********************/
	/*** piped template ***/
	/**********************/


template<class T, unsigned int d>
class piped :			/* template class for 'piped' variables */
	public _specc::piped_base	/* is based on piped_base */
{
public:
T		Value[d+1];	/* static, FIFO-type storage */


inline piped<T,d>(void);	/* constructor */

inline ~piped(void);		/* destructor */


inline operator T() const;	/* conversion operator (read access) */


inline piped<T,d> &operator=(	/* assignment operator #1 */
	T		Var);

inline piped<T,d> &operator=(	/* assignment operator #1b */
	const piped<T,d> &PipedVar);

inline piped<T,d> &operator+=(	/* assignment operator #2 */
	T		Var);

inline piped<T,d> &operator-=(	/* assignment operator #3 */
	T		Var);

inline piped<T,d> &operator*=(	/* assignment operator #4 */
	T		Var);

inline piped<T,d> &operator/=(	/* assignment operator #5 */
	T		Var);

inline piped<T,d> &operator%=(	/* assignment operator #6 */
	T		Var);

inline piped<T,d> &operator^=(	/* assignment operator #7 */
	T		Var);

inline piped<T,d> &operator&=(	/* assignment operator #8 */
	T		Var);

inline piped<T,d> &operator|=(	/* assignment operator #9 */
	T		Var);

inline piped<T,d> &operator<<=(	/* assignment operator #10 */
	T		Var);

inline piped<T,d> &operator>>=(	/* assignment operator #11 */
	T		Var);


inline piped<T,d> &operator++();/* increment operator #1 (pre-) */

inline piped<T,d> operator++(	/* increment operator #2 (post-) */
	int);

inline piped<T,d> &operator--();/* decrement operator #1 (pre-) */

inline piped<T,d> operator--(	/* decrement operator #2 (post-) */
	int);


void update(void);		/* shift the data in the pipeline */
};


/*** exported variables *************************************************/


	/* (none) */


/************************************************************************/
/*** prototype definitions for exported functions		      ***/
/************************************************************************/


	/* (none) */


/*** "implementation" ***************************************************/


#include "piped.h"	/* insert 'piped' template implementation */


#endif /* __SPECC_H */

/* EOF specc.h */
