/************************************************************************/
/* Win32Thread.cc: SpecC thread implementation based on Windows Threads	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 12/06/01 */
/************************************************************************/

/* last update: 12/28/01 */

/* modifications: (most recent first)
 *
 * 12/28/01 RD	removed obsolete to-do markers
 * 12/13/01 RD	added alternative to SignalObjectAndWait for Windows 95,98,ME
 * 12/11/01 RD	fixed a race condition between parent and new born child
 * 12/06/01 RD	initial version (based on PosixThread.cc)
 */

#include "specc.h"
#include "thread.h"
#include "Win32Thread.h"

#include <assert.h>


/*** constants and macros ***********************************************/


//#define SIM_W32THREAD_DEBUG	/* enable/disable debugging */

#ifdef SIM_W32THREAD_DEBUG
#include <stdio.h>
#endif /* SIM_W32THREAD_DEBUG */

#ifdef WIN_NT
#define SIM_WIN32_COMPLETE_API	/* Windows NT, 2000, XP */
#endif /* WIN_NT */


/*** internal type declarations *****************************************/


	/* (none) */


/*** internal function prototypes ***************************************/


static DWORD WThreadWrapper(	/* wrapper around the actual task */
	void		*Arg);


/*** internal variables *************************************************/


static _specc::thread_base	*RunningThread;	/* the running thread */


/*** exported variables *************************************************/


	/* none */


/************************************************************************/
/*** implementation of internal functions			      ***/
/************************************************************************/


static DWORD WThreadWrapper(	/* wrapper around the actual task */
	void		*Arg)
{
_specc::thread_base	*Myself;

assert(Arg != NULL);
Myself = (_specc::thread_base*) Arg;

#ifdef SIM_W32THREAD_DEBUG
puts("SIM_W32THREAD_DEBUG: Thread born, initializing...");
#endif /* SIM_W32THREAD_DEBUG */

#ifdef SIM_W32THREAD_DEBUG
puts("SIM_W32THREAD_DEBUG: New thread signalling the creator...");
#endif /* SIM_W32THREAD_DEBUG */

assert(RunningThread == Myself);	/* it's my initial turn now */
assert(Myself->ThreadCreator != NULL);	/* this is my creator */

Myself->ThreadRun(		/* transfer control back to the creator  */
	Myself->ThreadCreator);	/* (the scheduler will wake me up later) */

Myself->ThreadCreator = NULL;	/* my creator probably has died by now */

#ifdef SIM_W32THREAD_DEBUG
puts("SIM_W32THREAD_DEBUG: New thread starts its work...");
#endif /* SIM_W32THREAD_DEBUG */

Myself->ThreadFunction(Myself->ThreadArg);	/* perform the work */

#ifdef SIM_W32THREAD_DEBUG
puts("SIM_W32THREAD_DEBUG: Thread is done, dying...");
#endif /* SIM_W32THREAD_DEBUG */

return(0);	/* dummy result */

} /* end of WThreadWrapper */


/*** class implementations **********************************************/


	/***************************/
	/*** _specc::thread_base ***/
	/***************************/


_specc::thread_base::thread_base(	/* constructor #1 */
	thread_base	*Creator)
{

_specc::thread_base::ThreadFunction	= NULL;
_specc::thread_base::ThreadArg		= NULL;
_specc::thread_base::ThreadCreator	= Creator;
_specc::thread_base::ThreadHandle	= NULL;
_specc::thread_base::ThreadEvent	= NULL;

} /* end of _specc::thread_base::thread_base */


_specc::thread_base::~thread_base(void)	/* destructor */
{

/* nothing to do */

} /* end of _specc::thread_base::~thread_base */


void _specc::thread_base::ThreadStart(	/* initialize thread usage */
	void)
{

#ifdef SIM_W32THREAD_DEBUG
puts("SIM_W32THREAD_DEBUG: Thread usage starting...");
#endif /* SIM_W32THREAD_DEBUG */

RunningThread = NULL;	/* initialization */

} /* end of _specc::thread_base::ThreadStart */


void _specc::thread_base::ThreadEnd(	/* clean up after thread usage */
	void)
{

#ifdef SIM_W32THREAD_DEBUG
puts("SIM_W32THREAD_DEBUG: Thread usage ending...");
#endif /* SIM_W32THREAD_DEBUG */

assert(RunningThread == NULL);	/* clean exit */

} /* end of _specc::thread_base::ThreadEnd */


void _specc::thread_base::ThreadCreate(	/* create this new thread */
	thread_fct	Function,	/* (NULL for root thread) */
	thread_arg	Arg)		/* (NULL for root thread) */
{
thread_base	*Creator;
DWORD		NewThreadID = 0;

#ifdef SIM_W32THREAD_DEBUG
puts("SIM_W32THREAD_DEBUG: Creating a new thread...");
#endif /* SIM_W32THREAD_DEBUG */

assert(ThreadFunction == NULL);	/* must be unused */
assert(ThreadArg == NULL);
ThreadFunction	= Function;	/* store the function */
ThreadArg	= Arg;		/* and the argument   */

if (!(ThreadEvent = CreateEvent(NULL, FALSE, FALSE, NULL)))
   { fatalError("Cannot create event for child thread", GetLastError());
    } /* fi */

if (Function)			/* create new child thread */
   { Creator = RunningThread;
     assert(Creator != NULL);
     RunningThread = this;	/* temporarily switch to child */
     if (!(ThreadHandle = CreateThread(NULL, 0, &WThreadWrapper,
				(void*)this, 0, &NewThreadID)))
	{ fatalError("Cannot create new thread", GetLastError());
	 } /* fi */
#ifdef SIM_W32THREAD_DEBUG
     puts("SIM_W32THREAD_DEBUG: Get new thread in starting position...");
#endif /* SIM_W32THREAD_DEBUG */
     if (WaitForSingleObject(Creator->ThreadEvent, INFINITE) == WAIT_FAILED)
	{ fatalError("Cannot wait on child to get started", GetLastError());
	 } /* fi */
     assert(RunningThread == Creator);
    } /* fi */
else				/* create root thread */
   { assert(RunningThread == NULL);	/* not initialized yet */
     /* we are already running */
     RunningThread = this;
    } /* esle */

#ifdef SIM_W32THREAD_DEBUG
printf("SIM_W32THREAD_DEBUG: Thread %lu created,\n",
				(unsigned long)ThreadHandle);
printf("SIM_W32THREAD_DEBUG: new thread ID = %lu.\n",
				NewThreadID);
#endif /* SIM_W32THREAD_DEBUG */

} /* end of _specc::thread_base::ThreadCreate */


void _specc::thread_base::ThreadDelete(	/* delete this thread */
	void)
{

if (! CloseHandle(ThreadEvent))
   { fatalError("Cannot close the event handle", GetLastError());
    } /* fi */
ThreadEvent = NULL;

if (ThreadFunction)
   { assert(RunningThread != this);
     if (! CloseHandle(ThreadHandle))
	{ fatalError("Cannot close the thread handle", GetLastError());
	 } /* fi */
     ThreadHandle = NULL;
     ThreadFunction = NULL;
     ThreadArg = NULL;
    } /* fi */
else	/* delete the root thread */
   { assert(RunningThread == this);
     ThreadArg = NULL;
     RunningThread = NULL;
    } /* esle */

} /* end of _specc::thread_base::ThreadDelete */


void _specc::thread_base::ThreadJoin(	/* wait for a thread to complete */
	thread_base	*Thread)
{

assert(RunningThread == this);

#ifdef SIM_W32THREAD_DEBUG
puts("SIM_W32THREAD_DEBUG: Start joining...");
#endif /* SIM_W32THREAD_DEBUG */

RunningThread = Thread;	/* temporary only */

#ifdef SIM_WIN32_COMPLETE_API

if (SignalObjectAndWait(Thread->ThreadEvent, Thread->ThreadHandle,
			INFINITE, FALSE) != WAIT_OBJECT_0)
   { fatalError("Cannot signal thread to terminate", GetLastError());
    } /* fi */

#else /* !SIM_WIN32_COMPLETE_API */

if (! SetEvent(Thread->ThreadEvent))
   { fatalError("Cannot signal thread to terminate", GetLastError());
    } /* fi */
if (WaitForSingleObject(Thread->ThreadHandle, INFINITE) != WAIT_OBJECT_0)
   { fatalError("Cannot wait for thread to terminate", GetLastError());
    } /* fi */

#endif /* SIM_WIN32_COMPLETE_API */

RunningThread = this;	/* restore */

#ifdef SIM_W32THREAD_DEBUG
puts("SIM_W32THREAD_DEBUG: Joining done...");
#endif /* SIM_W32THREAD_DEBUG */

} /* end of _specc::thread_base::ThreadJoin */


void _specc::thread_base::ThreadAbort(	/* abort this thread */
	void)
{

#ifdef SIM_W32THREAD_DEBUG
puts("SIM_W32THREAD_DEBUG: Start thread termination...");
#endif /* SIM_W32THREAD_DEBUG */

assert(RunningThread != this);	/* must not commit suicide! */

if (! TerminateThread(ThreadHandle, 0))
   { fatalError("Cannot terminate thread", GetLastError());
    } /* fi */

#ifdef SIM_W32THREAD_DEBUG
puts("SIM_W32THREAD_DEBUG: Termination successful.");
#endif /* SIM_W32THREAD_DEBUG */

} /* end of _specc::thread_base::ThreadAbort */


void _specc::thread_base::ThreadRun(	/* transfer control to another thread */
	thread_base	*Thread)
{

assert(Thread != NULL);
assert(RunningThread == this);

#ifdef SIM_W32THREAD_DEBUG
printf("SIM_W32THREAD_DEBUG: Transferring control from thread %lu to %lu...\n",
			(unsigned long)ThreadHandle,
			(unsigned long)Thread->ThreadHandle);
#endif /* SIM_W32THREAD_DEBUG */

if (Thread == this)	/* continue to run myself? */
   { return;		/* nothing to change */
    } /* fi */

RunningThread = Thread;

#ifdef SIM_WIN32_COMPLETE_API

if (SignalObjectAndWait(Thread->ThreadEvent, ThreadEvent,
			INFINITE, FALSE) != WAIT_OBJECT_0)
   { fatalError("Cannot block myself", GetLastError());
    } /* fi */

#else /* !SIM_WIN32_COMPLETE_API */

if (! SetEvent(Thread->ThreadEvent))
   { fatalError("Cannot signal thread to resume", GetLastError());
    } /* fi */
if (WaitForSingleObject(ThreadEvent, INFINITE) != WAIT_OBJECT_0)
   { fatalError("Cannot block myself", GetLastError());
    } /* fi */

#endif /* SIM_WIN32_COMPLETE_API */

assert(RunningThread == this);

#ifdef SIM_W32THREAD_DEBUG
printf("SIM_W32THREAD_DEBUG: Thread %lu continuing...\n",
			(unsigned long)ThreadHandle);
#endif /* SIM_W32THREAD_DEBUG */

} /* end of _specc::thread_base::ThreadRun */


/************************************************************************/
/*** implementation of exported functions			      ***/
/************************************************************************/


	/* none */


/* EOF Win32Thread.cc */
