/************************************************************************/
/* Design.cc: SpecC Internal Representation, Design Class		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/24/97 */
/************************************************************************/

/* last update: 11/21/01 */

/* modifications: (most recent first)
 *
 * 11/21/01 RD	took out default arguments from function definitions
 * 11/09/01 RD	added line wrapping in code generation
 * 11/08/01 RD	switched code generation to use GL_IO layer
 * 10/04/01 RD	improved indentation of generated code
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/26/01 RD	bug fix for source code 'import'
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	eliminated support for automatic IP wrapping (scc -ip option)
 * 05/04/01 RD	added support for import of SpecC source (.sc) files
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/16/01 RD	added inclusion of SIR_LLONG_HEADER_NAME into C++ header
 * 04/06/01 RD	added support for non-native 'long long' type
 * 04/06/01 RD	started this header (last update was 04/08/99)
 */

#include "IntRep/Design.h"
#include "IntRep/Extern.h"

#include <stdlib.h>
#include <string.h>
#include <time.h>


/*** constants and macros ***********************************************/


	/* options to output 'long long' types */

#ifdef HAVE_LLONG
//#define LONGLONG_USE_NATIVE_OUTPUT_ROUTINES
#define LONGLONG_USE_SPECC_OUTPUT_ROUTINES
#else /* HAVE_LLONG */
#define LONGLONG_USE_SPECC_OUTPUT_ROUTINES
#endif /* HAVE_LLONG */


/*** internal type declarations *****************************************/


	/* (none) */


/*** prototypes of internal functions ***********************************/


static BOOL NeedForRangeChecker(	/* checks if range checker is needed */
	sir_design	*Design);

static ERROR DeclareRangeChecker(	/* generate decl. of range checker */
	GL_HANDLE	*IO);

static ERROR DefineRangeChecker(	/* generate def. of range checker */
	GL_HANDLE	*IO);


/*** exported variables *************************************************/


	/* none */


/*** internal variables *************************************************/


	/* none */


/*** class implementations **********************************************/


	/******************/
	/*** SIR_Design ***/
	/******************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Design::SIR_Design(			/* constructor #1 */
	const char	*Name)
{

assert(Name != NULL);
assert(*Name != 0);

SIR_Design::Name		= Name;
SIR_Design::FileList		= new SIR_FileList();
SIR_Design::ImportList		= new SIR_ImportList();
SIR_Design::Types		= new SIR_Types(this);
SIR_Design::Symbols		= new SIR_Symbols(NULL, SIR_SCOPE_GLOBAL);
SIR_Design::Notes		= NULL;

} /* end of SIR_Design::SIR_Design #1 */


SIR_Design::SIR_Design(			/* constructor #3 (duplicator) */
	sir_design	*Original)
{

SIR_Design::Name		= Original->Name;
SIR_Design::FileList		= new SIR_FileList(Original->FileList);
SIR_Design::ImportList		= new SIR_ImportList(Original->ImportList);
SIR_Design::Types		= new SIR_Types(Original->Types, this);
SIR_Design::Symbols		= new SIR_Symbols(Original->Symbols);
SIR_Design::Notes		= Original->Notes ?
					new SIR_Notes(Original->Notes) :
					NULL;

} /* end of SIR_Design::SIR_Design #3 */


SIR_Design::~SIR_Design(void)		/* destructor */
{

delete FileList;
delete ImportList;
delete Types;
delete Symbols;
delete Notes;

} /* end of SIR_Design::~SIR_Design */


ERROR SIR_Design::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

assert(MemberFct != 0);

/* this is not a node */

/* there are no nodes in FileList */
/* there are no nodes in ImportList */
/* there are no nodes in Types */
if (Symbols)
   { if ((SIR_Error = Symbols->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::DFS_ForAllNodes */


ERROR SIR_Design::DFS_ForAllSymbols(	/* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{

assert(MemberFct != 0);

/* this is not a symbol */

/* there are no symbols in FileList */
/* there are no symbols in ImportList */
/* there are no symbols in Types */
if (Symbols)
   { if ((SIR_Error = Symbols->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no symbols in Notes */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::DFS_ForAllSymbols */


ERROR SIR_Design::DFS_ForAllUserTypes(	/* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{

assert(MemberFct != 0);

/* this is not a usertype */

/* there are no usertypes in FileList */
/* there are no usertypes in ImportList */
/* there are no usertypes in Types (only links) */
if (Symbols)
   { if ((SIR_Error = Symbols->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no usertypes in Notes */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::DFS_ForAllUserTypes */


ERROR SIR_Design::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

assert(MemberFct != 0);

/* this is not a note */

/* there are no notes in FileList */
/* there are no notes in ImportList */
/* there are no notes in Types */
if (Symbols)
   { if ((SIR_Error = Symbols->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::DFS_ForAllNotes */


ERROR SIR_Design::DFS_ForAllStatements(	/* iterator over all statements */
	sir_stmnt_mptr	MemberFct,	/* (depth first) */
	sir_stmnt_marg	MemberFctArg)
{
sir_symbol	*Symbol1,
		*Symbol2;

Symbol1 = Symbols->First();	/* find all global functions */
while(Symbol1)
   { if (Symbol1->FctBody)
	{ if ((SIR_Error = Symbol1->FctBody->DFS_ForAllStatements(
						MemberFct, MemberFctArg)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
     if (  (Symbol1->IsBehaviorDefinition())
	 ||(Symbol1->IsChannelDefinition()))
	{ assert(Symbol1->ClassScope != NULL);
	  Symbol2 = Symbol1->ClassScope->First();	/* find all methods */
	  while(Symbol2)
	     { if (Symbol2->FctBody)
		  { if ((SIR_Error = Symbol2->FctBody->DFS_ForAllStatements(
						MemberFct, MemberFctArg)))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       Symbol2 = Symbol2->Succ();
	      } /* elihw */
	 } /* fi */
     Symbol1 = Symbol1->Succ();
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::DFS_ForAllStatements */


ERROR SIR_Design::Annotate(	/* attach a (global) note to the design */
	sir_note	*NewNote)	/* (consumes NewNote) */
{

assert(NewNote != NULL);

NewNote->Symbol = NULL;		/* global Note (no back pointers) */
NewNote->UserType = NULL;
NewNote->Label = NULL;

if (Notes)
   { if (! Notes->Insert(NewNote))
	{ delete NewNote;
	  return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { Notes = new SIR_Notes(NewNote);
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::Annotate */


ERROR SIR_Design::Integrate(		/* integrates an imported design */
	sir_design	*Imported,	/* (consumes the imported design) */
	const char	*ImportFile /* = NULL */, /* (default: no warnings) */
	const char	*SourceFile /* = NULL */,
	unsigned int	LineNumber /* = 0 */)
{
sir_import	*NewImport,
		*DoubleImport;
sir_symbol	*Symbol,
		*SymbolSucc;
sir_usertype	*UserType,
		*UserTypeSucc;

assert(Imported != NULL);

	/* step 0: check if we already know that design */

if (ImportList->Find(Imported->Name.chars()))
   { delete Imported;
     return(SIR_ERROR_NO_ERROR);	/* already imported before */
    } /* fi */

	/* step 1: create import entry and pointers to it */

NewImport = ImportList->Insert(Imported->Name.chars());
SIR_Error = Imported->DFS_ForAllNotes(
			&SIR_Note::SetImported, NewImport);
assert(SIR_Error == SIR_ERROR_NO_ERROR);
SIR_Error = Imported->DFS_ForAllSymbols(
			&SIR_Symbol::SetImported, NewImport);
assert(SIR_Error == SIR_ERROR_NO_ERROR);
SIR_Error = Imported->DFS_ForAllUserTypes(
			&SIR_UserType::SetImported, NewImport);
assert(SIR_Error == SIR_ERROR_NO_ERROR);

	/* step 2: integrate file list */

if ((SIR_Error = FileList->Integrate(		/* merge file lists */
			Imported->FileList)))	/* (aliasing doubles) */
   { delete Imported;
     return(SIR_Error);
    } /* fi */
if ((SIR_Error = Imported->DFS_ForAllNodes(
			&SIR_Node::UnaliasFilePointer, NULL)))
   { delete Imported;
     return(SIR_Error);
    } /* fi */

	/* step 3: integrate import list and delete double imports */

if ((SIR_Error = ImportList->Integrate(		/* merge import lists */
			Imported->ImportList)))	/* (marking doubles    */
   { delete Imported;				/* with alias pointer) */
     return(SIR_Error);
    } /* fi */

if (ImportFile)	/* issue helpful warnings */
   { assert(SourceFile != NULL);
     DoubleImport = Imported->ImportList->First();
     while(DoubleImport)
	{ GL_PrintWarningFmt(GL_WARN_PEDANTIC,
		"In file \"%s\" in line %u," GL_WARN_MSG_NEWLINE
		"when importing file \"%s\"," GL_WARN_MSG_NEWLINE
		"nested design '%s' is already imported",
		SourceFile, LineNumber,
		ImportFile,
		DoubleImport->ImportName.chars());
	  DoubleImport = DoubleImport->Succ();
	 } /* elihw */
    } /* fi */

if((SIR_Error = Imported->DFS_ForAllNotes(	/* remove double notes */
			&SIR_Note::RemoveDoubleImport, NULL)))
   { delete Imported;
     return(SIR_Error);
    } /* fi */

	/* mark double symbols and usertypes with alias */

Symbol = Imported->Symbols->First();	/* move notes still left at global */
while(Symbol)				/* imported symbols to our symbols */
   { assert(Symbol->Imported != NULL);	/* we just set this */
     if (Symbol->Imported->Alias)	/* is double-imported? */
	{ if (!(Symbol->Alias =		/* find its alias */
		Symbols->FindLocally(Symbol->Name.chars())))
	     { delete Imported;
	       return(SIR_ERROR_DOUBLE_IMP_SYMBOL_NOT_FOUND);
	      } /* fi */
	  Symbol->SetAlias(Symbol->Alias);	/* set alias recursively */
	  if ((SIR_Error = SIR_Notes::Merge(&Symbol->Alias->Notes,
						&Symbol->Notes)))
	     { delete Imported;
	       return(SIR_Error);
	      } /* fi */
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */
UserType = Imported->Symbols->UserTypes->First();
while(UserType)				/* the same for global usertypes... */
   { assert(UserType->Imported != NULL);
     if (UserType->Imported->Alias)
	{ if (UserType->Name)
	     { if (!(UserType->Alias =	/* find named user-type */
		Symbols->UserTypes->FindLocally(UserType->Name->chars())))
		  { delete Imported;
		    return(SIR_ERROR_DOUBLE_IMP_USRTP_NOT_FOUND);
		   } /* fi */
	      } /* fi */
	  else
	     { if (! UserType->TypeDef)	/* unnamed usertype without typedef */
		  { UserType = UserType->Succ();	/* skip it! */
		    continue;
		   } /* fi */
	       if (!(UserType->Alias =	/* find unnamed usertype via typedef */
		Symbols->UserTypes->FindLocallyViaTypeDef(
					UserType->TypeDef->Name.chars())))
		  { delete Imported;
		    return(SIR_ERROR_DOUBLE_IMP_USRTP_NOT_FOUND_2);
		   } /* fi */
	      } /* esle */
	  UserType->SetAlias(UserType->Alias);	/* set alias recursively */
	  if ((SIR_Error = SIR_Notes::Merge(&UserType->Alias->Notes,
						&UserType->Notes)))
	     { delete Imported;
	       return(SIR_Error);
	      } /* fi */
	 } /* fi */
     UserType = UserType->Succ();
    } /* elihw */

	/* unalias everything else */

Imported->Types->UnAlias();		/* unalias imported types */
Symbol = Imported->Symbols->First();
while(Symbol)				/* unalias imported, non-dbl. symbols */
   { assert(Symbol->Imported != NULL);
     if (! Symbol->Imported->Alias)
	{ Symbol->UnAlias(Symbols);
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */
UserType = Imported->Symbols->UserTypes->First();
while(UserType)				/* unalias imported, non-dbl. usrtps. */
   { assert(UserType->Imported != NULL);
     if (! UserType->Imported->Alias)
	{ UserType->UnAlias(Symbols);
	 } /* fi */
     UserType = UserType->Succ();
    } /* elihw */

	/* delete double-imported symbols, double-imported usertypes	*/
	/* (and their types) and double-imported local types		*/

Symbol = Imported->Symbols->First();
while(Symbol)
   { SymbolSucc = Symbol->Succ();
     if (Symbol->Imported->Alias)
	{ if ((SIR_Error = Symbol->DFS_ForAllUserTypes(
				&SIR_UserType::DeleteYourType, NULL)))
	     { delete Imported;
	       return(SIR_Error);
	      } /* fi */
	  Symbol->Remove();
	  delete Symbol;
	 } /* fi */
     Symbol = SymbolSucc;
    } /* elihw */
UserType = Imported->Symbols->UserTypes->First();
while(UserType)
   { UserTypeSucc = UserType->Succ();
     if (UserType->Imported->Alias)
	{ if ((SIR_Error = UserType->DFS_ForAllUserTypes(
				&SIR_UserType::DeleteYourType, NULL)))
	     { delete Imported;
	       return(SIR_Error);
	      } /* fi */
	  UserType->Remove();
	  delete UserType;
	 } /* fi */
     UserType = UserTypeSucc;
    } /* elihw */

	/* step 4: integrate user-defined types */

if ((SIR_Error = Symbols->UserTypes->Integrate(	/* merge top-level UserTypes */
			Imported->Symbols->UserTypes)))
   { delete Imported;
     return(SIR_Error);
    } /* fi */
/* we postpone pointer-unaliasing after type integration */

	/* step 5: integrate types */

if ((SIR_Error = Types->Integrate(		/* merge type tables */
			Imported->Types)))
   { delete Imported;
     return(SIR_Error);
    } /* fi */
Symbols->UserTypes->UnAlias(Symbols);	/* merged usertypes need unaliasing */
Types->UnAlias();			/* merged types need unaliasing */
Imported->Symbols->UnAlias(Symbols);	/* to-be-integrated symbols too */

	/* step 6: integrate symbols */

if ((SIR_Error = Symbols->Integrate(		/* merge top-level symbols */
			Imported->Symbols)))
   { delete Imported;
     return(SIR_Error);
    } /* fi */
Types->UnAlias();		/* unalias the merged design */
Symbols->UnAlias(Symbols);
if ((SIR_Error = Symbols->CheckClassContainment()))
   { delete Imported;
     return(SIR_Error);
    } /* fi */

	/* step 7: integrate global notes */

if ((SIR_Error = SIR_Notes::Merge(		/* merge global notes */
			&Notes, &Imported->Notes)))
   { delete Imported;
     return(SIR_Error);
    } /* fi */

	/* step 8: delete the skeleton of the imported design */

/* note: everything, that is left in the imported design, is the	*/
/*	 raw skeleton (list headers, etc.) and some meat (class		*/
/*	 bodies) which was already there in the original design;	*/
/*	 the left-over meat is marked with alias-pointers		*/
/*	 (pointing into the original design); in the merged design,	*/
/*	 all alias pointers (at symbols, usertypes, types, files,	*/
/*	 and imports) are NULL;						*/

delete Imported;

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::Integrate */


void SIR_Design::GarbageCollection(	/* garbage collector */
	void)
{

FileList->GarbageCollection(this);	/* clean up list of source files */
ImportList->GarbageCollection(this);	/* clean up list of imported files */
Types->GarbageCollection();		/* clean up type table */

} /* end of SIR_Design::GarbageCollection */


ERROR SIR_Design::WriteSC(	/* (re-) generates SpecC source code */
	GL_HANDLE	*IO,
	const char	*Filename,	/* (full path) */
	BOOL		WriteNotes /* = TRUE */,
	BOOL		WriteLines /* = TRUE */)
{
time_t		SysTime;
unsigned int	i;

assert(IO != NULL);
assert(Filename != NULL);

time(&SysTime);		/* read current time and date */

IO->PutS("//");
for(i=IO->GetColNo(); i<IO->GetLineWrap(); i++)
   { IO->PutC('/');
    } /* rof */
IO->PutNL();
IO->PutS("// SpecC source code generated by " GL_SPECC_NAME " V"
						GL_SPECC_VERSION_STRING "\n");
IO->PutS("// Design: ");
IO->PutS(Name);
IO->PutNL();
IO->PutS("// File:   ");
IO->PutS(GL_GetFileFromPath(Filename));
IO->PutNL();
IO->PutS("// Time:   ");
IO->PutS(ctime(&SysTime));
IO->PutS("//");
for(i=IO->GetColNo(); i<IO->GetLineWrap(); i++)
   { IO->PutC('/');
    } /* rof */
IO->PutNL();

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

if (! WriteNotes)
   { IO->PutNL();
     IO->PutS("// Note: Notes are suppressed in this file.\n");
    } /* fi */
if (! WriteLines)
   { IO->PutNL();
     IO->PutS("// Note: Line infos are suppressed in this file.\n");
    } /* fi */

SIR_LineInfo::InitWriteSC(WriteLines, Filename);

if (Notes)			/* write all global notes */
   { if ((SIR_Error = Notes->WriteSC(IO, WriteNotes)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

if (Symbols)			/* write all global decls. and defs. */
   { if ((SIR_Error = Symbols->WriteSC(IO, WriteNotes)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

if (IO->GetColNo() > 0)
   { IO->PutNL();
    } /* fi */
IO->PutNL();
IO->PutS("//");
for(i=IO->GetColNo(); i<IO->GetLineWrap(); i++)
   { IO->PutC('/');
    } /* rof */
IO->PutNL();
IO->PutS("// End of file ");
IO->PutS(GL_GetFileFromPath(Filename));
IO->PutNL();
IO->PutS("//");
for(i=IO->GetColNo(); i<IO->GetLineWrap(); i++)
   { IO->PutC('/');
    } /* rof */
IO->PutNL();

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);	/* successful */

} /* end of SIR_Design::WriteSC */


ERROR SIR_Design::WriteH(	/* generates C++ source code: header file */
	GL_HANDLE	*IO,
	const char	*Filename,	/* (full path) */
	BOOL		WriteNotes /* = TRUE */,
	BOOL		WriteLines /* = TRUE */)
{
time_t		SysTime;
unsigned int	i;

assert(IO != NULL);
assert(Filename != NULL);

time(&SysTime);		/* read current time and date */

IO->PutS("//");
for(i=IO->GetColNo(); i<IO->GetLineWrap(); i++)
   { IO->PutC('/');
    } /* rof */
IO->PutNL();
IO->PutS("// C++ header file generated by " GL_SPECC_NAME " V"
						GL_SPECC_VERSION_STRING "\n");
IO->PutS("// Design: ");
IO->PutS(Name);
IO->PutNL();
IO->PutS("// File:   ");
IO->PutS(GL_GetFileFromPath(Filename));
IO->PutNL();
IO->PutS("// Time:   ");
IO->PutS(ctime(&SysTime));
IO->PutS("//");
for(i=IO->GetColNo(); i<IO->GetLineWrap(); i++)
   { IO->PutC('/');
    } /* rof */
IO->PutNL();

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

if (! WriteNotes)
   { IO->PutNL();
     IO->PutS("// Note: Notes are suppressed in this file.\n");
    } /* fi */
if (! WriteLines)
   { IO->PutNL();
     IO->PutS("// Note: Line infos are suppressed in this file.\n");
    } /* fi */

IO->PutNL();
IO->PutS("#include <" SIR_SIM_HEADER_NAME ">");
IO->PutNL();
IO->PutS("#include <" SIR_LLONG_HEADER_NAME ">");
IO->PutNL();
IO->PutS("#include <" SIR_BIT_HEADER_NAME ">\n");

SIR_LineInfo::InitWriteSC(WriteLines, Filename);

if (Notes)			/* write all global notes */
   { if ((SIR_Error = Notes->WriteSC(IO, WriteNotes, TRUE)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

if (Symbols)			/* translate all global declarations */
   { if ((SIR_Error = Symbols->WriteH(IO, WriteNotes)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

if (NeedForRangeChecker(this))
   { if (Symbols->FindLocally(SIR_CXX_RANGE_NAME))
	{ GL_PrintWarning(GL_WARN_IMPORTANT,
		"Skipping generation of function '" SIR_CXX_RANGE_NAME "';"
							GL_WARN_MSG_NEWLINE
		"global symbol '" SIR_CXX_RANGE_NAME "' already exists");
	 } /* fi */
     else
	{ if ((SIR_Error = DeclareRangeChecker(IO)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* esle */
    } /* fi */

if (IO->GetColNo() > 0)
   { IO->PutNL();
    } /* fi */
IO->PutNL();
IO->PutS("//");
for(i=IO->GetColNo(); i<IO->GetLineWrap(); i++)
   { IO->PutC('/');
    } /* rof */
IO->PutNL();
IO->PutS("// End of file ");
IO->PutS(GL_GetFileFromPath(Filename));
IO->PutNL();
IO->PutS("//");
for(i=IO->GetColNo(); i<IO->GetLineWrap(); i++)
   { IO->PutC('/');
    } /* rof */
IO->PutNL();

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);	/* successful */

} /* end of SIR_Design::WriteH */


ERROR SIR_Design::WriteCC(	/* generates C++ source code: main file */
	GL_HANDLE	*IO,
	const char	*Filename,	/* (full path) */
	const char	*Headername,	/* (full path) */
	BOOL		WriteNotes /* = TRUE */,
	BOOL		WriteLines /* = TRUE */)
{
time_t		SysTime;
unsigned int	i;

assert(IO != NULL);
assert(Filename != NULL);
assert(Headername != NULL);

time(&SysTime);		/* read current time and date */

IO->PutS("//");
for(i=IO->GetColNo(); i<IO->GetLineWrap(); i++)
   { IO->PutC('/');
    } /* rof */
IO->PutNL();
IO->PutS("// C++ source file generated by " GL_SPECC_NAME " V"
						GL_SPECC_VERSION_STRING "\n");
IO->PutS("// Design: ");
IO->PutS(Name);
IO->PutNL();
IO->PutS("// File:   ");
IO->PutS(GL_GetFileFromPath(Filename));
IO->PutNL();
IO->PutS("// Time:   ");
IO->PutS(ctime(&SysTime));
IO->PutS("//");
for(i=IO->GetColNo(); i<IO->GetLineWrap(); i++)
   { IO->PutC('/');
    } /* rof */
IO->PutNL();

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

if (! WriteNotes)
   { IO->PutNL();
     IO->PutS("// Note: Notes are suppressed in this file.\n");
    } /* fi */
if (! WriteLines)
   { IO->PutNL();
     IO->PutS("// Note: Line infos are suppressed in this file.\n");
    } /* fi */

IO->PutNL();
IO->PrintF("#include \"%s\"\n", GL_GetFileFromPath(Headername));

SIR_LineInfo::InitWriteSC(WriteLines, Filename);

if (Symbols)			/* translate all global definitions */
   { if ((SIR_Error = Symbols->WriteCC(IO, WriteNotes)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

if (  (NeedForRangeChecker(this))
    &&(!(Symbols->FindLocally(SIR_CXX_RANGE_NAME))))
   { if ((SIR_Error = DefineRangeChecker(IO)))
	{ return(SIR_Error);
	 } /* fi */
    } /* esle */

if (IO->GetColNo() > 0)
   { IO->PutNL();
    } /* fi */
IO->PutNL();
IO->PutS("//");
for(i=IO->GetColNo(); i<IO->GetLineWrap(); i++)
   { IO->PutC('/');
    } /* rof */
IO->PutNL();
IO->PutS("// End of file ");
IO->PutS(GL_GetFileFromPath(Filename));
IO->PutNL();
IO->PutS("//");
for(i=IO->GetColNo(); i<IO->GetLineWrap(); i++)
   { IO->PutC('/');
    } /* rof */
IO->PutNL();

if ((SIR_Error = IO->Check()))
   { return(SIR_Error);
    } /* fi */

return(SIR_ERROR_NO_ERROR);	/* successful */

} /* end of SIR_Design::WriteCC */


/*** exported functions *************************************************/


	/* none */


/*** internal functions *************************************************/


BOOL NeedForRangeChecker(	/* checks whether range checker is needed */
	sir_design	*Design)
{

assert(Design != NULL);
assert(SIR_Error == SIR_ERROR_NO_ERROR);	/* needed for recursion */

if (SIR_ERROR_FLAG_YES == Design->DFS_ForAllStatements(
					&SIR_Statement::NeedsRangeChecker,
					(sir_stmnt_marg) NULL))
   { SIR_Error = SIR_ERROR_NO_ERROR;	/* clear the fake error */
     return(TRUE);
    } /* fi */

assert(SIR_Error == SIR_ERROR_FLAG_NO);
assert(SIR_Error == SIR_ERROR_NO_ERROR);

return(FALSE);	/* don't need it */

} /* end of NeedForRangeChecker */


ERROR DeclareRangeChecker(	/* generate declaration of range checker */
	GL_HANDLE	*IO)
{

SIR_LineInfo::WriteNewSC(IO, true);

IO->PutS("void " SIR_CXX_RANGE_NAME "("
	SIR_CXX_TIME_TYPE ", "
	SIR_CXX_TIME_TYPE ", "
	SIR_CXX_TIME_TYPE ", "
	"bool, "
	SIR_CXX_TIME_TYPE ", "
	"bool, "
	"const char *, "
	"const char *, "
	"const char *, "
	"unsigned int);");

return(SIR_ERROR_NO_ERROR);

} /* end of DeclareRangeChecker */


#ifdef LONGLONG_USE_NATIVE_OUTPUT_ROUTINES

#ifdef SOLARIS
#define FMT_LLD	"%lld"
#endif /* SOLARIS */
#ifdef SUNOS4
#define FMT_LLD	"%ld"	/* probably looses precision! */
#endif /* SUNOS4 */
#ifdef NETBSD
#define FMT_LLD	"%qd"
#endif /* NETBSD */
#ifdef LINUX
#define FMT_LLD	"%qd"
#endif /* LINUX */
#ifdef GNUWIN32
#define FMT_LLD	"%qd"
#endif /* GNUWIN32 */
#ifndef SOLARIS
#   ifndef SUNOS4
#      ifndef NETBSD
#         ifndef LINUX
#            ifndef GNUWIN32
#error "Unknown system! Don't know how to print 'long long int' values!"
#            endif /* GNUWIN32 */
#         endif /* LINUX */
#      endif /* NETBSD */
#   endif /* SUNOS4 */
#endif /* SOLARIS */

#endif /* LONGLONG_USE_NATIVE_OUTPUT_ROUTINES */


#ifdef LONGLONG_USE_SPECC_OUTPUT_ROUTINES

#define FMT_LLD	"%s"

#endif /* LONGLONG_USE_SPECC_OUTPUT_ROUTINES */


ERROR DefineRangeChecker(	/* generate definition of range checker */
	GL_HANDLE	*IO)
{

SIR_LineInfo::WriteVSPACE(IO);
SIR_LineInfo::WriteNewSC(IO, true);

IO->PutS("void " SIR_CXX_RANGE_NAME "(");
IO->TabStepUp();
SIR_LineInfo::WriteNL(IO);
IO->PutS(SIR_CXX_TIME_TYPE " Time1, ");
SIR_LineInfo::WriteNL(IO);
IO->PutS(SIR_CXX_TIME_TYPE " Time2, ");
SIR_LineInfo::WriteNL(IO);
IO->PutS(SIR_CXX_TIME_TYPE " Min, ");
SIR_LineInfo::WriteNL(IO);
IO->PutS("bool CheckMin, ");
SIR_LineInfo::WriteNL(IO);
IO->PutS(SIR_CXX_TIME_TYPE " Max, ");
SIR_LineInfo::WriteNL(IO);
IO->PutS("bool CheckMax, ");
SIR_LineInfo::WriteNL(IO);
IO->PutS("const char *Label1, ");
SIR_LineInfo::WriteNL(IO);
IO->PutS("const char *Label2, ");
SIR_LineInfo::WriteNL(IO);
IO->PutS("const char *File, ");
SIR_LineInfo::WriteNL(IO);
IO->PutS("unsigned int Line)");
SIR_LineInfo::WriteNL(IO, true, false); // IO->PutC('{');
SIR_LineInfo::WriteNL(IO);

#ifdef LONGLONG_USE_SPECC_OUTPUT_ROUTINES
IO->PutS("char BuffMin[40],BuffMax[40],BuffT2[40],BuffT1[40],BuffT[40];");
SIR_LineInfo::WriteNL(IO);
#endif /* LONGLONG_USE_SPECC_OUTPUT_ROUTINES */

IO->PutS("if (((CheckMin)&&(Time2-Time1 < Min))"
	" || ((CheckMax)&&(Time2-Time1 > Max)))");
IO->TabStepUp();
SIR_LineInfo::WriteNL(IO, true, false); // IO->PutC('{');
IO->PutS(SIR_SIM_ABORT "(");
IO->TabStepUp();
SIR_LineInfo::WriteNL(IO);
IO->PutS("\"ERROR:\\tRange check failed in line %d\\n\"");
SIR_LineInfo::WriteNL(IO);
IO->PutS("\"\\tin file \\\"%s\\\":\\n\"");
SIR_LineInfo::WriteNL(IO);
IO->PutS("\"\\trange(%s; %s; " FMT_LLD "; " FMT_LLD ") is "
		FMT_LLD " - " FMT_LLD " = " FMT_LLD ".\\n\"");
SIR_LineInfo::WriteNL(IO);
IO->PutS("\"Simulation aborted.\\n\",");
SIR_LineInfo::WriteNL(IO);
IO->PutS("Line, File,");
SIR_LineInfo::WriteNL(IO);

#ifdef LONGLONG_USE_SPECC_OUTPUT_ROUTINES

				/* 'time' is 'unsigned long long' */
IO->PutS("Label1, Label2, ull2str(10,&BuffMin[39],Min),"
			" ull2str(10,&BuffMax[39],Max),");
SIR_LineInfo::WriteNL(IO);
IO->PutS("ull2str(10,&BuffT2[39],Time2), ull2str(10,&BuffT1[39],Time1),"
			" ull2str(10,&BuffT[39],Time2 - Time1));");

#else /* not LONGLONG_USE_SPECC_OUTPUT_ROUTINES */

IO->PutS("Label1, Label2, Min, Max,");
SIR_LineInfo::WriteNL(IO);
IO->PutS("Time2, Time1, Time2 - Time1);");

#endif /* LONGLONG_USE_SPECC_OUTPUT_ROUTINES */

IO->TabStepDown();
SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
IO->TabStepDown();
SIR_LineInfo::WriteNL(IO, false, true); // IO->PutC('}');
IO->TabStepDown();

return(SIR_ERROR_NO_ERROR);

} /* end of DefineRangeChecker */


#undef FMT_LLD	/* needed only locally */


/* EOF Design.cc */
