/************************************************************************/
/* IntRep/List2.h: SpecC Internal Representation, List Implementation	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/24/97 */
/************************************************************************/

/* last update: 11/21/01 */

/* modifications: (most recent first)
 *
 * 11/21/01 RD	took out default arguments from function definitions
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	introduced this header (last change was 04/12/99)
 */

/* NOTE: This file contains the _implementation_ of the list template.	*/
/*       Although it is a header file, it is more like a ".c" file!	*/


#ifndef INTREP_LIST2_H
#define INTREP_LIST2_H


#include "IntRep/Error.h"

#include <assert.h>


/*** internal type declarations *****************************************/


	/* (none) */


/*** container template class implementation ****************************/
/*** (general purpose double-linked list)    ****************************/


	/****************/
	/*** SIR_List ***/
	/****************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


template <class item_type>
SIR_List<item_type>::SIR_List(		/* constructor (#1) */
	item_type	*FirstElement /* = NULL */)
{

FirstItem	= FirstElement;
LastItem	= FirstElement;
CurrItem	= FirstElement;
if (FirstElement)
   { NumberItems = 1;
     FirstElement->SuccItem = NULL;
     FirstElement->PredItem = NULL;
     FirstElement->ListHead = this;
    } /* fi */
else
   { NumberItems = 0;
    } /* fi */

} /* end of SIR_List<item_type>::SIR_List #1 */


template <class item_type>
SIR_List<item_type>::~SIR_List(void)	/* destructor */
{
item_type	*Curr,
		*Succ;

Curr = FirstItem;
while(Curr)
   { Succ = Curr->SuccItem;
     delete Curr;
     Curr = Succ;
    } /* elihw */

} /* end of SIR_List<item_type>::~SIR_List */


template <class item_type>
inline bool SIR_List<item_type>::Empty(void)	/* test for empty list? */
{

return(NumberItems == 0);

} /* end of SIR_List<item_type>::Empty */


template <class item_type>
inline unsigned int
	SIR_List<item_type>::NumElements(void)	/* number of elements */
{

return(NumberItems);

} /* end of SIR_List<item_type>::NumElements */


template <class item_type>
inline item_type
	*SIR_List<item_type>::First(void)	/* first element (or NULL) */
{

return(CurrItem = FirstItem);

} /* end of SIR_List<item_type>::First */


template <class item_type>
inline item_type
	*SIR_List<item_type>::Last(void)	/* last element (or NULL) */
{

return(CurrItem = LastItem);

} /* end of SIR_List<item_type>::Last */


template <class item_type>
item_type *SIR_List<item_type>::Previous(void)	/* previous element (or NULL) */
{

if (CurrItem)
   { return(CurrItem = CurrItem->PredItem);
    } /* fi */

return(NULL);

} /* end of SIR_List<item_type>::Previous */


template <class item_type>
inline item_type
	*SIR_List<item_type>::Curr(void)	/* current element (or NULL) */
{

return(CurrItem);

} /* end of SIR_List<item_type>::Curr */


template <class item_type>
item_type *SIR_List<item_type>::Next(void)	/* next element (or NULL) */
{

if (CurrItem)
   { return(CurrItem = CurrItem->SuccItem);
    } /* fi */

return(NULL);

} /* end of SIR_List<item_type>::Next */


template <class item_type>
item_type *SIR_List<item_type>::Prepend(	/* prepend one element */
	item_type	*Elem)
{

assert(Elem != NULL);

Elem->SuccItem = FirstItem;
Elem->PredItem = NULL;
Elem->ListHead = this;
if (FirstItem)
   { FirstItem->PredItem = Elem;
    } /* fi */
FirstItem = Elem;
if (LastItem == NULL)
   { LastItem = Elem;
    } /* fi */
NumberItems++;

return(CurrItem = Elem);

} /* end of SIR_List<item_type>::Prepend */


template <class item_type>
item_type *SIR_List<item_type>::Append(		/* append one element */
	item_type	*Elem)
{

assert(Elem != NULL);

Elem->SuccItem = NULL;
Elem->PredItem = LastItem;
Elem->ListHead = this;
if (LastItem)
   { LastItem->SuccItem = Elem;
    } /* fi */
if (FirstItem == NULL)
   { FirstItem = Elem;
    } /* fi */
LastItem = Elem;
NumberItems++;

return(CurrItem = Elem);

} /* end of SIR_List<item_type>::Append */


template <class item_type>
item_type *SIR_List<item_type>::InsertBefore(	/* insert before current (#1) */
	item_type	*Elem)
{

assert(CurrItem != NULL);
assert(Elem != NULL);

Elem->SuccItem = CurrItem;
Elem->PredItem = CurrItem->PredItem;
Elem->ListHead = this;
if (Elem->PredItem)
   { Elem->PredItem->SuccItem = Elem;
    } /* fi */
CurrItem->PredItem = Elem;
if (FirstItem == CurrItem)
   { FirstItem = Elem;
    } /* fi */
NumberItems++;

return(CurrItem = Elem);

} /* end of SIR_List<item_type>::InsertBefore (#1) */


template <class item_type>
item_type *SIR_List<item_type>::InsertBefore(	/* insert before given (#2) */
	item_type	*Elem,
	item_type	*Succ)
{

assert(Elem != NULL);
assert(Succ != NULL);
assert(Succ->ListHead == this);

Elem->SuccItem = Succ;
Elem->PredItem = Succ->PredItem;
Elem->ListHead = this;
if (Elem->PredItem)
   { Elem->PredItem->SuccItem = Elem;
    } /* fi */
Succ->PredItem = Elem;
if (FirstItem == Succ)
   { FirstItem = Elem;
    } /* fi */
NumberItems++;

return(CurrItem = Elem);

} /* end of SIR_List<item_type>::InsertBefore (#2) */


template <class item_type>
item_type *SIR_List<item_type>::InsertAfter(	/* insert after current (#1) */
	item_type	*Elem)
{

assert(CurrItem != NULL);
assert(Elem != NULL);

Elem->SuccItem = CurrItem->SuccItem;
Elem->PredItem = CurrItem;
Elem->ListHead = this;
if (Elem->SuccItem)
   { Elem->SuccItem->PredItem = Elem;
    } /* fi */
CurrItem->SuccItem = Elem;
if (LastItem == CurrItem)
   { LastItem = Elem;
    } /* fi */
NumberItems++;

return(CurrItem = Elem);

} /* end of SIR_List<item_type>::InsertAfter (#1) */


template <class item_type>
item_type *SIR_List<item_type>::InsertAfter(	/* insert after given (#2) */
	item_type	*Elem,
	item_type	*Pred)
{

assert(Elem != NULL);
assert(Pred != NULL);
assert(Pred->ListHead == this);

Elem->SuccItem = Pred->SuccItem;
Elem->PredItem = Pred;
Elem->ListHead = this;
if (Elem->SuccItem)
   { Elem->SuccItem->PredItem = Elem;
    } /* fi */
Pred->SuccItem = Elem;
if (LastItem == Pred)
   { LastItem = Elem;
    } /* fi */
NumberItems++;

return(CurrItem = Elem);

} /* end of SIR_List<item_type>::InsertAfter (#2) */


template <class item_type>
item_type *SIR_List<item_type>::Remove(		/* remove one element */
	item_type	*Elem)
{

assert(Elem != NULL);
assert(Elem->ListHead == this);

if (Elem->PredItem)
   { Elem->PredItem->SuccItem = Elem->SuccItem;
    } /* fi */
else
   { this->FirstItem = Elem->SuccItem;
    } /* esle */
if (Elem->SuccItem)
   { Elem->SuccItem->PredItem = Elem->PredItem;
    } /* fi */
else
   { this->LastItem = Elem->PredItem;
    } /* esle */
if (CurrItem == Elem)
   { CurrItem = NULL;
    } /* fi */
NumberItems--;
Elem->SuccItem = NULL;
Elem->PredItem = NULL;
Elem->ListHead = NULL;

return(Elem);

} /* end of SIR_List<item_type>::Remove */


template <class item_type>
SIR_ListElem<item_type> *SIR_List<item_type>::Remove(	/* remove one element */
	SIR_ListElem<item_type>	*Elem)
{

assert(Elem != NULL);
assert(Elem->ListHead == this);

if (Elem->PredItem)
   { Elem->PredItem->SuccItem = Elem->SuccItem;
    } /* fi */
else
   { this->FirstItem = Elem->SuccItem;
    } /* esle */
if (Elem->SuccItem)
   { Elem->SuccItem->PredItem = Elem->PredItem;
    } /* fi */
else
   { this->LastItem = Elem->PredItem;
    } /* esle */
if (CurrItem == Elem)
   { CurrItem = NULL;
    } /* fi */
NumberItems--;
Elem->SuccItem = NULL;
Elem->PredItem = NULL;
Elem->ListHead = NULL;

return(Elem);

} /* end of SIR_List<item_type>::Remove */


template <class item_type>
item_type *SIR_List<item_type>::Remove(void)	/* remove current element */
{
item_type	*RemovedElem;

assert(CurrItem != NULL);

if (CurrItem->PredItem)
   { CurrItem->PredItem->SuccItem = CurrItem->SuccItem;
    } /* fi */
else
   { this->FirstItem = CurrItem->SuccItem;
    } /* esle */
if (CurrItem->SuccItem)
   { CurrItem->SuccItem->PredItem = CurrItem->PredItem;
    } /* fi */
else
   { this->LastItem = CurrItem->PredItem;
    } /* esle */
NumberItems--;
CurrItem->SuccItem = NULL;
CurrItem->PredItem = NULL;
CurrItem->ListHead = NULL;

RemovedElem = CurrItem;
CurrItem = NULL;

return(RemovedElem);

} /* end of SIR_List<item_type>::Remove */


template <class item_type>
SIR_List<item_type> *SIR_List<item_type>::Concat(	/* concat two lists */
	SIR_List<item_type> *Appendix)
{

while(Appendix->First())
   { Append(Appendix->Remove());
    } /* elihw */

return(this);

} /* end of SIR_List<item_type>::Concat */


template <class item_type>
SIR_List<item_type> *SIR_List<item_type>::Precat(	/* precat two lists */
	SIR_List<item_type> *Prependix)
{

while(Prependix->Last())
   { Prepend(Prependix->Remove());
    } /* elihw */

return(this);

} /* end of SIR_List<item_type>::Precat */


	/********************/
	/*** SIR_ListElem ***/
	/********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


template <class item_type>
SIR_ListElem<item_type>::SIR_ListElem(void)	/* constructor #1 */
{

SuccItem = NULL;
PredItem = NULL;
ListHead = NULL;

} /* end of SIR_ListElem<item_type>::SIR_ListElem */


template <class item_type>
inline SIR_ListElem<item_type>::~SIR_ListElem(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_ListElem<item_type>::~SIR_ListElem */


template <class item_type>
inline item_type *SIR_ListElem<item_type>::Succ(void)	/* Successor */
{

return(SuccItem);

} /* end of SIR_ListElem<item_type>::Succ */


template <class item_type>
inline item_type *SIR_ListElem<item_type>::Pred(void)	/* Predecessor */
{

return(PredItem);

} /* end of SIR_ListElem<item_type>::Pred */


template <class item_type>
inline SIR_List<item_type> *SIR_ListElem<item_type>::Head(void)	/* List head */
{

return(ListHead);

} /* end of SIR_ListElem<item_type>::Head */


template <class item_type>
inline void SIR_ListElem<item_type>::Remove(void)	/* remove myself */
{

ListHead->Remove(this);

} /* end of SIR_ListElem<item_type>::Succ */


#endif /* INTREP_LIST2_H */

/* EOF IntRep/List2.h */
