// ============================================================================
// sys/bit.h: Bit Vector Library, system level
// ============================================================================
//
// 06/01/01 <gerstl>	Resolved conflict of method names with ANSI (keywords?)
// 05/24/01 <gerstl>	Initial version
//
// ----------------------------------------------------------------------------

#ifndef __BITSYS
#define __BITSYS

#include <limits.h>
#include <string.h>

#include <longlong.h>


/* Enable assertions only if -DDEBUG (but executed only if not -DNDEBUG ...) */
#ifdef DEBUG
# define _BITASSERT
#endif


/* Enable assertions in code? */
#ifdef _BITASSERT
# include <assert.h>
# define ASSERT(ex)              assert(ex)
#else
# define ASSERT(ex)
#endif


/* --- Some macros --- */

// Length of a bit vector, given boundaries
#define _BITLEN(l, r)           ((l) > (r) ? ((l)-(r)+1) : ((r)-(l)+1))

// Mapping of bit index to data chunks
#define _BITCHUNK(i)            ((int)((i) / _BITCHUNK_SIZE))

// Min / Max operations
#define _MAXLEN(a,b)	        ((a) > (b) ? (a) : (b))
#define _MINLEN(a,b)	        ((a) < (b) ? (a) : (b))

//// Mapping of arbitrary bit index to index into zero-aligned vector
#define _BITMAP(l,r,i)          ((l) > (r)? ((i) - (r)) : ((r) - (i)))

// sign[] function
#define _BITSIGN(l,r)           ((l) == (r)? 0 : ((l) > (r)? 1 : -1))

// Number of data chunks in a bit vector of given length
#define _BITCHUNKS(len)         (_BITCHUNK((len)-1)+1)

// Size of data chunk array
#define _SIZEOFBIT(len)         (_BITCHUNKS(len)*sizeof(_bit::chunk))

// Size of bit chunk
#define _BITCHUNK_SIZE          (int)((sizeof(_bit::chunk)*CHAR_BIT))

// Size of the complete bit vector class (including virtual table pointer, 
// same as sizeof(bit<len>) but at runtime)
// ## Determine base size during compile time through test program
//    instead of hard-coding size of virtual table pointer.
#define _SIZEOFBITCLASS(len)    (_SIZEOFBIT(len)+sizeof(void*))

// Masking of a single bit
#define _BITMASK(i)		(0x01 << (i))
  
// Masks for groups of bits
#define _BITMASK_MSB_SET(n)	((~(unsigned)0) << (_BITCHUNK_SIZE - (n)))
#define _BITMASK_LSB_SET(n)	((~(unsigned)0) >> (_BITCHUNK_SIZE - (n)))
#define _BITMASK_MSB_CLR(n)	((~(unsigned)0) >> (n))
#define _BITMASK_LSB_CLR(n)	((~(unsigned)0) << (n))


/* --- Constants for integral-to-bit conversions --- */

#define _BITLEN_BOOL         (1)
#define _BITLEN_CHAR         (CHAR_BIT)
#define _BITLEN_UCHAR        (CHAR_BIT)
#define _BITLEN_SHORT        (sizeof(short int)*CHAR_BIT)
#define _BITLEN_USHORT       (sizeof(short int)*CHAR_BIT)
#define _BITLEN_INT          (sizeof(int)*CHAR_BIT)
#define _BITLEN_UINT         (sizeof(int)*CHAR_BIT)
#define _BITLEN_LONG         (sizeof(long int)*CHAR_BIT)
#define _BITLEN_ULONG        (sizeof(long int)*CHAR_BIT)
#define _BITLEN_LLONG        (sizeof(LONG_LONG)*CHAR_BIT)
#define _BITLEN_ULLONG       (sizeof(LONG_LONG)*CHAR_BIT)

#define _BITUSGN_BOOL         (true)
#define _BITUSGN_CHAR         (false)
#define _BITUSGN_UCHAR        (true)
#define _BITUSGN_SHORT        (false)
#define _BITUSGN_USHORT       (true)
#define _BITUSGN_INT          (false)
#define _BITUSGN_UINT         (true)
#define _BITUSGN_LONG         (false)
#define _BITUSGN_ULONG        (true)
#define _BITUSGN_LLONG        (false)
#define _BITUSGN_ULLONG       (true)
#define _BITUSGN_DBL          (false)
#define _BITUSGN_LDBL         (false)


/* --- Macros for operation result types --- */

#define _BITLEN_UNARY(len, usgn)                                          (len)
#define _BITLEN_SLICE(len1, usgn1, l, r)                         (_BITLEN(l,r))
#define _BITLEN_CONCAT(len1, usgn1, len2, usgn2)              ((len1) + (len2))
#define _BITLEN_BINARY(len1, usgn1, len2, usgn2)           (_MAXLEN(len1,len2))

#define _BITUSGN_UNARY(len, usgn)                                        (usgn)
#define _BITUSGN_SLICE(len, usgn, l, r)                                  (usgn)
#define _BITUSGN_CONCAT(len1, usgn1, len2, usgn2)                       (usgn1)
#define _BITUSGN_BINARY(len1, usgn1, len2, usgn2) \
    ( (len1)==(len2)? ((usgn1)||(usgn2)) : (((len1)>(len2))? (usgn1):(usgn2)) )



/* --- Machine dependent stuff? --- */
/*     (at least according to draft C++ standard...) */

#if ((-1 >> 8) == -1) && ((-1l >> 8) == -1l)
#define _RSH_IS_SIGN_EXTENDING     // Is (int)-1 >> 8 == -1 ?
#endif


/* --- Forward declarations --- */

class _bitslice;


/* ------------------------------------------------------------------------ */
/* Dynamic Bit Vectors */


class _bit
{
public:
  // Type definition of bit vector chunks
  union chunk {
    signed int   sval;
    unsigned int uval;
  };

protected:
  chunk *vec;
  const int  len;
  const bool usgn;
  
public:
  // Copy constructor
  _bit(const _bit& c): len(c.len), usgn(c.usgn), isDynamic(init())
               { for (int i = 0; i < _BITCHUNKS(len); i++) vec[i] = c.vec[i]; }

  // Constructor for an empty bit vector
  _bit(int length, bool usign): len(length), usgn(usign), isDynamic(init()) { }

  // Constructor for a reference to an existing vector
  _bit(chunk* ref, int length, bool usign)
    : vec(ref), len(length), usgn(usign), isDynamic(false)                  { }
     
  // Construct from a bit slice
  _bit(const _bitslice& s, int length, bool usign)
    : len(length), usgn(usign), isDynamic(init())                   { set(s); }
  
  // Construct an operation result
  inline _bit(const _bit& op1, char type);                    // Unary
  inline _bit(const _bit& op1, char type, int sh);            // Shift
  inline _bit(const _bit& op1, char type, const _bit& op2);   // Binary
  inline _bit(const _bit& op1, const _bit& op2);              // Concatenation

#ifdef _BIT_TEMPLATE_OPERATORS
  // Operation results as a reference
  inline _bit(chunk* ref, int length, bool usign,
	      const _bit& op1, char type);                    // Unary
  inline _bit(chunk* ref, int length, bool usign,
	      const _bit& op1, char type, int sh);            // Shift
  inline _bit(chunk* ref, int length, bool usign, 
	      const _bit& op1, char type, const _bit& op2);   // Binary
  inline _bit(chunk* ref, int length, bool usign,
	      const _bit& op1, const _bit& op2);              // Concatenation
#endif  
  
  // Construct from bit string
  inline _bit(const char* str, int length = 0, bool usign = false,
	      unsigned int base = 2);

  // Integer Initializers, length=0 picks native length
  inline _bit(bool b,               int length=0, bool usign=_BITUSGN_BOOL);
  inline _bit(signed char i,        int length=0, bool usign=_BITUSGN_CHAR);
  inline _bit(unsigned char i,      int length=0, bool usign=_BITUSGN_UCHAR);
  inline _bit(short int i,          int length=0, bool usign=_BITUSGN_SHORT);
  inline _bit(unsigned short int i, int length=0, bool usign=_BITUSGN_USHORT);
  inline _bit(int i,                int length=0, bool usign=_BITUSGN_INT);
  inline _bit(unsigned int i,       int length=0, bool usign=_BITUSGN_UINT);
  inline _bit(long i,               int length=0, bool usign=_BITUSGN_LONG);
  inline _bit(unsigned long i,      int length=0, bool usign=_BITUSGN_ULONG);
  inline _bit(LONG_LONG i,          int length=0, bool usign=_BITUSGN_LLONG);
  inline _bit(UNSIGNED_LONG_LONG i, int length=0, bool usign=_BITUSGN_ULLONG);
//!LLONG!  inline _bit(_longlong_<> i,       int length=0, bool usign=_BITUSGN_LLONG);
//!LLONG!  inline _bit(_longlong_<true> i,   int length=0, bool usign=_BITUSGN_ULLONG);
  inline _bit(double d,             int length,   bool usign=_BITUSGN_DBL);
  inline _bit(long double d,        int length,   bool usign=_BITUSGN_LDBL);
  
  // Destructor: delete if dynamic
  ~_bit()                                       { if(isDynamic) delete[] vec; }

  // Slicing
  inline _bitslice operator[](int i) const;                 // [i]
  inline _bitslice operator()(int sl, int sr) const;        // [left:right]
  inline _bitslice operator()(int l, int r, int i) const;   // mapped versions
  inline _bitslice operator()(int l, int r, int sl, int sr) const;

  // Assignments
  _bit& operator= (const _bit& op)                          { return set(op); }
  _bit& operator= (const _bitslice& s)                      { return set(s);  }
  _bit& operator= (const char* s)                           { return set(s);  }
  _bit& operator= (int i)                                   { return set(i);  }
  _bit& operator= (unsigned int i)                          { return set(i);  }
  _bit& operator= (long i)                                  { return set(i);  }
  _bit& operator= (unsigned long i)                         { return set(i);  }
  _bit& operator= (LONG_LONG i)                             { return set(i);  }
  _bit& operator= (UNSIGNED_LONG_LONG i)                    { return set(i);  }
//!LLONG!  _bit& operator= (_longlong_<> i)     { return operator=((_longlong<>&)    i);  }
//!LLONG!  _bit& operator= (_longlong_<true> i) { return operator=((_longlong<true>&)i);  }
  _bit& operator= (double d)                                { return set(d);  }
  _bit& operator= (long double d)                           { return set(d);  }

  // String assignment
  _bit& set(const char* str, bool usign = false, unsigned int base = 0);
  
  // Bit operators
  _bit& operator^= (const _bit& op)                { return xorOp(*this, op); }
  _bit& operator&= (const _bit& op)                { return andOp(*this, op); }
  _bit& operator|= (const _bit& op)                { return orOp (*this, op); }

  // Arithmetics operators
  _bit& operator+= (const _bit& op)                { return addOp(*this, op); }
  _bit& operator*= (const _bit& op)                { return mulOp(*this, op); }
  _bit& operator-= (const _bit& op)                { return subOp(*this, op); }
  _bit& operator/= (const _bit& op)                { return divOp(*this, op); }
  _bit& operator%= (const _bit& op)          { return divOp(*this, op, this); }

  // Shift operators
  _bit& operator<<= (int sh)                       { return shlOp(*this, sh); }
  _bit& operator>>= (int sh)                       { return shrOp(*this, sh); }

  // Flip the bits in the vector (bv = ~bv)
  _bit& flip(void)                                     { return invOp(*this); }

  // Increment (prefix & postfix)
  _bit& operator++ ()                              { return incOp(*this, +1); }
  _bit  operator++ (int)       { _bit res(*this); incOp(res, +1); return res; }

  // Decrement (prefix & postfix)
  _bit& operator-- ()                              { return incOp(*this, -1); }
  _bit  operator-- (int)       { _bit res(*this); incOp(res, -1); return res; }

  // Test for != 0
  inline bool test(void) const;

  // Concatenation
  inline static const _bit concat(const _bit& op1, const _bit& op2);
    
  // Copy slice of bits from some other vector
  void copyBits(const _bit& src, int sl, int sr, int dl, int dr);
  
  // Explicit conversions
  inline int                toInt(void) const           { return vec[0].sval; }
  inline unsigned int       toUInt(void) const          { return vec[0].uval; }
  inline long               toLong(void) const;
  inline unsigned long      toULong(void) const;
  inline LONG_LONG          toLLong(void) const;
  inline UNSIGNED_LONG_LONG toULLong(void) const;
         double             toDouble(void) const;
         long double        toLDouble(void) const;

  // Get attributes
  chunk* addr(void) const                                      { return vec;  }
  bool   isUnsigned(void) const                                { return usgn; }
  int    length(void) const                                    { return len;  }
  
protected:
  // Dynamically allocated storage?
  const bool isDynamic;
  
  // Init vec: if larger than static chunk, allocate new dynamic memory
  bool init()                { vec = new chunk[_BITCHUNKS(len)]; return true; }
  
  // Comparison
  friend bool operator== (const _bit&, const _bit&);
  friend bool operator<  (const _bit&, const _bit&);
  friend bool operator>  (const _bit&, const _bit&);

  // Assignments
  inline _bit& set(const _bit& op);
  inline _bit& set(const _bitslice& s);
  inline _bit& set(int i);
  inline _bit& set(unsigned int i);
  inline _bit& set(long i);
  inline _bit& set(unsigned long i);
  inline _bit& set(LONG_LONG i);    
  inline _bit& set(UNSIGNED_LONG_LONG i);
         _bit& set(double d);
         _bit& set(long double d);
  
  // Operation evaluation
  inline _bit& eval(const _bit& op1, char type);                  // Unary
  inline _bit& eval(const _bit& op1, char type, int sh);          // Shift
  inline _bit& eval(const _bit& op1, char type, const _bit& op2); // Binary
  inline _bit& eval(const _bit& op1, const _bit& op2);            // Concat
  
  // Three-address operations ( op1.len >= len ! )
  inline _bit& invOp(const _bit& op1);
  inline _bit& negOp(const _bit& op1);
  inline _bit& incOp(const _bit& op1, int n);
  inline _bit& andOp(const _bit& op1, const _bit& op2);
  inline _bit& orOp (const _bit& op1, const _bit& op2);
  inline _bit& xorOp(const _bit& op1, const _bit& op2);
  inline _bit& addOp(const _bit& op1, const _bit& op2);
  inline _bit& subOp(const _bit& op1, const _bit& op2);
  inline _bit& subPo(const _bit& op1, const _bit& op2);
  inline _bit& concOp(const _bit& op1, const _bit& op2);
         _bit& shlOp(const _bit& op, int sh, bool mask = false);
         _bit& shrOp(const _bit& op, int sh);
         _bit& mulOp(const _bit& op1, const _bit& op2);
         _bit& divOp(const _bit& n, const _bit& d, _bit* r = 0);

  // Sign extend vector
  inline chunk extend(int chunk) const;

  // Return sign bit
  inline chunk sign() const;

  // Correct sign bits of partly filled, most significant chunk
  inline _bit& fill(int chunk);
  
  // Fill bits with zeros or ones
  void fill(int l, int r, int val);
};



/* ------------------------------------------------------------------------ */
/* Bit vector slice */


/* --- Base class, common representation --- */

class __bitslice
{
protected:
  _bit ref;        // Reference to original vector
  int  sl, sr;     // Indices of slice of original
  
public:
  // Constructors
  __bitslice(const _bit& o, int l, int r)
    : ref(o.addr(), o.length(), o.isUnsigned()), sl(l), sr(r)               { }
  __bitslice(_bit::chunk* ovec, int olen, bool ousgn, int l, int r)
    : ref(ovec, olen, ousgn), sl(l), sr(r)                                  { }
            
  // Get attributes
  int length(void) const  
              { return _BITLEN_SLICE(ref.length(), ref.isUnsigned(), sl, sr); }
};



/* --- Element of a bitbus (concatenated bit slices) --- */

class _bitbus_element: public __bitslice
{
public:
  // Next element in list
  _bitbus_element *next;
  
public:    
  // Bitbus element constructor
  _bitbus_element(_bit& o, int l, int r, _bitbus_element* lst = 0)
    : __bitslice(o, l, r), next(lst)                                        { }
  _bitbus_element(_bit::chunk* ovec, int olen, bool ousgn, 
		  int l, int r, _bitbus_element* lst = 0)
    : __bitslice(ovec, olen, ousgn, l, r), next(lst)                        { }
  
  // Concatenation, slicing of bitbus list starting from this element
  _bitbus_element* concat(_bitbus_element* list);
  _bitbus_element* slice(int l, int r);              // returns new list
    
  // Sync whole list starting with this element with the original vector
  void put(const _bit& buf);
  void get(const _bit& buf);
};



/* --- Slice of a bit vector --- */

class _bitslice: public __bitslice
{
protected:
  const bool usgn;
  
  // In case of a slice of a bitbus: list of bitbus elements
  _bitbus_element* const lst;
  
public:
  // Bit vector slice constructors
  _bitslice(_bit& o, int l, int r, bool usign, _bitbus_element* bus = 0)
    : __bitslice(o, l, r), usgn(usign), lst(bus? bus->slice(l, r) : 0)      { }
  _bitslice(_bit::chunk* ovec, int olen, bool ousgn, 
	     int l, int r, bool usign, _bitbus_element* bus = 0)
    : __bitslice(ovec, olen, ousgn, l, r), usgn(usign),
      lst(bus? bus->slice(l, r) : 0)                                        { }
   
  // Destructor
  ~_bitslice() { 
    for(_bitbus_element *t, *e(lst); e; e = t)       { t = e->next; delete e; }
  }
  
  // Type cast back to a _bit object
  operator const _bit() const                                 { return res(); }
  
  // Overloaded operators
  _bitslice& operator= (const _bit& b)                       { return put(b); }
  _bitslice& operator= (const _bitslice& s)    { if(lst||s.lst) put(_bit(s));
                  else ref.copyBits(s.ref, s.sl, s.sr, sl, sr); return *this; }

  _bitslice& operator^=(const _bit& op)            { return put(res() ^= op); }
  _bitslice& operator&=(const _bit& op)            { return put(res() &= op); }
  _bitslice& operator|=(const _bit& op)            { return put(res() |= op); }
  _bitslice& operator+=(const _bit& op)            { return put(res() += op); }
  _bitslice& operator-=(const _bit& op)            { return put(res() -= op); }
  _bitslice& operator*=(const _bit& op)            { return put(res() *= op); }
  _bitslice& operator/=(const _bit& op)            { return put(res() /= op); }
  _bitslice& operator%=(const _bit& op)            { return put(res() %= op); }
  _bitslice& operator<<= (int sh)                  { return put(res()<<= sh); }
  _bitslice& operator>>= (int sh)                  { return put(res()>>= sh); }

  _bitslice& flip(void)                           { return put(res().flip()); }
  
  _bitslice& operator++ ()                           { return put(++(res())); }
  _bit  operator++ (int i)           { _bit r(*this); put(++res()); return r; }

  _bitslice& operator-- ()                           { return put(--(res())); }
  _bit  operator-- (int i)           { _bit r(*this); put(--res()); return r; }

  // Slicing
  inline _bitslice operator[](int i);
  inline _bitslice operator()(int sl, int sr);
  inline _bitslice operator()(int l, int r, int i);
  inline _bitslice operator()(int l, int r, int sl, int sr);
  inline const _bitslice operator[](int i) const;
  inline const _bitslice operator()(int sl, int sr) const;
  inline const _bitslice operator()(int l, int r, int i) const;
  inline const _bitslice operator()(int l, int r, int sl, int sr) const;

  // Zero test
  bool test(void) const                                { return res().test(); }
  
  // Conversions
  int                toInt(void) const            { return res().toInt();     }
  unsigned int       toUInt(void) const           { return res().toUInt();    }
  long               toLong(void) const           { return res().toLong();    }
  unsigned long      toULong(void) const          { return res().toULong();   }
  LONG_LONG          toLLong(void) const          { return res().toLLong();   }
  UNSIGNED_LONG_LONG toULLong(void) const         { return res().toULLong();  }
  double             toDouble(void) const         { return res().toDouble();  }
  long double        toLDouble(void) const        { return res().toLDouble(); }
  
  // Get attributes
  bool isUnsigned(void) const                                  { return usgn; }
  
  // Sync with original vector
  _bitslice& put(const _bit& buf)                    { if(lst) lst->put(buf);
             else ref.copyBits(buf, buf.length()-1, 0, sl, sr); return *this; }
  const _bitslice& get(_bit& buf) const              { if(lst) lst->get(buf);
         else buf.copyBits(ref, sl, sr, buf.length()-1, 0);     return *this; }
  
protected:
  // Create slice result
  _bit res(void) const          { return _bit(*this, length(), isUnsigned()); }
};



/* ------------------------------------------------------------------------ */
/* Some external operators */


/* --- Bitwise --- */

// Exclusive Or
inline const _bit operator^ (const _bit& op1, const _bit& op2) { 
  return _bit(op1, '^', op2);
}

// And
inline const _bit operator& (const _bit& op1, const _bit& op2) { 
  return _bit(op1, '&', op2);
}

// Or
inline const _bit operator| (const _bit& op1, const _bit& op2) { 
  return _bit(op1, '|', op2);
}


// Inversion
inline const _bit operator~ (const _bit& op) {
  return _bit(op, '~');
}


/* --- Arithmetic --- */

// Addition
inline const _bit operator+ (const _bit& op1, const _bit& op2) {
  return _bit(op1, '+', op2);
}

// Unary plus
inline const _bit& operator+ (const _bit& op) {
  return op;
}


// Subtraction
inline const _bit operator- (const _bit& op1, const _bit& op2) {
  return _bit(op1, '-', op2);
}

// Negation
inline const _bit operator- (const _bit& op) {
  return _bit(op, '-');
}


// Multiplication
inline const _bit operator* (const _bit& op1, const _bit& op2) {
  return _bit(op1, '*', op2);
}


// Division
inline const _bit operator/ (const _bit& op1, const _bit& op2) {
  return _bit(op1, '/', op2);
}


// Modulo
inline const _bit operator% (const _bit& op1, const _bit& op2) {
  return _bit(op1, '%', op2);
}


/* --- Shifting --- */

inline const _bit operator>> (const _bit& op1, int sh) {
  return _bit(op1, '>', sh);
}

inline const _bit operator<< (const _bit& op1, int sh) {
  return _bit(op1, '<', sh);
}


/* --- Concatenation --- */

inline const _bit _bit::concat(const _bit& op1, const _bit& op2) {
  return _bit(op1, op2);
}




/* --- Relational --- */

// NOTE: in contrast to ANSI-C, comparisons are always signed (where
//       unsigned vectors simply have positive sign)!

// Equality
inline bool operator== (const _bit& op1, const _bit& op2) {
  for (int i = 0; i <= _BITCHUNK(_MAXLEN(op1.len, op2.len) - 1); i++) {
    if (op1.extend(i).uval != op2.extend(i).uval) return false;
  }
  return true;
}

// Inequality
inline bool operator!= (const _bit& op1, const _bit& op2) {
  return !operator==(op1, op2);
}


// Greater than
inline bool operator> (const _bit& op1, const _bit& op2) 
{
  const _bit::chunk s1 = op1.sign();
  const _bit::chunk s2 = op2.sign();

  // Check for same sign
  if (s1.uval != s2.uval) { 
    if (s1.uval) return false; else return true;
  }

  for (int i = _BITCHUNK(_MAXLEN(op1.len, op2.len) - 1); i >= 0; i--) 
  {
    if (op1.extend(i).uval > op2.extend(i).uval) return true;
    if (op1.extend(i).uval < op2.extend(i).uval) return false;
  }
  
  // Vectors are equal
  return false;
}


// Less than or equal
inline bool operator<= (const _bit& op1, const _bit& op2) {
  return !operator>(op1, op2);
}


// Less than
inline bool operator< (const _bit& op1, const _bit& op2) 
{
  const _bit::chunk s1 = op1.sign();
  const _bit::chunk s2 = op2.sign();

  // Check for same sign
  if (s1.uval != s2.uval) { 
    if (s1.uval) return true; else return false;
  }

  for (int i = _BITCHUNK(_MAXLEN(op1.len, op2.len) - 1); i >= 0; i--) 
  {
    if (op1.extend(i).uval < op2.extend(i).uval) return true;
    if (op1.extend(i).uval > op2.extend(i).uval) return false;
  }
  
  // Vectors are equal
  return false;
}


// Greater than or equal
inline bool operator>= (const _bit& op1, const _bit& op2) {
  return !operator<(op1, op2);
}



/* ------------------------------------------------------------------------- */
/* Class _bit Inlined deferred methods */


/* --- Integer Initializers --- */

inline _bit::_bit(bool b, int length, bool usign) 
  : len(length? length : _BITLEN_BOOL), usgn(usign), isDynamic(init()) {
  set((int) b);
}

inline _bit::_bit(signed char i, int length, bool usign) 
  : len(length? length : _BITLEN_CHAR), usgn(usign), isDynamic(init()) {
  set((signed int)i);
}

inline _bit::_bit(unsigned char i, int length, bool usign)
  : len(length? length : _BITLEN_UCHAR), usgn(usign), isDynamic(init()) {
  set((unsigned int)i);
}

inline _bit::_bit(short int i, int length, bool usign) 
  : len(length? length : _BITLEN_SHORT), usgn(usign), isDynamic(init()) {
  set((int) i);
}

inline _bit::_bit(unsigned short int i, int length, bool usign) 
  : len(length? length : _BITLEN_USHORT), usgn(usign), isDynamic(init()) {
  set((unsigned int) i);
}

inline _bit::_bit(int i, int length, bool usign) 
  : len(length? length : _BITLEN_INT), usgn(usign), isDynamic(init()) {
  set(i);
}

inline _bit::_bit(unsigned int i, int length, bool usign) 
  : len(length? length : _BITLEN_UINT), usgn(usign), isDynamic(init()) {
  set(i);
}

inline _bit::_bit(long i, int length, bool usign) 
  : len(length? length : _BITLEN_LONG), usgn(usign), isDynamic(init()) {
  set(i);
}

inline _bit::_bit(unsigned long i, int length, bool usign) 
  : len(length? length : _BITLEN_ULONG), usgn(usign), isDynamic(init()) {
  set(i);
}

inline _bit::_bit(LONG_LONG i, int length, bool usign) 
  : len(length? length : _BITLEN_LLONG), usgn(usign), isDynamic(init()) {
  set(i);
}

inline _bit::_bit(UNSIGNED_LONG_LONG i, int length, bool usign) 
  : len(length? length : _BITLEN_ULLONG), usgn(usign), isDynamic(init()) {
  set(i);
}

inline _bit::_bit(double d, int length, bool usign)
  : len(length), usgn(usign), isDynamic(init()) {
  set(d);
}

inline _bit::_bit(long double d, int length, bool usign) 
  : len(length), usgn(usign), isDynamic(init()) {
  set(d);
}

//!LLONG!inline _bit::_bit(_longlong_<> i, int length, bool usign) 
//!LLONG!  : len(length? length : _BITLEN_LLONG), usgn(usign), isDynamic(init()) {
//!LLONG!  set((LONG_LONG&)i);
//!LLONG!}
//!LLONG!
//!LLONG!inline _bit::_bit(_longlong_<true> i, int length, bool usign) 
//!LLONG!  : len(length? length : _BITLEN_ULLONG), usgn(usign), isDynamic(init()) {
//!LLONG!  set((UNSIGNED_LONG_LONG&)i);
//!LLONG!}
//!LLONG!


/* --- String Initializier --- */

inline _bit::_bit(const char* str, int length, bool usign, unsigned int base)
  : len(length? length : ((str[strlen(str)-1] == 'u')? (strlen(str)-1) : (strlen(str)))),
    usgn(usign || (length? false : (str[len] == 'u'))),
    isDynamic(init())
{
  // Assign string
  set(str, usign, base);
}


/* --- Operation results --- */

inline _bit::_bit(const _bit& op1, char type)
  : len (_BITLEN_UNARY (op1.length(),op1.isUnsigned())),
    usgn(_BITUSGN_UNARY(op1.length(),op1.isUnsigned())),
    isDynamic(init())
{
  eval(op1, type);
}

inline _bit::_bit(const _bit& op1, char type, int sh)
  : len (_BITLEN_UNARY (op1.length(),op1.isUnsigned())),
    usgn(_BITUSGN_UNARY(op1.length(),op1.isUnsigned())),
    isDynamic(init())
{
  eval(op1, type, sh);
}

inline _bit::_bit(const _bit& op1, const _bit& op2)
  : len (_BITLEN_CONCAT (op1.length(),op1.isUnsigned(),			 
			 op2.length(),op2.isUnsigned())),
    usgn(_BITUSGN_CONCAT(op1.length(),op1.isUnsigned(),
			 op2.length(),op2.isUnsigned())),
    isDynamic(init())
{
  eval(op1, op2);
}

inline _bit::_bit(const _bit& op1, char type, const _bit& op2)
  : len (_BITLEN_BINARY (op1.length(),op1.isUnsigned(),
			 op2.length(),op2.isUnsigned())),
    usgn(_BITUSGN_BINARY(op1.length(),op1.isUnsigned(),
			 op2.length(),op2.isUnsigned())),
    isDynamic(init())
{
  eval(op1, type, op2);
}


#ifdef _BIT_TEMPLATE_OPERATORS

/* --- Operation results as a reference --- */

inline _bit::_bit(chunk* ref, int length, bool usign,
		  const _bit& op1, char type)
  : vec(ref), len(length), usgn(usign), isDynamic(false) 
{
  eval(op1, type);
}

inline _bit::_bit(chunk* ref, int length, bool usign,
		  const _bit& op1, char type, int sh) 
  : vec(ref), len(length), usgn(usign), isDynamic(false) 
{
  eval(op1, type, sh);
}

inline _bit::_bit(chunk* ref, int length, bool usign,
		  const _bit& op1, const _bit& op2)
  : vec(ref), len(length), usgn(usign), isDynamic(false) 
{
  eval(op1, op2);
}

inline _bit::_bit(chunk* ref, int length, bool usign,
		  const _bit& op1, char type, const _bit& op2)
  : vec(ref), len(length), usgn(usign), isDynamic(false) 
{
  eval(op1, type, op2);
}

#endif


/* --- Slicing --- */

inline _bitslice _bit::operator[] (int i) const {
  return _bitslice(vec, len, usgn, i, i, true);
}							    

inline _bitslice _bit::operator()(int sl, int sr) const {
  return _bitslice(vec, len, usgn, sl, sr, _BITUSGN_SLICE(len,usgn,sl,sr));
}


inline _bitslice _bit::operator()(int l, int r, int i) const {
  ASSERT(_BITLEN(l,r)==len);  
  return operator[](_BITMAP(l,r, i));
}							    

inline _bitslice _bit::operator()(int l, int r, int sl, int sr) const {
  ASSERT(_BITLEN(l,r)==len);  
  return operator()(_BITMAP(l,r, sl), _BITMAP(l,r, sr));
}



/* --- Assignments --- */


inline _bit& _bit::set(const _bit& op) {
  int i; for (i = 0; i < _BITCHUNKS(len); i++) vec[i] = op.extend(i);
  return fill(i-1);
}

inline _bit& _bit::set(const _bitslice& s) {
  s.get(*this);
  return *this;
}


inline _bit& _bit::set(int i) {
  vec[0].sval = i;
#ifdef _RSH_IS_SIGN_EXTENDING
  i >>= _BITCHUNK_SIZE - 1;
#else
  if (i < 0) i = -1; else i = 0;
#endif
  int j; for (j = 1; j < _BITCHUNKS(len); j++) vec[j].sval = i;
  return fill(j-1);
}

inline _bit& _bit::set(unsigned int i) {
  vec[0].uval = i;
  int j; for (j = 1; j < _BITCHUNKS(len); j++) vec[j].uval = 0;
  return fill(j-1);
}


#if (LONG_MAX == INT_MAX)

inline _bit& _bit::set(long i) {
  return set((int) i);
}
inline _bit& _bit::set(unsigned long i) {
  return set((unsigned int) i);
}

#else // #if (LONG_MAX == INT_MAX)

inline _bit& _bit::set(long i) {
#ifndef _RSH_IS_SIGN_EXTENDING
  bool neg = (i < 0);
#endif
  int j; for (j = 0; j < _BITCHUNKS(_MINLEN(len,(int)_BITLEN_LONG))); j++) {
    vec[j].uval = i;
    i >>= _BITCHUNK_SIZE;
  }
#ifndef _RSH_IS_SIGN_EXTENDING
  if (neg) {
    i = -1;
    vec[j-1].uval |= _BITMASK_LSB_CLR(_BITLEN_LONG % _BITCHUNK_SIZE);
  }
#endif
  for ( ; j < _BITCHUNKS(len); j++) vec[j].sval = i;
  return fill(j-1);
}

inline _bit& _bit::set(unsigned long i) {
  int j; for (j = 0; j < _BITCHUNKS(_MINLEN(len,(int)_BITLEN_ULONG))); j++) {
    vec[j].uval = i;
    i >>= _BITCHUNK_SIZE;
  }
  for ( ; j < _BITCHUNKS(len); j++) vec[j].uval = 0;
  return fill(j-1);
}

#endif // #if (LONG_MAX == INT_MAX) 


inline _bit& _bit::set(LONG_LONG i) {
#ifndef _RSH_IS_SIGN_EXTENDING
  bool neg = (i < 0);
#endif
  int j; for (j = 0; j < _BITCHUNKS(_MINLEN(len,(int)_BITLEN_LLONG)); j++) {
    vec[j].uval = ll2uint(i);
    i >>= _BITCHUNK_SIZE;
  }
#ifndef _RSH_IS_SIGN_EXTENDING
  if (neg) {
    i = -1;
    vec[j-1].uval |= _BITMASK_LSB_CLR(_BITLEN_LLONG % _BITCHUNK_SIZE);
  }
#endif
  for ( ; j < _BITCHUNKS(len); j++) vec[j].sval = ll2int(i);
  return fill(j-1);
}

inline _bit& _bit::set(UNSIGNED_LONG_LONG i) {
  int j; for (j = 0; j < _BITCHUNKS(_MINLEN(len,(int)_BITLEN_ULLONG)); j++) {
    vec[j].uval = ll2uint(i);
    i >>= _BITCHUNK_SIZE;
  }
  for ( ; j < _BITCHUNKS(len); j++) vec[j].uval = 0;
  return fill(j-1);
}


/* --- Operation evaluation --- */

inline _bit& _bit::eval(const _bit& op1, char type) {
  switch (type) {
    case '~': return invOp(op1);
    case '-': return negOp(op1);
    default: ASSERT(false);
  }
  return *this;
}

inline _bit& _bit::eval(const _bit& op1, char type, int sh) {
  switch (type) {
    case '<': return shlOp(op1, sh);
    case '>': return shrOp(op1, sh);
    default: ASSERT(false);
  }
  return *this;
}

inline _bit& _bit::eval(const _bit& op1, const _bit& op2) {
  return concOp(op1, op2);
}

inline _bit& _bit::eval(const _bit& op1, char type, const _bit& op2) {
  bool s = op1.length() < op2.length();
  switch (type) {
    case '^': if(s) return xorOp(op2, op1); return xorOp(op1, op2);
    case '|': if(s) return orOp (op2, op1); return orOp (op1, op2);
    case '&': if(s) return andOp(op2, op1); return andOp(op1, op2);
    case '+': if(s) return addOp(op2, op1); return addOp(op1, op2);
    case '-': if(s) return subPo(op2, op1); return subOp(op1, op2);
    case '*': if(s) return mulOp(op2, op1); return mulOp(op1, op2);
    case '/': return divOp(op1, op2);
    case '%': return divOp(op1, op2, this);
    default: ASSERT(false);
  }
  return *this;
}


/* --- Bitwise operations --- */

// Exclusive Or
inline _bit& _bit::xorOp(const _bit& op1, const _bit& op2) {
  int i;
  ASSERT(op1.len >= len);
  for(i = 0; i < _BITCHUNKS(len); i++) {
    vec[i].uval = op1.vec[i].uval ^ op2.extend(i).uval;
  }
  return fill(i-1);
}

// Or
inline _bit& _bit::orOp(const _bit& op1, const _bit& op2) {
  int i;
  ASSERT(op1.len >= len);
  for(i = 0; i < _BITCHUNKS(len); i++) {
    vec[i].uval = op1.vec[i].uval | op2.extend(i).uval;
  }
  return fill(i-1);
}

// And
inline _bit& _bit::andOp(const _bit& op1, const _bit& op2) {
  int i;
  ASSERT(op1.len >= len);
  for(i = 0; i < _BITCHUNKS(len); i++) {
    vec[i].uval = op1.vec[i].uval & op2.extend(i).uval;
  }
  return fill(i-1);
}


/* --- Arithmetic operations --- */


// Addition
inline _bit& _bit::addOp(const _bit& op1, const _bit& op2) {
  int i;
  ASSERT(op1.len >= len);
  UNSIGNED_LONG_LONG tmp, carry = 0;
  for (i = 0; i < _BITCHUNKS(len); i++) {
    tmp = (carry + op1.vec[i].uval) + op2.extend(i).uval;
    vec[i].uval = ll2uint(tmp);
    carry = tmp >> _BITCHUNK_SIZE;
  }
  return fill(i-1);
}

// Subtraction
inline _bit& _bit::subOp(const _bit& op1, const _bit& op2) {
  int i;
  ASSERT(op1.len >= len);
  UNSIGNED_LONG_LONG tmp, carry = 1;
  for (i = 0; i < _BITCHUNKS(len); i++) {
    tmp = (carry + op1.vec[i].uval) + (~(op2.extend(i).uval));
    vec[i].uval = ll2uint(tmp);
    carry = tmp >> _BITCHUNK_SIZE;
  }
  return fill(i-1);
}
inline _bit& _bit::subPo(const _bit& op2, const _bit& op1) {
  int i;
  ASSERT(op2.len >= len);
  UNSIGNED_LONG_LONG tmp, carry = 1;
  for (i = 0; i < _BITCHUNKS(len); i++) {
    tmp = (carry + op1.extend(i).uval) + (~(op2.vec[i].uval));
    vec[i].uval = ll2uint(tmp);
    carry = tmp >> _BITCHUNK_SIZE;
  }
  return fill(i-1);
}

// Negation
inline _bit& _bit::negOp(const _bit& op1) {
  ASSERT(op1.len >= len);
  UNSIGNED_LONG_LONG tmp, carry = 1;
  int i;
  for (i = 0; i < _BITCHUNKS(len); i++) {
    tmp = ~(op1.vec[i].uval) + carry;
    vec[i].uval = ll2uint(tmp);
    carry = tmp >> _BITCHUNK_SIZE;
  }
  return fill(i-1);
}


// Flip bits
inline _bit& _bit::invOp(const _bit& op1)  {
  int i;
  ASSERT(op1.len >= len);
  for (i = 0; i < _BITCHUNKS(len); i++) {
    vec[i].uval = ~(op1.vec[i].uval);
  }
  return fill(i-1);
}


// Increment & Decrement
inline _bit& _bit::incOp(const _bit& op1, int n) {
  int i;
  ASSERT(op1.len >= len);
  LONG_LONG tmp = n;
  for (i = 0; i < _BITCHUNKS(len); i++) {
    tmp += op1.vec[i].uval;
    vec[i].sval = ll2int(tmp);
    tmp >>= _BITCHUNK_SIZE;
  }
  return fill(i-1);
}


/* --- Zero test --- */


inline bool _bit::test (void) const {
  for (int i = 0; i < _BITCHUNKS(len); i++) {
    if (vec[i].sval) return true;
  }
  return false;
}



/* --- Concatenation --- */

inline _bit& _bit::concOp(const _bit& op1, const _bit& op2) {
  // Set lower part
  for (int i = 0; i < _MINLEN(_BITCHUNKS(len), _BITCHUNKS(op2.len)); i++) {
    vec[i].uval = op2.vec[i].uval;
  }
  if (op2.len >= len) return fill(_BITCHUNK(len-1));
  
  // Shift and fit in upper part
  return shlOp(op1, op2.len, true);
}



/* --- Conversions --- */

#if (LONG_MAX == INT_MAX)

inline long _bit::toLong (void) const {
  return vec[0].sval;
}

inline unsigned long _bit::toULong (void) const {
  return vec[0].uval;
}

#else

inline long _bit::toLong (void) const {
  long res;
  int  i = _BITCHUNK(_MINLEN(len,(int)_BITLEN_LONG)-1);
  
  if(usgn) res = vec[i].uval; else res = vec[i].sval;
  for (--i; i >= 0; i--) {
    res <<= _BITCHUNK_SIZE;
    res += vec[i].uval;
  }
  return res;
}

inline unsigned long _bit::toULong (void) const {
  unsigned long res;
  int i = _BITCHUNK(_MINLEN(len,(int)_BITLEN_ULONG)-1);

  if(usgn) res = vec[i].uval; else res = vec[i].sval;
  for (--i; i >= 0; i--) {
    res <<= _BITCHUNK_SIZE;
    res += vec[i].uval;
  }
  return res;
}

#endif // #if (LONG_MAX == INT_MAX)


inline LONG_LONG _bit::toLLong (void) const {
  LONG_LONG res;
  int i = _BITCHUNK(_MINLEN(len,(int)_BITLEN_LLONG)-1);

  if(usgn) res = vec[i].uval; else res = vec[i].sval;
  for (--i; i >= 0; i--) {
    res <<= _BITCHUNK_SIZE;
    res += vec[i].uval;
  }
  return res;
}

inline UNSIGNED_LONG_LONG _bit::toULLong (void) const {
  UNSIGNED_LONG_LONG res;
  int i = _BITCHUNK(_MINLEN(len,(int)_BITLEN_ULLONG)-1);

  if(usgn) res = vec[i].uval; else res = vec[i].sval;
  for (--i; i >= 0; i--) {
    res <<= _BITCHUNK_SIZE;
    res += vec[i].uval;
  }
  return res;
}



/* --- Helper routines --- */


// Sign extend vector
inline _bit::chunk _bit::extend(int n) const {
  if (n < _BITCHUNKS(len)) return vec[n];
  return sign();
}  

// Return sign bit
inline _bit::chunk _bit::sign() const {
  _bit::chunk s;  s.uval = 0;
  if (!usgn && vec[_BITCHUNK(len-1)].sval < 0) s.sval = -1;
  return s;
}

#ifdef RSH_IS_SIGN_EXTENDING

// Correct sign bits of partly filled, most significant chunk
inline _bit& _bit::fill(int chunk) {
  if (!(len % _BITCHUNK_SIZE)) return *this;
  int sh = _BITCHUNK_SIZE - (len % _BITCHUNK_SIZE);
  vec[chunk].uval <<= sh;
  if( usgn ) vec[chunk].uval >>= sh; else vec[chunk].sval >>= sh;
  return *this;
}

#else

// Correct sign bits of partly filled, most significant chunk
inline _bit& _bit::fill(int chunk) {
  if (!(len % _BITCHUNK_SIZE)) return *this;
  if ( (usgn) || !(vec[chunk].uval & _BITMASK((len-1) % _BITCHUNK_SIZE)) )
    vec[chunk].uval &= _BITMASK_LSB_SET(len % _BITCHUNK_SIZE);
  else
    vec[chunk].uval |= _BITMASK_LSB_CLR(len % _BITCHUNK_SIZE);
  return *this;
}

#endif

    
/* ------------------------------------------------------------------------ */
/* Class _bitslice inlined methods */
    
    
inline  _bitslice _bitslice::operator[] (int i) {
  return _bitslice(ref, _BITMAP(sl,sr,i), _BITMAP(sl,sr,i), true, lst);
}

inline _bitslice _bitslice::operator()(int sl, int sr) {
  return _bitslice(ref, _BITMAP(this->sl,this->sr, sl),
		        _BITMAP(this->sl,this->sr, sr),
		        _BITUSGN_SLICE(ref.length(),isUnsigned(),sl,sr), 
		        lst);
}

inline _bitslice _bitslice::operator()(int l, int r, int i) {
  ASSERT(_BITLEN(l,r)==length());
  return operator[](_BITMAP(l,r, i));
}

inline _bitslice _bitslice::operator()(int l, int r, int sl, int sr) {
  ASSERT(_BITLEN(l,r)==length()); 
  return operator()(_BITMAP(l,r, sl), _BITMAP(l,r, sr));
}

inline const _bitslice _bitslice::operator[] (int i) const {
  return const_cast<_bitslice*>(this)->operator[](i);  
}

inline const _bitslice _bitslice::operator()(int sl, int sr) const {
  return const_cast<_bitslice*>(this)->operator()(sl, sr);
}

inline const _bitslice _bitslice::operator()(int l, int r, int i) const {
  return const_cast<_bitslice*>(this)->operator()(l, r, i);
}

inline const _bitslice _bitslice::operator()(int l, int r, int sl, int sr) const {
  return const_cast<_bitslice*>(this)->operator()(l, r, sl, sr);
}


#endif
