/************************************************************************/
/* PortMap.cc: SpecC Internal Representation, PortMapping Classes	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 06/24/98 */
/************************************************************************/

/* last update: 05/31/01 */

/* modifications: (most recent first)
 *
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/27/01 RD	additional mapping checking against directions of class ports
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/22/01 RD	fixed checking of constant mappings (allow const ports)
 * 04/30/01 RD	changed use of len member to length() method for bitvectors
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/25/01 RD	fixed checking of constant mappings (no const flag comparison);
 *		added parenthesis around constant mappings
 * 02/14/01 RD	added member function IsOpenMapping to SIR_PortMap
 * 02/14/01 RD	added member Constant to SIR_BitSlice in order to support
 *		port mapping to constants and open port mappings, as
 *		introduced with SpecC LRM1.0
 * 02/13/01 RD	created modifications list (last update was 04/12/99)
 */

#include "IntRep/PortMap.h"
#include "IntRep/Extern.h"
#include "IntRep/Symbol.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* (none) */


/*** internal type declarations *****************************************/


	/* (none) */


/*** class implementations **********************************************/


	/*******************/
	/*** SIR_PortMap ***/
	/*******************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_PortMap::SIR_PortMap(		/* constructor #1 (general) */
	sir_bitslices	*BitSlices)
{

assert(BitSlices != NULL);

SIR_PortMap::BitSlices	= BitSlices;

} /* end of SIR_PortMap::SIR_PortMap #1 */


SIR_PortMap::SIR_PortMap(		/* constructor #3 (duplicator) */
	sir_portmap	*Original)
{

SIR_PortMap::BitSlices	= Original->BitSlices ?
				new SIR_BitSlices(Original->BitSlices) :
				NULL;

} /* end of SIR_PortMap::SIR_PortMap #3 */


SIR_PortMap::~SIR_PortMap(void)		/* destructor */
{

delete BitSlices;

} /* end of SIR_PortMap::~SIR_PortMap */


sir_portmap *SIR_PortMap::New(	/* create a new direct port mapping (#2b) */
	sir_symbol	*DirectMapping)	/* (returns NULL if SIR_Error) */
{

/* DirectMapping may be NULL meaning open connection */

return(new SIR_PortMap(
		new SIR_BitSlices(
			new SIR_BitSlice(DirectMapping))));

} /* end of SIR_PortMap::New #2b */


ERROR SIR_PortMap::DFS_ForAllNodes(	/* iterator for over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

/* this is not a node, but there are nodes below */

if (BitSlices)
   { if ((SIR_Error = BitSlices->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_PortMap::DFS_ForAllNodes */


void SIR_PortMap::UnAlias(void)/* unalias all type, usertype, symbol links */
{

if (BitSlices)
   { BitSlices->UnAlias();
    } /* fi */

} /* end of SIR_PortMap::UnAlias */


void SIR_PortMap::WriteCC_Decl(		/* generates C++ source code (part 1) */
	FILE		*File,
	sir_type	*PortType,
	int		*BitBusNum,
	int		*OpenPortNum,
	int		*ConstPortNum)
{
static string	Buffer;
sir_bitslice	*BitSlice;
sir_types	*TypeTable;
sir_type	*ConstType;

assert(File != NULL);
assert(PortType != NULL);

if (IsOpenMapping())		/* create dummy for open mappings */
   { SIR_LineInfo::WriteNL(File);
     Buffer.form(SIR_CXX_OPEN_PORT_NAME, (*OpenPortNum)++);
     fprintf(File, "%s;", PortType->PrettyString(&Buffer, FALSE, TRUE, FALSE));
     return;
    } /* fi */

assert(BitSlices != NULL);
BitSlice = BitSlices->First();	/* create dummies for constant mappings */
while(BitSlice)
   { if (BitSlice->Constant)
	{ assert(BitSlice->Symbol == NULL);
	  SIR_LineInfo::WriteNL(File);
	  Buffer.form(SIR_CXX_CONST_PORT_NAME, (*ConstPortNum)++);
	  TypeTable = PortType->GetTable();
	  assert(TypeTable != NULL);
	  ConstType = TypeTable->FindOrInsert(BitSlice->Constant);
	  assert(ConstType != NULL);
	  ConstType = TypeTable->FindOrInsertModifiedType(ConstType,
							false); /* no const */
	  assert(ConstType != NULL);
	  fprintf(File, "%s;", ConstType->PrettyString(&Buffer,
							FALSE, TRUE, FALSE));
	 } /* fi */
     BitSlice = BitSlice->Succ();
    } /* elihw */

if (IsDirectlyMappable(PortType))	/* determine mapping type */
   { return;	/* nothing to write for straight mapping */
    } /* fi */
else
   { /* indirect mapping required */
     SIR_LineInfo::WriteNL(File);
     fprintf(File, SIR_BIT_BUS_CLASS_NAME "<"
			SIR_BIT_LENGTH_NAME "(%d,%d),%s> ",
			PortType->LeftBound, PortType->RightBound,
			((PortType->Type == SIR_TYPE_BIT) ?
				SIR_BIT_SIGNED_FLAG : SIR_BIT_UNSIGNED_FLAG));
     fprintf(File, SIR_CXX_SLICED_BUS_NAME ";", (*BitBusNum)++);
    } /* esle */

} /* end of SIR_PortMap::WriteCC_Decl */


const char *SIR_PortMap::WriteCC_Init(	/* generates C++ source code (part 2) */
	sir_type	*PortType,	/* (returns NULL for "nothing to do") */
	int		*BitBusNum,
	int		*OpenPortNum,
	int		*ConstPortNum)
{
static string	Buffer;
string		TmpString,
		PipedAccess;
sir_bitslice	*BitSlice;
sir_type	*MapType;
int		MapLeft,
		MapRight;
int		StoreConstPortNum;

assert(PortType != NULL);

if (IsOpenMapping())		/* initialize dummy for open mappings */
   { Buffer.form(SIR_CXX_OPEN_PORT_NAME "(0)", (*OpenPortNum)++);
     return(Buffer.chars());
    } /* fi */

StoreConstPortNum = *ConstPortNum;	/* we may need it twice */

assert(BitSlices != NULL);
BitSlice = BitSlices->First();	/* initialize dummies for constant mappings */
Buffer = "";
while(BitSlice)
   { if (BitSlice->Constant)
	{ assert(BitSlice->Symbol == NULL);
	  if (Buffer.length() > 0)
	     { Buffer += ", ";
	      } /* fi */
	  TmpString.form(SIR_CXX_CONST_PORT_NAME "(%s)",
			(*ConstPortNum)++, BitSlice->Constant->Print(TRUE));
	  Buffer += TmpString;
	 } /* fi */
     BitSlice = BitSlice->Succ();
    } /* elihw */

if (! IsDirectlyMappable(PortType))	/* indirect mapping required */
   { *ConstPortNum = StoreConstPortNum;	/* restore the counter */
     if (Buffer.length() > 0)
	{ Buffer += ", ";
	 } /* fi */
     TmpString.form(SIR_CXX_SLICED_BUS_NAME "(", (*BitBusNum)++);
     Buffer += TmpString;
     BitSlice = BitSlices->First();
     while(BitSlice)
	{ PipedAccess = "";
	  if (  (BitSlice->Symbol)
	      &&(BitSlice->Symbol->StorageClass == SIR_STORAGE_PIPED))
	     { switch(PortType->Direction)
		  { case SIR_PORT_NONE:
		    case SIR_PORT_INOUT:
		       { assert(FALSE);	/* type checked! */
			}
		    case SIR_PORT_IN:
		       { PipedAccess.form(SIR_SIM_PIPED_IN_ACCESS(
					BitSlice->Symbol->PipeStages));
			 break;
			}
		    case SIR_PORT_OUT:
		       { PipedAccess.form(SIR_SIM_PIPED_OUT_ACCESS(
					BitSlice->Symbol->PipeStages));
			 break;
			}
		    default:
		       { assert(FALSE);	/* bad direction */
			}
		   } /* hctiws */
	      } /* fi */
	  if (BitSlice->Pred())
	     { Buffer += "->" SIR_BIT_BUS_CONCAT_NAME "(";
	      } /* fi */
	  if (BitSlice->Symbol)
	     { TmpString.form("%s%s." SIR_BIT_BUS_SLICE_NAME "(%d,%d, %d,%d)",
				BitSlice->Symbol->Name.chars(),
				PipedAccess.chars(),
				BitSlice->Symbol->Type->LeftBound,
				BitSlice->Symbol->Type->RightBound,
				BitSlice->LeftBound,
				BitSlice->RightBound);
	       Buffer += TmpString;
	      } /* fi */
	  else
	     { assert(BitSlice->Constant != NULL);
	       MapType = PortType->GetTable()->FindOrInsert(BitSlice->Constant);
	       MapLeft = MapType->LeftBound;
	       MapRight = MapType->RightBound;
	       TmpString.form(SIR_CXX_CONST_PORT_NAME "."
				SIR_BIT_BUS_SLICE_NAME "(%d,%d, %d,%d)",
				(*ConstPortNum)++,
				MapLeft, MapRight,
				MapLeft, MapRight);
	       Buffer += TmpString;
	      } /* esle */
	  if (BitSlice->Pred())
	     { Buffer += ")";
	      } /* fi */
	  BitSlice = BitSlice->Succ();
	 } /* elihw */
     Buffer += ")";
    } /* fi */

if (Buffer.length() > 0)
   { return(Buffer.chars());
    } /* fi */
else
   { return(NULL);	/* nothing to write (i.e. straight mapping) */
    } /* esle */

} /* end of SIR_PortMap::WriteCC_Init */


void SIR_PortMap::WriteCC(		/* generates C++ source code (part 3) */
	FILE		*File,
	sir_type	*PortType,
	int		*BitBusNum,
	int		*OpenPortNum,
	int		*ConstPortNum)
{
sir_bitslice	*BitSlice;

assert(File != NULL);
assert(PortType != NULL);

if (IsOpenMapping())		/* map to dummy for open mappings */
   { fprintf(File, SIR_CXX_OPEN_PORT_NAME, (*OpenPortNum)++);
     return;
    } /* fi */

if (IsDirectlyMappable(PortType))	/* determine mapping type */
   { /* do straight mapping */
     BitSlice = BitSlices->First();
     if (BitSlice->Constant)
	{ assert(BitSlice->Symbol == NULL);
	  fprintf(File, SIR_CXX_CONST_PORT_NAME, (*ConstPortNum)++);
	  return;
	 } /* fi */
     assert(BitSlice->Symbol != NULL);
     if (BitSlice->Symbol->StorageClass == SIR_STORAGE_PIPED)
	{ switch(PortType->Direction)
	     { case SIR_PORT_NONE:
	       case SIR_PORT_INOUT:
		  { assert(FALSE);	/* type checked! */
		   }
	       case SIR_PORT_IN:
		  { fputs(BitSlice->Symbol->Name.chars(), File);
		    fprintf(File, SIR_SIM_PIPED_IN_ACCESS(
					BitSlice->Symbol->PipeStages));
		    break;
		   }
	       case SIR_PORT_OUT:
		  { fputs(BitSlice->Symbol->Name.chars(), File);
		    fprintf(File, SIR_SIM_PIPED_OUT_ACCESS(
					BitSlice->Symbol->PipeStages));
		    break;
		   }
	       default:
		  { assert(FALSE);	/* bad direction */
		   }
	      } /* hctiws */
	 } /* fi */
     else
	{ fputs(BitSlice->Symbol->Name.chars(), File);
	 } /* esle */
    } /* fi */
else
   { /* indirect mapping required */
     fprintf(File, SIR_CXX_SLICED_BUS_NAME, (*BitBusNum)++);
    } /* esle */

} /* end of SIR_PortMap::WriteCC */


BOOL SIR_PortMap::IsOpenMapping(void)	/* determines whether mapping is open */
{
sir_bitslice	*BitSlice;

assert(BitSlices != NULL);

BitSlice = BitSlices->First();
assert(BitSlice != NULL);
if (BitSlice->Symbol)
   { assert(BitSlice->Constant == NULL);
     return(FALSE);	/* (std.) symbol mapping */
    } /* fi */

if (BitSlice->Constant)
   { return(FALSE);	/* (at least one) constant mapping */
    } /* fi */

assert(BitSlice->Succ() == NULL);	/* grammar assures these */
assert(BitSlice->Pred() == NULL);

return(TRUE);

} /* end of SIR_PortMap::IsOpenMapping */


BOOL SIR_PortMap::IsDirectlyMappable(	/* determines port mapping type */
	sir_type	*PortType)
{
sir_bitslice	*BitSlice;
sir_type	*MapType;
int		PortLeft,
		PortRight,
		MapLeft,
		MapRight;

assert(PortType != NULL);
assert(BitSlices != NULL);

BitSlice = BitSlices->First();
assert(BitSlice != NULL);
if (  (! BitSlice->Symbol)
    &&(! BitSlice->Constant))
   { return(FALSE);	/* open mapping */
    } /* fi */

if (BitSlice->Symbol)
   { assert(BitSlice->Constant == NULL);
     MapType = BitSlice->Symbol->Type;
     MapLeft = BitSlice->LeftBound;
     MapRight = BitSlice->RightBound;
    } /* fi */
else
   { assert(BitSlice->Constant != NULL);
     MapType = PortType->GetTable()->FindOrInsert(BitSlice->Constant);
     MapLeft = MapType->LeftBound;
     MapRight = MapType->RightBound;
    } /* esle */
assert(MapType != NULL);

if (PortType->TypeClass() != SIR_TYPECLASS_BITVECTOR)
   { assert(BitSlices->NumElements() == 1);			/* type check */
     assert(MapType->TypeClass() != SIR_TYPECLASS_BITVECTOR);	/* type check */
     return(TRUE);
    } /* fi */

	/* bitvector mapping */

if (BitSlices->NumElements() != 1)
   { return(FALSE);	/* concatenated bitslices */
    } /* fi */

if (  (MapLeft != MapType->LeftBound)
    ||(MapRight != MapType->RightBound))
   { return(FALSE);	/* sliced and/or reversed mapping */
    } /* fi */

PortLeft = PortType->LeftBound;
PortRight = PortType->RightBound;
assert((ABS((PortLeft)-(PortRight))+1)	/* width must match */
		== (ABS((MapLeft)-(MapRight))+1));

return(SIR_BIT_DIRECTLY_MAPPABLE(	/* mapping supported by bit library */
		PortLeft, PortRight,
		(PortType->Type == SIR_TYPE_UBIT),
		MapLeft, MapRight,
		(MapType->Type == SIR_TYPE_UBIT)));

} /* end of SIR_PortMap::IsDirectlyMappable */


ERROR SIR_PortMap::Check(	/* semantically check the port mapping */
	sir_type_ptr	*Port,
	unsigned int	PortNum,
	const char	*SourceFile = NULL,	/* default: no warnings */
	unsigned int	SourceLine = 0)
{
sir_bitslice	*BitSlice;
sir_type	*PortMapType;
int		BusWidth;

assert(Port != NULL);
assert(PortNum > 0);

if (Port->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
   { BusWidth = 0;
     BitSlice = BitSlices->First();
     while(BitSlice)
	{ if (BitSlice->Symbol)
	     { assert(BitSlice->Constant == NULL);
	       if (BitSlice->Symbol->Type->TypeClass() !=
						SIR_TYPECLASS_BITVECTOR)
		  { SIR_ErrMsg.form("Mismatch in port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
				"bitvector type expected",
				PortNum);
		    return(SIR_ERROR_PORT_MAPPING_MISMATCH_5);
		   } /* fi */
	       if (  (BitSlice->Symbol->StorageClass == SIR_STORAGE_PIPED)
		   &&(  (Port->Type->Direction != SIR_PORT_IN)
		   &&(Port->Type->Direction != SIR_PORT_OUT)))
		  { SIR_ErrMsg.form("Illegal port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
				"mapping to piped variable requires "
					"port direction 'in' or 'out'",
				PortNum);
		    return(SIR_ERROR_PORT_MAPPING_MISMATCH_6);
		   } /* fi */
	       PortMapType = BitSlice->Symbol->Type;
	       if (  (  (Port->Type->Direction == SIR_PORT_IN)
		      &&(PortMapType->Direction == SIR_PORT_OUT))
		   ||(  (Port->Type->Direction == SIR_PORT_INOUT)
		      &&(  (PortMapType->Direction == SIR_PORT_IN)
			 ||(PortMapType->Direction == SIR_PORT_OUT)))
		   ||(  (Port->Type->Direction == SIR_PORT_OUT)
		      &&(PortMapType->Direction == SIR_PORT_IN)))
		  { SIR_ErrMsg.form("Mismatch in port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
				"instance '%s' port mapped to class '%s' port",
			PortNum,
			SIR_Type::PrintDirection(Port->Type->Direction),
			SIR_Type::PrintDirection(PortMapType->Direction));
		    return(SIR_ERROR_PORT_MAPPING_MISMATCH_10);
		   } /* fi */
	       BusWidth += ABS(BitSlice->LeftBound - BitSlice->RightBound) + 1;
	      } /* fi */
	  else
	     { if (BitSlice->Constant)
		  { if (Port->Type->Direction != SIR_PORT_IN)
		       { SIR_ErrMsg.form("Mismatch in port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
				"constant mapping only allowed for 'in' ports",
				PortNum);
			 return(SIR_ERROR_PORT_MAPPING_MISMATCH_8);
			} /* fi */
		    if (  (BitSlice->Constant->Type == SIR_CONST_BIT)
			||(BitSlice->Constant->Type == SIR_CONST_UBIT))
		       { BusWidth += BitSlice->Constant->BIT_Value->length();
			} /* fi */
		    else
		       { SIR_ErrMsg.form("Mismatch in port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
				"bitvector type expected",
				PortNum);
			 return(SIR_ERROR_PORT_MAPPING_MISMATCH_5);
			} /* esle */
		   } /* fi */
	       else
		  { /* open connection */
		    assert(BitSlice->Succ() == NULL); /* grammar assures this */
		    assert(BitSlice->Pred() == NULL);
		    if (Port->Type->Direction != SIR_PORT_OUT)
		       { SIR_ErrMsg.form("Mismatch in port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
				"open mapping only allowed for 'out' ports",
				PortNum);
			 return(SIR_ERROR_PORT_MAPPING_MISMATCH_9);
			} /* fi */
		    return(SIR_ERROR_NO_ERROR);	/* OK, done */
		   } /* esle */
	      } /* esle */
	  BitSlice = BitSlice->Succ();
	 } /* elihw */
     if (ABS(Port->Type->LeftBound - Port->Type->RightBound) + 1 != BusWidth)
	{ SIR_ErrMsg.form("Mismatch in port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
			"bus width (%d) does not match port width (%d)",
			PortNum, BusWidth,
			ABS(Port->Type->LeftBound - Port->Type->RightBound) +1);
	  return(SIR_ERROR_PORT_MAPPING_MISMATCH_4);
	 } /* fi */
     if (  (SourceFile)
	 &&(BitSlices->NumElements() == 1)
	 &&(BitSlices->First()->Symbol)
	 &&(BitSlices->First()->Symbol->Type->Type != Port->Type->Type))
	{ if (Port->Type->Type == SIR_TYPE_BIT)
	     { assert(BitSlices->First()->Symbol->Type->Type == SIR_TYPE_UBIT);
	       GL_PrintWarningFmt(GL_WARN_INFORMATIVE,
			"In file '%s', line %u," GL_WARN_MSG_NEWLINE
			"signed bitvector port %u mapped to unsigned type",
			SourceFile, SourceLine, PortNum);
	      } /* fi */
	  else
	     { assert(BitSlices->First()->Symbol->Type->Type == SIR_TYPE_BIT);
	       GL_PrintWarningFmt(GL_WARN_INFORMATIVE,
			"In file '%s', line %u," GL_WARN_MSG_NEWLINE
			"unsigned bitvector port %u mapped to signed type",
			SourceFile, SourceLine, PortNum);
	      } /* esle */
	 } /* fi */
     if (  (SourceFile)
	 &&(BitSlices->NumElements() == 1)
	 &&(BitSlices->First()->Constant))
	{ if (  (Port->Type->Type == SIR_TYPE_BIT)
	      &&(BitSlices->First()->Constant->Type == SIR_CONST_UBIT))
	     { GL_PrintWarningFmt(GL_WARN_INFORMATIVE,
			"In file '%s', line %u," GL_WARN_MSG_NEWLINE
			"signed bitvector port %u mapped to unsigned constant",
			SourceFile, SourceLine, PortNum);
	      } /* fi */
	  if (  (Port->Type->Type == SIR_TYPE_UBIT)
	      &&(BitSlices->First()->Constant->Type == SIR_CONST_BIT))
	     { GL_PrintWarningFmt(GL_WARN_INFORMATIVE,
			"In file '%s', line %u," GL_WARN_MSG_NEWLINE
			"unsigned bitvector port %u mapped to signed constant",
			SourceFile, SourceLine, PortNum);
	      } /* fi */
	 } /* fi */
    } /* fi */
else
   { if (BitSlices->NumElements() > 1)
	{ SIR_ErrMsg.form("Illegal port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
			"non-bitvector port prohibits concatenated mapping",
			PortNum);
	  return(SIR_ERROR_PORT_MAPPING_MISMATCH_3);
	 } /* fi */
     BitSlice = BitSlices->First();
     assert(BitSlice != NULL);
     if (BitSlice->Symbol)
	{ PortMapType = BitSlice->Symbol->Type;
	  if (  (  (Port->Type->Direction == SIR_PORT_IN)
		 &&(PortMapType->Direction == SIR_PORT_OUT))
	      ||(  (Port->Type->Direction == SIR_PORT_INOUT)
		 &&(  (PortMapType->Direction == SIR_PORT_IN)
		    ||(PortMapType->Direction == SIR_PORT_OUT)))
	      ||(  (Port->Type->Direction == SIR_PORT_OUT)
		 &&(PortMapType->Direction == SIR_PORT_IN)))
	     { SIR_ErrMsg.form("Mismatch in port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
				"instance '%s' port mapped to class '%s' port",
			PortNum,
			SIR_Type::PrintDirection(Port->Type->Direction),
			SIR_Type::PrintDirection(PortMapType->Direction));
	       return(SIR_ERROR_PORT_MAPPING_MISMATCH_10);
	      } /* fi */
	 } /* fi */
     else
	{ if (BitSlice->Constant)
	     { if (Port->Type->Direction != SIR_PORT_IN)
		  { SIR_ErrMsg.form("Mismatch in port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
				"constant mapping only allowed for 'in' ports",
				PortNum);
		    return(SIR_ERROR_PORT_MAPPING_MISMATCH_8);
		   } /* fi */
	       PortMapType = Port->Type->GetTable()->FindOrInsert(
							BitSlice->Constant);
	       if (! Port->Type->Const)	/* allow non-const ports be */
		  { PortMapType =	/* mapped onto constants    */
			PortMapType->Modified(false);	/* no const flag! */
		   } /* fi */
	      } /* fi */
	  else
	     { /* open connection */
	       if (Port->Type->Direction != SIR_PORT_OUT)
		  { SIR_ErrMsg.form("Mismatch in port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
				"open mapping only allowed for 'out' ports",
				PortNum);
		    return(SIR_ERROR_PORT_MAPPING_MISMATCH_9);
		   } /* fi */
	       if (Port->Type->Type == SIR_TYPE_INTERFACE)
		  { SIR_ErrMsg.form("Illegal port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
				"port of interface type cannot be left open",
				PortNum);
		    return(SIR_ERROR_PORT_MAPPING_MISMATCH_7);
		   } /* fi */
	       return(SIR_ERROR_NO_ERROR);	/* OK, done */
	      } /* esle */
	 } /* esle */
     assert(PortMapType != NULL);
     if (0 != SIR_Types::TypeCmp(		/* exact type matching! */
				PortMapType, Port->Type,
				true))	/* (ignore direction) */
	{ if (  (Port->Type->Type != SIR_TYPE_INTERFACE)
	      ||(PortMapType->Type != SIR_TYPE_CHANNEL)
	/* note: channels can be mapped to interface ports, behaviors cannot! */
	      ||(! PortMapType->ClassSymbol->Interfaces->Find(
						Port->Type->ClassSymbol)))
	     { SIR_ErrMsg.form("Type mismatch in port mapping for port %u",
				PortNum);
	       return(SIR_ERROR_PORT_MAPPING_MISMATCH_2);
	      } /* fi */
	 } /* fi */
     if (  (BitSlice->Symbol)
	 &&(BitSlice->Symbol->StorageClass == SIR_STORAGE_PIPED)
	 &&(  (Port->Type->Direction != SIR_PORT_IN)
	    &&(Port->Type->Direction != SIR_PORT_OUT)))
	{ SIR_ErrMsg.form("Illegal port mapping for port %u:"
							GL_ERROR_MSG_NEWLINE
				"mapping to piped variable requires "
					"port direction 'in' or 'out'",
				PortNum);
	  return(SIR_ERROR_PORT_MAPPING_MISMATCH_6);
	 } /* fi */
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_PortMap::Check */


	/********************/
	/*** SIR_PortMaps ***/
	/********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_PortMaps::SIR_PortMaps(		/* constructor #1 */
	sir_portmap	*FirstEntry = NULL) :
		SIR_List<SIR_PortMap>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_PortMaps::SIR_PortMaps #1 */


SIR_PortMaps::SIR_PortMaps(		/* constructor #2 (duplicator) */
	sir_portmaps	*Original)
{
sir_portmap	*Map;

Map = Original->First();
while(Map)
   { Append(new SIR_PortMap(Map));
     Map = Map->Succ();
    } /* elihw */

} /* end of SIR_PortMaps::SIR_PortMaps #2 */


SIR_PortMaps::~SIR_PortMaps(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_PortMaps::~SIR_PortMaps */


ERROR SIR_PortMaps::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_portmap	*PortMap,
		*Succ;

/* this is not a node, but there are nodes below */

PortMap = First();
while(PortMap)
   { Succ = PortMap->Succ();
     if ((SIR_Error = PortMap->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     PortMap = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_PortMaps::DFS_ForAllNodes */


void SIR_PortMaps::UnAlias(void) /* unalias all type, usertype, symbol lks. */
{
sir_portmap	*PortMap;

PortMap = First();
while(PortMap)
   { PortMap->UnAlias();
     PortMap = PortMap->Succ();
    } /* elihw */

} /* end of SIR_PortMaps::UnAlias */


ERROR SIR_PortMaps::Check(	/* semantically check the port mappings */
	sir_type_ptrs	*PortList,
	const char	*SourceFile = NULL,	/* default: no warnings */
	unsigned int	SourceLine = 0)
{
sir_portmap	*PortMap;
sir_type_ptr	*Port;
unsigned int	PortNum;

assert(PortList != NULL);

PortNum = PortList->NumElements();
if (  (PortNum == 1)
    &&(PortList->First()->Type->Type == SIR_TYPE_VOID))
   { PortNum = 0;
    } /* fi */
if (NumElements() != PortNum)
   { SIR_ErrMsg.form("Mismatch in port mapping: %d mappings for %d ports",
			this->NumElements(), PortNum);
     return(SIR_ERROR_PORT_MAPPING_MISMATCH_1);
    } /* fi */

PortMap = First();
Port = PortList->First();
PortNum = 1;
while(PortMap)
   { assert(Port != NULL);
     if ((SIR_Error = PortMap->Check(Port, PortNum, SourceFile, SourceLine)))
	{ return(SIR_Error);
	 } /* fi */
     Port = Port->Succ();
     PortMap = PortMap->Succ();
     PortNum++;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_PortMaps::Check */


ERROR SIR_PortMaps::WriteSC(		/* (re-) generates SpecC source code */
	FILE		*File)
{
sir_portmap	*PortMap;
sir_bitslice	*BitSlice;

assert(this != NULL);	/* always exists */

if (! Empty())
   { fputc('(', File);
     PortMap = First();
     while(PortMap)
	{ assert(PortMap->BitSlices != NULL);
	  BitSlice = PortMap->BitSlices->First();
	  while(BitSlice)
	     { if (BitSlice->Symbol)
		  { fputs(BitSlice->Symbol->Name.chars(), File);
		    if (BitSlice->Symbol->Type->TypeClass()
					== SIR_TYPECLASS_BITVECTOR)
		       { fprintf(File, "[%d:%d]", BitSlice->LeftBound,
						BitSlice->RightBound);
			} /* fi */
		   } /* fi */
	       else
		  { if (BitSlice->Constant)
		       { fputs("(", File);
			 fputs(BitSlice->Constant->Print(), File);
			 fputs(")", File);
			} /* fi */
		    else
		       { fputs("/* open */", File);
			} /* fi */
		   } /* esle */
	       BitSlice = BitSlice->Succ();
	       if (BitSlice)
		  { fputs(" @ ", File);
		   } /* fi */
	      } /* elihw */
	  PortMap = PortMap->Succ();
	  if (PortMap)
	     { fputs(", ", File);
	      } /* fi */
	 } /* elihw */
     fputc(')', File);
    } /* fi */

if (ferror(File))
   { SIR_ErrMsg.form("Writing to file failed%s",
			GL_SystemErrorMessage());
     return(SIR_ERROR_WRITING_SPECC_FILE);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_PortMaps::WriteSC */


	/********************/
	/*** SIR_BitSlice ***/
	/********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_BitSlice::SIR_BitSlice(		/* constructor #1 (standard) */
	sir_symbol	*Symbol,	/* NULL for open connection */
	int		LeftBound = 0,
	int		RightBound = 0)
{

SIR_BitSlice::Symbol		= Symbol;
SIR_BitSlice::Constant		= NULL;
SIR_BitSlice::LeftBound		= LeftBound;
SIR_BitSlice::RightBound	= RightBound;

} /* end of SIR_BitSlice::SIR_BitSlice #1 */


SIR_BitSlice::SIR_BitSlice(		/* constructor #2 (constant) */
	sir_constant	*Constant)
{

SIR_BitSlice::Symbol		= NULL;
SIR_BitSlice::Constant		= Constant;
SIR_BitSlice::LeftBound		= 0;
SIR_BitSlice::RightBound	= 0;

} /* end of SIR_BitSlice::SIR_BitSlice #2 */


SIR_BitSlice::SIR_BitSlice(		/* constructor #4 (duplicator) */
	sir_bitslice	*Original)
{

SIR_BitSlice::Symbol		= Original->Symbol;
SIR_BitSlice::Constant		= Original->Constant ?
					new SIR_Constant(
						Original->Constant) :
					NULL;
SIR_BitSlice::LeftBound		= Original->LeftBound;
SIR_BitSlice::RightBound	= Original->RightBound;

} /* end of SIR_BitSlice::SIR_BitSlice #4 */


SIR_BitSlice::~SIR_BitSlice(void)		/* destructor */
{

delete Constant;

} /* end of SIR_BitSlice::~SIR_BitSlice */


sir_bitslice *SIR_BitSlice::New(	/* create a new bitslice #1b */
	sir_symbol	*Mapping,	/* (returns NULL if SIR_Error) */
	int		LeftBound = 0,	/* bounds for bitvectors */
	int		RightBound = 0)
{
int		Left,
		Right;

/* Mapping may be NULL indicating open connection */

if (  (Mapping)
    &&(Mapping->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR))
   { Left = Mapping->Type->LeftBound;
     Right = Mapping->Type->RightBound;
     if (  (  (Left > Right)
	    &&(  (LeftBound > Left)  || (LeftBound < Right)
	       ||(RightBound > Left) || (RightBound < Right)))
         ||(  (Left <= Right)
	    &&(  (LeftBound < Left)  || (LeftBound > Right)
	       ||(RightBound < Left) || (RightBound > Right))))
	{ SIR_ErrMsg.form("Bitslice [%d:%d] out of range for %s[%d:%d]",
				LeftBound, RightBound,
				Mapping->Name.chars(), Left, Right);
	  SIR_Error = SIR_ERROR_BITSLICE_OUT_OF_RANGE;
	  return(NULL);
	 } /* fi */
    } /* fi */

return(new SIR_BitSlice(Mapping, LeftBound, RightBound));

} /* end of SIR_BitSlice::New #1b */


ERROR SIR_BitSlice::DFS_ForAllNodes(	/* iterator for over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

/* this is not a node, but there are nodes below */

if (Constant)
   { if ((SIR_Error = Constant->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_BitSlice::DFS_ForAllNodes */


	/*********************/
	/*** SIR_BitSlices ***/
	/*********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_BitSlices::SIR_BitSlices(		/* constructor #1 */
	sir_bitslice	*FirstEntry = NULL) :
		SIR_List<SIR_BitSlice>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_BitSlices::SIR_BitSlices #1 */


SIR_BitSlices::SIR_BitSlices(		/* constructor #2 (duplicator) */
	sir_bitslices	*Original)
{
sir_bitslice	*Slice;

Slice = Original->First();
while(Slice)
   { Append(new SIR_BitSlice(Slice));
     Slice = Slice->Succ();
    } /* elihw */

} /* end of SIR_BitSlices::SIR_BitSlices #2 */


SIR_BitSlices::~SIR_BitSlices(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_BitSlices::~SIR_BitSlices */


ERROR SIR_BitSlices::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_bitslice	*Slice,
		*Succ;

/* this is not a node, but there are nodes below */

Slice = First();
while(Slice)
   { Succ = Slice->Succ();
     if ((SIR_Error = Slice->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Slice = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_BitSlices::DFS_ForAllNodes */


void SIR_BitSlices::UnAlias(void) /* unalias all type, usertype, symbol lks. */
{
sir_bitslice	*BitSlice;

BitSlice = First();
while(BitSlice)
   { if (  (BitSlice->Symbol)
	 &&(BitSlice->Symbol->Alias))
	{ BitSlice->Symbol = BitSlice->Symbol->Alias;
	 } /* fi */
     BitSlice = BitSlice->Succ();
    } /* elihw */

} /* end of SIR_BitSlices::UnAlias */


/*** exported functions *************************************************/


	/* none */


/* EOF PortMap.cc */
