/************************************************************************/
/* Compiler.c: The SpecC Compiler					*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 08/31/97 */
/************************************************************************/

/* last update: 05/25/01 */

/* modifications: (most recent first)
 *
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	eliminated support for automatic IP wrapping (scc -ip option)
 * 05/25/01 RD	first version adjusted for SCRC
 */


#include "Compiler.h"
#include "CcDriver.h"
#include "IntRep.h"
#include "Parser.h"

#include <stdlib.h>
#include <strings.h>


/*** constants and macros ***********************************************/


#ifdef DEBUG
#ifndef SCC_DEBUG
#define SCC_DEBUG
#endif /* SCC_DEBUG */
#endif /* DEBUG */


#define SCC_SPARE_MEMORY_SIZE	256*1024	/* 256 KB */


/*** internal type declarations *****************************************/


	/* none */


/*** prototypes of internal functions ***********************************/


static const char *SCC_ErrorText(	/* returns message of error code */
	ERROR		ErrorNumber);

static void PrintHeader(		/* print program header */
	FILE		*File);

static void PrintUsage(			/* print usage info */
	FILE		*File);

static ERROR GetArguments(		/* parse command line arguments */
	int		argc,
	char		**argv);

static ERROR GetCommand(		/* parse the command argument */
	char		*SuffixCmd);


/*** exported variables *************************************************/


	/* none */


/*** internal variables *************************************************/


static ERROR	SCC_Error = SCC_ERROR_NO_ERROR;	/* last error of this module */
static string	ErrorMessage;


	/*** command line options and derivations (and their defaults) ***/

static string		DesignName;	/* no default */

static string		FilenameSCin;	/* these will be build */
static string		FilenameSCout;	/* based on DesignName */
static string		FilenameSI;
static string		FilenameCC;
static string		FilenameH;
static string		FilenameO;
static string		FilenameOUT;

static BOOL		CallPreprocessor	= TRUE;	/* based on command */
static BOOL		CallParser		= TRUE;
static BOOL		CallDeparser		= FALSE;
static BOOL		CallTranslator		= TRUE;
static BOOL		CallCompiler		= TRUE;
static BOOL		CallLinker		= TRUE;

static string		*InputFilename  = &FilenameSCin;/* based on command */
static string		*OutputFilename = &FilenameOUT;

static GL_STRINGLIST	*PDirList = NULL,
			*PDirListInsert = NULL;
static GL_STRINGLIST	*DDefList = NULL,
			*DDefListLast = NULL;
static GL_STRINGLIST	*UDefList = NULL,
			*UDefListLast = NULL;
static GL_STRINGLIST	*IDirList = NULL,
			*IDirListLast = NULL;
static GL_STRINGLIST	*LDirList = NULL,
			*LDirListLast = NULL;
static GL_STRINGLIST	*lLibList = NULL,
			*lLibListLast = NULL;

static BOOL		UsageInfoOnly	= FALSE;
static int		VerbosityLevel	= 0;
static int		WarningLevel	= 0;
static BOOL		DebugMode	= FALSE;
static BOOL		OptimizeMode	= FALSE;
static BOOL		GenerateLines	= TRUE;
static BOOL		GenerateNotes	= TRUE;

static BOOL		OptionIorOrequired = FALSE;
static BOOL		GenerateH	= TRUE;
static BOOL		GenerateCC	= TRUE;


	/*** the design ***/

sir_design		*TheDesign = NULL;


/************************************************************************/
/*** implementation of exported functions			      ***/
/************************************************************************/


	/* none */


/************************************************************************/
/*** implementation of internal functions			      ***/
/************************************************************************/


	/*** error handling ***/


static const char *SCC_ErrorText(	/* returns message of error code */
	ERROR	ErrorNumber)
{

if (  (ErrorNumber >= GL_ERROR_BASE)
    &&(ErrorNumber <= GL_ERROR_BASE + GL_ERROR_RANGE))
   { return(GL_ErrorText(ErrorNumber));
    } /* fi */
if (  (ErrorNumber >= CCD_ERROR_BASE)
    &&(ErrorNumber <= CCD_ERROR_BASE + CCD_ERROR_RANGE))
   { return(CCD_ErrorText(ErrorNumber));
    } /* fi */
if (  (ErrorNumber >= SIR_ERROR_BASE)
    &&(ErrorNumber <= SIR_ERROR_BASE + SIR_ERROR_RANGE))
   { return(SIR_ErrorText(ErrorNumber));
    } /* fi */
if (  (ErrorNumber >= PRS_ERROR_BASE)
    &&(ErrorNumber <= PRS_ERROR_BASE + PRS_ERROR_RANGE))
   { return(PRS_ErrorText(ErrorNumber));
    } /* fi */

switch (ErrorNumber)
   { case SCC_ERROR_NO_ERROR:
	  return("No error (internal)");
     case SCC_ERROR_NO_DESIGN_NAME:
	  return("Bad arguments: design name missing");
     case SCC_ERROR_UNKNOWN_COMMAND:
	  return(ErrorMessage.chars()
		/* "Unknown command -%s" */);
     case SCC_ERROR_BAD_DESIGN_NAME:
	  return(ErrorMessage.chars()
		/* "'%s' is not a legal design name" */);
     case SCC_ERROR_BAD_ARGUMENT:
	  return(ErrorMessage.chars()
		/* "Bad argument '%s'" */);
     case SCC_ERROR_UNKNOWN_OPTION:
	  return(ErrorMessage.chars()
		/* "Unknown option %s" */);
     case SCC_ERROR_OPTION_NEEDS_ARGUMENT:
	  return(ErrorMessage.chars()
		/* "Option %s requires an argument" */);
     case SCC_ERROR_EMPTY_ARGUMENT:
	  return(ErrorMessage.chars()
		/* "Empty argument for option -%s" */);
     case SCC_ERROR_OUTPUT_FILENAME_AMBIGUOUS:
	  return("Specification of output filename is ambiguous");
     case SCC_ERROR_OPTION_IO_REQUIRED:
	  return("Explicit input or output filename required"
							GL_ERROR_MSG_NEWLINE
		"(use option -i or -o to avoid overriding the input file)");
    } /* hctiws */

return("Unknown error number (internal)");

} /* end of SCC_ErrorText */


	/*** header and usage info ***/


static void PrintHeader(		/* print program header */
	FILE		*File)
{

fputs(	"scrc: SpecC Reference Compiler V " GL_SPECC_VERSION_STRING "\n"
	GL_SPECC_COPYRIGHT_STRING "\n\n",
		File);

} /* end of PrintHeader */


static void PrintUsage(			/* print usage info */
	FILE		*File)
{

fputs(
"Usage: scrc <design> [<command>] [<option>...]\n"
"   or: scrc -h  (print version and usage info and quit)\n"
"\n"
"Command:\n"
"-<suffix>2<suffix>  where <suffix> = sc|si|cc|h|cch|o|out\n"
"                    specifies the compilation steps to be performed;\n"
"e.g. -sc2out        compiles SpecC source code into an executable file\n"
"                    (this is the default command)\n"
"\n"
"Options:         Description:                   Default:\n"
"-v|-vv|-vvv      increase verbosity level       silent\n"
"-w|-ww|-www      increase warning level         no warnings\n"
"-g|-O            enable debugging/optimization  none\n"
"-sl              suppress gen. of line infos    enabled\n"
"-sn              suppress gen. of annotations   enabled\n"
"-i <infile|->    specify input filename (%i)    <design>.<suffix>\n"
"-o <outfile|->   specify output filename (%o)   <design>.<suffix>\n"
"-D               do not define standard macros  " CCD_USAGE_DEFINES "\n"
"-D<macrodef>     define a preprocessor macro    none\n"
"-U               do not use standard undefines  " CCD_USAGE_UNDEFINES "\n"
"-U<undef>        undefine a preprocessor macro  none\n"
"-I               clear standard include path    " CCD_USAGE_STD_INCLUDE "\n"
"-I<dir>          insert dir in include path     none\n"
"-L               clear standard library path    " CCD_USAGE_STD_LIBRARY "\n"
"-L<dir>          insert dir in library path     none\n"
"-l               do not use standard libraries  " CCD_USAGE_STD_LIBRARIES "\n"
"-l<lib>          link against library lib       none\n"
"-P               clear standard import path     <current dir>\n"
"-P<dir>          insert dir in import path      none\n"
"-xpp <pp call>   override preprocessor call     " CCD_USAGE_PP_CALL "\n"
"-xcc <cc call>   override compiler call         " CCD_USAGE_CC_CALL "\n"
"-xld <ld call>   override linker call           " CCD_USAGE_LD_CALL "\n"
"-xp <pp option>  pass preprocessor opt. (%p)    " CCD_USAGE_PP_OPTS "\n"
"-xc <cc option>  pass compiler option (%c)      " CCD_USAGE_CC_OPTS "\n"
"-xl <ld option>  pass linker option (%l)        " CCD_USAGE_LD_OPTS "\n"
	"\n", File);

} /* end of PrintUsage */


	/*** command line argument handling ***/


static ERROR GetArguments(		/* parse command line arguments */
	int	argc,
	char	**argv)
{
GL_STRINGLIST	*Succ,
		*TmpDirList;
ERROR		Error;
int		Arg;

if (argc < 2)
   { return(SCC_ERROR_NO_DESIGN_NAME);
    } /* fi */

if (  (argc == 2)
    &&(0 == strcmp("-h", argv[1])))
   { UsageInfoOnly = TRUE;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */

Arg = 1;
DesignName = argv[Arg];
if (GL_CheckIdentifier(DesignName))
   { GL_Reset();
     ErrorMessage.form("'%s' is not a legal design name", argv[Arg]);
     return(SCC_ERROR_BAD_DESIGN_NAME);
    } /* elihw */

FilenameSCin	= DesignName + ".sc";	/* default file names */
FilenameSCout	= DesignName + ".sc";
FilenameSI	= DesignName + ".si";
FilenameCC	= DesignName + ".cc";
FilenameH	= DesignName + ".h";
FilenameO	= DesignName + ".o";
#ifdef GNUWIN32
FilenameOUT	= DesignName + ".exe";
#else /* not GNUWIN32 */
FilenameOUT	= DesignName;
#endif /* GNUWIN32 */

assert(PDirList == NULL);		/* default import path */
assert(PDirListInsert == NULL);
PDirList = new GL_STRINGLIST();
PDirList->Succ = NULL;
PDirList->String = ".";

Arg++;
if (Arg >= argc)
   { return(SCC_ERROR_NO_ERROR);	/* no more arguments */
    } /* fi */

if (argv[Arg][0] == '-')
   { if (index(&argv[Arg][1], '2'))	/* is this a command option? */
	{ if ((Error = GetCommand(&argv[Arg][1])))
	     { return(Error);
	      } /* fi */
	  Arg++;
	 } /* fi */
    } /* fi */

while(Arg < argc)			/* process all options */
   { if (argv[Arg][0] != '-')
	{ ErrorMessage.form("Bad argument '%s'", argv[Arg]);
	  return(SCC_ERROR_BAD_ARGUMENT);
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "v"))
	{ VerbosityLevel++;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "vv"))
	{ VerbosityLevel += 2;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "vvv"))
	{ VerbosityLevel += 3;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "w"))
	{ WarningLevel++;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "ww"))
	{ WarningLevel += 2;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "www"))
	{ WarningLevel += 3;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "g"))
	{ DebugMode = TRUE;
	  if (OptimizeMode)
	     { GL_PrintWarning(GL_WARN_STANDARD,
				"Option -g disables option -O");
	       OptimizeMode = FALSE;
	      } /* fi */
	  if ((Error = CCD_SetCompilerMode(CCD_MODE_DEBUG)))
	     { return(Error);
	      } /* fi */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "O"))
	{ OptimizeMode = TRUE;
	  if (DebugMode)
	     { GL_PrintWarning(GL_WARN_STANDARD,
				"Option -O disables option -g");
	       DebugMode = FALSE;
	      } /* fi */
	  if ((Error = CCD_SetCompilerMode(CCD_MODE_OPTIMIZE)))
	     { return(Error);
	      } /* fi */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "sl"))
	{ GenerateLines = FALSE;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "sn"))
	{ GenerateNotes = FALSE;
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "i"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -i requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (argv[Arg][0] == 0)
	     { ErrorMessage = "Empty argument for option -i";
	       return(SCC_ERROR_EMPTY_ARGUMENT);
	      } /* fi */
	  *InputFilename = argv[Arg++];
	  if ((SCC_Error = GL_CheckPathFilename(*InputFilename)))
	     { return(SCC_Error);
	      } /* fi */
	  if (OptionIorOrequired)
	     { OptionIorOrequired = FALSE;
	      } /* fi */
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "o"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -o requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! OutputFilename)
	     { return(SCC_ERROR_OUTPUT_FILENAME_AMBIGUOUS);
	      } /* fi */
	  if (argv[Arg][0] == 0)
	     { ErrorMessage = "Empty argument for option -o";
	       return(SCC_ERROR_EMPTY_ARGUMENT);
	      } /* fi */
	  *OutputFilename = argv[Arg++];
	  if ((SCC_Error = GL_CheckPathFilename(*OutputFilename)))
	     { return(SCC_Error);
	      } /* fi */
	  if (OptionIorOrequired)
	     { OptionIorOrequired = FALSE;
	      } /* fi */
	  continue;
	 } /* fi */
     if (argv[Arg][1] == 'D')
	{ if (argv[Arg][2] == 0)
	     { if ((SCC_Error = CCD_NoStdMacroDefs()))
		  { return(SCC_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (DDefListLast)
		  { DDefListLast->Succ = new GL_STRINGLIST();
		    DDefListLast = DDefListLast->Succ;
		   } /* fi */
	       else
		  { assert(DDefList == NULL);
		    DDefList = new GL_STRINGLIST();
		    DDefListLast = DDefList;
		   } /* esle */
	       DDefListLast->Succ = NULL;
	       DDefListLast->String = &argv[Arg][2];
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (argv[Arg][1] == 'U')
	{ if (argv[Arg][2] == 0)
	     { if ((SCC_Error = CCD_NoStdUnDefs()))
		  { return(SCC_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (UDefListLast)
		  { UDefListLast->Succ = new GL_STRINGLIST();
		    UDefListLast = UDefListLast->Succ;
		   } /* fi */
	       else
		  { assert(UDefList == NULL);
		    UDefList = new GL_STRINGLIST();
		    UDefListLast = UDefList;
		   } /* esle */
	       UDefListLast->Succ = NULL;
	       UDefListLast->String = &argv[Arg][2];
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (argv[Arg][1] == 'I')
	{ if (argv[Arg][2] == 0)
	     { if ((SCC_Error = CCD_NoStdIncludePath()))
		  { return(SCC_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (IDirListLast)
		  { IDirListLast->Succ = new GL_STRINGLIST();
		    IDirListLast = IDirListLast->Succ;
		   } /* fi */
	       else
		  { assert(IDirList == NULL);
		    IDirList = new GL_STRINGLIST();
		    IDirListLast = IDirList;
		   } /* esle */
	       IDirListLast->Succ = NULL;
	       IDirListLast->String = &argv[Arg][2];
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (argv[Arg][1] == 'L')
	{ if (argv[Arg][2] == 0)
	     { if ((SCC_Error = CCD_NoStdLibraryPath()))
		  { return(SCC_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (LDirListLast)
		  { LDirListLast->Succ = new GL_STRINGLIST();
		    LDirListLast = LDirListLast->Succ;
		   } /* fi */
	       else
		  { assert(LDirList == NULL);
		    LDirList = new GL_STRINGLIST();
		    LDirListLast = LDirList;
		   } /* esle */
	       LDirListLast->Succ = NULL;
	       LDirListLast->String = &argv[Arg][2];
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (argv[Arg][1] == 'l')
	{ if (argv[Arg][2] == 0)
	     { if ((SCC_Error = CCD_NoStdLibraries()))
		  { return(SCC_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (lLibListLast)
		  { lLibListLast->Succ = new GL_STRINGLIST();
		    lLibListLast = lLibListLast->Succ;
		   } /* fi */
	       else
		  { assert(lLibList == NULL);
		    lLibList = new GL_STRINGLIST();
		    lLibListLast = lLibList;
		   } /* esle */
	       lLibListLast->Succ = NULL;
	       lLibListLast->String = &argv[Arg][2];
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (argv[Arg][1] == 'P')
	{ if (argv[Arg][2] == 0)
	     { while(PDirList)
		  { Succ = PDirList->Succ;
		    delete PDirList;
		    PDirList = Succ;
		   } /* eliwh */
	       PDirList = NULL;	/* empty import dir list */
	       PDirListInsert = NULL;
	      } /* fi */
	  else
	     { TmpDirList = new GL_STRINGLIST();
	       TmpDirList->Succ = NULL;
	       TmpDirList->String = &argv[Arg][2];
	       if (PDirList)
		  { if (PDirListInsert)
		       { TmpDirList->Succ = PDirListInsert->Succ;
			 PDirListInsert->Succ = TmpDirList;
			 PDirListInsert = TmpDirList;
			} /* fi */
		    else
		       { TmpDirList->Succ = PDirList;
			 PDirList = TmpDirList;
			 PDirListInsert = TmpDirList;
			} /* esle */
		   } /* fi */
	       else
		  { PDirList = TmpDirList;
		    assert(PDirListInsert == NULL);
		   } /* esle */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "xpp"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -xpp requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! CallPreprocessor)
	     { GL_PrintWarning(GL_WARN_STANDARD,
			"Option -xpp ignored because preprocessing not done");
	      } /* fi */
	  else
	     { if (argv[Arg][0] == 0)
		  { ErrorMessage = "Empty argument for option -xpp";
		    return(SCC_ERROR_EMPTY_ARGUMENT);
		   } /* fi */
	       if ((Error = CCD_SetPreprocessorCall(argv[Arg])))
		  { return(Error);
		   } /* fi */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "xcc"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -xcc requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! CallCompiler)
	     { GL_PrintWarning(GL_WARN_STANDARD,
			"Option -xcc ignored because compiling not done");
	      } /* fi */
	  else
	     { if (argv[Arg][0] == 0)
		  { ErrorMessage = "Empty argument for option -xcc";
		    return(SCC_ERROR_EMPTY_ARGUMENT);
		   } /* fi */
	       if ((Error = CCD_SetCompilerCall(argv[Arg])))
		  { return(Error);
		   } /* fi */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "xld"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -xld requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! CallLinker)
	     { GL_PrintWarning(GL_WARN_STANDARD,
			"Option -xld ignored because linking not done");
	      } /* fi */
	  else
	     { if (argv[Arg][0] == 0)
		  { ErrorMessage = "Empty argument for option -xld";
		    return(SCC_ERROR_EMPTY_ARGUMENT);
		   } /* fi */
	       if ((Error = CCD_SetLinkerCall(argv[Arg])))
		  { return(Error);
		   } /* fi */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "xp"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -xp requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! CallPreprocessor)
	     { GL_PrintWarning(GL_WARN_STANDARD,
			"Option -xp ignored because preprocessing not done");
	      } /* fi */
	  else
	     { if ((Error = CCD_SetPreprocessorOption(argv[Arg])))
		  { return(Error);
		   } /* fi */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "xc"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -xc requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! CallCompiler)
	     { GL_PrintWarning(GL_WARN_STANDARD,
			"Option -xc ignored because compiling not done");
	      } /* fi */
	  else
	     { if ((Error = CCD_SetCompilerOption(argv[Arg])))
		  { return(Error);
		   } /* fi */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */
     if (0 == strcmp(&argv[Arg][1], "xl"))
	{ if (++Arg >= argc)
	     { ErrorMessage = "Option -xl requires an argument";
	       return(SCC_ERROR_OPTION_NEEDS_ARGUMENT);
	      } /* fi */
	  if (! CallLinker)
	     { GL_PrintWarning(GL_WARN_STANDARD,
			"Option -xl ignored because linking not done");
	      } /* fi */
	  else
	     { if ((Error = CCD_SetLinkerOption(argv[Arg])))
		  { return(Error);
		   } /* fi */
	      } /* esle */
	  Arg++;
	  continue;
	 } /* fi */

     ErrorMessage.form("Unknown option %s", argv[Arg]);
     return(SCC_ERROR_UNKNOWN_OPTION);
    } /* elihw */

if (OptionIorOrequired)
   { return(SCC_ERROR_OPTION_IO_REQUIRED);
    } /* fi */

if ((Error = CCD_SetMacroDefs(DDefList)))	/* set lists in driver */
   { return(Error);
    } /* fi */
if ((Error = CCD_SetUnDefs(UDefList)))
   { return(Error);
    } /* fi */
if ((Error = CCD_SetIncludePath(IDirList)))
   { return(Error);
    } /* fi */
if ((Error = CCD_SetLibraryPath(LDirList)))
   { return(Error);
    } /* fi */
if ((Error = CCD_SetLibraries(lLibList)))
   { return(Error);
    } /* fi */

return(SCC_ERROR_NO_ERROR);	/* all arguments are ok */

} /* end of GetArguments */


static ERROR GetCommand(		/* parse the command argument */
	char	*SuffixCmd)
{

	/* ok, first we assume doing nothing */
CallPreprocessor	= FALSE;
CallParser		= FALSE;
CallDeparser		= FALSE;
CallTranslator		= FALSE;
CallCompiler		= FALSE;
CallLinker		= FALSE;

if (0 == strcmp(SuffixCmd, "sc2si"))	/* start from a .sc file */
   { CallPreprocessor	= TRUE;
     InputFilename  = &FilenameSCin;
     OutputFilename = &FilenameSI;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "sc2cc"))
   { CallPreprocessor	= TRUE;
     CallParser		= TRUE;
     CallTranslator	= TRUE;
     InputFilename  = &FilenameSCin;
     OutputFilename = &FilenameCC;
     GenerateH		= FALSE;	/* no .h file */
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "sc2h"))
   { CallPreprocessor	= TRUE;
     CallParser		= TRUE;
     CallTranslator	= TRUE;
     InputFilename  = &FilenameSCin;
     OutputFilename = &FilenameH;
     GenerateCC		= FALSE;	/* no .cc file */
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "sc2cch"))
   { CallPreprocessor	= TRUE;
     CallParser		= TRUE;
     CallTranslator	= TRUE;
     InputFilename  = &FilenameSCin;
     OutputFilename = NULL;	/* no -o option allowed */
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "sc2o"))
   { CallPreprocessor	= TRUE;
     CallParser		= TRUE;
     CallTranslator	= TRUE;
     CallCompiler	= TRUE;
     InputFilename  = &FilenameSCin;
     OutputFilename = &FilenameO;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "sc2out"))
   { CallPreprocessor	= TRUE;
     CallParser		= TRUE;
     CallTranslator	= TRUE;
     CallCompiler	= TRUE;
     CallLinker		= TRUE;
     InputFilename  = &FilenameSCin;
     OutputFilename = &FilenameOUT;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "sc2sc"))
   { OptionIorOrequired = TRUE;
     CallPreprocessor	= TRUE;
     CallParser		= TRUE;
     CallDeparser	= TRUE;
     InputFilename  = &FilenameSCin;
     OutputFilename = &FilenameSCout;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */

if (0 == strcmp(SuffixCmd, "si2cc"))
   { CallParser		= TRUE;
     CallTranslator	= TRUE;
     InputFilename  = &FilenameSI;
     OutputFilename = &FilenameCC;
     GenerateH		= FALSE;	/* no .h file */
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "si2h"))
   { CallParser		= TRUE;
     CallTranslator	= TRUE;
     InputFilename  = &FilenameSI;
     OutputFilename = &FilenameH;
     GenerateCC		= FALSE;	/* no .cc file */
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "si2cch"))
   { CallParser		= TRUE;
     CallTranslator	= TRUE;
     InputFilename  = &FilenameSI;
     OutputFilename = NULL;	/* no -o option allowed */
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "si2o"))
   { CallParser		= TRUE;
     CallTranslator	= TRUE;
     CallCompiler	= TRUE;
     InputFilename  = &FilenameSI;
     OutputFilename = &FilenameO;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "si2out"))
   { CallParser		= TRUE;
     CallTranslator	= TRUE;
     CallCompiler	= TRUE;
     CallLinker		= TRUE;
     InputFilename  = &FilenameSI;
     OutputFilename = &FilenameOUT;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "si2sc"))
   { CallParser		= TRUE;
     CallDeparser	= TRUE;
     InputFilename  = &FilenameSI;
     OutputFilename = &FilenameSCout;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */

if (0 == strcmp(SuffixCmd, "cc2o"))	/* start from a .cc file */
   { CallCompiler	= TRUE;
     InputFilename  = &FilenameCC;
     OutputFilename = &FilenameO;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */
if (0 == strcmp(SuffixCmd, "cc2out"))
   { CallCompiler	= TRUE;
     CallLinker		= TRUE;
     InputFilename  = &FilenameCC;
     OutputFilename = &FilenameOUT;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */

if (0 == strcmp(SuffixCmd, "o2out"))	/* start from a .o file */
   { CallLinker		= TRUE;
     InputFilename  = &FilenameO;
     OutputFilename = &FilenameOUT;
     return(SCC_ERROR_NO_ERROR);
    } /* fi */

ErrorMessage = "Unknown command -";
ErrorMessage += SuffixCmd;
return(SCC_ERROR_UNKNOWN_COMMAND);

} /* end of GetCommand */


	/*** clean up ***/


void CleanUp(void)
{
GL_STRINGLIST	*Succ;

while(PDirList)			/* clean up the import path */
   { Succ = PDirList->Succ;
     delete PDirList;
     PDirList = Succ;
    } /* elihw */
PDirListInsert = NULL;

while(DDefList)			/* clean up the macro definitions */
   { Succ = DDefList->Succ;
     delete DDefList;
     DDefList = Succ;
    } /* elihw */
DDefListLast = NULL;

while(UDefList)			/* clean up the macro un-definitions */
   { Succ = UDefList->Succ;
     delete UDefList;
     UDefList = Succ;
    } /* elihw */
UDefListLast = NULL;

while(IDirList)			/* clean up the include path */
   { Succ = IDirList->Succ;
     delete IDirList;
     IDirList = Succ;
    } /* elihw */
IDirListLast = NULL;

while(LDirList)			/* clean up the library path */
   { Succ = LDirList->Succ;
     delete LDirList;
     LDirList = Succ;
    } /* elihw */
LDirListLast = NULL;

while(lLibList)			/* clean up the library list */
   { Succ = lLibList->Succ;
     delete lLibList;
     lLibList = Succ;
    } /* elihw */
lLibListLast = NULL;

delete TheDesign;		/* delete the design data structure */

CCD_SetDefaultOptions();	/* clean up compiler driver */

GL_FreeSpareMemory();

} /* end of CleanUp */


/************************************************************************/
/*** main							      ***/
/************************************************************************/


int main(int argc, char **argv)
{
int		i;
ERROR		Error;
FILE		*InFile,
		*OutFile;
#ifdef SCC_DEBUG
GL_STRINGLIST	*StringPtr;
#endif /* SCC_DEBUG */

	/* initialization */

CCD_SetDefaultOptions();	/* initialize CCD module options */

	/* parse the arguments */

SCC_Error = GetArguments(argc, argv);
if (UsageInfoOnly)
   { PrintHeader(stdout);
     PrintUsage(stdout);
     CleanUp();
     return(RESULT_OK);
    } /* fi */

if ((VerbosityLevel >= 1) || (SCC_Error))
   { PrintHeader(stderr);
    } /* fi */

if ((VerbosityLevel >= 3) || (SCC_Error))
   { fprintf(stderr, "      %s", argv[0]);
     for(i=1; i<argc; i++)
	{ if (index(argv[i], ' '))
	     { fprintf(stderr, " \"%s\"", argv[i]);
	      } /* fi */
	  else
	     { fprintf(stderr, " %s", argv[i]);
	      } /* esle */
	 } /* rof */
     fputs("\n\n", stderr);
    } /* fi */

#ifdef SCC_DEBUG	/* dump the evaluated arguments and options */

fprintf(stderr, "SCC_DEBUG: DesignName     = \"%s\"\n", DesignName.chars());
fprintf(stderr, "SCC_DEBUG: FilenameSCin   = \"%s\"\n", FilenameSCin.chars());
fprintf(stderr, "SCC_DEBUG: FilenameSCout  = \"%s\"\n", FilenameSCout.chars());
fprintf(stderr, "SCC_DEBUG: FilenameSI     = \"%s\"\n", FilenameSI.chars());
fprintf(stderr, "SCC_DEBUG: FilenameCC     = \"%s\"\n", FilenameCC.chars());
fprintf(stderr, "SCC_DEBUG: FilenameH      = \"%s\"\n", FilenameH.chars());
fprintf(stderr, "SCC_DEBUG: FilenameO      = \"%s\"\n", FilenameO.chars());
fprintf(stderr, "SCC_DEBUG: FilenameOUT    = \"%s\"\n", FilenameOUT.chars());
fprintf(stderr, "SCC_DEBUG: Tasks are %s%s%s%s%s%s\b\b.\n",
	(CallPreprocessor ? "Preprocessor, " : ""),
	(CallParser ? "Parser, " : ""),
	(CallDeparser ? "Deparser, " : ""),
	(CallTranslator ? "Translator, " : ""),
	(CallCompiler ? "Compiler, " : ""),
	(CallLinker ? "Linker, " : ""));
fprintf(stderr, "SCC_DEBUG: VerbosityLevel = %d\n", VerbosityLevel);
fprintf(stderr, "SCC_DEBUG: WarningLevel   = %d\n", WarningLevel);
fprintf(stderr, "SCC_DEBUG: DebugMode      = %d\n", DebugMode);
fprintf(stderr, "SCC_DEBUG: OptimizeMode   = %d\n", OptimizeMode);
fprintf(stderr, "SCC_DEBUG: GenerateLines  = %d\n", GenerateLines);
fprintf(stderr, "SCC_DEBUG: GenerateNotes  = %d\n", GenerateNotes);
fprintf(stderr, "SCC_DEBUG: GenerateH      = %d\n", GenerateH);
fprintf(stderr, "SCC_DEBUG: GenerateCC     = %d\n", GenerateCC);
fprintf(stderr, "SCC_DEBUG: ImportPath     = ");
StringPtr = PDirList;
while(StringPtr)
   { fprintf(stderr, "\"%s\"%s",
		StringPtr->String,
		(StringPtr->Succ ? ", " : ""));
     StringPtr = StringPtr->Succ;
    } /* elihw */
fputs("\n", stderr);
fprintf(stderr, "SCC_DEBUG: IncludePath    = ");
StringPtr = IDirList;
while(StringPtr)
   { fprintf(stderr, "\"%s\"%s",
		StringPtr->String,
		(StringPtr->Succ ? ", " : ""));
     StringPtr = StringPtr->Succ;
    } /* elihw */
fputs("\n", stderr);
fprintf(stderr, "SCC_DEBUG: LibraryPath    = ");
StringPtr = LDirList;
while(StringPtr)
   { fprintf(stderr, "\"%s\"%s",
		StringPtr->String,
		(StringPtr->Succ ? ", " : ""));
     StringPtr = StringPtr->Succ;
    } /* elihw */
fputs("\n", stderr);
fprintf(stderr, "SCC_DEBUG: LibraryList    = ");
StringPtr = lLibList;
while(StringPtr)
   { fprintf(stderr, "\"%s\"%s",
		StringPtr->String,
		(StringPtr->Succ ? ", " : ""));
     StringPtr = StringPtr->Succ;
    } /* elihw */
fputs("\n", stderr);
#endif /* SCC_DEBUG */

if (SCC_Error)
   { GL_PrintError(SCC_Error, SCC_ErrorText(SCC_Error));
     PrintUsage(stderr);
     fputs("Aborted.\n", stderr);
     CleanUp();
     return(RESULT_ERROR);
    } /* fi */

GL_VerbosityLevel = VerbosityLevel;	/* publish the VerbosityLevel */
GL_WarningLevel = WarningLevel;		/* and the WarningLevel */

	/* enable smart memory management */

if ((Error = GL_ReserveSpareMemory(SCC_SPARE_MEMORY_SIZE)))
   { GL_PrintError(Error, GL_ErrorText(Error));
     fputs("Aborted.\n", stderr);
     CleanUp();
     return(RESULT_ERROR);
    } /* fi */


	/* call the preprocessor */

if (CallPreprocessor)
   { if (VerbosityLevel >= 1)
	{ fputs("Preprocessing...\n", stderr);
	 } /* fi */
     if (VerbosityLevel >= 2)
	{ fprintf(stderr, "   Input:  \"%s\"\n", FilenameSCin.chars());
	  fprintf(stderr, "   Output: \"%s\"\n", FilenameSI.chars());
	 } /* fi */
     if ((Error = CCD_Preprocessor(FilenameSCin.chars(),
					FilenameSI.chars())))
	{ GL_PrintError(Error, CCD_ErrorText(Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
    } /* fi */

	/* call the parser */

if (CallParser)
   { if (VerbosityLevel >= 1)
	{ fputs("Parsing...\n", stderr);
	 } /* fi */
     if (VerbosityLevel >= 2)
	{ fprintf(stderr, "   Input:  \"%s\"\n", FilenameSI.chars());
	  fprintf(stderr, "   Output: <internal representation>\n");
	 } /* fi */
     if (!(InFile = GL_OpenFileForReading(FilenameSI.chars())))
	{ GL_PrintError(GL_Error, GL_ErrorText(GL_Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
     assert(TheDesign == NULL);
     if (!(TheDesign = PRS_ParseDesign(InFile, DesignName.chars(),
				FilenameSI.chars(), PDirList)))
	{ GL_PrintError(PRS_Error, PRS_ErrorText(PRS_Error));
	  fputs("Aborted.\n", stderr);
	  GL_CloseFile(InFile);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
     if ((Error = GL_CloseFileWithCheck(InFile)))
	{ GL_PrintError(Error, GL_ErrorText(Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
    } /* fi */

	/* call the translator */

if (CallTranslator)
   { if (VerbosityLevel >= 1)
	{ fputs("Translating...\n", stderr);
	 } /* fi */
     if (VerbosityLevel >= 2)
	{ fprintf(stderr, "   Input:  <internal representation>\n");
	 } /* fi */
     if (GenerateH)
	{ if (VerbosityLevel >= 2)
	     { fprintf(stderr, "   Output: \"%s\"\n", FilenameH.chars());
	      } /* fi */
	  if (!(OutFile = GL_OpenFileForWriting(FilenameH.chars())))
	     { GL_PrintError(GL_Error, GL_ErrorText(GL_Error));
	       fputs("Aborted.\n", stderr);
	       CleanUp();
	       return(RESULT_ERROR);
	      } /* fi */
	  assert(TheDesign != NULL);
	  if ((Error = TheDesign->WriteH(OutFile,
			FilenameH.chars(),
			GenerateNotes, GenerateLines)))
	     { GL_PrintError(Error, SIR_ErrorText(Error));
	       fputs("Aborted.\n", stderr);
	       GL_CloseFile(OutFile);
	       CleanUp();
	       return(RESULT_ERROR);
	      } /* fi */
	  if ((Error = GL_CloseFileWithCheck(OutFile)))
	     { GL_PrintError(Error, GL_ErrorText(Error));
	       fputs("Aborted.\n", stderr);
	       CleanUp();
	       return(RESULT_ERROR);
	      } /* fi */
	 } /* fi */
     if (GenerateCC)
	{ if (VerbosityLevel >= 2)
	     { fprintf(stderr, "   Output: \"%s\"\n", FilenameCC.chars());
	      } /* fi */
	  if (!(OutFile = GL_OpenFileForWriting(FilenameCC.chars())))
	     { GL_PrintError(GL_Error, GL_ErrorText(GL_Error));
	       fputs("Aborted.\n", stderr);
	       CleanUp();
	       return(RESULT_ERROR);
	      } /* fi */
	  assert(TheDesign != NULL);
	  if ((Error = TheDesign->WriteCC(OutFile,
			FilenameCC.chars(), FilenameH.chars(),
			GenerateNotes, GenerateLines)))
	     { GL_PrintError(Error, SIR_ErrorText(Error));
	       fputs("Aborted.\n", stderr);
	       GL_CloseFile(OutFile);
	       CleanUp();
	       return(RESULT_ERROR);
	      } /* fi */
	  if ((Error = GL_CloseFileWithCheck(OutFile)))
	     { GL_PrintError(Error, GL_ErrorText(Error));
	       fputs("Aborted.\n", stderr);
	       CleanUp();
	       return(RESULT_ERROR);
	      } /* fi */
	 } /* fi */
    } /* fi */

	/* call the compiler */

if (CallCompiler)
   { if (VerbosityLevel >= 1)
	{ fputs("Compiling...\n", stderr);
	 } /* fi */
     if (VerbosityLevel >= 2)
	{ fprintf(stderr, "   Input:  \"%s\"\n", FilenameCC.chars());
	  fprintf(stderr, "   Output: \"%s\"\n", FilenameO.chars());
	 } /* fi */
     if ((Error = CCD_Compiler(FilenameCC.chars(), FilenameO.chars())))
	{ GL_PrintError(Error, CCD_ErrorText(Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
    } /* fi */

	/* call the linker */

if (CallLinker)
   { if (VerbosityLevel >= 1)
	{ fputs("Linking...\n", stderr);
	 } /* fi */
     if (VerbosityLevel >= 2)
	{ fprintf(stderr, "   Input:  \"%s\"\n", FilenameO.chars());
	  fprintf(stderr, "   Output: \"%s\"\n", FilenameOUT.chars());
	 } /* fi */
     if ((Error = CCD_Linker(FilenameO.chars(), FilenameOUT.chars())))
	{ GL_PrintError(Error, CCD_ErrorText(Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
    } /* fi */

	/* call the deparser */

if (CallDeparser)
   { if (VerbosityLevel >= 1)
	{ fputs("Deparsing...\n", stderr);
	 } /* fi */
     if (VerbosityLevel >= 2)
	{ fprintf(stderr, "   Input:  <internal representation>\n");
	  fprintf(stderr, "   Output: \"%s\"\n", FilenameSCout.chars());
	 } /* fi */
     if (!(OutFile = GL_OpenFileForWriting(FilenameSCout.chars())))
	{ GL_PrintError(GL_Error, GL_ErrorText(GL_Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
     assert(TheDesign != NULL);
     if ((Error = TheDesign->WriteSC(OutFile,
			FilenameSCout.chars(),
			GenerateNotes, GenerateLines)))
	{ GL_PrintError(Error, SIR_ErrorText(Error));
	  fputs("Aborted.\n", stderr);
	  GL_CloseFile(OutFile);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
     if ((Error = GL_CloseFileWithCheck(OutFile)))
	{ GL_PrintError(Error, GL_ErrorText(Error));
	  fputs("Aborted.\n", stderr);
	  CleanUp();
	  return(RESULT_ERROR);
	 } /* fi */
    } /* fi */

if (VerbosityLevel >= 1)
   { fputs("Done.\n", stderr);
    } /* fi */

CleanUp();

return(RESULT_OK);	/* success! */

} /* end of main */


/* EOF Compiler.c */
