// ============================================================================
// bit.h: Bit Vector Library, user level
// ============================================================================
// 
// 05/24/01 <gerstl>	Initial version
//
// ----------------------------------------------------------------------------

#ifndef __BIT
#define __BIT

#include <sys/bit.h>



/* ------------------------------------------------------------------------ */
/* Bit Vectors */



/* --- Bit vector representation --- */

template<int len> class _bitrep
{
protected:
  // Array of chunks
  _bit::chunk  vec[_BITCHUNKS(len)];
};


/* --- Bit vector class --- */

template<int len, bool usign=false> class bit: public _bitrep<len>
{
public:
  // Constructors
  bit()                                                                     { }
  bit(const _bit& c)                                             { ref() = c; }
  bit(const _bitslice& s)                                        { ref() = s; }
  template<int l, bool us> bit(const bit<l,us>& c)               { ref() = c; }
  bit(const char* s)                                             { ref() = s; }
  bit(int i)                                                     { ref() = i; }
  bit(unsigned int i)                                            { ref() = i; }
  bit(long i)                                                    { ref() = i; }
  bit(unsigned long i)                                           { ref() = i; }
  bit(long long i)                                               { ref() = i; }
  bit(unsigned long long i)                                      { ref() = i; }

  // Assignments
  bit& operator=(const _bit& o)                    { ref() = o; return put(); }
  bit& operator=(const _bitslice& o)               { ref() = o; return put(); }
  template<int l, bool us> 
    bit& operator=(const bit<l,us>& c)             { ref() = c; return put(); }
  bit& operator=(const char* s)                    { ref() = s; return put(); }
  bit& operator=(int i)                            { ref() = i; return put(); }
  bit& operator=(unsigned int i)                   { ref() = i; return put(); }
  bit& operator=(long i)                           { ref() = i; return put(); }
  bit& operator=(unsigned long i)                  { ref() = i; return put(); }
  bit& operator=(long long i)                      { ref() = i; return put(); }
  bit& operator=(unsigned long long i)             { ref() = i; return put(); }
  bit& operator=(double d)                         { ref() = d; return put(); }
  bit& operator=(long double d)                    { ref() = d; return put(); }

  // Bit operators
  bit& operator^=(const _bit& op)         { get(); ref() ^= op; return put(); }
  bit& operator&=(const _bit& op)         { get(); ref() &= op; return put(); }
  bit& operator|=(const _bit& op)         { get(); ref() |= op; return put(); }

  // Arithmetic operators
  bit& operator+=(const _bit& op)         { get(); ref() += op; return put(); }
  bit& operator-=(const _bit& op)         { get(); ref() -= op; return put(); }
  bit& operator*=(const _bit& op)         { get(); ref() *= op; return put(); }
  bit& operator/=(const _bit& op)         { get(); ref() /= op; return put(); }
  bit& operator%=(const _bit& op)         { get(); ref() %= op; return put(); }

  // Shift operators
  bit& operator<<=(int sh)               { get(); ref() <<= sh; return put(); }
  bit& operator>>=(int sh)               { get(); ref() >>= sh; return put(); }

  // Flip the bits in the vector
  bit& flip(void)                        { get(); ref().flip(); return put(); }

  // Increment & Decrement
  bit& operator++ ()                        { get(); ++(ref()); return put(); }
  bit& operator-- ()                        { get(); --(ref()); return put(); }
  bit  operator++ (int)           { bit r(get()); ++(ref()); put(); return r; }
  bit  operator-- (int)           { bit r(get()); --(ref()); put(); return r; }
  
  // Test for != 0
  bool test(void) const                                { return val().test(); }

  // Explicit conversions
  int toInt(void) const                           { return val().toInt();     }
  unsigned int toUInt(void) const                 { return val().toUInt();    }
  long toLong(void) const                         { return val().toLong();    }
  unsigned long toULong(void) const               { return val().toULong();   }
  long long toLLong(void) const                   { return val().toLLong();   }
  unsigned long long toULLong(void) const         { return val().toULLong();  }
  double toDouble(void) const                     { return val().toDouble();  }
  long double toLDouble(void) const               { return val().toLDouble(); }
                                  
  // Slicing
  _bitslice operator[](int i)                     { return slice(i, i, true); }
  _bitslice operator()(int sl, int sr)                { return slice(sl, sr); }
  const _bitslice operator[](int i) const         { return slice(i, i, true); }
  const _bitslice operator()(int sl, int sr) const    { return slice(sl, sr); }

  // Mapped slicing 
  _bitslice operator()(int l, int r, int i)    
              { ASSERT(_BITLEN(l,r)==len); return operator[](_BITMAP(l,r,i)); }
  _bitslice operator()(int l, int r, int sl, int sr) 
    {ASSERT(_BITLEN(l,r)==len); return slice(_BITMAP(l,r,sl),_BITMAP(l,r,sr));}
  const _bitslice operator()(int l, int r, int i) const 
              { ASSERT(_BITLEN(l,r)==len); return operator[](_BITMAP(l,r,i)); }
  const _bitslice operator()(int l, int r, int sl, int sr) const 
    {ASSERT(_BITLEN(l,r)==len); return slice(_BITMAP(l,r,sl),_BITMAP(l,r,sr));}
                              
  // Implicit cast to working (rhs) bit vector representation
  operator const _bit() const                { const_cast<bit*>(this)->get();
                      return _bit(const_cast<_bit::chunk*>(vec), len, usign); }
  
  // Create a piece for a bit vector bus
  virtual _bitbus_element* bus_slice(int sl, int sr) 
                       { return new _bitbus_element(vec, len, usign, sl, sr); }
  _bitbus_element* bus_slice(int l, int r, int sl, int sr) 
                        { return bus_slice(_BITMAP(l,r,sl), _BITMAP(l,r,sr)); }

protected:
  // Convert to __bit base class
  _bit ref(void)                              { return _bit(vec, len, usign); }
  const _bit val(void) const                  { return operator const _bit(); }
  
  // Create a slice
  virtual _bitslice slice(int l, int r, bool usgn = usign)
                             { return _bitslice(vec, len, usign, l, r, usgn); }
  virtual const _bitslice slice(int l, int r, bool usgn = usign) const
   { return _bitslice(const_cast<_bit::chunk*>(vec), len, usign, l, r, usgn); }
  
  // Functions for synchronization to be overloaded by bit busses
  virtual bit& put()                                          { return *this; }
  virtual bit& get()                                          { return *this; }
};



/* --- Bit vector bus (concatenated bit slices) --- */

template<int len, bool usign=false> class _bitbus: public bit<len,usign>
{
protected:
  _bitbus_element* lst;

public:
  // Constructor
  _bitbus(_bitbus_element* list): lst(list)                                 { }
  
  // Desctructor
  virtual ~_bitbus() 
     { for(_bitbus_element* el=lst; el; el=lst)  { lst=el->next; delete el; } }
  
  // Create a piece for a bit vector bus
  virtual _bitbus_element* bus_slice(int sl, int sr) 
                                                 { return lst->slice(sl, sr); }
protected:  
  // Create a slice
  virtual _bitslice slice(int l, int r, bool usgn = usign)
                        { return _bitslice(vec, len, usign, l, r, usgn, lst); }
  virtual const _bitslice slice(int l, int r, bool usgn = usign) const
                            { return _bitslice(const_cast<_bit::chunk*>(vec), 
                    		                len, usign, l, r, usgn, lst); }
  // Sync with original vectors
  virtual _bitbus& put()                     { lst->put(ref()); return *this; }
  virtual _bitbus& get()                     { lst->get(ref()); return *this; }
};



/* --- Shadow class for use inside unions --- */

template<int len, bool usign=false> class _bit_: public _bitrep<len>
{
public:  
  /* Cast to real bit vector
     Will fail miserably, cast from non-virtual to virtual class!
  operator const bit<len,usign>& (void) const 
                   { return *reinterpret_cast<const bit<len,usign>*>(this); }
  */
  
  // Cast to working bit vector
  operator const _bit (void) const 
                    { return _bit(const_cast<_bit::chunk*>(vec), len, usign); }
  
  // Assignments
  _bit_& operator=(const _bit& c)                  { ref() = c; return *this; }
  _bit_& operator=(const _bitslice& s)             { ref() = s; return *this; }
  template<int l, bool us> 
    _bit_& operator=(const bit<l,us>& c)           { ref() = c; return *this; }
  _bit_& operator=(const char* s)                  { ref() = s; return *this; }
  _bit_& operator=(int i)                          { ref() = i; return *this; }
  _bit_& operator=(unsigned int i)                 { ref() = i; return *this; }
  _bit_& operator=(long i)                         { ref() = i; return *this; }
  _bit_& operator=(unsigned long i)                { ref() = i; return *this; }
  _bit_& operator=(long long i)                    { ref() = i; return *this; }
  _bit_& operator=(unsigned long long i)           { ref() = i; return *this; }
  _bit_& operator=(double d)                       { ref() = d; return *this; }
  _bit_& operator=(long double d)                  { ref() = d; return *this; }

  // Bit operators
  _bit_& operator^=(const _bit& op)              { ref() ^= op; return *this; }
  _bit_& operator&=(const _bit& op)              { ref() &= op; return *this; }
  _bit_& operator|=(const _bit& op)              { ref() |= op; return *this; }

  // Arithmetic operators
  _bit_& operator+=(const _bit& op)              { ref() += op; return *this; }
  _bit_& operator-=(const _bit& op)              { ref() -= op; return *this; }
  _bit_& operator*=(const _bit& op)              { ref() *= op; return *this; }
  _bit_& operator/=(const _bit& op)              { ref() /= op; return *this; }
  _bit_& operator%=(const _bit& op)              { ref() %= op; return *this; }

  // Shift operators
  _bit_& operator<<=(int sh)                    { ref() <<= sh; return *this; }
  _bit_& operator>>=(int sh)                    { ref() >>= sh; return *this; }

  // Flip the bits in the vector
  _bit_& flip(void)                             { ref().flip(); return *this; }

  // Increment & Decrement
  _bit_& operator++ ()                             { ++(ref()); return *this; }
  _bit_& operator-- ()                             { --(ref()); return *this; }
  _bit_  operator++ (int)              { _bit_ r(*this); ++(ref()); return r; }
  _bit_  operator-- (int)              { _bit_ r(*this); --(ref()); return r; }
  
  // Test for != 0
  bool test(void) const                                { return val().test(); }

  // Explicit conversions
  int toInt(void) const                           { return val().toInt();     }
  unsigned int toUInt(void) const                 { return val().toUInt();    }
  long toLong(void) const                         { return val().toLong();    }
  unsigned long toULong(void) const               { return val().toULong();   }
  long long toLLong(void) const                   { return val().toLLong();   }
  unsigned long long toULLong(void) const         { return val().toULLong();  }
  double toDouble(void) const                     { return val().toDouble();  }
  long double toLDouble(void) const               { return val().toLDouble(); }
                                  
  // Slicing
  _bitslice operator[](int i)                     { return slice(i, i, true); }
  _bitslice operator()(int sl, int sr)                { return slice(sl, sr); }
  const _bitslice operator[](int i) const         { return slice(i, i, true); }
  const _bitslice operator()(int sl, int sr) const    { return slice(sl, sr); }

  // Mapped slicing 
  _bitslice operator()(int l, int r, int i)    
              { ASSERT(_BITLEN(l,r)==len); return operator[](_BITMAP(l,r,i)); }
  _bitslice operator()(int l, int r, int sl, int sr) 
    {ASSERT(_BITLEN(l,r)==len); return slice(_BITMAP(l,r,sl),_BITMAP(l,r,sr));}
  const _bitslice operator()(int l, int r, int i) const 
              { ASSERT(_BITLEN(l,r)==len); return operator[](_BITMAP(l,r,i)); }
  const _bitslice operator()(int l, int r, int sl, int sr) const 
    {ASSERT(_BITLEN(l,r)==len); return slice(_BITMAP(l,r,sl),_BITMAP(l,r,sr));}
  
protected:
  // Convert to __bit base class
  _bit ref(void)                              { return _bit(vec, len, usign); }
  const _bit val(void) const                  { return operator const _bit(); }

  // Create a slice
  _bitslice slice(int l, int r, bool usgn = usign) 
                             { return _bitslice(vec, len, usign, l, r, usgn); }
  const _bitslice slice(int l, int r, bool usgn = usign) const
   { return _bitslice(const_cast<_bit::chunk*>(vec), len, usign, l, r, usgn); }

private:
  // Simulate room needed for virtual method table pointer in real class
  void*  vptr;
};



#ifdef _BIT_TEMPLATE_OPERATORS


/* ------------------------------------------------------------------------ */
/* Result of an operation */


template<int len, bool usign> class _bitres: public _bit_<len,usign>
{
public:
  // Constructors
  _bitres(const _bit& op1, char type) 
                                          { _bit(vec, len, usign, op1, type); }
  _bitres(const _bit& op1, char type, int sh)
                                      { _bit(vec, len, usign, op1, type, sh); }
  _bitres(const _bit& op1, char type, const _bit& op2)
                                     { _bit(vec, len, usign, op1, type, op2); }
  _bitres(const _bit& op1, const _bit& op2)
                                           { _bit(vec, len, usign, op1, op2); }
};



/* ------------------------------------------------------------------------- */
/* Operators */


#define _BIT_UNARY(len,usign)	\
  _bitres<_BITLEN_UNARY(len,usign),_BITUSGN_UNARY(len,usign)>

#define _BIT_BINARY(len1,usign1,len2,usign2)	\
  _bitres<_BITLEN_BINARY(len1,usign1,len2,usign2),  \
          _BITUSGN_BINARY(len1,usign1,len2,usign2)>


template<int len1, bool usign1>
  const bit<len1,usign1>&
    operator+(const bit<len1,usign1>& op1) {
      return op1;
}

template<int len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator~(const bit<len1,usign1>& op1) {
      return _BIT_UNARY(len1,usign1)(op1, '~');
}

template<int len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator-(const bit<len1,usign1>& op1) {
      return _BIT_UNARY(len1,usign1)(op1, '-');
}

template<int len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator>>(const bit<len1,usign1>& op1, int sh) {
      return _BIT_UNARY(len1,usign1)(op1, '>', sh);
}

template<int len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator<<(const bit<len1,usign1>& op1, int sh) {
      return _BIT_UNARY(len1,usign1)(op1, '<', sh);
}
  
template<int len1, bool usign1, int len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator+(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '+', op2);
}

template<int len1, bool usign1, int len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator-(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '-', op2);
}

template<int len1, bool usign1, int len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator*(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '*', op2);
}

template<int len1, bool usign1, int len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator/(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '/', op2);
}

template<int len1, bool usign1, int len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator%(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '%', op2);
}

template<int len1, bool usign1, int len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator&(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '&', op2);
}

template<int len1, bool usign1, int len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator|(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '|', op2);
}

template<int len1, bool usign1, int len2, bool usign2>
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator^(const bit<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '^', op2);
}



/* ------------------------------------------------------------------------- */
/* Operators with shadow class*/

template<int len1, bool usign1>
  const _bit_<len1,usign1>&
    operator+(const _bit_<len1,usign1>& op1) {
      return op1;
}

template<int len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator~(const _bit_<len1,usign1>& op1) {
      return _BIT_UNARY(len1,usign1)(op1, '~');
}

template<int len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator-(const _bit_<len1,usign1>& op1) {
      return _BIT_UNARY(len1,usign1)(op1, '-');
}

template<int len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator<<(const _bit_<len1,usign1>& op1, int sh) {
      return _BIT_UNARY(len1,usign1)(op1, '<', sh);
}

template<int len1, bool usign1>
  const _BIT_UNARY(len1,usign1)
    operator>>(const _bit_<len1,usign1>& op1, int sh) {
      return _BIT_UNARY(len1,usign1)(op1, '>', sh);
}

template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator+(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '+', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator+(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '+', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator+(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '+', op2);
}

template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator-(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '-', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator-(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '-', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator-(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '-', op2);
}

template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator*(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '*', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator*(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '*', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator*(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '*', op2);
}

template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator/(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '/', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator/(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '/', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator/(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '/', op2);
}

template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator%(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '%', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator%(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '%', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator%(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '%', op2);
}

template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator&(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '&', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator&(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '&', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator&(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '&', op2);
}

template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator|(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '|', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator|(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '|', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator|(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '|', op2);
}

template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator^(const _bit_<len1,usign1>& op1, const bit<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '^', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator^(const bit<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '^', op2);
}
template<int len1, bool usign1, int len2, bool usign2> 
  const _BIT_BINARY(len1,usign1,len2,usign2)
    operator^(const _bit_<len1,usign1>& op1, const _bit_<len2,usign2>& op2) {
      return _BIT_BINARY(len1,usign1,len2,usign2)(op1, '^', op2);
}

#endif // #ifdef _BIT_TEMPLATE_OPERATORS


/* ------------------------------------------------------------------------- */
/* Global functions */


// Conversion to string
char *bit2str(unsigned int base, char *endptr, const _bit& b);
char *ubit2str(unsigned int base, char *endptr, const _bit& b);

// Conversion from string
void str2bit(unsigned int base, const char *str, _bit *bptr);
void str2ubit(unsigned int base, const char *str, _bit *ubptr);


#endif






