/************************************************************************/
/* Parser.c: SpecC parser						*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 07/02/97 */
/************************************************************************/

/* last update: 05/30/01 */

/* modifications: (most recent first)
 *
 * 05/30/01 RD	reduced contents to reference compiler needs
 * 05/30/01 RD	eliminated level 2 of SIR API
 * 05/27/01 RD	added semantic checking for event port accesses
 * 05/04/01 RD	added support for import of SpecC source (.sc) files
 * 05/03/01 RD	added PRS_ERROR_IMPORT_FILE_NOT_FOUND
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/18/01 RD	added PRS_ERROR_NEGATIVE_ARRAY_SIZE
 * 02/12/01 RD	added PRS_ERROR_BITVECTOR_LENGTH
 */


#include "Parser.h"
#include "PRS_Scanner.h"
#include "CcDriver.h"

#include <stdlib.h>


/*** constants and macros ***********************************************/


#ifdef DEBUG
#ifndef PRS_DEBUG
#define PRS_DEBUG
#endif /* PRS_DEBUG */
#endif /* DEBUG */


/*** internal type declarations *****************************************/


	/* none */


/*** prototypes of internal functions ***********************************/


	/* none */


/*** exported variables *************************************************/


ERROR		PRS_Error = PRS_ERROR_NO_ERROR;	/* last error of this module */

unsigned int	PRS_NumberOfLines = 0;		/* number lines processed */


/*** internal variables *************************************************/


static string	ErrorMessage;


/************************************************************************/
/*** implementation of exported functions			      ***/
/************************************************************************/


	/* error handling */

void PRS_Reset(void)		/* recover from error status */
{

GL_Reset();
SIR_Reset();
CCD_Reset();

PRS_Error = PRS_ERROR_NO_ERROR;

PRS_NumberOfLines = 0;

} /* end of PRS_Reset */


const char *PRS_ErrorText(	/* returns message of error code */
	ERROR	ErrorNumber)
{

if (  (ErrorNumber >= GL_ERROR_BASE)
    &&(ErrorNumber <= GL_ERROR_BASE + GL_ERROR_RANGE))
   { return(GL_ErrorText(ErrorNumber));
    } /* fi */
if (  (ErrorNumber >= CCD_ERROR_BASE)
    &&(ErrorNumber <= CCD_ERROR_BASE + CCD_ERROR_RANGE))
   { return(CCD_ErrorText(ErrorNumber));
    } /* fi */
if (  (ErrorNumber >= SIR_ERROR_BASE)
    &&(ErrorNumber <= SIR_ERROR_BASE + SIR_ERROR_RANGE))
   { return(SIR_ErrorText(ErrorNumber));
    } /* fi */

switch (ErrorNumber)
   { case PRS_ERROR_NO_ERROR:
	  return("No error (internal)");
     case PRS_ERROR_OUT_OF_MEMORY:
	  return("Out of memory");
     case PRS_ERROR_PARSE_ERROR:
     case PRS_ERROR_UNDECLARED_IDENTIFIER:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
			"Undeclared identifier '%s'" */ );
     case PRS_ERROR_UNDECLARED_IDENTIFIER_2:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
			"Undeclared identifier '%s' in port mapping" */ );
     case PRS_ERROR_SEMANTIC_ERROR:
	  return(ErrorMessage.chars()
		/* "Semantic error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
			"%s (#%d)" */ );
     case PRS_ERROR_TYPE_CONST_TWICE:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
			"Type modifier 'const' specified twice" */ );
     case PRS_ERROR_TYPE_VOLATILE_TWICE:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
			"Type modifier 'volatile' specified twice" */ );
     case PRS_ERROR_PARAMETER_STORAGE_CLASS:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
	"Parameter storage class must be empty, 'auto' or 'register'" */ );
     case PRS_ERROR_FUNCTION_DECLARATOR_EXPECTED:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
			"Function declarator expected" */ );
     case PRS_ERROR_BAD_STORAGECLASS_FOR_FUNCTION:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
	"Storage class of function definition must be empty or 'static'" */ );
     case PRS_ERROR_UNDEFINED_SYMBOL_FOR_NOTE:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
		"Undefined symbol '%s' to attach note '%s' to" */ );
     case PRS_ERROR_EVENT_TYPE_MISMATCH:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
		"Type mismatch: '%s' is not an event" */ );
     case PRS_ERROR_ILLEGAL_BITFIELD_SIZE:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
			"Size of bit field (%d) out of range %d to %d" */ );
     case PRS_ERROR_ILLEGAL_BITFIELD_TYPE:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
			"Type of bitfield must be integer" */ );
     case PRS_ERROR_ILLEGAL_TYPE:
	  return(ErrorMessage.chars()
		/* "Illegal type error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
			"%s (#%04d)" */ );
     case PRS_ERROR_CONST_EXPR_NOT_INTEGER:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
		"Constant expression not integer" */ );
     case PRS_ERROR_EMPTY_FILENAME:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
		"Empty filename" */ );
     case PRS_ERROR_NOT_BEHAVIOR_INSTANCE:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
		"Type mismatch: '%s' is not a behavior instance" */ );
     case PRS_ERROR_STATE_LABEL_DEFINED_TWICE:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
		"State label '%s' defined twice" */ );
     case PRS_ERROR_STATE_LABEL_NOT_DEFINED:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
		"State label '%s' not defined" */ );
     case PRS_ERROR_PORT_MAPPING_1:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
		"Identifier '%s' is not of bitvector type" */ );
     case PRS_ERROR_PORT_MAPPING_2:
	  return(ErrorMessage.chars()
		/* "Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
		"Bitslice [%d:%d] out of range (%s[%d:%d])" */ );
     case PRS_ERROR_USERTYPE_IN_PORT_LIST:
	  return("User-type definition inside port list not supported;"
							GL_ERROR_MSG_NEWLINE
		"its scope would be only this definition or declaration,"
							GL_ERROR_MSG_NEWLINE
		"which is probably not what you want");
     case PRS_ERROR_NOTES_RESTRICTED_TO_FCTBODY:
	  return("Illegal annotation at function;" GL_ERROR_MSG_NEWLINE
		"notes are restricted to function body");
     case PRS_ERROR_NOTES_RESTRICTED_TO_FCTBODY2:
	  return("Illegal external annotation;" GL_ERROR_MSG_NEWLINE
		"notes are restricted to function body");
     case PRS_ERROR_CONSTANT_PARSING_FAILED:
	  return(ErrorMessage.chars()
		/* "String conversion to SpecC constant failed;"
						GL_ERROR_MSG_NEWLINE
		"parse error in '%s'" */ );
     case PRS_ERROR_TYPE_MISMATCH_IN_RETURN:
	  return(ErrorMessage.chars()
		/* "Type mismatch in return statement" */ );
     case PRS_ERROR_UNDEFINED_LABEL_IN_RANGE:
	  return(ErrorMessage.chars()
		/* "Undefined label '%s' in range statement" */ );
     case PRS_ERROR_REDEFINED_CONSTRAINT:
	  return(ErrorMessage.chars()
		/* "Redefined constraint between labels" GL_ERROR_MSG_NEWLINE
		"'%s' and '%s'" */ );
     case PRS_ERROR_CONSTRAINT_LABEL_NOT_IN_DO:
	  return(ErrorMessage.chars()
		/* "Constraint label '%s'" GL_ERROR_MSG_NEWLINE
		"not in do-statement list" */ );
     case PRS_ERROR_BITVECTOR_LENGTH:
	  return(ErrorMessage.chars()
		/* "Length of bit vector must be a positive number" */ );
     case PRS_ERROR_NEGATIVE_ARRAY_SIZE:
	  return(ErrorMessage.chars()
		/* "Negative array size (%d) specified" */ );
     case PRS_ERROR_IMPORT_FILE_NOT_FOUND:
	  return(ErrorMessage.chars()
		/* "Import file \"%s\" not found" GL_ERROR_MSG_NEWLINE
		"in import search path" */ );
     case PRS_ERROR_CYCLIC_IMPORT:
	  return(ErrorMessage.chars()
		/* "Cyclic import of design '%s'" */ );
     case PRS_ERROR_IMPORT_FAILED:
	  return(ErrorMessage.chars()
		/* "%s (#%d)" GL_ERROR_MSG_NEWLINE
		"(file \"%s\" is imported from %s)" */ );
     case PRS_ERROR_NOTIFY_INPUT_PORT_1:
     case PRS_ERROR_NOTIFY_INPUT_PORT_2:
	  return(ErrorMessage.chars()
		/* "Notified event '%s' must not be input port" */ );
     case PRS_ERROR_WAIT_OUTPUT_PORT:
	  return(ErrorMessage.chars()
		/* "Waiting event '%s' must not be output port" */ );
     case PRS_ERROR_TRAP_OUTPUT_PORT:
	  return(ErrorMessage.chars()
		/* "Trap event '%s' must not be output port" */ );
     case PRS_ERROR_INTERRUPT_OUTPUT_PORT:
	  return(ErrorMessage.chars()
		/* "Interrupt event '%s' must not be output port" */ );
    } /* hctiws */

return("Unknown error number (internal)");

} /* end of PRS_ErrorText */


void PRS_error(			/* note the error message #1 */
	const char	*Msg)
{

PRS_error(Msg, PRS_ERROR_PARSE_ERROR, PRS_lineno, PRS_CurrentFile.chars());

} /* end of PRS_error #1 */


void PRS_error(			/* note the error message #2 */
	const char	*Msg,
	ERROR		Error,
	unsigned int	Line,
	const char	*File)
{

if (PRS_Error)	/* if an error is already reported */
   { return;	/* ignore the second message       */
    } /* fi */

ErrorMessage.form(	"Parse error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
			"%s",
			Line, File, Msg);

PRS_Error = Error;

} /* end of PRS_error #2 */


void PRS_error(			/* note the error message #3 */
	const char	*Msg,
	ERROR		Error)
{

ErrorMessage = Msg;
PRS_Error = Error;

} /* end of PRS_error #3 */


void PRS_SemanticError(		/* handle sub-module error messages */
	ERROR		Error,
	sir_node	*Node1 = NULL,
	sir_node	*Node2 = NULL,
	sir_node	*Node3 = NULL)
{
unsigned int	Line;
const char	*File;

if (  (Node3)	/* find the most accurate line-information */
    &&(Node3->LineInfo))
   { Line = Node3->LineInfo->Line;
     File = Node3->LineInfo->File->Filename.chars();
    } /* fi */
else
   { if (  (Node2)
	 &&(Node2->LineInfo))
	{ Line = Node2->LineInfo->Line;
	  File = Node2->LineInfo->File->Filename.chars();
	 } /* fi */
     else
	{ if (  (Node1)
	      &&(Node1->LineInfo))
	     { Line = Node1->LineInfo->Line;
	       File = Node1->LineInfo->File->Filename.chars();
	      } /* fi */
	  else
	     { Line = PRS_lineno;
	       File = PRS_CurrentFile.chars();
	      } /* esle */
	 } /* esle */
    } /* esle */

ErrorMessage.form(	"Semantic error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
			"%s (#%d)",
			Line, File, PRS_ErrorText(Error), Error);

PRS_Error = PRS_ERROR_SEMANTIC_ERROR;

} /* end of PRS_SemanticError */


void PRS_PrintWarning(	/* print a warning message */
	int		WarnLevel,
	const char	*Msg)
{

GL_PrintWarningFmt(WarnLevel,
		"In line %u, file \"%s\":" GL_WARN_MSG_NEWLINE
		"%s",
		PRS_lineno, PRS_CurrentFile.chars(), Msg);

} /* end of PRS_PrintWarning */


/*** parser supporting functions ****************************************/


ERROR PRS_CheckTypeDefinition(		/* checks if type is legal here */
	sir_type	*TypeEntry,
	bool		VoidIsOK = false,
	bool		IsPort = false,
	SIR_STORAGE	StorageClass = SIR_STORAGE_NONE,
	unsigned int	Line = 0,
	const char	*Filename = NULL)
{
ERROR	SirError;

assert(TypeEntry != NULL);

if ((SirError = TypeEntry->Check(VoidIsOK, IsPort,
			(  (StorageClass == SIR_STORAGE_EXTERN)
			 ||(StorageClass == SIR_STORAGE_TYPEDEF)))))
   { ErrorMessage.form(	"Illegal type error in line %u" GL_ERROR_MSG_NEWLINE
			"in file \"%s\":" GL_ERROR_MSG_NEWLINE
			"%s (#%d)",
			(Line ? Line : PRS_lineno),
			(Filename ? Filename : PRS_CurrentFile.chars()),
			SIR_ErrorText(SirError),
			SirError);
     return(PRS_Error = PRS_ERROR_ILLEGAL_TYPE);
    } /* fi */

return(PRS_ERROR_NO_ERROR);

} /* end of PRS_CheckTypeDefinition */


/*** main functionality *************************************************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


sir_design *PRS_ParseDesign(		/* parse a whole design (#1) */
	FILE		*FromFile,	/* from a file */
	const char	*DesignName,	/* (returns a new SIR data structure */
	const char	*FileName,	/*  or NULL and PRS_Error) */
	GL_STRINGLIST	*ImportDirs = NULL)
{

PRS_TheDesign = new SIR_Design(DesignName);	/* start with a new design */
PRS_CurrScope = PRS_TheDesign->Symbols;		/* start with global scope */
PRS_NumberOfLines = 1;		/* number of lines processed */
PRS_lineno = 1;			/* current line */
PRS_CurrentFile = FileName;	/* current filename */
PRS_CurrFileInfo = PRS_TheDesign->FileList->FindOrInsert(FileName);
PRS_ImportDirList = ImportDirs;	/* publish the dirs to search for import */
PRS_in = FromFile;		/* publish the file descriptor */
PRS_Initialize(PRS_in,		/* and (re-)start the scanner  */
		PRS_PARSE_LANGUAGE);
PRS_Error = PRS_ERROR_NO_ERROR;	/* clean error state */

assert(PRS_FileStack == NULL);	/* start with an empty stack */

#ifdef YYDEBUG
PRS_debug = 1;	/* turn debugging on */
#endif /* YYDEBUG */

#ifdef PRS_DEBUG
puts("PRS_DEBUG: Starting parsing...");
#endif /* PRS_DEBUG */
if (PRS_parse())		/* call the parser */
   { /* exit due to error condition */
     PRS_ClearStack();
     delete PRS_TheDesign;
     PRS_TheDesign = NULL;
     return(NULL);	/* error is in PRS_Error */
    } /* fi */

assert(PRS_FileStack == NULL);	/* leave with an empty stack */

#ifdef PRS_DEBUG
puts("PRS_DEBUG: Parsing successful!");
#endif /* PRS_DEBUG */

PRS_TheDesign->GarbageCollection();	/* clean up the design */

return(PRS_TheDesign);	/* return the complete design */

} /* end of PRS_ParseDesign #1 */


/************************************************************************/
/*** implementation of internal functions			      ***/
/************************************************************************/


	/* none */


/************************************************************************/
/*** main (for debugging only)					      ***/
/************************************************************************/


#ifdef DEBUG	/* module self test */

int main(int argc, char **argv)
{
char			*FromName,
			*ToName;
FILE			*FromFile,
			*ToFile;
sir_design		*NewDesign;
static GL_STRINGLIST	ImportDir = { NULL, "." };
ERROR			Error;

printf("DEBUGGING: Parser ('%s')\n\n", argv[0]);

if (argc != 3)
   { printf("ERROR: Bad args! (usage: %s <from> <to>\n", argv[0]);
     return(RESULT_ERROR);
    } /* fi */

FromName	= argv[1];
ToName		= argv[2];

GL_WarningLevel = GL_WARN_STANDARD;	/* enable standard warnings */

printf("Parsing '%s'...\n", FromName);
if (!(FromFile = GL_OpenFileForReading(FromName)))
   { GL_PrintError(GL_Error, GL_ErrorText(GL_Error));
     exit(RESULT_ERROR);
    } /* fi */
if (!(NewDesign = PRS_ParseDesign(FromFile, GL_GetBaseName(FromName),
					FromName, &ImportDir)))
   { GL_PrintError(PRS_Error, PRS_ErrorText(PRS_Error));
     fclose(FromFile);
     exit(RESULT_ERROR);
    } /* fi */
GL_CloseFile(FromFile);
printf("Successfully parsed %d lines.\n", PRS_NumberOfLines);

printf("\nDumping '%s'...\n", ToName);
if (!(ToFile = GL_OpenFileForWriting(ToName)))
   { GL_PrintError(PRS_Error, PRS_ErrorText(PRS_Error));
     delete NewDesign;
     exit(RESULT_ERROR);
    } /* fi */
if ((Error = NewDesign->WriteSC(ToFile, ToName)))
   { GL_PrintError(Error, SIR_ErrorText(Error));
     fclose(ToFile);
     delete NewDesign;
     exit(RESULT_ERROR);
    } /* fi */
GL_CloseFile(ToFile);
puts("Dumping successful.");

delete NewDesign;	/* we don't need that any more */

puts("\nDone.");
return(RESULT_OK);	/* success! */

} /* end of main */

#endif /* DEBUG */


/* EOF Parser.c */
