/************************************************************************/
/* IntRep/Symbol.h: SpecC Internal Representation, Symbol Classes	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/08/97 */
/************************************************************************/

/* last update: 05/31/01 */

/* modifications: (most recent first)
 *
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	eliminated support for automatic IP wrapping (scc -ip option)
 * 05/25/01 RD	introduced this header (last change was 05/10/99)
 */

#ifndef INTREP_SYMBOL_H
#define INTREP_SYMBOL_H


#include "Global.h"
#include "IntRep/Statement.h"
#include "IntRep/Initializer.h"
#include "IntRep/UserType.h"
#include "IntRep/PortMap.h"


/*** constants and macros ***********************************************/


// #define SIR_PUT_COMMENTS_IN_SOURCE	// put comments in generated source code


/*** enumeration types **************************************************/


enum SIR_SymbolClass		/* supported symbol classes */
{				/* (see class SIR_Symbol) */
SIR_SYMBOL_IDENTIFIER,
SIR_SYMBOL_TYPEDEF,
SIR_SYMBOL_BEHAVIOR,
SIR_SYMBOL_CHANNEL,
SIR_SYMBOL_INTERFACE
};

enum SIR_StorageClass		/* storage class */
{
SIR_STORAGE_NONE,
SIR_STORAGE_TYPEDEF,
SIR_STORAGE_EXTERN,
SIR_STORAGE_STATIC,
SIR_STORAGE_AUTO,
SIR_STORAGE_REGISTER,
SIR_STORAGE_PIPED
};

enum SIR_ScopeInfo		/* scope information */
{
SIR_SCOPE_GLOBAL,
SIR_SCOPE_CLASS,
SIR_SCOPE_PARAMETER,
SIR_SCOPE_STATEMENT,
SIR_SCOPE_USERTYPE
};


/*** type definitions ***************************************************/


typedef enum SIR_SymbolClass		SIR_SYMBOLCLASS;
typedef enum SIR_StorageClass		SIR_STORAGE;
typedef enum SIR_ScopeInfo		SIR_SCOPE;

typedef class SIR_Symbol		sir_symbol;
typedef SIR_List<sir_symbol>		sir_symbol_list;
typedef class SIR_Symbols		sir_symbols;
typedef void *				sir_symbol_marg;
typedef ERROR (SIR_Symbol::*sir_symbol_mptr)(void*);

typedef class SIR_SymbolPtr		sir_symbol_ptr;
typedef SIR_List<sir_symbol_ptr>	sir_symbol_ptr_list;
typedef class SIR_SymbolPtrs		sir_symbol_ptrs;

typedef class SIR_SymbolPtr		SIR_Event;	/* alias type */
typedef class SIR_SymbolPtrs		SIR_Events;	/* alias type */
typedef class SIR_SymbolPtr		sir_event;	/* alias type */
typedef class SIR_SymbolPtrs		sir_events;	/* alias type */

typedef class SIR_Declarator		sir_declarator;	/* cyclic link */
typedef class SIR_DeclSpec		sir_declspec;	/* cyclic link */
typedef class SIR_Design		sir_design;	/* cyclic link */


/*** class declarations *************************************************/


	/******************/
	/*** SIR_Symbol ***/
	/******************/


class SIR_Symbol :		/* symbol (entry in a symbol table) */
	public SIR_Node,		/* is a node */
	public SIR_ListElem<SIR_Symbol>	/* and a list element */
{
public:
SIR_SYMBOLCLASS	Class;		/* class of this symbol (see above) */
string		Name;		/* name of this symbol */
SIR_STORAGE	StorageClass;	/* storage class (see above) */
int		PipeStages;	/* number of pipeline stages (if PIPED) */
sir_type	*Type;		/* link to type table entry */
sir_notes	*Notes;		/* notes for this symbol (NULL if none) */
sir_initializer	*Initializer;	/* initialization info (or NULL) */
int		EnumValue;	/* value (if this is an enum-constant) */
sir_parameters	*Parameters;	/* parameter or port list (or NULL) */
sir_symbols	*ParamScope;	/* parameter scope (or NULL) */
sir_symbols	*ClassScope;	/* class scope (or NULL) */
sir_labels	*Labels;	/* label list (for functions) */
sir_statement	*FctBody;	/* function body (or NULL) */
sir_symbol_ptrs	*Interfaces;	/* list of implemented interfaces (or NULL) */
sir_portmaps	*PortMappings;	/* port mapping list for insts. (or NULL) */
sir_import	*Imported;	/* link to import file (NULL if not imported) */

sir_symbol	*Alias;		/* alias pointer (temporary only) */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Symbol(			/* constructor #1 (general) */
	SIR_SYMBOLCLASS	Class,
	const char	*Name,
	sir_type	*Type = NULL,
	SIR_STORAGE	StorageClass = SIR_STORAGE_NONE,
	int		PipeStages = 0,
	sir_initializer	*Initializer = NULL,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Symbol(			/* constructor #2 (functions) */
	const char	*Name,
	sir_type	*Type,
	SIR_STORAGE	StorageClass,
	sir_parameters	*Parameters,
	sir_symbols	*ParamScope,
	sir_labels	*Labels,
	sir_statement	*FctBody,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Symbol(			/* constructor #3 (classes) */
	SIR_SYMBOLCLASS	Class,
	const char	*Name,
	sir_type	*Type,
	sir_parameters	*Parameters,
	sir_symbols	*ClassScope,
	sir_symbol_ptrs	*Interfaces,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Symbol(			/* constructor #4 (instantiations) */
	const char	*Name,
	sir_type	*Type,
	sir_portmaps	*PortMappings,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Symbol(			/* constructor #6 (special for parser) */
	sir_declspec	*DeclSpec,
	sir_declarator	*Declarator,
	sir_type	*Type,
	BOOL		StoreParameters = false);

SIR_Symbol(			/* constructor #7 (special for parser) */
	SIR_SYMBOLCLASS	Class,
	sir_declarator	*Declarator,
	sir_type	*Type,
	sir_symbol_ptrs	*Interfaces,
	BOOL		StoreParameters = false);

SIR_Symbol(			/* constructor #8 (duplicator) */
	sir_symbol	*Original);

~SIR_Symbol(void);		/* destructor */


BOOL IsVariable(void);		/* check if IDENTIFIER and !TYPE_FUNCTION   */
				/* and !TYPE_BEHAVIOR and !TYPE_CHANNEL and */
				/* !TYPE_INTERFACE and Direction==PORT_NONE */
				/* and !IsEnumMember			    */

BOOL IsVariableDefinition(void);/* check if IsVariable and !STORAGE_EXTERN */

BOOL IsFunction(void);		/* check if IDENTIFIER and TYPE_FUNCTION */

BOOL IsFunctionDefinition(void);/* check if IsFunction and FctBody!=NULL */

BOOL IsBehaviorInstance(void);	/* check if IDENTIFIER and TYPE_BEHAVIOR */

BOOL IsChannelInstance(void);	/* check if IDENTIFIER and TYPE_CHANNEL */

BOOL IsTypeDef(void);		/* check if TYPEDEF */

BOOL IsBehavior(void);		/* check if BEHAVIOR */

BOOL IsBehaviorDefinition(void);/* check if IsBehavior and ClassScope!=NULL */

BOOL IsChannel(void);		/* check if CHANNEL */

BOOL IsChannelDefinition(void);	/* check if IsChannel and ClassScope!=NULL */

BOOL IsInterface(void);		/* check if INTERFACE */

BOOL IsInterfaceDefinition(void);/* check if IsInt. and ClassScope!=NULL */

BOOL IsClass(void);		/* check if BEHAVIOR, CHANNEL or INTERFACE */

BOOL IsPort(void);		/* check if IDENTIFIER and TYPE_INTERFACE */
				/* or Direction!=PORT_NONE		  */

BOOL IsParameter(void);		/* check if defined in SCOPE_PARAMETER */

BOOL IsEnumMember(void);	/* check if this is an enum member */


BOOL IsDirectDependant(		/* checks if this symbol depends on that */
	sir_symbol	*ThatSymbol,	/* (directly) */
	SIR_DEPENDENCY	*Reason = NULL);

BOOL IsIndirectDependant(	/* checks if this symbol depends on that */
	sir_symbol	*ThatSymbol,	/* (indirectly) */
	SIR_DEPENDENCY	*Reason = NULL,
	sir_statement	**DepStmnt = NULL,
	sir_expression	**DepExpr = NULL);

BOOL MapsPortTo(		/* checks for port-instantiation */
	sir_symbol	*Symbol);	/* (instantiations only) */

BOOL Instantiates(		/* checks for instantiation */
	sir_symbol	*ClassSymbol);	/* (behaviors and channels only) */

BOOL Contains(			/* checks for containment */
	sir_symbol	*ClassSymbol);	/* (behaviors and channels only) */


sir_symbols *GetScope(void);	/* determines the scope of this symbol */
				/* (returns NULL if not in a table) */

sir_design *GetDesign(void);	/* determines the design this symbol is in */
				/* (returns NULL if type is not in design) */


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);

ERROR DFS_ForAllSymbols(	/* iterator over all symbols (depth first) */
	sir_symbol_mptr	MemberFct,
	sir_symbol_marg	MemberFctArg);

ERROR DFS_ForAllUserTypes(	/* iterator over all usertypes (depth f.) */
	sir_usertp_mptr	MemberFct,
	sir_usertp_marg	MemberFctArg);

ERROR DFS_ForAllNotes(		/* iterator over all notes (depth first) */
	sir_note_mptr	MemberFct,
	sir_note_marg	MemberFctArg);


ERROR SetImported(		/* sets the Imported pointer (if unset) */
	sir_symbol_marg	ImportPtr);	/* or resets it if NULL */

ERROR TakeOverImport(		/* takes over an imported symbol */
	sir_symbol_marg	ImportPtr);

ERROR MarkImportEntry(		/* marks its import entry being used */
	sir_symbol_marg	/* Unused */);

ERROR MarkUsedTypes(		/* marks the type entries used here */
	sir_symbol_marg	/* Unused */);


BOOL DFS_FindDependant(		/* searches for dependants (depth first) */
	sir_symbol	**SkipSymbol,	/* (internal, use the next ones!) */
	sir_symbol	*ThatSymbol,
	sir_symbol	**DepSymbol,
	sir_statement	**DepStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason);

BOOL FindDependant(		/* finds symbols that depend on this one #1 */
	sir_symbol	**DepSymbol = NULL,	/* return symbol found */
	sir_statement	**DepStmnt = NULL,	/* return statement found */
	sir_expression	**DepExpr = NULL,	/* return expression found */
	SIR_DEPENDENCY	*Reason = NULL,		/* return type of dependency */
	sir_symbols	*SearchScope = NULL,	/* search in this scope */
	sir_symbol	*SearchAfter = NULL);	/* continue search here */

BOOL FindDependant(		/* finds symbols that depend on this one #2 */
	sir_statement	*SearchStmnt,		/* search in this statement */
	sir_statement	**DepStmnt = NULL,	/* return statement found */
	sir_expression	**DepExpr = NULL,	/* return expression found */
	SIR_DEPENDENCY	*Reason = NULL);	/* return type of dependency */


void SetAlias(			/* sets all type, usertype, symbol alias' */
	sir_symbol	*Alias);/* (iterates over symbols and usertypes) */

void UnAlias(			/* unalias all type, usertype, symbol lnks.*/
	sir_symbols	*GlobalSymbols);


const char *PrintStorageClass(		/* prints SpecC storage class #1 */
	BOOL		AppendSpace = false); /* ("extern" or "extern ") */

static const char *PrintStorageClass(	/* prints SpecC storage class #2 */
	SIR_STORAGE	StorageClass,	/* (eg. "extern" or "extern ") */
	int		PipeStages = 0,
	BOOL		AppendSpace = false);


ERROR WriteSC(			/* (re-) generates SpecC source code */
	FILE		*File,	/* (for non-class symbols) */
	BOOL		WriteNotes,
	BOOL		AsDeclaration = FALSE,
	BOOL		CplusplusMode = FALSE,
	BOOL		AsReference = FALSE);

ERROR WriteSC2(			/* (re-) generates SpecC source code */
	FILE		*File,	/* (for class symbols) */
	BOOL		WriteNotes,
	BOOL		AsDeclaration = FALSE);

ERROR WriteSC3(			/* (re-) generates SpecC source code */
	FILE		*File,	/* (for function definitions) */
	BOOL		WriteNotes,
	BOOL		CplusplusMode = FALSE);

ERROR WriteCC2(			/* generates C++ source code */
	FILE		*File,	/* (for class symbols) */
	BOOL		WriteNotes,
	BOOL		AsDeclaration = FALSE);

ERROR WriteCC2b(		/* generates C++ source code */
	FILE		*File,	/* (for classes, method implementations) */
	BOOL		WriteNotes);

ERROR WriteCC2IP(		/* generates C++ source code */
	FILE		*File);	/* (for IP classes) */

ERROR WriteSC4ClassBody(	/* generates SpecC code for class bodies */
	FILE		*File,	/* (for class definitions only) */
	BOOL		WriteNotes = TRUE,
	BOOL		WriteLines = TRUE,
	const char	*ThisFilename = NULL,	/* default location if */
	unsigned int	ThisLineNumber = 1);	/* line info unknown   */


ERROR CheckInitializer(		/* checks whether initializer is legal */
	sir_initializer	*Initializer = NULL,	/* (if not specified, */
	sir_type	*Type = NULL,		/*  own data is used) */
	SIR_STORAGE	*StorageClass = NULL,
	int		*PipeStages = NULL);

ERROR CheckStorageClass(	/* checks whether storage class is legal */
	SIR_STORAGE	*StorageClass = NULL,	/* (if not specified, */
	int		*PipeStages = NULL,	/*  own data is used) */
	SIR_SYMBOLCLASS	*Class = NULL,
	SIR_SCOPE	*ScopeInfo = NULL);


ERROR Annotate(			/* attach a note to the symbol */
	sir_note	*NewNote);	/* (consumes NewNote) */

void Touch(			/* modification update (remove import info) */
	BOOL		ImportOnly = FALSE);/* (dflt: remove source info too) */

void Strip(			/* remove source location for the tree below */
	BOOL		LocalOnly = FALSE); /* (dflt: work recursively) */
};


	/*******************/
	/*** SIR_Symbols ***/
	/*******************/


class SIR_Symbols :		/* symbol "table" (alias: Scope) */
	public SIR_List<SIR_Symbol>	/* is simply a list of symbols */
{					/* with additional member(function)s */
public:
sir_symbols	*Parent;	/* link to parent symbol table (or NULL) */
sir_usertypes	*UserTypes;	/* separate table for user type symbols */
SIR_SCOPE	ScopeInfo;	/* scope information (see above) */
sir_symbol	*ParentSymbol;	/* link to fct/cls. with this scope (or NULL) */
sir_usertype	*ParentUType;	/* link to usertype with this scope (or NULL) */
sir_statement	*ParentStmnt;	/* link to statement with this scope (or NULL)*/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Symbols(			/* constructor #1 */
	sir_symbols	*Parent,	/* (creates UserTypes automatically) */
	SIR_SCOPE	ScopeInfo,
	sir_symbol	*ParentSymbol = NULL,
	sir_usertype	*ParentUType = NULL,
	sir_statement	*ParentStmnt = NULL);

SIR_Symbols(			/* constructor #3 (duplicator) */
	sir_symbols	*Original);

~SIR_Symbols(void);		/* destructor */


BOOL IsAncestorOf(		/* checks if the symbol can see this scope */
	sir_symbol	*Symbol);


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);

ERROR DFS_ForAllSymbols(	/* iterator over all symbols (depth first) */
	sir_symbol_mptr	MemberFct,
	sir_symbol_marg	MemberFctArg);

ERROR DFS_ForAllUserTypes(	/* iterator over all usertypes (depth first) */
	sir_usertp_mptr	MemberFct,
	sir_usertp_marg	MemberFctArg);

ERROR DFS_ForAllNotes(		/* iterator over all notes (depth first) */
	sir_note_mptr	MemberFct,
	sir_note_marg	MemberFctArg);

BOOL DFS_FindDependant(	/* searches for dependants (depth first) */
	sir_symbol	**SkipSymbol,
	sir_symbol	*ThatSymbol,
	sir_symbol	**DepSymbol,
	sir_statement	**DepStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason);


void SetAlias(			/* sets all type, usertype, symbol alias' */
	sir_symbols	*Alias);/* (iterates over symbols and usertypes) */

void UnAlias(			/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols);

ERROR Integrate(		/* integrates imported symbol table */
	sir_symbols	*Imported);

ERROR CheckClassContainment(	/* checks class hierarchy for cycles */
	void);


sir_symbol *Find(		/* find an entry globally (here and above) */
	const char	*Name);	/* (returns NULL if not found) */

sir_symbol *FindLocally(	/* find an entry locally (only in this table) */
	const char	*Name);	/* (returns NULL if not found) */

sir_symbol *FindLocally(	/* find a symbol with this scope locally */
	sir_symbols	*Scope);	/* (returns NULL if not found) */

sir_symbol *Insert(		/* inserts a new symbol */
	sir_symbol	*NewSymbol);	/* (the argument is consumed!) */

void FindName(			/* find a symbol or usertype globally */
	const char	*Name,
	sir_symbol	**SymbolFound,		/* the symbol found (or NULL) */
	sir_usertype	**UserTypeFound);	/* user-type found (or NULL) */

void FindNameLocally(		/* find a symbol or usertype locally */
	const char	*Name,
	sir_symbol	**SymbolFound,		/* the symbol found (or NULL) */
	sir_usertype	**UserTypeFound);	/* user-type found (or NULL) */


sir_symbol *Declare(		/* declares a new symbol (and prepares def.) */
	sir_symbol	*NewSymbol);	/* (the argument is consumed!) */
					/* (may return NULL and SIR_Error) */

ERROR DefineVariable(		/* makes variable declaration a definition */
	sir_symbol	*VariableDecl,	/* (consumes Initializer) */
	SIR_STORAGE	DeclStorageClass,
	sir_initializer	*Initializer);

ERROR DefineFunction(		/* makes function declaration a definition */
	sir_symbol	*FunctionDecl,	/* (consumes FctBody) */
	sir_statement	*FctBody);

ERROR DefineClass(		/* makes class declaration a definition */
	sir_symbol	*ClassDecl);


ERROR WriteSC(			/* (re-) generates SpecC source code */
	FILE		*File,
	BOOL		WriteNotes);

ERROR WriteOrderedSC1(		/* writes ordered class instantiations */
	sir_symbol	*Current,	/* (recursively) */
	FILE		*File,
	BOOL		WriteNotes);

ERROR WriteOrderedSC2(		/* writes ordered class definitions */
	sir_symbol	*Current,	/* (recursively) */
	FILE		*File,
	BOOL		WriteNotes);

ERROR WriteH(			/* generates C++ header code */
	FILE		*File,
	BOOL		WriteNotes);

ERROR WriteCC(			/* generates C++ main code */
	FILE		*File,
	BOOL		WriteNotes);

ERROR WriteOrderedCC2(		/* writes ordered C++ class definitions */
	sir_symbol	*Current,	/* (recursively) */
	FILE		*File,
	BOOL		WriteNotes);
};


	/*********************/		/*****************/
	/*** SIR_SymbolPtr ***   alias	 *** SIR_Event ***/
	/*********************/		/*****************/


class SIR_SymbolPtr :		/* symbol pointer (alias event) */
	public SIR_ListElem<SIR_SymbolPtr>	/* is a list element */
{
public:
sir_symbol	*Symbol;	/* link to the symbol */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_SymbolPtr(			/* constructor #1 */
	sir_symbol	*Symbol);

SIR_SymbolPtr(			/* constructor #3 (duplicator) */
	sir_symbol_ptr	*Original);

~SIR_SymbolPtr(void);		/* destructor */
};


	/**********************/	/******************/
	/*** SIR_SymbolPtrs ***  alias	 *** SIR_Events ***/
	/**********************/	/******************/


class SIR_SymbolPtrs :		/* list of symbol pointers (alias event list) */
	public SIR_List<SIR_SymbolPtr>	/* is basically a list */
{					/* with additional functions */
public:

//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_SymbolPtrs(			/* constructor #1 */
	sir_symbol_ptr	*FirstEntry = NULL);

SIR_SymbolPtrs(			/* constructor #2 (duplicator) */
	sir_symbol_ptrs	*Original);

~SIR_SymbolPtrs(void);		/* destructor */


void UnAlias(void);		/* unalias all type, usertype, symbol links */


int CmpSymbolNames(		/* compare all symbol names */
	sir_symbol_ptrs	*Others);	/* (similar to strcmp) */

sir_symbol_ptr *Find(		/* searches for a specific entry */
	sir_symbol	*Symbol);	/* (returns NULL if not found) */
};


#endif /* INTREP_SYMBOL_H */

/* EOF IntRep/Symbol.h */
