/************************************************************************/
/* GL_String.h: Replacement for GNU's libg++ String class... 	     	*/
/************************************************************************/
/* Author: Andreas Gerstlauer <gerstl>		first version: 03/02/00 */
/************************************************************************/

/* last update: 04/27/01 by A.G. */

#ifndef GL_STRING_H
#define GL_STRING_H

#include <stdlib.h>
#include <string.h>
#include <iostream.h>
#include <assert.h>


class SubString;


class String
{
   friend class SubString;
  
protected:
   unsigned short  _len;      // string length
   unsigned short  _size;     // allocated size
   char*   _str;
   
public:
   // constructors & assignment
   String();
   String(const String& x);
   String(const SubString& x);
   String(const char* t);
   String(char c);

   ~String();

   // assignment
   String&           operator =  (const String&     y);
   String&           operator =  (const SubString&  y);
   String&           operator =  (const char* y);
   String&           operator =  (char        c);

   // format string
   // Note: only plain %%, %c, %s, and %[d,u,i,o,x,X] are supported!
   String&           form (const char* format, ...);
  
   // concatenation
   String&           operator += (const String&     y); 
   String&           operator += (const SubString&  y);
   String&           operator += (const char* t);
   String&           operator += (char        c);
  
   void              prepend(const String&     y); 
   void              prepend(const SubString&  y);
   void              prepend(const char* t);
   void              prepend(char        c);

   
   //  return number of occurences of target in String
   int               freq(char        c) const; 
   int               freq(const char* t) const;

   // Substring extraction
   SubString   at(int pos, int len);
   SubString   at(const char* t, int startpos = 0);
   SubString   at(char        c, int startpos = 0);

   SubString   before(int pos);
   SubString   before(const char*  t, int startpos = 0);
   SubString   before(char         c, int startpos = 0);

   SubString   after(int pos);
   SubString   after(const char*  t, int startpos = 0);
   SubString   after(char         c, int startpos = 0);
   
   // deletion
   void              del(const char* t);
   void              del(char        c);

 
   // conversion
   operator const char*() const;
   const char*       chars() const;

   // element extraction  
   char&             operator [] (int i);
   const char&       operator [] (int i) const;
   
   // status
   unsigned int      length() const;
   int               empty() const;

   // IO
   friend inline ostream&   operator<<(ostream& s, const String& x);
   friend inline ostream&   operator<<(ostream& s, const SubString& x);
   friend istream&   operator>>(istream& s, String& x);

protected:
   char* salloc (int len, int keep = 0, bool force = false);
   void  scopy  (const char* t, int startpos, int len);
   void  sinsert(const char* t, int pos, int len, int repl = 0);
   void  sinsert(char c, int pos, int repl = 0);
};



class SubString
{
  friend class String;
  
protected:
  String&           _str;      // The String I'm a substring of
  unsigned short    _pos;      // starting position in S's rep
  unsigned short    _len;      // length of substring

  SubString(String& x, int p, int l);
  SubString(const SubString& x);

public:
  ~SubString();

  SubString&        operator =  (const String&     y);
  SubString&        operator =  (const SubString&  y);
  SubString&        operator =  (const char* t);
  SubString&        operator =  (char        c);

  // status
  unsigned int      length() const;
  int               empty() const;
  const char*       chars() const;
  operator const char*() const;
  // Note that these don't necessarily return a null-terminated char* !
  // Only a pointer into the parent String, valid up to length()
   
  // I/O
  friend inline ostream&   operator<<(ostream& s, const SubString& x);
};



// ------------------- String implementation ---------------------

extern const char* const _nullString;

// inlined constructors & assignment

inline String::String() : _len(0), _size(0), _str((char*) _nullString) {
}
inline String::String(const String& x) 
 : _len(0), _size(0), _str((char*) _nullString) { operator= (x); }
inline String::String(const SubString& x) 
 : _len(0), _size(0), _str((char*) _nullString) { operator= (x); }
inline String::String(const char* t) 
 : _len(0), _size(0), _str((char*) _nullString) { operator= (t); }
inline String::String(char c) 
 : _len(0), _size(0), _str((char*) _nullString) { operator= (c); }

inline String::~String() {
   if (_str != _nullString) free (_str);
}

inline String& String::operator= (const String& y) {
  sinsert(y.chars(), 0, y.length(), _len);
  return *this;
}
inline String& String::operator= (const SubString& y) {
  sinsert(y.chars(), 0, y.length(), _len);
  return *this;
}
inline String& String::operator= (const char* y) {
  y? sinsert(y, 0, strlen(y), _len) : sinsert("", 0, 0, _len);
  return *this;
}
inline String& String::operator= (const char c) {
  sinsert(c, 0, _len);
  return *this;
}


// inlined methods

inline String& String::operator+= (const String& y) {
  sinsert(y.chars(), _len, y.length());
  return *this;
}
inline String& String::operator+= (const SubString& y) {
  sinsert(y.chars(), _len, y.length());
  return *this;
}
inline String& String::operator+= (const char* y) {
  if (y) sinsert(y, _len, strlen(y));
  return *this;
}
inline String& String::operator+= (const char c) {
  sinsert(c, _len);
  return *this;
}


inline void String::prepend(const String&     y) {
  sinsert(y.chars(), 0, y.length());
}
inline void String::prepend(const SubString&  y) {
  sinsert(y.chars(), 0, y.length());
}
inline void String::prepend(const char* t) {
  if (t) sinsert(t, 0, strlen(t));
}
inline void String::prepend(char        c) {
  sinsert(c, 0);
}


inline SubString String::at(int pos, int len) {
  return SubString(*this, pos, len);
}

inline SubString String::before(int pos) {
  return SubString(*this, 0, pos);
}

inline SubString String::after(int pos) {
  return SubString(*this, pos + 1, _len - pos - 1);
}

inline void String::del(const char* t) {
  at(t).operator= ((const char*)0);
}
inline void String::del(char        c) {
  at(c).operator= ((const char*)0);
}



inline const char* String::chars() const {
   return _str;
}
inline String::operator const char*() const {
   return chars();
}

inline unsigned int String::length() const {
   return _len;
}
inline int String::empty() const {
   return (_len == 0);
}


inline char&  String::operator [] (int i) 
{ 
  assert (((unsigned)i) < length());
  return _str[i];
}

inline const char&  String::operator [] (int i) const
{ 
  assert (((unsigned)i) < length());
  return _str[i];
}
 

// ------------------- SubString implementation ---------------------

// inlined constructors & assignment


inline SubString::SubString(String& x, int p, int l)
  : _str(x), _pos(p), _len(l) 
{
  if (_pos + _len > x._len) _len = x._len - _pos;
  assert((p >= 0) && (p <= x._len));
  assert((l >= 0) && (p + _len <= x._len));
}
inline SubString::SubString(const SubString& x) 
  : _str(x._str), _pos(x._pos), _len(x._len) {
}

inline SubString::~SubString() {
}


// inlined methods

inline const char* SubString::chars() const {
   return &(_str.chars())[_pos];
}
inline SubString::operator const char*() const {
   return chars();
}

inline unsigned int SubString::length() const {
   return _len;
}
inline int SubString::empty() const {
   return (_len == 0);
}


// ------------------- Global functions ---------------------

inline int compare(const String& x, const String& y) {
  return strcmp(x.chars(), y.chars());
}
inline int compare(const String& x, const char* t) {
  return strcmp(x.chars(), t);
}
int compare(const SubString& x, const String& y);
int compare(const String& x, const SubString&  y);
int compare(const SubString& x, const SubString&  y);
int compare(const SubString& x, const char* t);

 
// addition operators

inline String operator + (const String& x, const String& y) {
   String r(x); r += y; return r;
}
inline String operator + (const String& x, const SubString& y) {
   String r(x); r += y; return r;
}
inline String operator + (const String& x, const char* y) {
   String r(x); r += y; return r;
}
inline String operator + (const char* x, String& y) {
   String r(x); r += y; return r;
}
inline String operator + (const String& x, char y) {
   String r(x); r += y; return r;
}
inline String operator + (const SubString& x, const String& y) {
   String r(x); r += y; return r;
}
inline String operator + (const SubString& x, const SubString& y) {
   String r(x); r += y; return r;
}
inline String operator + (const SubString& x, const char* y) {
   String r(x); r += y; return r;
}
inline String operator + (const char* x, SubString& y) {
   String r(x); r += y; return r;
}
inline String operator + (const SubString& x, char y) {
   String r(x); r += y; return r;
}


// I/O operators  
  
inline ostream& operator<<(ostream& s, const String& x) {
   s << x.chars(); return s;
}
inline ostream& operator<<(ostream& s, const SubString& x) {
  s.write(x.chars(), x.length());
  return s;
}


// a zillion comparison operators

inline int operator==(const String& x, const String& y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(const String& x, const String& y)
{
  return compare(x, y) != 0; 
}

inline int operator>(const String& x, const String& y)
{
  return compare(x, y) > 0; 
}

inline int operator>=(const String& x, const String& y)
{
  return compare(x, y) >= 0; 
}

inline int operator<(const String& x, const String& y)
{
  return compare(x, y) < 0; 
}

inline int operator<=(const String& x, const String& y)
{
  return compare(x, y) <= 0; 
}

inline int operator==(const String& x, const SubString&  y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(const String& x, const SubString&  y)
{
  return compare(x, y) != 0; 
}

inline int operator>(const String& x, const SubString&  y)      
{
  return compare(x, y) > 0; 
}

inline int operator>=(const String& x, const SubString&  y)
{
  return compare(x, y) >= 0; 
}

inline int operator<(const String& x, const SubString&  y) 
{
  return compare(x, y) < 0; 
}

inline int operator<=(const String& x, const SubString&  y)
{
  return compare(x, y) <= 0; 
}

inline int operator==(const String& x, const char* t) 
{
  return compare(x, t) == 0; 
}

inline int operator!=(const String& x, const char* t) 
{
  return compare(x, t) != 0; 
}

inline int operator>(const String& x, const char* t)  
{
  return compare(x, t) > 0; 
}

inline int operator>=(const String& x, const char* t) 
{
  return compare(x, t) >= 0; 
}

inline int operator<(const String& x, const char* t)  
{
  return compare(x, t) < 0; 
}

inline int operator<=(const String& x, const char* t) 
{
  return compare(x, t) <= 0; 
}

inline int operator==(const SubString& x, const String& y) 
{
  return compare(y, x) == 0; 
}

inline int operator!=(const SubString& x, const String& y)
{
  return compare(y, x) != 0;
}

inline int operator>(const SubString& x, const String& y)      
{
  return compare(y, x) < 0;
}

inline int operator>=(const SubString& x, const String& y)     
{
  return compare(y, x) <= 0;
}

inline int operator<(const SubString& x, const String& y)      
{
  return compare(y, x) > 0;
}

inline int operator<=(const SubString& x, const String& y)     
{
  return compare(y, x) >= 0;
}

inline int operator==(const SubString& x, const SubString&  y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(const SubString& x, const SubString&  y)
{
  return compare(x, y) != 0;
}

inline int operator>(const SubString& x, const SubString&  y)      
{
  return compare(x, y) > 0;
}

inline int operator>=(const SubString& x, const SubString&  y)
{
  return compare(x, y) >= 0;
}

inline int operator<(const SubString& x, const SubString&  y) 
{
  return compare(x, y) < 0;
}

inline int operator<=(const SubString& x, const SubString&  y)
{
  return compare(x, y) <= 0;
}

inline int operator==(const SubString& x, const char* t) 
{
  return compare(x, t) == 0; 
}

inline int operator!=(const SubString& x, const char* t) 
{
  return compare(x, t) != 0;
}

inline int operator>(const SubString& x, const char* t)  
{
  return compare(x, t) > 0; 
}

inline int operator>=(const SubString& x, const char* t) 
{
  return compare(x, t) >= 0; 
}

inline int operator<(const SubString& x, const char* t)  
{
  return compare(x, t) < 0; 
}

inline int operator<=(const SubString& x, const char* t) 
{
  return compare(x, t) <= 0; 
}
  
  





#endif
