/************************************************************************/
/* thread.cc: SpecC run-time simulation library, thread management	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 03/26/01 */
/************************************************************************/

/* last update: 06/01/01 */

/* modifications: (most recent first)
 *
 * 06/01/01 RD	bug fix for bug showing up on linux
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/16/01 RD	resolved name space competition with user by moving more
 *		definitions into _specc name space and adding prefixes
 * 05/16/01 RD	separated common thread code (in thread.[h|cc]) from
 *		platform-specific thread code (in PosixThread.[h|cc])
 * 05/16/01 RD	unlocked the mutex for root when exiting
 * 05/15/01 RD	cleaned up the source code
 * 05/14/01 RD	clean up messy exception data structures
 * 05/12/01 RD	fixed memory leaks and some bugs with empty constructs
 * 05/11/01 RD	added support for exception handling
 * 04/16/01 RD	renamed "SIM_Time" to "sim_time"
 * 04/06/01 RD	added deadlock handling
 * 04/05/01 RD	bug fix in queue::Remove
 * 04/02/01 RD	replaced event queue and ready queue with generic queue
 * 03/30/01 RD	added generic queue implementation
 * 03/30/01 RD	added event queue (support for 'waitfor')
 * 03/29/01 RD	first working version ('par' support only)
 * 03/26/01 RD	initial version
 */

#include "thread.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>


/*** constants and macros ***********************************************/


//#define SIM_THREAD_DEBUG	/* enable/disable debugging */


/*** internal type declarations *****************************************/


	/* dummy behavior for empty exception handlers */

class _SIM_EmptyBhvr : public _specc::behavior
{
public:
_SIM_EmptyBhvr(void);		/* constructor */

virtual ~_SIM_EmptyBhvr(void);	/* destructor */

void main(void);		/* do nothing */
};


/*** internal function prototypes ***************************************/


static void *ThreadWrapper(	/* wrapper around the actual behavior */
	void		*Arg);


/*** internal variables *************************************************/


static _SIM_EmptyBhvr	EmptyHandler;	/* empty exception handler */


/*** exported variables *************************************************/


	/* (none */


/************************************************************************/
/*** implementation of internal functions			      ***/
/************************************************************************/


	/* dummy behavior for empty exception handlers */

_SIM_EmptyBhvr::_SIM_EmptyBhvr(void)	/* constructor */
{

/* nothing */

} /* end of _SIM_EmptyBhvr::_SIM_EmptyBhvr */


_SIM_EmptyBhvr::~_SIM_EmptyBhvr(void)	/* destructor */
{

/* nothing */

} /* end of _SIM_EmptyBhvr::~_SIM_EmptyBhvr */


void _SIM_EmptyBhvr::main(void)		/* do nothing */
{

/* nothing */

} /* end of _SIM_EmptyBhvr::main */


static void *ThreadWrapper(	/* wrapper around the actual behavior */
	void		*Arg)
{
_specc::thread	*Myself,
		*MyParent,
		*TryTree;

assert(Arg != NULL);
Myself = (_specc::thread*) Arg;

#ifdef SIM_THREAD_DEBUG
puts("SIM_THREAD_DEBUG: Child born, initializing...");
#endif /* SIM_THREAD_DEBUG */

Myself->CreationTime = _specc::CurrentTime;	/* remember my birthday! */

assert(_specc::CurrentThread == Myself);	/* it's my initial turn now */
MyParent = Myself->Parent;
assert(MyParent != NULL);			/* I need to know my parent! */

if (MyParent->Status == SIM_STATE_ABORTING)	/* am I an abortion handler? */
   { assert(MyParent->NumChildren == 2);
     assert(MyParent->LastChild == Myself);
     TryTree = MyParent->FirstChild;
     TryTree->Abort();				/* abort my sibling! */
     TryTree->Delete();				/* delete my sibling */
     assert(MyParent->NumChildren == 1);
     assert(MyParent->FirstChild == Myself);
    } /* fi */

#ifdef SIM_THREAD_DEBUG
puts("SIM_THREAD_DEBUG: Child behavior starts...");
#endif /* SIM_THREAD_DEBUG */

Myself->Behavior->main();	/* execute the concurrent behavior */

#ifdef SIM_THREAD_DEBUG
puts("SIM_THREAD_DEBUG: Child behavior done...");
#endif /* SIM_THREAD_DEBUG */

switch(MyParent->Status)
   { case SIM_STATE_JOIN:
	{ MyParent->NumCompleted++;
	  if (MyParent->NumChildren == MyParent->NumCompleted) /* last child? */
	     { MyParent->Status = SIM_STATE_READY;	/* wake up parent */
	       _specc::ReadyQueue.ADD(MyParent);
	      } /* fi */
	  else						/* wait for siblings */
	     { assert(MyParent->NumChildren > MyParent->NumCompleted);
	      } /* esle */
	  break;
	 }
     case SIM_STATE_TRYING:
     case SIM_STATE_ABORTING:
	{ MyParent->NumCompleted++;
	  assert(MyParent->NumChildren == 1);
	  assert(MyParent->NumCompleted == 1);
	  MyParent->Status = SIM_STATE_READY;	/* wake up parent */
	  _specc::TryQueue.Remove(MyParent);
	  _specc::ReadyQueue.ADD(MyParent);
	  break;
	 }
     case SIM_STATE_INTERRUPTED:
	{ MyParent->NumCompleted++;
	  assert(MyParent->NumChildren == 2);
	  assert(MyParent->NumCompleted == 1);
	  MyParent->Status = SIM_STATE_READY;	/* wake up parent */
	  _specc::TryQueue.Remove(MyParent);
	  _specc::ReadyQueue.ADD(MyParent);
	  break;
	 }
     case SIM_STATE_CREATED:
     case SIM_STATE_RUNNING:
     case SIM_STATE_READY:
     case SIM_STATE_NOTIFIED:
     case SIM_STATE_WAITFOR:
     case SIM_STATE_WAIT:
     case SIM_STATE_COMPLETE:
     case SIM_STATE_SUSPENDED:
     case SIM_STATE_SUSPENDED2:
	{ assert(false);	/* must not appear here */
	 }
     default:
	{ assert(false);	/* illegal status */
	 }
    } /* hctiws */

Myself->Status = SIM_STATE_COMPLETE;
Myself->Schedule();		/* call the scheduler */

return(NULL);	/* dummy result */

} /* end of ThreadWrapper */


/*** class implementations **********************************************/


	/*********************/
	/*** _specc::queue ***/
	/*********************/


_specc::queue::queue(		/* constructor #1 */
	void)
{

_specc::queue::First	= NULL;
_specc::queue::Last	= NULL;
_specc::queue::Length	= 0;

} /* end of _specc::queue::queue #1 */


_specc::queue::queue(		/* constructor #2 */
	thread	*First)
{

assert(First != NULL);

_specc::queue::First	= First;
_specc::queue::Last	= First;
_specc::queue::Length	= 1;

} /* end of _specc::queue::queue #2 */


_specc::queue::~queue(		/* destructor */
	void)
{

/* nothing to do */

} /* end of _specc::queue::~queue */


bool _specc::queue::IsEmpty(	/* check whether this queue is empty */
	void)
{

return(Length == 0);

} /* end of _specc::queue::IsEmpty */


void _specc::queue::Append(	/* append the thread to this queue */
	thread	*Thread)	/* (insert it as the last element) */
{

assert(Thread != NULL);		/* must exist */
assert(Thread->Next == NULL);	/* must not be in any queue */
assert(Thread->Prev == NULL);
assert(Thread->Queue == NULL);

if (Last)
   { assert(First != NULL);
     Last->Next = Thread;
     Thread->Prev = Last;
     Last = Thread;
    } /* fi */
else
   { assert(First == NULL);
     First = Thread;
     Last = Thread;
    } /* esle */

Thread->Queue = this;
Length++;

} /* end of _specc::queue::Append */


void _specc::queue::Prepend(	/* prepend the thread to this queue */
	thread	*Thread)	/* (insert it as the first element) */
{

assert(Thread != NULL);		/* must exist */
assert(Thread->Next == NULL);	/* must not be in any queue */
assert(Thread->Prev == NULL);
assert(Thread->Queue == NULL);

if (First)
   { assert(Last != NULL);
     First->Prev = Thread;
     Thread->Next = First;
     First = Thread;
    } /* fi */
else
   { assert(Last == NULL);
     First = Thread;
     Last = Thread;
    } /* esle */

Thread->Queue = this;
Length++;

} /* end of _specc::queue::Prepend */


void _specc::queue::Insert(		/* insert thread into the queue */
	thread		*Thread)	/* according to the time stamp */
{
thread		*Prev;

assert(Thread != NULL);		/* must exist */
assert(Thread->Next == NULL);	/* must not be in any queue */
assert(Thread->Prev == NULL);
assert(Thread->Queue == NULL);

if (First)
   { assert(Last != NULL);
     if (SIM_CMP_TIME(Thread->TimeStamp, First->TimeStamp))
	{ Prev = First;		/* search predecessor */
	  while(  (Prev->Next)
		&&(SIM_CMP_TIME(Thread->TimeStamp, Prev->Next->TimeStamp)))
	     { Prev = Prev->Next;
	      } /* elihw */
	  Thread->Next = Prev->Next;	/* insert here */
	  Thread->Prev = Prev;
	  if (Prev->Next)
	     { Prev->Next->Prev = Thread;
	      } /* fi */
	  else
	     { assert(Last == Prev);
	       Last = Thread;
	      } /* esle */
	  Prev->Next = Thread;
	 } /* fi */
     else	/* insert at the very front */
	{ Thread->Next = First;
	  First->Prev = Thread;
	  First = Thread;
	 } /* esle */
    } /* fi */
else	/* insert as the only element */
   { assert(Last == NULL);
     First = Thread;
     Last = Thread;
    } /* esle */

Thread->Queue = this;
Length++;

} /* end of _specc::queue::Insert */


void _specc::queue::Remove(	/* remove the thread from this queue */
	thread	*Thread)
{

assert(Thread != NULL);		/* must exist */

if (! Thread->Queue)
   { return;	/* not in any queue */
    } /* fi */

assert(Thread->Queue == this);	/* must be in this queue */

if (First == Thread)
   { First = Thread->Next;
    } /* fi */
if (Last == Thread)
   { Last = Thread->Prev;
    } /* fi */
if (Thread->Next)
   { Thread->Next->Prev = Thread->Prev;
    } /* fi */
if (Thread->Prev)
   { Thread->Prev->Next = Thread->Next;
    } /* fi */

Thread->Next = NULL;
Thread->Prev = NULL;
Thread->Queue = NULL;
Length--;

} /* end of _specc::queue::Remove */


	/**************************/
	/*** _specc::queue_elem ***/
	/**************************/


_specc::queue_elem::queue_elem(		/* constructor #1 */
	void)
{

_specc::queue_elem::Next	= NULL;
_specc::queue_elem::Prev	= NULL;
_specc::queue_elem::Queue	= NULL;
_specc::queue_elem::QueueBackup	= NULL;
_specc::queue_elem::TimeStamp	= 0;

} /* end of _specc::queue_elem::queue_elem #1 */


_specc::queue_elem::~queue_elem(	/* destructor */
	void)
{

/* nothing to do */

} /* end of _specc::queue_elem::~queue_elem */


bool _specc::queue_elem::IsInQueue(	/* check if thread is in any queue */
	void)
{

return(Queue != NULL);

} /* end of _specc::queue_elem::IsInQueue */


bool _specc::queue_elem::IsInQueue(	/* check if thread is in that queue */
	queue	*Queue)
{

return(_specc::queue_elem::Queue == Queue);

} /* end of _specc::queue_elem::IsInQueue */


	/**********************/
	/*** _specc::thread ***/
	/**********************/


_specc::thread::thread(		/* constructor #1 */
	behavior	*Behavior,
	thread		*Creator,
	thread		*Parent = NULL) : thread_base(Creator)
{

_specc::thread::Status		= SIM_STATE_CREATED;
_specc::thread::StatusBackup	= SIM_STATE_CREATED;
_specc::thread::Behavior	= Behavior;
_specc::thread::Parent		= Parent;
_specc::thread::NextSibling	= NULL;
_specc::thread::FirstChild	= NULL;
_specc::thread::LastChild	= NULL;
_specc::thread::NumChildren	= 0;
_specc::thread::NumCompleted	= 0;
_specc::thread::CreationTime	= 0;
_specc::thread::EventList	= NULL;
_specc::thread::Exceptions	= NULL;
_specc::thread::ExceptionChild	= NULL;

} /* end of _specc::thread::thread */


_specc::thread::~thread(void)	/* destructor */
{

if (Queue)
   { Queue->Remove(this);
    } /* fi */

delete EventList;
delete Exceptions;

} /* end of _specc::thread::~thread */


void _specc::thread::Start(	/* initialize thread usage */
	void)
{

#ifdef SIM_THREAD_DEBUG
puts("SIM_THREAD_DEBUG: Initializing simulation time...");
#endif /* SIM_THREAD_DEBUG */
CurrentTime = 0;		/* reset the simulation time */

assert(RootThread == NULL);	/* must not have been started before */
assert(CurrentThread == NULL);
assert(ReadyQueue.IsEmpty());	/* queues must be empty */
assert(WaitforQueue.IsEmpty());
assert(WaitQueue.IsEmpty());
assert(NotifiedQueue.IsEmpty());
assert(TryQueue.IsEmpty());
assert(SuspendQueue.IsEmpty());
assert(NotifiedEvents == NULL);	/* event lists must be empty */
assert(Notify1List == NULL);

#ifdef SIM_THREAD_DEBUG
puts("SIM_THREAD_DEBUG: Initializing thread package...");
#endif /* SIM_THREAD_DEBUG */
ThreadStart();

} /* end of _specc::thread::Start */


void _specc::thread::End(void)	/* clean up after thread usage */
{

#ifdef SIM_THREAD_DEBUG
puts("SIM_THREAD_DEBUG: Cleaning up thread package...");
#endif /* SIM_THREAD_DEBUG */
ThreadEnd();

#ifdef SIM_THREAD_DEBUG
puts("SIM_THREAD_DEBUG: Cleaning up data structures...");
#endif /* SIM_THREAD_DEBUG */

assert(RootThread == NULL);	/* must not exist any more */
assert(CurrentThread == NULL);
assert(ReadyQueue.IsEmpty());	/* queues must be empty */
assert(WaitforQueue.IsEmpty());
assert(WaitQueue.IsEmpty());
assert(NotifiedQueue.IsEmpty());
assert(TryQueue.IsEmpty());
assert(SuspendQueue.IsEmpty());

if (NotifiedEvents)		/* clean up notification lists */
   { NotifiedEvents->Reset();
     NotifiedEvents = NULL;
    } /* fi */
delete Notify1List;
Notify1List = NULL;

} /* end of _specc::thread::End */


_specc::thread *_specc::thread::Create(	/* create a new (child) thread */
	behavior	*Behavior,	/* (NULL for root thread) */
	thread		*Parent)	/* (NULL for root thread) */
{
thread		*NewThread;

#ifdef SIM_THREAD_DEBUG
puts("SIM_THREAD_DEBUG: Creating a new thread...");
#endif /* SIM_THREAD_DEBUG */
if (!(NewThread = new thread(Behavior, CurrentThread)))
   { fatalError("Cannot create new thread data", ENOMEM);
    } /* fi */

if (Behavior)			/* create new child thread */
   { assert(Parent != NULL);
     if (Parent->LastChild)
	{ assert(Parent->LastChild->NextSibling == NULL);
	  Parent->LastChild->NextSibling = NewThread;
	  Parent->LastChild = NewThread;
	 } /* fi */
     else
	{ Parent->FirstChild = NewThread;
	  Parent->LastChild = NewThread;
	 } /* esle */
     NewThread->Parent = Parent;
     Parent->NumChildren++;
     NewThread->ThreadCreate(&ThreadWrapper, (thread_arg)NewThread);
     NewThread->Status = SIM_STATE_READY;
     ReadyQueue.ADD(NewThread);
    } /* fi */
else				/* create root thread */
   { assert(Parent == NULL);
     NewThread->ThreadCreate(NULL, NULL);
     assert(RootThread == NULL); /* must be unused */
     RootThread = NewThread;	/* store the root thread */
     CurrentThread = NewThread;	/* already running */
     NewThread->Status = SIM_STATE_RUNNING;
    } /* esle */

#ifdef SIM_THREAD_DEBUG
puts("SIM_THREAD_DEBUG: Thread creation successful...");
#endif /* SIM_THREAD_DEBUG */

return(NewThread);

} /* end of _specc::thread::Create */


void _specc::thread::Delete(	/* delete this (completed) thread */
	void)
{
thread		*Pred;

assert(NumChildren == 0); /* must not have children any more */

if (Parent)
   { assert(Parent->FirstChild != NULL); /* must have at least this child */
     if (Parent->FirstChild == this)
	{ Parent->FirstChild = NextSibling;
	  if (Parent->LastChild == this)
	     { Parent->LastChild = NULL;
	      } /* fi */
	 } /* fi */
     else
        { Pred = Parent->FirstChild;
	  while(Pred->NextSibling)
	     { if (Pred->NextSibling == this)
		  { Pred->NextSibling = NextSibling;
		   } /* fi */
	       else
		  { Pred = Pred->NextSibling;
		   } /* esle */
	      } /* elihw */
	  Parent->LastChild = Pred;
	 } /* esle */
     Parent->NumChildren--;
     Parent->NumCompleted--;
    } /* fi */
else
   { assert(NextSibling == NULL); /* must not have any siblings */
    } /* esle */

if (RootThread == this)
   { RootThread = NULL;
    } /* fi */
if (CurrentThread == this)
   { CurrentThread = NULL;
    } /* fi */

ThreadDelete();
delete this;

} /* end of _specc::thread::Delete */


void _specc::thread::DeleteChildren(	/* delete all child threads */
	void)
{
thread		*Child,
		*Next;

Child = FirstChild;
while(Child)
   { Next = Child->NextSibling;
     assert(Child != CurrentThread);	/* must not commit suicide */
     assert(Child->NumChildren == 0);	/* must not have any grand children */
     delete Child;
     Child = Next;
    } /* elihw */

FirstChild = NULL;
LastChild = NULL;
NumChildren = 0;
NumCompleted = 0;

} /* end of _specc::thread::DeleteChildren */


void _specc::thread::Join(	/* wait for a child thread to complete */
	thread		*Thread)	/* (NULL joins will all children) */
{
thread		*Child;

assert(Status == SIM_STATE_RUNNING);	/* we just woke up */

#ifdef SIM_THREAD_DEBUG
puts("SIM_THREAD_DEBUG: Start joining...");
#endif /* SIM_THREAD_DEBUG */

if (Thread)		/* joining with a specific child */
   { assert(Thread->Parent == this);	/* must be parent of this child */
     ThreadJoin(Thread);
    } /* fi */
else			/* joining with all children */
   { Child = FirstChild;
     while (Child)
	{ ThreadJoin(Child);
	  Child = Child->NextSibling;
	 } /* elihw */
    } /* esle */

#ifdef SIM_THREAD_DEBUG
puts("SIM_THREAD_DEBUG: Joining done...");
#endif /* SIM_THREAD_DEBUG */

} /* end of _specc::thread::Join */


void _specc::thread::Suspend(void)		/* suspend this thread tree */
{						/* (recursion!) */

assert(this != NULL);

if (Status == SIM_STATE_SUSPENDED)	/* already suspended? */
   { Status = SIM_STATE_SUSPENDED2;	/* mark as end of this suspension */
     if (NextSibling)
	{ NextSibling->Suspend();	/* recursively suspend all siblings */
	 } /* fi */
     /* recursion stops here, child subtree is already suspended */
    } /* fi */
else
   { assert(QueueBackup == NULL); /* must be fresh */
     assert(StatusBackup == SIM_STATE_CREATED);
     QueueBackup = Queue;		/* backup this thread's status */
     StatusBackup = Status;
     if (Queue)				/* remove myself from any queues */
	{ Queue->Remove(this);
	 } /* fi */
     Status = SIM_STATE_SUSPENDED;	/* change my status to suspension */
     SuspendQueue.Append(this);
     if (FirstChild)
	{ FirstChild->Suspend();	/* recursively suspend all children */
	 } /* fi */
     if (NextSibling)
	{ NextSibling->Suspend();	/* recursively suspend all siblings */
	 } /* fi */
    } /* esle */

} /* end of _specc::thread::Suspend */


void _specc::thread::Resume(		/* resume this thread tree     */
	sim_time	Delay)		/* (taking delay into account) */
{					/* (recursion!) */

assert(this != NULL);

if (Status == SIM_STATE_SUSPENDED)
   { Status = StatusBackup;		/* restore status before suspension */
     StatusBackup = SIM_STATE_CREATED;
     SuspendQueue.Remove(this);
     if (QueueBackup)
	{ /* OK, this thread must have been in one of these queues:	*/
	  /* ReadyQueue, WaitforQueue, WaitQueue, or TryQueue		*/
	  /* (NotifiedQueue and SuspendQueue are not possible);		*/
	  if (QueueBackup == &ReadyQueue)
	     { ReadyQueue.ADD(this);
	      } /* fi */
	  else if (QueueBackup == &WaitforQueue)
	     { TimeStamp += Delay;
	       WaitforQueue.Insert(this);
	      } /* fi esle */
	  else if (QueueBackup == &WaitQueue)
	     { WaitQueue.Append(this);
	      } /* fi esle */
	  else
	     { assert(QueueBackup == &TryQueue);
	       TryQueue.Append(this);
	      } /* esle */
	  QueueBackup = NULL;
	 } /* fi */
     else
	{ assert(Queue == NULL); /* not in any queue */
	 } /* esle */
     if (FirstChild)
	{ FirstChild->Resume(Delay);	/* recursively resume all children */
	 } /* fi */
     if (NextSibling)
	{ NextSibling->Resume(Delay);	/* recursively resume all siblings */
	 } /* fi */
    } /* fi */
else
   { assert(Status == SIM_STATE_SUSPENDED2);
     Status = SIM_STATE_SUSPENDED;
     if (NextSibling)
	{ NextSibling->Resume(Delay);	/* recursively resume all siblings */
	 } /* fi */
     /* recursion stops here, child subtree was multiple times suspended */
    } /* esle */

} /* end of _specc::thread::Resume */


void _specc::thread::Abort(	/* abort this thread tree (force completion) */
	void)
{
thread		*Child;

assert(CurrentThread != this);	/* must not commit suicide! */

Child = FirstChild;
while(Child)
   { Child->Abort();		/* recursively abort all children */
     Child = Child->NextSibling;
    } /* elihw */
DeleteChildren();		/* delete all aborted children */

ThreadAbort();			/* abort this! */

Parent->NumCompleted++;

} /* end of _specc::thread::Abort */


void _specc::thread::Run(	/* transfer control to a specific thread */
	thread		*Thread)
{

#ifdef SIM_THREAD_DEBUG
puts("SIM_THREAD_DEBUG: Task switching...");
#endif /* SIM_THREAD_DEBUG */

assert(Thread != NULL);
assert(CurrentThread == this);

CurrentThread = Thread;
ThreadRun(Thread);

} /* end of _specc::thread::Run */


void _specc::thread::Schedule(	/* transfer control to another thread */
	void)
{
thread		*NextThread,
		*Thread,
		*Succ,
		*SelectedThread,
		*NextThreadPred;
event_ptr	*EventPtr;
event_list	*Notify1,
		**Notify1Ptr;
exception	*Exception;
sim_time	NextTime;

assert(CurrentThread == this);		/* this is the current thread */
assert(! IsInQueue(&ReadyQueue));	/* it is not in the ready queue */
assert(  (Status == SIM_STATE_RUNNING)	/* it is in state RUNNING or */
       ||(Status == SIM_STATE_WAITFOR)	/* was RUNNING just before   */
       ||(Status == SIM_STATE_WAIT)
       ||(Status == SIM_STATE_JOIN)
       ||(Status == SIM_STATE_COMPLETE)
       ||(Status == SIM_STATE_TRYING));


	/*** scheduler part 1: handle the ready queue ***/


if (Status == SIM_STATE_RUNNING)	/* possible context switch: */
   {
#ifdef SIM_MINIMIZE_CONTEXT_SWITCHES
     return;				/* (a) don't switch context */
#else /* ! SIM_MINIMIZE_CONTEXT_SWITCHES */
     Status = SIM_STATE_READY;		/* (b) allow context switch */
     ReadyQueue.ADD(this);
#endif /* SIM_MINIMIZE_CONTEXT_SWITCHES */
    } /* fi */

if ((NextThread = ReadyQueue.First))	/* any thread ready to run? */
   { ReadyQueue.Remove(NextThread);
     NextThread->Status = SIM_STATE_RUNNING;
     Run(NextThread);			/* run the next thread */
     return;
    } /* fi */


	/*** scheduler part 2: handle any exceptions ***/


assert(ReadyQueue.IsEmpty());	/* ready queue is empty here */

Thread = TryQueue.First;/* (note: the order of the try queue obeys the */
while(Thread)		/*        hierarchical priority of exceptions) */
   { assert(Thread->Status == SIM_STATE_TRYING);
     NextThreadPred = Thread;	/* walk through the queue */
     if (Notify1List)		/* exception handling for 'notifyone' */
	{ Notify1Ptr = &Notify1List;
	  Notify1 = Notify1List;
	  while(Notify1)
	     { if ((Exception = Thread->Exceptions->FirstMatch(
						Notify1->EventList)))
		  { NextThreadPred = Thread->Prev; /* keep location in queue */
		    Thread->HandleException(Exception);
		    *Notify1Ptr = Notify1->Next; /* consume this 'notifyone' */
		    Notify1->Next = NULL;
		    delete Notify1;
		    break; // next thread
		   } /* fi */
	       else
		  { Notify1Ptr = &Notify1->Next; /* skip to next */
		    Notify1 = Notify1->Next;
		   } /* esle */
	      } /* elihw */
	 } /* fi */
     if (  (NotifiedEvents)	/* exception handling for 'notify' */
	 &&(Thread->Status == SIM_STATE_TRYING)) /* (not triggered yet) */
	{ if ((Exception = Thread->Exceptions->FirstMatch()))
	     { NextThreadPred = Thread->Prev; /* keep location in queue */
	       Thread->HandleException(Exception);
	       // next thread
	      } /* fi */
	 } /* fi */
     if (NextThreadPred)	/* continue with next sensitive thread */
	{ Thread = NextThreadPred->Next;
	 } /* fi */
     else
	{ Thread = TryQueue.First;
	 } /* esle */
    } /* elihw */


	/*** scheduler part 3: handle any notifications ***/


if (Notify1List)	/* perform notification handling for 'notifyone' */
   { Notify1 = Notify1List;
     while(Notify1)
	{ SelectedThread = NULL;
#ifdef SIM_NOTIFYONE_SELECT_FIRST_WAITING_THREAD
	  Thread = WaitQueue.First;
	  while(Thread)
	     { if (Notify1->Triggers(Thread->EventList))
		  { SelectedThread = Thread;
		    break;
		   } /* fi */
	       Thread = Thread->Next;
	      } /* elihw */
#endif /* SIM_NOTIFYONE_SELECT_FIRST_WAITING_THREAD */
#ifdef SIM_NOTIFYONE_SELECT_LAST_WAITING_THREAD
	  Thread = WaitQueue.Last;
	  while(Thread)
	     { if (Notify1->Triggers(Thread->EventList))
		  { SelectedThread = Thread;
		    break;
		   } /* fi */
	       Thread = Thread->Prev;
	      } /* elihw */
#endif /* SIM_NOTIFYONE_SELECT_LAST_WAITING_THREAD */
	  if (SelectedThread)
	     { assert(SelectedThread->Status == SIM_STATE_WAIT);
	       SelectedThread->Status = SIM_STATE_NOTIFIED;
	       WaitQueue.Remove(SelectedThread);
	       NotifiedQueue.Append(SelectedThread);
	      } /* fi */
	  else
	     { /* we could search further in the NotifiedQueue, */
	       /* but since those threads are already notified, */
	       /* nothing would change; so, we stop here;	*/
	      } /* esle */
	  Notify1 = Notify1->Next;
	 } /* elihw */
     delete Notify1List;	/* reset all 'notifyone' events */
     Notify1List = NULL;
    } /* fi */

if (NotifiedEvents)	/* perform notification handling for 'notify' */
   { Thread = WaitQueue.First;
     while(Thread)
	{ Succ = Thread->Next;
	  EventPtr = Thread->EventList;
	  while(EventPtr)
	     { if (EventPtr->Event->Notified)
		  { assert(Thread->Status == SIM_STATE_WAIT);
		    Thread->Status = SIM_STATE_NOTIFIED;
		    WaitQueue.Remove(Thread);
		    NotifiedQueue.Append(Thread);
		    break;
		   } /* fi */
	       EventPtr = EventPtr->Next;
	      } /* elihw */
	  Thread = Succ;
	 } /* elihw */
     NotifiedEvents->Reset();	/* reset all 'notify' events */
     NotifiedEvents = NULL;
    } /* fi */

Thread = NotifiedQueue.First;	/* set all notified threads ready to run */
while(Thread)
   { Succ = Thread->Next;
     assert(Thread->Status == SIM_STATE_NOTIFIED);
     Thread->Status = SIM_STATE_READY;
     NotifiedQueue.Remove(Thread);
     ReadyQueue.ADD(Thread);
     Thread = Succ;
    } /* elihw */


	/*** scheduler part 4: handle synchronization cycle ***/


if ((NextThread = ReadyQueue.First))	/* any thread ready to run? */
   { ReadyQueue.Remove(NextThread);
     NextThread->Status = SIM_STATE_RUNNING;
     Run(NextThread);			/* run the next thread */
     return;
    } /* fi */


	/*** scheduler part 5: advance simulation time ***/


assert(ReadyQueue.IsEmpty());	/* ready queue is empty here */

if (! WaitforQueue.IsEmpty())	/* anybody waiting for time advance? */
   { Thread = WaitforQueue.First;
     assert(Thread != NULL);
     NextTime = Thread->TimeStamp;
     assert(NextTime >= CurrentTime);	/* never go backwards in time! */
     CurrentTime = NextTime;		/* advance the simulation time */
     do { Succ = Thread->Next;
	  WaitforQueue.Remove(Thread);	/* enable threads that wake up now */
	  Thread->TimeStamp = 0;
	  assert(Thread->Status == SIM_STATE_WAITFOR);
	  Thread->Status = SIM_STATE_READY;
	  ReadyQueue.ADD(Thread);
	  Thread = Succ;
	 } while(  (Thread)
		 &&(Thread->TimeStamp == CurrentTime));
     NextThread = ReadyQueue.First;
     assert(NextThread != NULL); /* at least one was just put in */
     ReadyQueue.Remove(NextThread);
     NextThread->Status = SIM_STATE_RUNNING;
     Run(NextThread);			/* run the next thread */
     return;
    } /* fi */


	/*** scheduler part 6: handle deadlocks ***/


assert(ReadyQueue.IsEmpty());		/* ready queue is empty here */
assert(WaitforQueue.IsEmpty());		/* waitfor queue is empty as well */
assert(NotifiedQueue.IsEmpty());	/* notify queue is empty anyways */
// TryQueue may or may not be empty
// WaitQueue may or may not be empty
// SuspendQueue may or may not be empty

/* note: typically, there is at least one thread stuck	*/
/*       in the WaitQueue or in the SuspendQueue;	*/
/*       however, both queues can also be empty when	*/
/*       an empty non-terminating 'pipe' is executed;	*/

fprintf(stderr,
	"\n"
	"libsim: Deadlock detected!\n"
	"libsim: Time %s, 0 threads ready, %u waiting, %u suspended.\n"
	"libsim: Exiting.\n",
	time2str(CurrentTime), WaitQueue.Length, SuspendQueue.Length);

exit(RESULT_DEADLOCK);

} /* end of _specc::thread::Schedule */


void _specc::thread::HandleException(		/* handle an exception */
	exception	*Exception)
{
thread		*TryChild;

assert(this != NULL);
assert(Exception != NULL);
assert(Status == SIM_STATE_TRYING);
assert(NumChildren == 1);

TryChild = FirstChild;

if (Exception->IsTrap)			/* handle an abortion exception */
   { Status = SIM_STATE_ABORTING;
     TryQueue.Remove(this);
/* Note: It's very tempting to call                        */
/*   TryChild->Abort();                                    */
/* directly at this point, but this will lead to a suicide */
/* of the CurrentThread if the CurrentThread is one of the */
/* threads being aborted (which can easily happen!).       */
/* Committing suicide, however, will lead to a deadlock    */
/* situation where no thread can proceed any more...       */
/* So, we simply suspend all threads for the time being    */
/* and let the abortion handler abort its 'try' sibling    */
/* (and all the 'try' children) before the handler starts  */
/* its actual work (see ThreadWrapper() above).            */
     TryChild->Suspend();
     assert(ExceptionChild == NULL);
     if (Exception->Behavior)
	{ ExceptionChild = Create(Exception->Behavior, this);
	 } /* fi */
     else
	{ ExceptionChild = Create(&EmptyHandler, this);
	 } /* esle */
    } /* fi */
else					/* handle an interrupt exception */
   { Status = SIM_STATE_INTERRUPTED;
     TryQueue.Remove(this);
     TryChild->Suspend();
     assert(ExceptionChild == NULL);
     if (Exception->Behavior)
	{ ExceptionChild = Create(Exception->Behavior, this);
	 } /* fi */
     else
	{ ExceptionChild = Create(&EmptyHandler, this);
	 } /* esle */
    } /* esle */

assert(ExceptionChild != NULL);

} /* end of _specc::thread::HandleException */


/************************************************************************/
/*** implementation of exported functions			      ***/
/************************************************************************/


	/* (none) */


/* EOF thread.cc */
