%{
/* PRS_Scanner.sed: PREPROCESSED BY SED */
SED+KANJI/* (Japanese Kanji encoding support enabled) */
SED-KANJI/* (Japanese Kanji encoding support disabled) */

/************************************************************************/
/* PRS_Scanner.l: Scanner for Parser.c					*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 07/02/97 */
/************************************************************************/

/* last update: 05/30/01 */

/* modifications: (most recent first)
 *
 * 05/30/01 RD	reduced contents to reference compiler needs
 * 05/29/01 RD	removed false error message when parsing from stdin
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/18/01 RD	removed an invalid assertion in PRS_Pop()
 * 05/05/01 RD	fixed error reporting at end of nested files (TOK_SCANNER_EOF)
 * 05/04/01 RD	added support for import of SpecC source (.sc) files
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/18/01 RD	moved LONG_LONG_MAX definitions out to longlong.h
 * 04/16/01 RD	added support for non-native 'long long' type;
 *		changed use of 'long long int' to TIME type
 * 01/23/01 RD	bug fix recognizing 1234lL as a valid long long constant
 */

/*
 * This scanner is specified based on:
 *
 * technical reports ICS-TR-97-15, ICS-TR-97-16, ICS-TR-97-26
 * of the University of California, Irvine
 *
 * and
 * 
 * cpp5.l, a C/C++ scanner written by James A. Roskind.
 * "Portions Copyright (c) 1989, 1990 James A. Roskind".
 * (http://www.idiom.com/free-compilers/,
 * ftp://ftp.infoseek.com/ftp/pub/c++grammar/*,
 * ftp://ftp.sra.co.jp/.a/pub/cmd/c++grammar2.0.tar.gz)
 */

/* note: this file must be pre-processed by sed in order select		*/
/*	 whether Japanese Kanji encoding shall be supported or not;	*/
/*	 for Kanji support, lines starting with SED+KANJI will		*/
/*	 be enabled, lines starting with SED-KANJI will be disabled,	*/
/*	 or vice-versa; (see below)					*/

#include "PRS_Scanner.h"
#include "PRS_Parser.h"
#include <limits.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>


/*** constants and macros ***********************************************/


	/* options to input 'long long' types */

#ifdef HAVE_LLONG
//#define LONGLONG_USE_NATIVE_INPUT_ROUTINES
#define LONGLONG_USE_SPECC_INPUT_ROUTINES
#else /* HAVE_LLONG */
#define LONGLONG_USE_SPECC_INPUT_ROUTINES
#endif /* HAVE_LLONG */


/*** optimize the generated scanner *************************************/


	/* modify input function to prohibit reading directories */

#undef	YY_INPUT
#define YY_INPUT(buf, result, max_size)	\
	if ((result = read(fileno(PRS_in), (char*) buf, max_size)) < 0 ) \
	   { result = 0; }

	/* forbid any output of the scanner */

#undef	ECHO
#define ECHO

	/* initialize the line number */

#undef	YY_USER_INIT
#define YY_USER_INIT

	/* stop when reaching the end of file */

#define PRS_wrap()	TRUE


#ifdef LONGLONG_USE_NATIVE_INPUT_ROUTINES

	/*** portability definitions ***/

#ifdef SOLARIS
#define str2ull(b,s)	strtoull(s, NULL, b)
#endif /* SOLARIS */

#ifdef SUNOS4
#define str2ull(b,s)	strtol(s, NULL, b)
#define ONLY_LONG_PRECISION
#endif /* SUNOS4 */

#ifdef NETBSD
#define str2ull(b,s)	strtouq(s, NULL, b)
#endif /* NETBSD */

#ifdef LINUX
#define str2ull(b,s)	strtouq(s, NULL, b)
#endif /* LINUX */

#ifdef GNUWIN32
#define str2ull(b,s)	strtol(s, NULL, b)
#define ONLY_LONG_PRECISION
#endif /* GNUWIN32 */

#ifndef str2ull
#error "Unknown system! str2ull() definition missing!"
#endif /* str2ull */

#endif /* LONGLONG_USE_NATIVE_INPUT_ROUTINES */


#ifdef LONGLONG_USE_SPECC_INPUT_ROUTINES

#ifdef str2ull
#error "str2ull definition overwrites SpecC function!"
#endif /* str2ull */

#ifdef ONLY_LONG_PRECISION
#error "ONLY_LONG_PRECISION defined with SpecC function!"
#endif /* ONLY_LONG_PRECISION */

#endif /* LONGLONG_USE_SPECC_INPUT_ROUTINES */


/*** internal variables *************************************************/


static string			TmpString;
static UNSIGNED_LONG_LONG	BigValue;
static YY_BUFFER_STATE		PRS_InitialBuffer,
				PRS_TextBuffer;
static prs_grammar		TheGrammarType;


/*** exported variables *************************************************/


sir_design	*PRS_TheDesign;		/* the current design */
sir_symbols	*PRS_CurrScope;		/* the current scope */
unsigned int	PRS_lineno;		/* current line number */
string		PRS_CurrentFile;	/* current filename */
sir_fileinfo	*PRS_CurrFileInfo;	/* current fileinfo */
GL_STRINGLIST	*PRS_ImportDirList;	/* paths to search for import */

prs_stackitem	*PRS_FileStack = NULL;	/* stack of nested files */


/*** macros for easier rule definition **********************************/
%}

delimiter	[ \t\b\r]
newline		[\n\f\v]
whitespace	{delimiter}+
ws		{delimiter}*
ucletter	[A-Z]
lcletter	[a-z]
letter		({ucletter}|{lcletter})
digit		[0-9]
bindigit	[01]
octdigit	[0-7]
hexdigit	[0-9a-fA-F]
identifier	(({letter}|"_")({letter}|{digit}|"_")*)
integer		{digit}+
binary		{bindigit}+
decinteger	[1-9]{digit}*
octinteger	"0"{octdigit}*
hexinteger	"0"[xX]{hexdigit}+
decinteger_u	{decinteger}[uU]
octinteger_u	{octinteger}[uU]
hexinteger_u	{hexinteger}[uU]
decinteger_l	{decinteger}[lL]
octinteger_l	{octinteger}[lL]
hexinteger_l	{hexinteger}[lL]
decinteger_ul	{decinteger}([uU][lL]|[lL][uU])
octinteger_ul	{octinteger}([uU][lL]|[lL][uU])
hexinteger_ul	{hexinteger}([uU][lL]|[lL][uU])
decinteger_ll	{decinteger}[lL][lL]
octinteger_ll	{octinteger}[lL][lL]
hexinteger_ll	{hexinteger}[lL][lL]
decinteger_ull	{decinteger}([uU][lL][lL]|[lL][lL][uU])
octinteger_ull	{octinteger}([uU][lL][lL]|[lL][lL][uU])
hexinteger_ull	{hexinteger}([uU][lL][lL]|[lL][lL][uU])
octchar		"\\"{octdigit}{1,3}
hexchar		"\\x"{hexdigit}+
exponent	[eE][+-]?{integer}
fraction	{integer}
float1		{integer}"."{fraction}?({exponent})?
float2		"."{fraction}({exponent})?
float3		{integer}{exponent}
float		{float1}|{float2}|{float3}
float_f		{float}[fF]
float_l		{float}[lL]
bitvector	{binary}[bB]
bitvector_u	{binary}([uU][bB]|[bB][uU])
cppstart	{ws}"#"{ws}
cppflag		{whitespace}[1-4]
cppfile		"\""[ !#-~]*"\""
cpplineno	^{cppstart}{integer}{whitespace}{cppfile}{cppflag}*{ws}{newline}
cpppragma	^{cppstart}"pragma"{ws}.*
cppdirective	^{cppstart}.*

SED+KANJI%{		/* Japanese 2-byte kanji encodings */
SED+KANJI%}
SED+KANJIkanji_jis	[\x81-\x9f\xe0-\xfc][\x40-\x7e\x80-\xfc]
SED+KANJIkanji_euc	[\xa1-\xfe][\xa1-\xfe]
SED+KANJIkanji		{kanji_jis}|{kanji_euc}


%x GRAMMAR
%x COMMENT1
%x COMMENT2
%x STRNGLIT

%%
%{
/*** rules to identify tokens *******************************************/
%}

%{
/*** handle scanner state GRAMMAR ***/

/* note: this is the actual initial state in order to select a grammar */
%}

<GRAMMAR>.|\n	{ BEGIN(INITIAL);	/* next we do some real work */
		  yyless(0);		/* start again with this character */
		  yy_current_buffer->yy_at_bol = 1; /* at beginning of line */
		  switch(TheGrammarType)
		     { case PRS_PARSE_LANGUAGE:
			  { return(TOK_PARSE_LANGUAGE);
			   }
		       default:
			  { assert(FALSE);
			   }
		      } /* hctiws */
		 }

%{
/*
 * Comments should be removed by the preprocessor, but we allow them anyway
 */

/*** handle scanner state COMMENT1 ***/
%}

<INITIAL>"/*"	{ BEGIN(COMMENT1);	/* begin C comment state */
#ifdef PRS_SCANNER_DEBUG_COMMENTS
		  printf("SCANNER_DEBUG_COMMENTS:\t%s", &PRS_text[0]);
#endif /* PRS_SCANNER_DEBUG_COMMENTS */
		 }

<COMMENT1>{
   "*/"		{ BEGIN(INITIAL);	/* end comment state, back INITIAL */
#ifdef PRS_SCANNER_DEBUG_COMMENTS
		  printf("%s\n", &PRS_text[0]);
#endif /* PRS_SCANNER_DEBUG_COMMENTS */
		 }

   "/*"		{ PRS_PrintWarning(GL_WARN_STANDARD,
					"Probably nested comments");
#ifdef PRS_SCANNER_DEBUG_COMMENTS
		  printf("%s", &PRS_text[0]);
#endif /* PRS_SCANNER_DEBUG_COMMENTS */
		 }

   <<EOF>>	{ PRS_error("Unterminated comment");
		  return(TOK_SCANNER_ERROR);
		 }

   [^*/\n]*	{ /* ignore every char except '*' and NL (performance!) */
#ifdef PRS_SCANNER_DEBUG_COMMENTS
		  printf("%s", &PRS_text[0]);
#endif /* PRS_SCANNER_DEBUG_COMMENTS */
		 }

   .		{ /* all single characters within comments are ignored */
#ifdef PRS_SCANNER_DEBUG_COMMENTS
		  printf("%s", &PRS_text[0]);
#endif /* PRS_SCANNER_DEBUG_COMMENTS */
		 }

   \n		{ PRS_lineno++;	/* count line feeds within comments */
		  PRS_NumberOfLines++;
#ifdef PRS_SCANNER_DEBUG_COMMENTS
		  printf("\nSCANNER_DEBUG_COMMENTS:\t");
#endif /* PRS_SCANNER_DEBUG_COMMENTS */
		 }
	}

%{
/*** handle scanner state COMMENT2 ***/

/* note: we are C based, but why not allow C++ style comments? */
%}


<INITIAL>"//"	{ BEGIN(COMMENT2);	/* begin C++ comment state */
#ifdef PRS_SCANNER_DEBUG_COMMENTS
		  printf("SCANNER_DEBUG_COMMENTS:\t%s", &PRS_text[0]);
#endif /* PRS_SCANNER_DEBUG_COMMENTS */
		 }

<COMMENT2>{
   \n		{ BEGIN(INITIAL);	/* end comment state, back INITIAL */
		  PRS_lineno++;		/* count the line feed */
		  PRS_NumberOfLines++;
#ifdef PRS_SCANNER_DEBUG_COMMENTS
		  printf("%s", &PRS_text[0]);
#endif /* PRS_SCANNER_DEBUG_COMMENTS */
		 }

   .*		{ /* all characters within comments are ignored */
#ifdef PRS_SCANNER_DEBUG_COMMENTS
		  printf("%s", &PRS_text[0]);
#endif /* PRS_SCANNER_DEBUG_COMMENTS */
		 }
	}

%{
/*** handle scanner state STRNGLIT ***/
%}

<INITIAL>"\""	{ BEGIN(STRNGLIT);	/* begin string literal state */
		  PRS_lval.String = new PRS_StringToken;
		 }

<STRNGLIT>{
   "\""		{ BEGIN(INITIAL);	/* end string state, back INITIAL */
#ifdef PRS_SCANNER_DEBUG_STRINGS
		  printf("SCANNER_DEBUG_STRINGS: \"%s\" (%d)\n",
				PRS_lval.String->CharString.chars(),
				PRS_lval.String->CharString.length());
#endif /* PRS_SCANNER_DEBUG_STRINGS */
		  return(TOK_STRING);
		 }

   <<EOF>>	{ delete PRS_lval.String;
		  PRS_error("Unterminated string constant");
		  return(TOK_SCANNER_ERROR);
		 }

   "\\n"	{ PRS_lval.String->CharString += '\n';	/* NL (0x0a) */
		 }

   "\\t"	{ PRS_lval.String->CharString += '\t';	/* HT (0x09) */
		 }

   "\\v"	{ PRS_lval.String->CharString += '\v';	/* VT (0x0b) */
		 }

   "\\b"	{ PRS_lval.String->CharString += '\b';	/* BS (0x08) */
		 }

   "\\r"	{ PRS_lval.String->CharString += '\r';	/* CR (0x0d) */
		 }

   "\\f"	{ PRS_lval.String->CharString += '\f';	/* FF (0x0c) */
		 }

   "\\a"	{ PRS_lval.String->CharString += '\a';	/* BEL (0x07) */
		 }

   {octchar}	{ PRS_lval.String->CharString +=	/* octal character */
			(char)strtol(&PRS_text[1], NULL, 8);
		 }

   {hexchar}	{ PRS_lval.String->CharString +=	/* hex character */
			(char)strtol(&PRS_text[2], NULL, 16);
		 }

   "\\".	{ PRS_lval.String->CharString += /* ignore the backslash */
				PRS_text[1];
		 }

SED+KANJI   {kanji}	{ PRS_lval.String->CharString += /* 2-byte codes */
SED+KANJI				&PRS_text[0];
SED+KANJI		 }

SED+KANJI   .		{ PRS_lval.String->CharString += /* standard stuff */
SED+KANJI				PRS_text[0];	 /* (slow version) */
SED+KANJI		 }

SED-KANJI   [^\\\"\n]*	{ PRS_lval.String->CharString += /* standard stuff */
SED-KANJI				&PRS_text[0];	 /* (fast version) */
SED-KANJI		 }

   \n		{ delete PRS_lval.String;
		  PRS_error("Unterminated string constant");
		  return(TOK_SCANNER_ERROR);
		 }
	}

%{
/*** the following defaults to scanner state INITIAL ***/
%}

{newline}	{ PRS_lineno++;
		  PRS_NumberOfLines++;
		 }
{whitespace}	{ /* skipped */
		 }

{cpplineno}	{ const char	*CharPtr;

		  PRS_NumberOfLines++;
#ifdef PRS_SCANNER_DEBUG_CPPLINES
		  printf("SCANNER_DEBUG_CPPLINES: %s", &PRS_text[0]);
#endif /* PRS_SCANNER_DEBUG_CPPLINES */
		  CharPtr = &PRS_text[0];
		  while(! isdigit(*CharPtr))
		     { CharPtr++;
		      } /* elihw */
		  PRS_lineno = atoi(CharPtr);
		  while(*CharPtr != '"')
		     { CharPtr++;
		      } /* elihw */
		  CharPtr++;
		  PRS_CurrentFile = CharPtr;
		  PRS_CurrentFile = PRS_CurrentFile.before('"', -1);
#ifdef PRS_SCANNER_DEBUG_CPPLINES
		  printf("SCANNER_DEBUG_CPPLINES: "
			"Line %u, File \"%s\".\n",
			PRS_lineno, PRS_CurrentFile.chars());
#endif /* PRS_SCANNER_DEBUG_CPPLINES */
		  if ((! PRS_CurrentFile.empty()) && (PRS_lineno >= 1))
		     { PRS_CurrFileInfo = PRS_TheDesign->FileList->FindOrInsert(
						PRS_CurrentFile.chars());
		      } /* fi */
		 }
{cpppragma}	{
#ifdef PRS_SCANNER_DEBUG_CPPLINES
		  printf("SCANNER_DEBUG_CPPLINES: %s\n", &PRS_text[0]);
#endif /* PRS_SCANNER_DEBUG_CPPLINES */
		  TmpString.form(
			"'pragma' directive found:" GL_WARN_MSG_NEWLINE
			"%s" GL_WARN_MSG_NEWLINE
			"Directive ignored",
			&PRS_text[0]);
		  PRS_PrintWarning(GL_WARN_INFORMATIVE, TmpString.chars());
		  TmpString = "";
		 }
{cppdirective}	{
#ifdef PRS_SCANNER_DEBUG_CPPLINES
		  printf("SCANNER_DEBUG_CPPLINES: %s\n", &PRS_text[0]);
#endif /* PRS_SCANNER_DEBUG_CPPLINES */
		  PRS_error("Preprocessor directive found");
		  return(TOK_SCANNER_ERROR);
		 }

%{
/*** keywords ***/
%}

"auto"		{ return(TOK_AUTO); }
"break"		{ return(TOK_BREAK); }
"case"		{ return(TOK_CASE); }
"char"		{ return(TOK_CHAR); }
"const"		{ return(TOK_CONST); }
"continue"	{ return(TOK_CONTINUE); }
"default"	{ return(TOK_DEFAULT); }
"do"		{ return(TOK_DO); }
"double"	{ return(TOK_DOUBLE); }
"else"		{ return(TOK_ELSE); }
"enum"		{ return(TOK_ENUM); }
"extern"	{ return(TOK_EXTERN); }
"float"		{ return(TOK_FLOAT); }
"for"		{ return(TOK_FOR); }
"goto"		{ return(TOK_GOTO); }
"if"		{ return(TOK_IF); }
"int"		{ return(TOK_INT); }
"long"		{ return(TOK_LONG); }
"register"	{ return(TOK_REGISTER); }
"return"	{ return(TOK_RETURN); }
"short"		{ return(TOK_SHORT); }
"signed"	{ return(TOK_SIGNED); }
"sizeof"	{ return(TOK_SIZEOF); }
"static"	{ return(TOK_STATIC); }
"struct"	{ return(TOK_STRUCT); }
"switch"	{ return(TOK_SWITCH); }
"typedef"	{ return(TOK_TYPEDEF); }
"union"		{ return(TOK_UNION); }
"unsigned"	{ return(TOK_UNSIGNED); }
"void"		{ return(TOK_VOID); }
"volatile"	{ return(TOK_VOLATILE); }
"while"		{ return(TOK_WHILE); }

"behavior"	{ return(TOK_BEHAVIOR); }
"bit"		{ return(TOK_BIT); }
"bool"		{ return(TOK_BOOL); }
"channel"	{ return(TOK_CHANNEL); }
"event"		{ return(TOK_EVENT); }
"false"		{ return(TOK_FALSE); }
"fsm"		{ return(TOK_FSM); }
"implements"	{ return(TOK_IMPLEMENTS); }
"import"	{ return(TOK_IMPORT); }
"in"		{ return(TOK_IN); }
"inout"		{ return(TOK_INOUT); }
"interface"	{ return(TOK_INTERFACE); }
"interrupt"	{ return(TOK_INTERRUPT); }
"note"		{ return(TOK_NOTE); }
"notify"	{ return(TOK_NOTIFY); }
"notifyone"	{ return(TOK_NOTIFYONE); }
"out"		{ return(TOK_OUT); }
"par"		{ return(TOK_PAR); }
"pipe"		{ return(TOK_PIPE); }
"piped"		{ return(TOK_PIPED); }
"range"		{ return(TOK_RANGE); }
"this"		{ return(TOK_THIS); }
"timing"	{ return(TOK_TIMING); }
"trap"		{ return(TOK_TRAP); }
"true"		{ return(TOK_TRUE); }
"try"		{ return(TOK_TRY); }
"wait"		{ return(TOK_WAIT); }
"waitfor"	{ return(TOK_WAITFOR); }

"asm"			|
"catch"			|
"class"			|
"const_cast"		|
"delete"		|
"dynamic_cast"		|
"explicit"		|
"export"		|
"friend"		|
"inline"		|
"mutable"		|
"namespace"		|
"new"			|
"operator"		|
"private"		|
"protected"		|
"public"		|
"reinterpret_cast"	|
"static_cast"		|
"template"		|
"throw"			|
"typeid"		|
"typename"		|
"using"			|
"virtual"	{ TmpString.form("'%s' is a reserved keyword",
					&PRS_text[0]);
		  PRS_error(TmpString.chars());
		  return(TOK_SCANNER_ERROR);
		 }

"->"		{ return(TOK_ARROW); }
"++"		{ return(TOK_INCR); }
"--"		{ return(TOK_DECR); }
"<<"		{ return(TOK_SHIFTLEFT); }
">>"		{ return(TOK_SHIFTRIGHT); }
"<="		{ return(TOK_LE); }
">="		{ return(TOK_GE); }
"=="		{ return(TOK_EQ); }
"!="		{ return(TOK_NE); }
"&&"		{ return(TOK_ANDAND); }
"||"		{ return(TOK_OROR); }
"..."		{ return(TOK_ELLIPSIS); }

"*="		{ return(TOK_MULTASSIGN); }
"/="		{ return(TOK_DIVASSIGN); }
"%="		{ return(TOK_MODASSIGN); }
"+="		{ return(TOK_PLUSASSIGN); }
"-="		{ return(TOK_MINUSASSIGN); }
"<<="		{ return(TOK_SLASSIGN); }
">>="		{ return(TOK_SRASSIGN); }
"&="		{ return(TOK_ANDASSIGN); }
"^="		{ return(TOK_EORASSIGN); }
"|="		{ return(TOK_ORASSIGN); }

{identifier}	{ PRS_lval.Name = new PRS_NameToken;
		  PRS_lval.Name->Name = &PRS_text[0];
		  PRS_lval.Name->Symbol = PRS_CurrScope->Find(&PRS_text[0]);
		  if (PRS_lval.Name->Symbol)
		     { switch(PRS_lval.Name->Symbol->Class)
			  { case SIR_SYMBOL_IDENTIFIER:
			      { return(TOK_IDENTIFIER);
			       }
			    case SIR_SYMBOL_TYPEDEF:
			      { return(TOK_TYPEDEFNAME);
			       }
			    case SIR_SYMBOL_BEHAVIOR:
			      { return(TOK_BEHAVIORNAME);
			       }
			    case SIR_SYMBOL_CHANNEL:
			      { return(TOK_CHANNELNAME);
			       }
			    case SIR_SYMBOL_INTERFACE:
			      { return(TOK_INTERFACENAME);
			       }
			    default:
			      { assert(FALSE);	/* bad symbol class */
			       }
			   } /* hctiws */
		      } /* fi */
		  return(TOK_IDENTIFIER);
		 }

{decinteger}	{ BigValue = str2ull(10, &PRS_text[0]);
		  if (BigValue > INT_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Decimal constant %s out of range for 'int'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_INT, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{octinteger}	{ BigValue = str2ull(8, &PRS_text[0]);
		  if (BigValue > UINT_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Octal constant %s out of range for 'int'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_INT, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{hexinteger}	{ BigValue = str2ull(16, &PRS_text[2]);
		  if (BigValue > UINT_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Hexadecimal constant %s out of range for 'int'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_INT, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{decinteger_u}	{ BigValue = str2ull(10, &PRS_text[0]);
		  if (BigValue > UINT_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Decimal constant %s out of range for 'unsigned int'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_UINT, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{octinteger_u}	{ BigValue = str2ull(8, &PRS_text[0]);
		  if (BigValue > UINT_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Octal constant %s out of range for 'unsigned int'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_UINT, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{hexinteger_u}	{ BigValue = str2ull(16, &PRS_text[2]);
		  if (BigValue > UINT_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
		"Hexadecimal constant %s out of range for 'unsigned int'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_UINT, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{decinteger_l}	{ BigValue = str2ull(10, &PRS_text[0]);
		  if (BigValue > LONG_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Decimal constant %s out of range for 'long'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_LONG, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{octinteger_l}	{ BigValue = str2ull(8, &PRS_text[0]);
		  if (BigValue > ULONG_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Octal constant %s out of range for 'long'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_LONG, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{hexinteger_l}	{ BigValue = str2ull(16, &PRS_text[2]);
		  if (BigValue > ULONG_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Hexadecimal constant %s out of range for 'long'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_LONG, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{decinteger_ul}	{ BigValue = str2ull(10, &PRS_text[0]);
		  if (BigValue > ULONG_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Decimal constant %s out of range for 'unsigned long'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_ULONG, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{octinteger_ul}	{ BigValue = str2ull(8, &PRS_text[0]);
		  if (BigValue > ULONG_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Octal constant %s out of range for 'unsigned long'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_ULONG, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{hexinteger_ul}	{ BigValue = str2ull(16, &PRS_text[2]);
		  if (BigValue > ULONG_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
		"Hexadecimal constant %s out of range for 'unsigned long'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_ULONG, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{decinteger_ll}	{ BigValue = str2ull(10, &PRS_text[0]);
#ifdef ONLY_LONG_PRECISION
		  PRS_PrintWarning(GL_WARN_IMPORTANT,
		"'long long' constant is parsed with only 'long' precision");
#else /* ONLY_LONG_PRECISION */
		  if (BigValue > LONG_LONG_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Decimal constant %s out of range for 'long long'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
#endif /* ONLY_LONG_PRECISION */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_LONGLONG, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{octinteger_ll}	{ BigValue = str2ull(8, &PRS_text[0]);
#ifdef ONLY_LONG_PRECISION
		  PRS_PrintWarning(GL_WARN_IMPORTANT,
		"'long long' constant is parsed with only 'long' precision");
#else /* ONLY_LONG_PRECISION */
		  if (BigValue > ULONG_LONG_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Octal constant %s out of range for 'long long'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
#endif /* ONLY_LONG_PRECISION */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_LONGLONG, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{hexinteger_ll}	{ BigValue = str2ull(16, &PRS_text[2]);
#ifdef ONLY_LONG_PRECISION
		  PRS_PrintWarning(GL_WARN_IMPORTANT,
		"'long long' constant is parsed with only 'long' precision");
#else /* ONLY_LONG_PRECISION */
		  if (BigValue > ULONG_LONG_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
			"Hexadecimal constant %s out of range for 'long long'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
#endif /* ONLY_LONG_PRECISION */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_LONGLONG, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{decinteger_ull} {BigValue = str2ull(10, &PRS_text[0]);
#ifdef ONLY_LONG_PRECISION
		  PRS_PrintWarning(GL_WARN_IMPORTANT,
	"'unsigned long long' constant is parsed with only 'long' precision");
#else /* ONLY_LONG_PRECISION */
		  if (BigValue > ULONG_LONG_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
		"Decimal constant %s out of range for 'unsigned long long'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
#endif /* ONLY_LONG_PRECISION */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_ULONGLONG, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{octinteger_ull} {BigValue = str2ull(8, &PRS_text[0]);
#ifdef ONLY_LONG_PRECISION
		  PRS_PrintWarning(GL_WARN_IMPORTANT,
	"'unsigned long long' constant is parsed with only 'long' precision");
#else /* ONLY_LONG_PRECISION */
		  if (BigValue > ULONG_LONG_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
		"Octal constant %s out of range for 'unsigned long long'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
#endif /* ONLY_LONG_PRECISION */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_ULONGLONG, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }
{hexinteger_ull} {BigValue = str2ull(16, &PRS_text[2]);
#ifdef ONLY_LONG_PRECISION
		  PRS_PrintWarning(GL_WARN_IMPORTANT,
	"'unsigned long long' constant is parsed with only 'long' precision");
#else /* ONLY_LONG_PRECISION */
		  if (BigValue > ULONG_LONG_MAX)
		     { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
			"In file \"%s\", line %u," GL_WARN_MSG_NEWLINE
		"Hexadecimal constant %s out of range for 'unsigned long long'",
				PRS_CurrentFile.chars(), PRS_lineno,
				&PRS_text[0]);
		      } /* fi */
#endif /* ONLY_LONG_PRECISION */
		  PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_ULONGLONG, BigValue,
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_INTEGER);
		 }

{float}		{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_DOUBLE,
					(long double) atof(&PRS_text[0]),
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_FLOATING);
		 }
{float_f}	{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_FLOAT,
					(long double) atof(&PRS_text[0]),
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_FLOATING);
		 }
{float_l}	{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_LONGDOUBLE, 0.0l,
					PRS_lineno, PRS_CurrFileInfo);
#ifdef SOLARIS
		  sscanf(&PRS_text[0], "%Lf", &PRS_lval.Const->Const->LD_Value);
#endif /* SOLARIS */
#ifdef SUNOS4
		  sscanf(&PRS_text[0], "%lf", &PRS_lval.Const->Const->D_Value);
		  PRS_lval.Const->Const->LD_Value =
						PRS_lval.Const->Const->D_Value;
		  if (sizeof(long double) > sizeof(double))
		     { PRS_PrintWarning(GL_WARN_IMPORTANT,
	"'long double' constant is parsed with only 'double' precision");
		      } /* fi */
#endif /* SUNOS4 */
#ifdef NETBSD
		  sscanf(&PRS_text[0], "%Lf", &PRS_lval.Const->Const->LD_Value);
#endif /* NETBSD */
#ifdef LINUX
		  sscanf(&PRS_text[0], "%Lf", &PRS_lval.Const->Const->LD_Value);
#endif /* LINUX */
#ifdef GNUWIN32
		  sscanf(&PRS_text[0], "%Lf", &PRS_lval.Const->Const->LD_Value);
#endif /* GNUWIN32 */
#ifndef SOLARIS
#   ifndef SUNOS4
#      ifndef NETBSD
#         ifndef LINUX
#            ifndef GNUWIN32
#error "Unknown system! Don't know how to parse 'long double' constants!"
#            endif /* GNUWIN32 */
#         endif /* LINUX */
#      endif /* NETBSD */
#   endif /* SUNOS4 */
#endif /* SOLARIS */
		  return(TOK_FLOATING);
		 }

{bitvector}	{ char		*Bitvector;
		  int		Length;

		  PRS_lval.Const = new PRS_ConstToken;
		  Length = strlen(&PRS_text[0]) - 1;	/* without "b" */
		  Bitvector = new char[Length+1];
		  strncpy(Bitvector, &PRS_text[0], Length);
		  *(Bitvector + Length) = 0;	/* append terminating 0 */
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_BIT, Bitvector,
					PRS_lineno, PRS_CurrFileInfo);
		  delete[] Bitvector;
		  return(TOK_BITVECTOR);
		 }
{bitvector_u}	{ char		*Bitvector;
		  int		Length;

		  PRS_lval.Const = new PRS_ConstToken;
		  Length = strlen(&PRS_text[0]) - 2;	/* without "ub" */
		  Bitvector = new char[Length+1];
		  strncpy(Bitvector, &PRS_text[0], Length);
		  *(Bitvector + Length) = 0;	/* append terminating 0 */
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_UBIT, Bitvector,
					PRS_lineno, PRS_CurrFileInfo);
		  delete[] Bitvector;
		  return(TOK_BITVECTOR);
		 }

"'\\n'"		{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_CHAR, '\n',	/* NL (LF) */
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_CHARACTER);
		 }
"'\\t'"		{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_CHAR, '\t',	/* HT */
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_CHARACTER);
		 }
"'\\v'"		{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_CHAR, '\v',	/* VT */
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_CHARACTER);
		 }
"'\\b'"		{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_CHAR, '\b',	/* BS */
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_CHARACTER);
		 }
"'\\r'"		{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_CHAR, '\r',	/* CR */
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_CHARACTER);
		 }
"'\\f'"		{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_CHAR, '\f',	/* FF */
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_CHARACTER);
		 }
"'\\a'"		{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_CHAR, '\a',	/* BEL */
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_CHARACTER);
		 }
"'"{octchar}"'"	{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_CHAR, /* octal character */
					(char)strtol(&PRS_text[2], NULL, 8),
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_CHARACTER);
		 }
"'"{hexchar}"'"	{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_CHAR, /* hex character */
					(char)strtol(&PRS_text[3], NULL, 16),
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_CHARACTER);
		 }
"'\\"."'"	{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_CHAR, /* ignore backslash */
					PRS_text[2],
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_CHARACTER);
		 }
"'"."'"		{ PRS_lval.Const = new PRS_ConstToken;
		  PRS_lval.Const->Const =
			new SIR_Constant(SIR_CONST_CHAR, /* standard char */
					PRS_text[1],
					PRS_lineno, PRS_CurrFileInfo);
		  return(TOK_CHARACTER);
		 }

.		{ return(PRS_text[0]);
		 }

<<EOF>>		{ if (PRS_FileStack)	/* check if we're in a nested file */
		     { return(TOK_SCANNER_EOF);	/* notify parser */
		      } /* fi */
		  else
		     { yyterminate();		/* done! */
		      } /* esle */
		 }

%%
/*** implementation of exported functions *******************************/


void PRS_Initialize(		/* initialize (reset) the scanner */
	FILE		*NewInputHandle,
	prs_grammar	GrammarType = PRS_PARSE_LANGUAGE)
{
static BOOL	HasBeenCalledBefore = FALSE;	/* flag for virginity */

PRS_restart(NewInputHandle);

if (! HasBeenCalledBefore)	/* first time? */
   { PRS_InitialBuffer = YY_CURRENT_BUFFER;
     HasBeenCalledBefore = TRUE;	/* not a virgin any more */
    } /* fi */

BEGIN(GRAMMAR);		/* we always start with grammar selection! */
TheGrammarType = GrammarType;

} /* end of PRS_Initialize */


ERROR PRS_Push(				/* start parsing a nested file */
	const char	*NestedName,	/* (import a SpecC source file) */
	const char	*DesignName,
	const char	*PreprocName)
{
ERROR		Error;
FILE		*NestedFile;
prs_stackitem	*NewStackItem;

assert(NestedName != NULL);
assert(DesignName != NULL);
assert(PreprocName != NULL);
assert(PRS_CurrScope == PRS_TheDesign->Symbols); /* must be at top level */

if (!(NestedFile = GL_OpenFileForReading(PreprocName)))
   { return(GL_Error);
    } /* fi */

NewStackItem = new PRS_StackItem(PRS_FileStack,
				NestedName, PreprocName,
				PRS_TheDesign,
				PRS_lineno, PRS_CurrentFile,
				PRS_CurrFileInfo, PRS_in,
				(void*)YY_CURRENT_BUFFER);
PRS_FileStack = NewStackItem;	/* push data onto the stack */

PRS_TheDesign = new SIR_Design(DesignName);	/* start new design */
PRS_CurrScope = PRS_TheDesign->Symbols;		/* start new global scope */
PRS_lineno = 1;					/* reset current line */
PRS_CurrentFile = NestedName;			/* reset current filename */
PRS_CurrFileInfo = PRS_TheDesign->FileList->FindOrInsert(NestedName);
PRS_in = NestedFile;				/* start new file descriptor */

yy_switch_to_buffer(yy_create_buffer(NestedFile, YY_BUF_SIZE));

return(PRS_ERROR_NO_ERROR);

} /* end of PRS_Push */


sir_design *PRS_Pop(		/* resume parsing after importing a file */
	string		*ImportFileName = NULL)
{
sir_design	*ImportedDesign;
prs_stackitem	*StackItem;

assert(PRS_FileStack != NULL);		/* stack must not be empty */

assert(PRS_in != NULL);
GL_CloseFile(PRS_in);

assert(PRS_TheDesign != NULL);
ImportedDesign = PRS_TheDesign;

StackItem = PRS_FileStack;	/* pop the data from the stack */
PRS_FileStack = StackItem->Next;

if (ImportFileName)
   { *ImportFileName = StackItem->NestedFileName;
    } /* fi */
GL_DeleteFile(StackItem->PreprocName.chars());	/* delete the temp. file */

PRS_TheDesign = StackItem->Design;		/* restore the design */
PRS_CurrScope = PRS_TheDesign->Symbols;		/* restore the scope */
PRS_lineno = StackItem->LineNo;			/* restore the line number */
PRS_CurrentFile = StackItem->FileName;		/* restore the file name */
PRS_CurrFileInfo = StackItem->FileInfo;		/* restore the file info */
PRS_in = StackItem->File;			/* restore the file handle */

yy_delete_buffer(YY_CURRENT_BUFFER);		/* delete the temp. buffer */
yy_switch_to_buffer((YY_BUFFER_STATE)StackItem->Buffer);

delete StackItem;

return(ImportedDesign);

} /* end of PRS_Pop */


void PRS_ClearStack(void)	/* clean up the stack (if any) */
{
prs_stackitem	*StackEntry;
sir_design	*ImportedDesign;
string		ImportErrorMsg,
		TmpString;

if (  (PRS_FileStack)	/* extend the error message (if any) */
    &&(PRS_Error))
   { StackEntry = PRS_FileStack;
     ImportErrorMsg.form("%s (#%d)" GL_ERROR_MSG_NEWLINE
			"(file \"%s\" is imported from" GL_ERROR_MSG_NEWLINE,
			PRS_ErrorText(PRS_Error), PRS_Error,
			PRS_FileStack->NestedFileName.chars());
     while(StackEntry)
	{ TmpString.form("file \"%s\", line %u",
			StackEntry->FileName.chars(),
			StackEntry->LineNo);
	  ImportErrorMsg += TmpString;
	  if (StackEntry->Next)
	     { ImportErrorMsg += ", which is imported from"
				GL_ERROR_MSG_NEWLINE;
	      } /* fi */
	  StackEntry = StackEntry->Next;
	 } /* elihw */
     ImportErrorMsg += ")";
     PRS_error(ImportErrorMsg, PRS_ERROR_IMPORT_FAILED);
    } /* fi */

while(PRS_FileStack)	/* perform the clean up */
   { ImportedDesign = PRS_Pop();
     delete ImportedDesign;
    } /* elihw */

} /* end of PRS_ClearStack */


/*** constructors and destructors ***/


PRS_ScannerToken::PRS_ScannerToken(void)
{

Line = PRS_lineno;
FileInfo = PRS_CurrFileInfo;

} /* end of PRS_ScannerToken::PRS_ScannerToken */


PRS_ScannerToken::~PRS_ScannerToken(void)
{

/* nothing to do */

} /* end of PRS_ScannerToken::~PRS_ScannerToken */


PRS_StackItem::PRS_StackItem(
	prs_stackitem	*Next,
	const char	*NestedFileName,
	const char	*PreprocName,
	sir_design	*Design,
	unsigned int	LineNo,
	string		FileName,
	sir_fileinfo	*FileInfo,
	FILE		*File,
	void		*Buffer)
{

PRS_StackItem::Next		= Next;
PRS_StackItem::NestedFileName	= NestedFileName;
PRS_StackItem::PreprocName	= PreprocName;
PRS_StackItem::Design		= Design;
PRS_StackItem::LineNo		= LineNo;
PRS_StackItem::FileName		= FileName;
PRS_StackItem::FileInfo		= FileInfo;
PRS_StackItem::File		= File;
PRS_StackItem::Buffer		= Buffer;

} /* end of PRS_StackItem::PRS_StackItem */


PRS_StackItem::~PRS_StackItem(void)
{

/* nothing to do */

} /* end of PRS_StackItem::~PRS_StackItem */


/* EOF PRS_Scanner.l */
