%{
/************************************************************************/
/* PRS_Parser.y: yacc parser for Parser.c				*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 07/02/97 */
/************************************************************************/

/* last update: 05/30/01 */

/* modifications: (most recent first)
 *
 * 05/30/01 RD	reduced contents to reference compiler needs
 * 05/27/01 RD	added semantic checking for event port accesses
 * 05/26/01 RD	bug fix for source code 'import'
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/05/01 RD	fixed error reporting at end of nested files (TOK_SCANNER_EOF)
 * 05/04/01 RD	added support for import of SpecC source (.sc) files
 * 05/03/01 RD	replaced GL_OpenFileWithDirList() by GL_FindFileInDirList()
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/25/01 RD	added rule for constant_expression in port mappings
 * 04/18/01 RD	bug fix: negative array sizes are forbidden
 * 04/16/01 RD	fixed broken PRS_SCANNER_DEBUG_TOKENS portions
 * 03/08/01 RD	added (optional) constant-check for 'waitfor' argument
 * 03/02/01 RD	added optional arguments to 'pipe' statement
 * 02/13/01 RD	allow constant and open port mappings
 * 02/12/01 RD	allowed "bit[8]" as a shortcut for "bit[7:0]"
 * 05/26/00 AG	Changed concatenation operator in port mappings to '@'
 */

/*
 * This parser is specified based on:
 *
 * technical reports ICS-TR-97-15, ICS-TR-97-16, ICS-TR-97-26
 * of the University of California, Irvine
 *
 * and
 *
 * Bjarne Stroustrup, "The C++ programming language" (third edition),
 * Addison Wesley, Reading, Massachusetts, 1997.
 *
 * and
 *
 * c5.y, a ANSI-C grammar written by James A. Roskind.
 * "Portions Copyright (c) 1989, 1990 James A. Roskind".
 * (http://www.idiom.com/free-compilers/,
 * ftp://ftp.infoseek.com/ftp/pub/c++grammar/*,
 * ftp://ftp.sra.co.jp/.a/pub/cmd/c++grammar2.0.tar.gz)
 */

#include "Parser.h"
#include "PRS_Scanner.h"
#include "CcDriver.h"

#include <stdlib.h>
#include <assert.h>


/*** token declaration **************************************************/
%}

/*** ANSI-C keywords ***/

%token	TOK_AUTO
%token	TOK_BREAK
%token	TOK_CASE
%token	TOK_CHAR
%token	TOK_CONST
%token	TOK_CONTINUE
%token	TOK_DEFAULT
%token	TOK_DO
%token	TOK_DOUBLE
%token	TOK_ELSE
%token	TOK_ENUM
%token	TOK_EXTERN
%token	TOK_FLOAT
%token	TOK_FOR
%token	TOK_GOTO
%token	TOK_IF
%token	TOK_INT
%token	TOK_LONG
%token	TOK_REGISTER
%token	TOK_RETURN
%token	TOK_SHORT
%token	TOK_SIGNED
%token	TOK_SIZEOF
%token	TOK_STATIC
%token	TOK_STRUCT
%token	TOK_SWITCH
%token	TOK_TYPEDEF
%token	TOK_UNION
%token	TOK_UNSIGNED
%token	TOK_VOID
%token	TOK_VOLATILE
%token	TOK_WHILE

/*** SpecC-only keywords ***/

%token	TOK_BEHAVIOR
%token	TOK_BIT
%token	TOK_BOOL
%token	TOK_CHANNEL
%token	TOK_EVENT
%token	TOK_FALSE
%token	TOK_FSM
%token	TOK_IMPLEMENTS
%token	TOK_IMPORT
%token	TOK_IN
%token	TOK_INOUT
%token	TOK_INTERFACE
%token	TOK_INTERRUPT
%token	TOK_NOTE
%token	TOK_NOTIFY
%token	TOK_NOTIFYONE
%token	TOK_OUT
%token	TOK_PAR
%token	TOK_PIPE
%token	TOK_PIPED
%token	TOK_RANGE
%token	TOK_THIS
%token	TOK_TIMING
%token	TOK_TRAP
%token	TOK_TRUE
%token	TOK_TRY
%token	TOK_WAIT
%token	TOK_WAITFOR

/*** multi-character operators ***/

%token	TOK_ARROW
%token	TOK_INCR
%token	TOK_DECR
%token	TOK_SHIFTLEFT
%token	TOK_SHIFTRIGHT
%token	TOK_LE
%token	TOK_GE
%token	TOK_EQ
%token	TOK_NE
%token	TOK_ANDAND
%token	TOK_OROR
%token	TOK_ELLIPSIS

/*** modifying assignment operators ***/

%token	TOK_MULTASSIGN
%token	TOK_DIVASSIGN
%token	TOK_MODASSIGN
%token	TOK_PLUSASSIGN
%token	TOK_MINUSASSIGN
%token	TOK_SLASSIGN
%token	TOK_SRASSIGN
%token	TOK_ANDASSIGN
%token	TOK_EORASSIGN
%token	TOK_ORASSIGN

/*** scanner parsed tokens (these have a value!) ***/

%token	<Name>		TOK_IDENTIFIER
%token	<Name>		TOK_TYPEDEFNAME
%token	<Name>		TOK_BEHAVIORNAME
%token	<Name>		TOK_CHANNELNAME
%token	<Name>		TOK_INTERFACENAME
%token	<Const>		TOK_INTEGER
%token	<Const>		TOK_FLOATING
%token	<Const>		TOK_CHARACTER
%token	<String>	TOK_STRING
%token	<Const>		TOK_BITVECTOR

/*** special scanner reports ***/

%token	TOK_SCANNER_ERROR	/* used by scanner to report errors */
%token	TOK_SCANNER_EOF		/* used by scanner to report end of import */

/*** parser mode (determined by the scanner) ***/

%token	TOK_PARSE_LANGUAGE


/*** priority, associativity, etc. definitions **************************/


%start	grammar

%expect 1	/* the famous "dangling `else'" ambiguity */
		/* results in one shift/reduce conflict   */
		/* that we don't want to be reported      */


/*** type declarations **************************************************/


%union		/* the different types of PRS_lval */
{
prs_nametoken	*Name;
prs_consttoken	*Const;
prs_stringtoken	*String;
prs_lineinfo	*LineInfo;
sir_constant	*Constant;
sir_expression	*Expression;
SIR_EXPRTYPE	ExpressionType;
sir_expressions	*ExpressionList;
sir_type	*Type;
sir_declarator	*Declarator;		/* declarator under construction */
sir_declarator	*AbstrDeclarator;	/* declarator with type info only */
sir_declarator	*MmbrDeclarator;	/* declarator for aggregate members */
sir_basic_type	*BasicType;
sir_parameter	*Parameter;
sir_parameters	*ParameterList;
sir_declspec	*DeclarationSpec;
sir_initializer	*Initializer;
sir_initials	*InitializerList;
sir_symbol	*Symbol;
sir_symbol_list	*SymbolList;
sir_symbols	*Scope;
sir_symbol_ptr	*SymbolPtr;
sir_symbol_ptrs	*SymbolPtrList;
sir_statement	*Statement;
sir_statements	*StatementList;
SIR_DIRECTION	Direction;
sir_transition	*Transition;
sir_transitions	*TransitionList;
sir_exception	*Exception;
sir_exceptions	*ExceptionList;
sir_constraint	*Constraint;
sir_constraints	*ConstraintList;
sir_usertype	*UserType;
SIR_USERTYPE	UserTypeClass;
sir_member	*Member;
sir_members	*MemberList;
sir_declspec	*MemberDeclSpec;
sir_portmaps	*PortMapList;
sir_bitslice	*BitSlice;
sir_bitslices	*BitSliceList;
}

	/* return types of rules */

%type <Name>		identifier
%type <Name>		typedef_name
%type <Name>		behavior_name
%type <Name>		channel_name
%type <Name>		interface_name
%type <Const>		integer
%type <Const>		floating
%type <Const>		character
%type <String>		string
%type <Const>		bitvector
%type <Constant>	constant
%type <String>		string_literal_list
%type <Expression>	primary_expression
%type <Expression>	postfix_expression
%type <Name>		member_name
%type <ExpressionList>	argument_expression_list
%type <Expression>	unary_expression
%type <ExpressionType>	unary_operator
%type <Expression>	cast_expression
%type <Expression>	concat_expression
%type <Expression>	multiplicative_expression
%type <Expression>	additive_expression
%type <Expression>	shift_expression
%type <Expression>	relational_expression
%type <Expression>	equality_expression
%type <Expression>	and_expression
%type <Expression>	exclusive_or_expression
%type <Expression>	inclusive_or_expression
%type <Expression>	logical_and_expression
%type <Expression>	logical_or_expression
%type <Expression>	conditional_expression
%type <Expression>	assignment_expression
%type <ExpressionType>	assignment_operator
%type <Expression>	comma_expression
%type <Expression>	constant_expression
%type <Expression>	comma_expression_opt
/* %type <void>		declaration */
%type <DeclarationSpec>	default_declaring_list
%type <DeclarationSpec>	declaring_list
%type <DeclarationSpec>	declaration_specifier
%type <Type>		type_specifier
%type <BasicType>	declaration_qualifier_list
%type <BasicType>	type_qualifier_list
%type <BasicType>	declaration_qualifier
%type <BasicType>	type_qualifier
%type <BasicType>	basic_declaration_specifier
%type <BasicType>	basic_type_specifier
%type <DeclarationSpec>	sue_declaration_specifier
%type <Type>		sue_type_specifier
%type <DeclarationSpec>	typedef_declaration_specifier
%type <Type>		typedef_type_specifier
%type <BasicType>	storage_class
%type <BasicType>	basic_type_name
%type <Type>		elaborated_type_name
%type <Type>		aggregate_name
%type <UserTypeClass>	aggregate_key
%type <MemberList>	member_declaration_list
%type <MemberList>	member_declaration
%type <MemberDeclSpec>	member_default_declaring_list
%type <MemberDeclSpec>	member_declaring_list
%type <MmbrDeclarator>	member_declarator
%type <MmbrDeclarator>	member_identifier_declarator
%type <Expression>	bit_field_size_opt
%type <Expression>	bit_field_size
%type <Type>		enum_name
%type <MemberList>	enumerator_list
%type <Expression>	enumerator_value_opt
%type <ParameterList>	parameter_type_list
%type <ParameterList>	parameter_list
%type <Parameter>	parameter_declaration
%type <Name>		identifier_or_typedef_name
%type <Type>		type_name
%type <Initializer>	initializer_opt
%type <Initializer>	initializer
%type <InitializerList>	initializer_list
%type <LineInfo>	line_info
%type <Statement>	statement
%type <Statement>	labeled_statement
%type <Statement>	compound_statement
%type <Statement>	compound_scope
/* %type <void>		declaration_list */
%type <StatementList>	statement_list
%type <Statement>	expression_statement
%type <Statement>	selection_statement
%type <Statement>	iteration_statement
%type <Statement>	jump_statement
/* %type <void>		translation_unit */
/* %type <void>		external_definition_list */
/* %type <void>		external_definition */
/* %type <void>		function_definition */
%type <Declarator>	declarator
%type <Declarator>	typedef_declarator
%type <Declarator>	parameter_typedef_declarator
%type <Declarator>	clean_typedef_declarator
%type <Declarator>	clean_postfix_typedef_declarator
%type <Declarator>	paren_typedef_declarator
%type <Declarator>	paren_postfix_typedef_declarator
%type <Declarator>	simple_paren_typedef_declarator
%type <Declarator>	identifier_declarator
%type <Declarator>	unary_identifier_declarator
%type <Declarator>	postfix_identifier_declarator
%type <Declarator>	paren_identifier_declarator
%type <AbstrDeclarator>	abstract_declarator
%type <AbstrDeclarator>	postfixing_abstract_declarator
%type <AbstrDeclarator>	array_abstract_declarator
%type <AbstrDeclarator>	unary_abstract_declarator
%type <AbstrDeclarator>	postfix_abstract_declarator
/* %type <void>		spec_c_definition */
/* %type <void>		import_definition */
/* %type <void>		behavior_declaration */
/* %type <void>		behavior_definition */
%type <Declarator>	behavior_specifier
/* %type <void>		channel_declaration */
/* %type <void>		channel_definition */
%type <Declarator>	channel_specifier
%type <ParameterList>	port_list_opt
%type <ParameterList>	port_list
%type <Parameter>	port_declaration
%type <Direction>	port_direction
%type <Parameter>	interface_parameter
%type <SymbolPtrList>	implements_interface_opt
%type <SymbolPtrList>	interface_list
/* %type <void>		internal_definition_list_opt */
/* %type <void>		internal_definition_list */
/* %type <void>		internal_definition */
/* %type <void>		instantiation */
%type <DeclarationSpec>	instance_declaring_list
%type <Declarator>	instance_declarator
%type <DeclarationSpec>	behavior_or_channel
%type <PortMapList>	port_mapping_list_opt
%type <PortMapList>	port_mapping_list
%type <BitSliceList>	port_mapping_opt
%type <BitSliceList>	port_mapping
%type <BitSlice>	bit_slice
/* %type <void>		interface_declaration */
/* %type <void>		interface_definition */
%type <Declarator>	interface_specifier
/* %type <void>		internal_declaration_list_opt */
/* %type <void>		internal_declaration_list */
/* %type <void>		internal_declaration */
/* %type <void>		note_definition */
%type <Constant>	note
%type <Name>		typedef_or_class_name
%type <Name>		any_name
%type <Statement>	spec_c_statement
%type <Statement>	concurrent_statement
%type <Statement>	fsm_statement
%type <TransitionList>	transition_list
%type <TransitionList>	transition
%type <TransitionList>	cond_branch_list
%type <Transition>	cond_branch
%type <Statement>	exception_statement
%type <ExceptionList>	exception_list_opt
%type <ExceptionList>	exception_list
%type <Exception>	exception
%type <SymbolPtrList>	paren_event_list
%type <SymbolPtrList>	event_list
%type <SymbolPtr>	event
%type <Statement>	timing_statement
%type <ConstraintList>	constraint_list_opt
%type <ConstraintList>	constraint_list
%type <Constraint>	constraint
%type <Constant>	time_opt
%type <Expression>	time
%type <Statement>	wait_statement
%type <Statement>	waitfor_statement
%type <Statement>	notify_statement


%{
/*** internal variables *************************************************/


static string		TmpErrMsg;	/* dynamic error messages */
static sir_usertype	*CurrEnumType	/* current type of enum members */
				= NULL;


/************************************************************************/
/*** rules **************************************************************/
/************************************************************************/
%}
%%

/*** Grammar selection **************************************************/


/* note: this rule is NOT part of the SpecC language!		*/
/*	 (this is just a trick to use the same parser for	*/
/*	 multiple inputs, e.g. the complete SpecC language,	*/
/*	 as well as only parts of it)				*/

grammar:			/* void */
	/*** standard SpecC language ***/
	TOK_PARSE_LANGUAGE translation_unit
	{ /* nothing to do */
	 };


/*** Token with values **************************************************/


identifier:			/* Name */
	TOK_IDENTIFIER
	{
#ifdef PRS_SCANNER_DEBUG_TOKENS
	  printf("SCANNER_DEBUG_TOKENS: Identifier = %s\n",
		$1->Name.chars());
	  printf("SCANNER_DEBUG_TOKENS:    in file \"%s\"\n"
		 "SCANNER_DEBUG_TOKENS:    in line %d\n",
			$1->FileInfo->Filename.chars(),
			$1->Line);
#endif /* PRS_SCANNER_DEBUG_TOKENS */
	  $$ = $1;
	 };

typedef_name:			/* Name */
	TOK_TYPEDEFNAME
	{
#ifdef PRS_SCANNER_DEBUG_TOKENS
	  printf("SCANNER_DEBUG_TOKENS: TypeDefName = %s\n",
		$1->Name.chars());
	  printf("SCANNER_DEBUG_TOKENS:    in file \"%s\"\n"
		 "SCANNER_DEBUG_TOKENS:    in line %d\n",
			$1->FileInfo->Filename.chars(),
			$1->Line);
#endif /* PRS_SCANNER_DEBUG_TOKENS */
	  assert($1->Symbol != 0);
	  $$ = $1;
	 };

behavior_name:			/* Name */
	TOK_BEHAVIORNAME
	{
#ifdef PRS_SCANNER_DEBUG_TOKENS
	  printf("SCANNER_DEBUG_TOKENS: BehaviorName = %s\n",
		$1->Name.chars());
	  printf("SCANNER_DEBUG_TOKENS:    in file \"%s\"\n"
		 "SCANNER_DEBUG_TOKENS:    in line %d\n",
			$1->FileInfo->Filename.chars(),
			$1->Line);
#endif /* PRS_SCANNER_DEBUG_TOKENS */
	  assert($1->Symbol != 0);
	  $$ = $1;
	 };

channel_name:			/* Name */
	TOK_CHANNELNAME
	{
#ifdef PRS_SCANNER_DEBUG_TOKENS
	  printf("SCANNER_DEBUG_TOKENS: ChannelName = %s\n",
		$1->Name.chars());
	  printf("SCANNER_DEBUG_TOKENS:    in file \"%s\"\n"
		 "SCANNER_DEBUG_TOKENS:    in line %d\n",
			$1->FileInfo->Filename.chars(),
			$1->Line);
#endif /* PRS_SCANNER_DEBUG_TOKENS */
	  assert($1->Symbol != 0);
	  $$ = $1;
	 };

interface_name:			/* Name */
	TOK_INTERFACENAME
	{
#ifdef PRS_SCANNER_DEBUG_TOKENS
	  printf("SCANNER_DEBUG_TOKENS: InterfaceName = %s\n",
		$1->Name.chars());
	  printf("SCANNER_DEBUG_TOKENS:    in file \"%s\"\n"
		 "SCANNER_DEBUG_TOKENS:    in line %d\n",
			$1->FileInfo->Filename.chars(),
			$1->Line);
#endif /* PRS_SCANNER_DEBUG_TOKENS */
	  assert($1->Symbol != 0);
	  $$ = $1;
	 };

integer:			/* Const */
	TOK_INTEGER
	{
#ifdef PRS_SCANNER_DEBUG_TOKENS
	  printf("SCANNER_DEBUG_TOKENS: Integer = Type %d, Value %s\n",
			$1->Const->Type, $1->Const->Print());
	  printf("SCANNER_DEBUG_TOKENS:    in file \"%s\"\n"
		 "SCANNER_DEBUG_TOKENS:    in line %d\n",
			$1->FileInfo->Filename.chars(),
			$1->Line);
#endif /* PRS_SCANNER_DEBUG_TOKENS */
	  $$ = $1;
	 };

floating:			/* Const */
	TOK_FLOATING
	{
#ifdef PRS_SCANNER_DEBUG_TOKENS
	  printf("SCANNER_DEBUG_TOKENS: Float = Type %d, Value %s\n",
			$1->Const->Type, $1->Const->Print());
	  printf("SCANNER_DEBUG_TOKENS:    in file \"%s\"\n"
		 "SCANNER_DEBUG_TOKENS:    in line %d\n",
			$1->FileInfo->Filename.chars(),
			$1->Line);
#endif /* PRS_SCANNER_DEBUG_TOKENS */
	  $$ = $1;
	 };

character:			/* Const */
	TOK_CHARACTER
	{
#ifdef PRS_SCANNER_DEBUG_TOKENS
	  printf("SCANNER_DEBUG_TOKENS: Character = Type %d, Value %s\n",
			$1->Const->Type, $1->Const->Print());
	  printf("SCANNER_DEBUG_TOKENS:    in file \"%s\"\n"
		 "SCANNER_DEBUG_TOKENS:    in line %d\n",
			$1->FileInfo->Filename.chars(),
			$1->Line);
#endif /* PRS_SCANNER_DEBUG_TOKENS */
	  $$ = $1;
	 };

string:				/* String */
	TOK_STRING
	{
#ifdef PRS_SCANNER_DEBUG_TOKENS
	  sir_constant	*TmpStringConst;
	  TmpStringConst = new SIR_Constant(SIR_CONST_CHARSTRING,
						$1->CharString.chars());
	  printf("SCANNER_DEBUG_TOKENS: String = Type %d, Value %s\n",
		TmpStringConst->Type, TmpStringConst->Print());
	  delete TmpStringConst;
	  printf("SCANNER_DEBUG_TOKENS:    in file \"%s\"\n"
		 "SCANNER_DEBUG_TOKENS:    in line %d\n",
			$1->FileInfo->Filename.chars(),
			$1->Line);
#endif /* PRS_SCANNER_DEBUG_TOKENS */
	  $$ = $1;
	 };

bitvector:			/* Const */
	TOK_BITVECTOR
	{
#ifdef PRS_SCANNER_DEBUG_TOKENS
	  printf("SCANNER_DEBUG_TOKENS: Bitvector = Type %d, Value %s\n",
		$1->Const->Type, $1->Const->Print());
	  printf("SCANNER_DEBUG_TOKENS:    in file \"%s\"\n"
		 "SCANNER_DEBUG_TOKENS:    in line %d\n",
			$1->FileInfo->Filename.chars(),
			$1->Line);
#endif /* PRS_SCANNER_DEBUG_TOKENS */
	  $$ = $1;
	 };


/*** Constants **********************************************************/


/* note: the following has been changed from the ANSI-C grammar:	*/
/*	- constant includes string_literal_list (cleaner)		*/

constant:			/* Constant */
	integer
	{ $$ = $1->Const;
	  delete $1;
	 }
	| floating
	{ $$ = $1->Const;
	  delete $1;
	 }
	| character
	{ $$ = $1->Const;
	  delete $1;
	 }
	/*** SpecC-only: boolean constants ***/
	| TOK_FALSE
	{ $$ = new SIR_Constant(SIR_CONST_BOOL, false,
				PRS_lineno, PRS_CurrFileInfo);
	 }
	| TOK_TRUE
	{ $$ = new SIR_Constant(SIR_CONST_BOOL, true,
				PRS_lineno, PRS_CurrFileInfo);
	 }
	/*** SpecC-only: bitvector constant ***/
	| bitvector
	{ $$ = $1->Const;
	  delete $1;
	 }
	| string_literal_list
	{ $$ = new SIR_Constant(SIR_CONST_CHARSTRING,
					$1->CharString.chars(),
					$1->Line, $1->FileInfo);
	  delete $1;
	 };

string_literal_list:		/* String */
	string
	{ $$ = $1;
	 }
	| string_literal_list string
	{ $$ = $1;
	  $$->CharString += $2->CharString;
	  delete $2;
	 };


/*** Expressions ********************************************************/


primary_expression:		/* Expression */
	identifier
	{ if (!($1->Symbol))
	     { TmpErrMsg.form("Undeclared identifier '%s'", $1->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_UNDECLARED_IDENTIFIER,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1;
	       YYABORT;
	      } /* fi */
	  if (!($$ = SIR_Expression::New($1->Symbol)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1;
	       YYABORT;
	      } /* fi */
	  $$->UpdateLineInfo($1->Line, $1->FileInfo);
	  delete $1;
	 }
	| constant
	{ $$ = SIR_Expression::New($1, PRS_TheDesign->Types);
	  assert($$ != NULL);	/* must be ok */
	  $$->SetLineInfo($1);
	 }
	| '(' comma_expression ')'
	{
#ifdef PRS_KEEP_REDUNDANT_PARENTHESES
	  if (!($$ = SIR_Expression::New(		/* keep these       */
			SIR_EXPR_PARENTHESES, $2)))	/* for readability! */
	     { PRS_SemanticError(SIR_Error, $2);
	       delete $2;
	       YYABORT;
	      } /* fi */
#else /* PRS_KEEP_REDUNDANT_PARENTHESES */
	  $$ = $2;				/* throw parentheses away */
#endif /* PRS_KEEP_REDUNDANT_PARENTHESES */
	 }
	/*** SpecC-only: class self reference ***/
	| TOK_THIS
	{ sir_symbols	*ClassScope;

	  ClassScope = PRS_CurrScope;
	  while(  (ClassScope)
		&&(ClassScope->ScopeInfo != SIR_SCOPE_CLASS))
	     { ClassScope = ClassScope->Parent;
	      } /* elihw */
	  if (! ClassScope)
	     { PRS_error("Use of 'this' out of class scope");
	       YYABORT;
	      } /* fi */
	  if (!($$ = SIR_Expression::New(SIR_EXPR_THIS, PRS_TheDesign->Types,
						ClassScope->ParentSymbol)))
	     { PRS_SemanticError(SIR_Error);
	       YYABORT;
	      } /* fi */
	  $$->UpdateLineInfo(PRS_lineno, PRS_CurrFileInfo);
	 };

postfix_expression:		/* Expression */
	primary_expression
	{ $$ = $1;
	 }
	| postfix_expression '[' comma_expression ']'
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_ARRAY_ACCESS, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 }
	| postfix_expression '(' ')'
	{ if (!($$ = SIR_Expression::New($1, new SIR_Expressions())))
	     { PRS_SemanticError(SIR_Error, $1);
	       delete $1;
	       YYABORT;
	      } /* fi */
	 }
	| postfix_expression '(' argument_expression_list ')'
	{ if (!($$ = SIR_Expression::New($1, $3)))
	     { PRS_SemanticError(SIR_Error, $1);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 }
	| postfix_expression '.' member_name
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_MEMBER_ACCESS,
						$1, $3->Name.chars())))
	     { PRS_SemanticError(SIR_Error, $1);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  delete $3;
	 }
	| postfix_expression TOK_ARROW member_name
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_MEMBER_POINTER,
						$1, $3->Name.chars())))
	     { PRS_SemanticError(SIR_Error, $1);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  delete $3;
	 }
	| postfix_expression TOK_INCR
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_POST_INCREMENT, $1)))
	     { PRS_SemanticError(SIR_Error, $1);
	       delete $1;
	       YYABORT;
	      } /* fi */
	 }
	| postfix_expression TOK_DECR
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_POST_DECREMENT, $1)))
	     { PRS_SemanticError(SIR_Error, $1);
	       delete $1;
	       YYABORT;
	      } /* fi */
	 }
	/*** SpecC-only: bitvector slicing ***/
	| postfix_expression '[' constant_expression ':' constant_expression ']'
	{ int		Const1,
			Const2;

	  Const1 = $3->IntegerEval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $3, $5);
	       delete $1; delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	  Const2 = $5->IntegerEval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $5);
	       delete $1; delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	  if (!($$ = SIR_Expression::New($1, Const1, Const2)))
	     { PRS_SemanticError(SIR_Error, $1, $3, $5);
	       delete $1; delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	  delete $3; delete $5;
	 };

member_name:			/* Name */
	identifier
	{ $$ = $1;
	 }
	| typedef_or_class_name
	{ $$ = $1;
	 };

argument_expression_list:	/* ExpressionList */
	assignment_expression
	{ $$ = new SIR_Expressions($1);
	 }
	| argument_expression_list ',' assignment_expression
	{ $1->Append($3);
	  $$ = $1;
	 };

unary_expression:		/* Expression */
	postfix_expression
	{ $$ =$1;
	 }
	| TOK_INCR unary_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_PRE_INCREMENT, $2)))
	     { PRS_SemanticError(SIR_Error, $2);
	       delete $2;
	       YYABORT;
	      } /* fi */
	 }
	| TOK_DECR unary_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_PRE_DECREMENT, $2)))
	     { PRS_SemanticError(SIR_Error, $2);
	       delete $2;
	       YYABORT;
	      } /* fi */
	 }
	| unary_operator cast_expression
	{ if (!($$ = SIR_Expression::New($1, $2)))
	     { PRS_SemanticError(SIR_Error, $2);
	       delete $2;
	       YYABORT;
	      } /* fi */
	 }
	| TOK_SIZEOF line_info unary_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_SIZEOF_EXPR, $3)))
	     { PRS_SemanticError(SIR_Error, $3);
	       delete $2; delete $3;
	       YYABORT;
	      } /* fi */
	  $$->UpdateLineInfo($2->Line, $2->FileInfo);
	  delete $2;
	 }
	| TOK_SIZEOF line_info '(' type_name ')'
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_SIZEOF_TYPE, $4)))
	     { PRS_SemanticError(SIR_Error);
	       delete $2;
	       YYABORT;
	      } /* fi */
	  $$->UpdateLineInfo($2->Line, $2->FileInfo);
	  delete $2;
	 };

unary_operator:			/* ExpressionType */
	'&'
	{ $$ = SIR_EXPR_ADDRESS_OF;
	 }
	| '*'
	{ $$ = SIR_EXPR_CONTENT_OF;
	 }
	| '+'
	{ $$ = SIR_EXPR_POSITIVE;
	 }
	| '-'
	{ $$ = SIR_EXPR_NEGATIVE;
	 }
	| '~'
	{ $$ = SIR_EXPR_NOT;
	 }
	| '!'
	{ $$ = SIR_EXPR_LOGICAL_NOT;
	 };

cast_expression:		/* Expression */
	unary_expression
	{ $$ = $1;
	 }
	| '(' type_name ')' cast_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_TYPE_CONVERSION, $2, $4)))
	     { PRS_SemanticError(SIR_Error);
	       delete $4;
	       YYABORT;
	      } /* fi */
	 };

	/*** SpecC-only: bitvector concatenation ***/
concat_expression:		/* Expression */
	cast_expression
	{ $$ = $1;
	 }
	| concat_expression '@' cast_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_CONCATENATION, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

/* note: for multiplicative_expression every "cast_expression"	*/
/*       is replaced with "concat_expression"			*/

multiplicative_expression:	/* Expression */
	concat_expression
	{ $$ = $1;
	 }
	| multiplicative_expression '*' concat_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_MULTIPLY, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 }
	| multiplicative_expression '/' concat_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_DIVIDE, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 }
	| multiplicative_expression '%' concat_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_MODULO, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

additive_expression:		/* Expression */
	multiplicative_expression
	{ $$ = $1;
	 }
	| additive_expression '+' multiplicative_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_ADD, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 }
	| additive_expression '-' multiplicative_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_SUBTRACT, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

shift_expression:		/* Expression */
	additive_expression
	{ $$ = $1;
	 }
	| shift_expression TOK_SHIFTLEFT additive_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_SHIFT_LEFT, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 }
	| shift_expression TOK_SHIFTRIGHT additive_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_SHIFT_RIGHT, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

relational_expression:		/* Expression */
	shift_expression
	{ $$ = $1;
	 }
	| relational_expression '<' shift_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_LESS, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 }
	| relational_expression '>' shift_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_GREATER, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 }
	| relational_expression TOK_LE shift_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_LESS_EQUAL, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 }
	| relational_expression TOK_GE shift_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_GREATER_EQUAL, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

equality_expression:		/* Expression */
	relational_expression
	{ $$ = $1;
	 }
	| equality_expression TOK_EQ relational_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_EQUAL, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 }
	| equality_expression TOK_NE relational_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_NOT_EQUAL, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

and_expression:			/* Expression */
	equality_expression
	{ $$ = $1;
	 }
	| and_expression '&' equality_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_AND, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

exclusive_or_expression:	/* Expression */
	and_expression
	{ $$ = $1;
	 }
	| exclusive_or_expression '^' and_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_EOR, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

inclusive_or_expression:	/* Expression */
	exclusive_or_expression
	{ $$ = $1;
	 }
	| inclusive_or_expression '|' exclusive_or_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_OR, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

logical_and_expression:		/* Expression */
	inclusive_or_expression
	{ $$ = $1;
	 }
	| logical_and_expression TOK_ANDAND inclusive_or_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_LOGICAL_AND, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

logical_or_expression:		/* Expression */
	logical_and_expression
	{ $$ = $1;
	 }
	| logical_or_expression TOK_OROR logical_and_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_LOGICAL_OR, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

conditional_expression:		/* Expression */
	logical_or_expression
	{ $$ = $1;
	 }
	| logical_or_expression '?' comma_expression ':'
		conditional_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_CONDITION, $1, $3, $5)))
	     { PRS_SemanticError(SIR_Error, $1, $3, $5);
	       delete $1; delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	 };

assignment_expression:		/* Expression */
	conditional_expression
	{ $$ = $1;
	 }
	| unary_expression assignment_operator assignment_expression
	{ if (!($$ = SIR_Expression::New($2, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

assignment_operator:		/* ExpressionType */
	'='
	{ $$ = SIR_EXPR_ASSIGNMENT;
	 }
	| TOK_MULTASSIGN
	{ $$ = SIR_EXPR_MUL_ASSIGN;
	 }
	| TOK_DIVASSIGN
	{ $$ = SIR_EXPR_DIV_ASSIGN;
	 }
	| TOK_MODASSIGN
	{ $$ = SIR_EXPR_MOD_ASSIGN;
	 }
	| TOK_PLUSASSIGN
	{ $$ = SIR_EXPR_ADD_ASSIGN;
	 }
	| TOK_MINUSASSIGN
	{ $$ = SIR_EXPR_SUB_ASSIGN;
	 }
	| TOK_SLASSIGN
	{ $$ = SIR_EXPR_SHL_ASSIGN;
	 }
	| TOK_SRASSIGN
	{ $$ = SIR_EXPR_SHR_ASSIGN;
	 }
	| TOK_ANDASSIGN
	{ $$ = SIR_EXPR_AND_ASSIGN;
	 }
	| TOK_EORASSIGN
	{ $$ = SIR_EXPR_EOR_ASSIGN;
	 }
	| TOK_ORASSIGN
	{ $$ = SIR_EXPR_OR_ASSIGN;
	 };

comma_expression:		/* Expression */
	assignment_expression
	{ $$ = $1;
	 }
	| comma_expression ',' assignment_expression
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_COMMA, $1, $3)))
	     { PRS_SemanticError(SIR_Error, $1, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	 };

constant_expression:		/* Expression */
	conditional_expression
	{ $$ = $1;
	 };

comma_expression_opt:		/* Expression */
	/* nothing */
	{ if (!($$ = SIR_Expression::New(SIR_EXPR_VOID, PRS_TheDesign->Types)))
	     { PRS_SemanticError(SIR_Error);
	       YYABORT;
	      } /* fi */
	  $$->UpdateLineInfo(PRS_lineno, PRS_CurrFileInfo);
	 }
	| comma_expression
	{ $$ = $1;
	 };


/*** Declarations *******************************************************/


declaration:			/* void */
	sue_declaration_specifier ';'
	{ if ($1->StorageClass != SIR_STORAGE_NONE)
	     { PRS_PrintWarning(GL_WARN_INFORMATIVE,
			"Storage class useless in empty declaration");
	      } /* fi */
	  delete $1;
	 }
	| sue_type_specifier ';'
	{ assert($1->UserType != NULL);	/* it's a usertype */
	  if (  ($1->UserType->Name == NULL)	/* unnamed? */
	      &&(  ($1->UserType->Class == SIR_USERTYPE_STRUCT)
		 ||($1->UserType->Class == SIR_USERTYPE_UNION)))
	     { PRS_PrintWarning(GL_WARN_INFORMATIVE,
			"Useless unnamed struct or union declaration");
	      } /* fi */
	 }
	| declaring_list ';'
	{ delete $1;
	 }
	| default_declaring_list ';'
	{ delete $1;
	 };

default_declaring_list:		/* DeclarationSpec */
	declaration_qualifier_list identifier_declarator
		{ /* !!! mid-rule action #1 !!! */
		  $<DeclarationSpec>$ = new SIR_DeclSpec(
					PRS_TheDesign->Types, $1);
		 }
		{ /* !!! mid-rule action #2 !!! */
		  sir_declspec	*DeclSpec;
		  sir_type	*TypeEntry;

		  DeclSpec = $<DeclarationSpec>3;
		  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
					DeclSpec->BaseType, $2->DeclTypes);
		  if (PRS_CheckTypeDefinition(TypeEntry,
				(DeclSpec->StorageClass == SIR_STORAGE_TYPEDEF),
				false, DeclSpec->StorageClass,
				$2->Line, $2->FileInfo->Filename.chars()))
		     { delete $1; delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		  if (!($<Symbol>$ = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $2, TypeEntry))))
		     { PRS_SemanticError(SIR_Error);
		       delete $1; delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		 }
		initializer_opt
	{ sir_declspec	*DeclSpec;
	  sir_symbol	*Symbol;
	  ERROR		Error;

	  DeclSpec = $<DeclarationSpec>3;
	  Symbol = $<Symbol>4;
	  if ((Error = PRS_CurrScope->DefineVariable(Symbol,
					DeclSpec->StorageClass, $5)))
	     { PRS_SemanticError(Error, Symbol);
	       delete $1; delete $2;
	       delete DeclSpec;
	       YYABORT;
	      } /* fi */
	  $$ = DeclSpec;
	  delete $1; delete $2;
	 }
	| type_qualifier_list identifier_declarator
		{ /* !!! mid-rule action #1 !!! */
		  $<DeclarationSpec>$ = new SIR_DeclSpec(
					PRS_TheDesign->Types, $1);
		 }
		{ /* !!! mid-rule action #2 !!! */
		  sir_declspec	*DeclSpec;
		  sir_type	*TypeEntry;

		  DeclSpec = $<DeclarationSpec>3;
		  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
					DeclSpec->BaseType, $2->DeclTypes);
		  if (PRS_CheckTypeDefinition(TypeEntry,
				(DeclSpec->StorageClass == SIR_STORAGE_TYPEDEF),
				false, DeclSpec->StorageClass,
				$2->Line, $2->FileInfo->Filename.chars()))
		     { delete $1; delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		  if (!($<Symbol>$ = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $2, TypeEntry))))
		     { PRS_SemanticError(SIR_Error);
		       delete $1; delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		 }
		initializer_opt
	{ sir_declspec	*DeclSpec;
	  sir_symbol	*Symbol;
	  ERROR		Error;

	  DeclSpec = $<DeclarationSpec>3;
	  Symbol = $<Symbol>4;
	  if ((Error = PRS_CurrScope->DefineVariable(Symbol,
					DeclSpec->StorageClass, $5)))
	     { PRS_SemanticError(Error, Symbol);
	       delete $1; delete $2;
	       delete DeclSpec;
	       YYABORT;
	      } /* fi */
	  $$ = DeclSpec;
	  delete $1; delete $2;
	 }
	| default_declaring_list ',' identifier_declarator
		{ /* !!! mid-rule action !!! */
		  sir_type	*TypeEntry;

		  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
						$1->BaseType, $3->DeclTypes);
		  if (PRS_CheckTypeDefinition(TypeEntry,
				($1->StorageClass == SIR_STORAGE_TYPEDEF),
				false, $1->StorageClass,
				$3->Line, $3->FileInfo->Filename.chars()))
		     { delete $1; delete $3;
		       YYABORT;
		      } /* fi */
		  if (!($<Symbol>$ = PRS_CurrScope->Declare(
				new SIR_Symbol($1, $3, TypeEntry))))
		     { PRS_SemanticError(SIR_Error);
		       delete $1; delete $3;
		       YYABORT;
		      } /* fi */
		 }
		initializer_opt
	{ sir_symbol	*Symbol;
	  ERROR		Error;

	  Symbol = $<Symbol>4;
	  if ((Error = PRS_CurrScope->DefineVariable(Symbol,
					$1->StorageClass, $5)))
	     { PRS_SemanticError(Error, Symbol);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  $$ = $1;
	  delete $3;
	 };

declaring_list:			/* DeclarationSpec */
	declaration_specifier declarator
		{ /* !!! mid-rule action !!! */
		  sir_type	*TypeEntry;

		  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
						$1->BaseType, $2->DeclTypes);
		  if (PRS_CheckTypeDefinition(TypeEntry,
				($1->StorageClass == SIR_STORAGE_TYPEDEF),
				false, $1->StorageClass,
				$2->Line, $2->FileInfo->Filename.chars()))
		     { delete $1; delete $2;
		       YYABORT;
		      } /* fi */
		  if (!($<Symbol>$ = PRS_CurrScope->Declare(
				new SIR_Symbol($1, $2, TypeEntry))))
		     { PRS_SemanticError(SIR_Error);
		       delete $1; delete $2;
		       YYABORT;
		      } /* fi */
		 }
		initializer_opt
	{ sir_symbol	*Symbol;
	  ERROR		Error;

	  Symbol = $<Symbol>3;
	  if ((Error = PRS_CurrScope->DefineVariable(Symbol,
					$1->StorageClass, $4)))
	     { PRS_SemanticError(Error, Symbol);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = $1;
	  delete $2;
	 }
	| type_specifier declarator
		{ /* !!! mid-rule action #1 !!! */
		  $<DeclarationSpec>$ = new SIR_DeclSpec($1);
		 }
		{ /* !!! mid-rule action #2 !!! */
		  sir_declspec	*DeclSpec;
		  sir_type	*TypeEntry;

		  DeclSpec = $<DeclarationSpec>3;
		  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
					DeclSpec->BaseType, $2->DeclTypes);
		  if (PRS_CheckTypeDefinition(TypeEntry,
				(DeclSpec->StorageClass == SIR_STORAGE_TYPEDEF),
				false, DeclSpec->StorageClass,
				$2->Line, $2->FileInfo->Filename.chars()))
		     { delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		  if (!($<Symbol>$ = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $2, TypeEntry))))
		     { PRS_SemanticError(SIR_Error);
		       delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		 }
		initializer_opt
	{ sir_declspec	*DeclSpec;
	  sir_symbol	*Symbol;
	  ERROR		Error;

	  DeclSpec = $<DeclarationSpec>3;
	  Symbol = $<Symbol>4;
	  if ((Error = PRS_CurrScope->DefineVariable(Symbol,
					DeclSpec->StorageClass, $5)))
	     { PRS_SemanticError(Error, Symbol);
	       delete $2;
	       delete DeclSpec;
	       YYABORT;
	      } /* fi */
	  $$ = DeclSpec;
	  delete $2;
	 }
	| declaring_list ',' declarator
		{ /* !!! mid-rule action !!! */
		  sir_type	*TypeEntry;

		  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
						$1->BaseType, $3->DeclTypes);
		  if (PRS_CheckTypeDefinition(TypeEntry,
				($1->StorageClass == SIR_STORAGE_TYPEDEF),
				false, $1->StorageClass,
				$3->Line, $3->FileInfo->Filename.chars()))
		     { delete $1; delete $3;
		       YYABORT;
		      } /* fi */
		  if (!($<Symbol>$ = PRS_CurrScope->Declare(
				new SIR_Symbol($1, $3, TypeEntry))))
		     { PRS_SemanticError(SIR_Error);
		       delete $1; delete $3;
		       YYABORT;
		      } /* fi */
		 }
		initializer_opt
	{ sir_symbol	*Symbol;
	  ERROR		Error;

	  Symbol = $<Symbol>4;
	  if ((Error = PRS_CurrScope->DefineVariable(Symbol,
					$1->StorageClass, $5)))
	     { PRS_SemanticError(Error, Symbol);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  $$ = $1;
	  delete $3;
	 };

declaration_specifier:		/* DeclarationSpec */
	basic_declaration_specifier
	{ $$ = new SIR_DeclSpec(PRS_TheDesign->Types, $1);
	  delete $1;
	 }
	| sue_declaration_specifier
	{ $$ = $1;
	 }
	| typedef_declaration_specifier
	{ $$ = $1;
	 };

type_specifier:			/* Type */
	basic_type_specifier
	{ $$ = PRS_TheDesign->Types->FindOrInsert($1);
	  delete $1;
	 }
	| sue_type_specifier
	{ $$ = $1;
	 }
	| typedef_type_specifier
	{ $$ = $1;
	 };

declaration_qualifier_list:	/* BasicType */
	storage_class
	{ $$ = $1;
	 }
	| type_qualifier_list storage_class
	{ if (!($$ = $1->Merge($2)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $2;
	 }
	| declaration_qualifier_list declaration_qualifier
	{ if (!($$ = $1->Merge($2)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $2;
	 };

type_qualifier_list:		/* BasicType */
	type_qualifier
	{ $$ = $1;
	 }
	| type_qualifier_list type_qualifier
	{ if (!($$ = $1->Merge($2)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $2;
	 };

declaration_qualifier:		/* BasicType */
	storage_class
	{ $$ = $1;
	 }
	| type_qualifier
	{ $$ = $1;
	 };

type_qualifier:			/* BasicType */
	TOK_CONST
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_UNKNOWN, false, false,
							true, false);
	 }
	| TOK_VOLATILE
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_UNKNOWN, false, false,
							false, true);
	 };

basic_declaration_specifier:	/* BasicType */
	declaration_qualifier_list basic_type_name
	{ if (!($$ = $1->Merge($2)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $2;
	 }
	| basic_type_specifier storage_class
	{ if (!($$ = $1->Merge($2)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $2;
	 }
	| basic_declaration_specifier declaration_qualifier
	{ if (!($$ = $1->Merge($2)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $2;
	 }
	| basic_declaration_specifier basic_type_name
	{ if (!($$ = $1->Merge($2)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $2;
	 };

basic_type_specifier:		/* BasicType */
	basic_type_name
	{ $$ = $1;
	 }
	| type_qualifier_list basic_type_name
	{ if (!($$ = $1->Merge($2)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $2;
	 }
	| basic_type_specifier type_qualifier
	{ if (!($$ = $1->Merge($2)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $2;
	 }
	| basic_type_specifier basic_type_name
	{ if (!($$ = $1->Merge($2)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $2;
	 };

sue_declaration_specifier:	/* DeclarationSpec */
	declaration_qualifier_list elaborated_type_name
	{ $$ = new SIR_DeclSpec(
		PRS_TheDesign->Types->FindOrInsertModifiedType($2,
					$1->IsConst, $1->IsVolatile),
		$1->StorageClass, $1->PipeStages);
	  delete $1;
	 }
	| sue_type_specifier storage_class
	{ $$ = new SIR_DeclSpec($1, /* storage class has no CONST or VOLATILE */
				$2->StorageClass, $2->PipeStages);
	  delete $2;
	 }
	| sue_declaration_specifier declaration_qualifier
	{ sir_basic_type	*TmpBasicType;

	  TmpBasicType = new SIR_BasicType(
				$1->StorageClass, $1->PipeStages,
				$1->BaseType->Const, $1->BaseType->Volatile);
	  if (!(TmpBasicType->Merge($2)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       delete TmpBasicType;
	       YYABORT;
	      } /* fi */
	  $1->StorageClass	= TmpBasicType->StorageClass;
	  $1->PipeStages	= TmpBasicType->PipeStages;
	  $1->BaseType = PRS_TheDesign->Types->FindOrInsertModifiedType(
				$1->BaseType,
				TmpBasicType->IsConst,
				TmpBasicType->IsVolatile);
	  $$ = $1;
	  delete TmpBasicType;
	  delete $2;
	 };

sue_type_specifier:		/* Type */
	elaborated_type_name
	{ $$ = $1;
	 }
	| type_qualifier_list elaborated_type_name
	{ $$ = PRS_TheDesign->Types->FindOrInsertModifiedType($2,
					$1->IsConst, $1->IsVolatile);
	  delete $1;
	 }
	| sue_type_specifier type_qualifier
	{ if ($1->Const && $2->IsConst)
	     { PRS_error("Type modifier 'const' specified twice",
			PRS_ERROR_TYPE_CONST_TWICE,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete $2;
	       YYABORT;
	      } /* fi */
	  if ($1->Volatile && $2->IsVolatile)
	     { PRS_error("Type modifier 'volatile' specified twice",
			PRS_ERROR_TYPE_VOLATILE_TWICE,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = PRS_TheDesign->Types->FindOrInsertModifiedType($1,
				$2->IsConst || $1->Const,
				$2->IsVolatile || $1->Volatile);
	  delete $2;
	 };

typedef_declaration_specifier:	/* DeclarationSpec */
	typedef_type_specifier storage_class
	{ $$ = new SIR_DeclSpec($1, /* storage class has no CONST or VOLATILE */
				$2->StorageClass, $2->PipeStages);
	  delete $2;
	 }
	| declaration_qualifier_list typedef_name
	{ assert($2->Symbol != NULL);	/* scanner has found this symbol */
	  assert($2->Symbol->Type != NULL);	/* typedef type */
	  $$ = new SIR_DeclSpec(
		PRS_TheDesign->Types->FindOrInsertModifiedType(
					$2->Symbol->Type,
					$1->IsConst, $1->IsVolatile),
		$1->StorageClass, $1->PipeStages);
	  delete $1; delete $2;
	 }
	| typedef_declaration_specifier declaration_qualifier
	{ sir_basic_type	*TmpBasicType;

	  TmpBasicType = new SIR_BasicType(
				$1->StorageClass, $1->PipeStages,
				$1->BaseType->Const, $1->BaseType->Volatile);
	  if (!(TmpBasicType->Merge($2)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       delete TmpBasicType;
	       YYABORT;
	      } /* fi */
	  $1->StorageClass	= TmpBasicType->StorageClass;
	  $1->PipeStages	= TmpBasicType->PipeStages;
	  $1->BaseType = PRS_TheDesign->Types->FindOrInsertModifiedType(
				$1->BaseType,
				TmpBasicType->IsConst,
				TmpBasicType->IsVolatile);
	  $$ = $1;
	  delete TmpBasicType;
	  delete $2;
	 };

typedef_type_specifier:		/* Type */
	typedef_name
	{ assert($1->Symbol != NULL);	/* scanner has found this symbol */
	  assert($1->Symbol->Type != NULL);	/* typedef type */
	  $$ = $1->Symbol->Type;
	  delete $1;
	 }
	| type_qualifier_list typedef_name
	{ assert($2->Symbol != NULL);	/* scanner has found this symbol */
	  assert($2->Symbol->Type != NULL);	/* typedef type */
	  if ($1->IsConst && $2->Symbol->Type->Const)
	     { PRS_error("Type modifier 'const' specified twice",
			PRS_ERROR_TYPE_CONST_TWICE,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  if ($1->IsVolatile && $2->Symbol->Type->Volatile)
	     { PRS_error("Type modifier 'volatile' specified twice",
			PRS_ERROR_TYPE_VOLATILE_TWICE,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = PRS_TheDesign->Types->FindOrInsertModifiedType(
				$2->Symbol->Type,
				$1->IsConst || $2->Symbol->Type->Const,
				$1->IsVolatile || $2->Symbol->Type->Volatile);
	  delete $1; delete $2;
	 }
	| typedef_type_specifier type_qualifier
	{ if ($1->Const && $2->IsConst)
	     { PRS_error("Type modifier 'const' specified twice",
			PRS_ERROR_TYPE_CONST_TWICE,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete $2;
	       YYABORT;
	      } /* fi */
	  if ($1->Volatile && $2->IsVolatile)
	     { PRS_error("Type modifier 'volatile' specified twice",
			PRS_ERROR_TYPE_VOLATILE_TWICE,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = PRS_TheDesign->Types->FindOrInsertModifiedType($1,
				$2->IsConst || $1->Const,
				$2->IsVolatile || $1->Volatile);
	  delete $2;
	 };

storage_class:			/* BasicType */
	TOK_TYPEDEF
	{ $$ = new SIR_BasicType(SIR_STORAGE_TYPEDEF);
	 }
	| TOK_EXTERN
	{ $$ = new SIR_BasicType(SIR_STORAGE_EXTERN);
	 }
	| TOK_STATIC
	{ $$ = new SIR_BasicType(SIR_STORAGE_STATIC);
	 }
	| TOK_AUTO
	{ $$ = new SIR_BasicType(SIR_STORAGE_AUTO);
	 }
	| TOK_REGISTER
	{ $$ = new SIR_BasicType(SIR_STORAGE_REGISTER);
	 }
	/*** SpecC-only: piped modifier ***/
	| TOK_PIPED
	{ $$ = new SIR_BasicType(SIR_STORAGE_PIPED);
	 };

basic_type_name:		/* BasicType */
	TOK_INT
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_INT);
	 }
	| TOK_CHAR
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_CHAR);
	 }
	| TOK_SHORT
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_SHORT);
	 }
	| TOK_LONG
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_LONG);
	 }
	| TOK_FLOAT
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_FLOAT);
	 }
	| TOK_DOUBLE
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_DOUBLE);
	 }
	| TOK_SIGNED
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_UNKNOWN, true, false);
	 }
	| TOK_UNSIGNED
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_UNKNOWN, false, true);
	 }
	| TOK_VOID
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_VOID);
	 }
	/*** SpecC-only: boolean type ***/
	| TOK_BOOL
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_BOOL);
	 }
	/*** SpecC-only: bit(vector) type ***/
	| TOK_BIT '[' constant_expression ':' constant_expression ']'
	{ int		Const1,
			Const2;

	  Const1 = $3->IntegerEval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $3, $5);
	       delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	  Const2 = $5->IntegerEval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $5);
	       delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_BasicType(SIR_BASICTYPE_BIT,
				false, false, false, false,
				Const1, Const2);
	  delete $3; delete $5;
	 }
	| TOK_BIT '[' constant_expression ']'
	{ int		Const;

	  Const = $3->IntegerEval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $3);
	       delete $3;
	       YYABORT;
	      } /* fi */
	  if (Const < 1)
	     { PRS_error("Length of bit vector must be a positive number",
				PRS_ERROR_BITVECTOR_LENGTH,
				PRS_lineno, PRS_CurrentFile.chars());
	       delete $3;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_BasicType(SIR_BASICTYPE_BIT,
				false, false, false, false,
				Const-1, 0);
	  delete $3;
	 }
	/*** SpecC-only: event type ***/
	| TOK_EVENT
	{ $$ = new SIR_BasicType(SIR_BASICTYPE_EVENT);
	 };

elaborated_type_name:		/* Type */
	aggregate_name
	{ $$ = $1;
	 }
	| enum_name
	{ $$ = $1;
	 };

aggregate_name:			/* Type */
	aggregate_key
		{ /* !!! mid-rule action !!! */
		  sir_symbols	*NewScope;
		  sir_usertype	*NewUserType;

		  /* note: we don't support anonymous unions as in C++ */
		  NewScope = new SIR_Symbols(PRS_CurrScope,
						SIR_SCOPE_USERTYPE);
		  if (!(NewUserType = PRS_CurrScope->UserTypes->Declare(
				$1,
				NULL,	/* unnamed */
				NewScope,
				PRS_TheDesign->Types)))
		     { PRS_SemanticError(SIR_Error);
		       YYABORT;
		      } /* fi */
		  PRS_CurrScope = NewScope;	/* enter local scope */
		  $<UserType>$ = NewUserType;
		 }
		'{' member_declaration_list '}'
	{ sir_usertype	*NewUserType;
	  ERROR		Error;

	  NewUserType = $<UserType>2;
	  PRS_CurrScope = PRS_CurrScope->Parent; /* back to outer scope */
	  if ((Error = PRS_CurrScope->UserTypes->Define(NewUserType, $4)))
	     { PRS_SemanticError(SIR_Error);
	       YYABORT;
	      } /* fi */
	  $$ = NewUserType->Type;
	 }
	| aggregate_key identifier_or_typedef_name
		{ /* !!! mid-rule action !!! */
		  sir_symbols	*NewScope;
		  sir_usertype	*NewUserType;

		  NewScope = new SIR_Symbols(PRS_CurrScope,
						SIR_SCOPE_USERTYPE);
		  if (!(NewUserType = PRS_CurrScope->UserTypes->Declare(
				$1,
				$2->Name.chars(),
				NewScope,
				PRS_TheDesign->Types)))
		     { PRS_SemanticError(SIR_Error);
		       delete $2;
		       YYABORT;
		      } /* fi */
		  PRS_CurrScope = NewScope;	/* enter local scope */
		  $<UserType>$ = NewUserType;
		 }
		'{' member_declaration_list '}'
	{ sir_usertype	*NewUserType;
	  ERROR		Error;

	  NewUserType = $<UserType>3;
	  PRS_CurrScope = PRS_CurrScope->Parent; /* back to outer scope */
	  if ((Error = PRS_CurrScope->UserTypes->Define(NewUserType, $5)))
	     { PRS_SemanticError(SIR_Error);
	       delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = NewUserType->Type;
	  delete $2;
	 }
	| aggregate_key identifier_or_typedef_name
	{ sir_usertype	*UserType;

	  if (  (UserType = PRS_CurrScope->UserTypes->Find(
					$2->Name.chars()))
	      &&(UserType->Class == $1))
	     { $$ = UserType->Type;
	      } /* fi */
	  else
	     { if (!(UserType = PRS_CurrScope->UserTypes->Declare(
				$1,
				$2->Name.chars(),
				NULL,
				PRS_TheDesign->Types)))
		  { PRS_SemanticError(SIR_Error);
		    delete $2;
		    YYABORT;
		   } /* fi */
	       $$ = UserType->Type;
	      } /* esle */
	  delete $2;
	 };

aggregate_key:			/* UserTypeClass */
	TOK_STRUCT
	{ $$ = SIR_USERTYPE_STRUCT;
	 }
	| TOK_UNION
	{ $$ = SIR_USERTYPE_UNION;
	 };

member_declaration_list:	/* MemberList */
	member_declaration
	{ $$ = $1;
	 }
	| member_declaration_list member_declaration
	{ /* note: for easier handling the "list of lists" is */
	  /*       flattened into one simple list of members  */
	  $1->Concat($2);
	  $$ = $1;
	  delete $2;
	 };

member_declaration:		/* MemberList */
	member_declaring_list ';'
	{ $$ = $1->MemberList;
	  $1->MemberList = NULL;
	  delete $1;
	 }
	| member_default_declaring_list ';'
	{ $$ = $1->MemberList;
	  $1->MemberList = NULL;
	  delete $1;
	 }
	/*** SpecC-only: note definition in member list ***/
	| note_definition
	{ $$ = new SIR_Members();	/* empty list */
	 };

member_default_declaring_list:	/* MemberDeclSpec */
	type_qualifier_list member_identifier_declarator
	{ sir_declspec	*DeclSpec;
	  sir_type	*TypeEntry;
	  sir_symbol	*SymbolEntry;

	  DeclSpec = new SIR_DeclSpec(PRS_TheDesign->Types, $1);
	  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
				DeclSpec->BaseType, $2->DeclTypes);
	  if (PRS_CheckTypeDefinition(TypeEntry, false,
				false, SIR_STORAGE_NONE,
				$2->Line, $2->FileInfo->Filename.chars()))
	     { delete $1; delete $2;
	       delete DeclSpec;
	       YYABORT;
	      } /* fi */
	  if ($2->BitFieldSize != SIR_BITFIELD_SIZE_NONE)
	     { if (  (TypeEntry->Type != SIR_TYPE_INT)
		   &&(TypeEntry->Type != SIR_TYPE_UINT))
		  { PRS_error("Type of bitfield must be integer",
				PRS_ERROR_ILLEGAL_BITFIELD_TYPE,
				$2->Line, $2->FileInfo->Filename.chars());
		    delete $1; delete $2;
		    delete DeclSpec;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  if ($2->SymbolName)
	     { if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $2, TypeEntry))))
		  { PRS_SemanticError(SIR_Error);
		    delete $1; delete $2;
		    delete DeclSpec;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  else
	     { SymbolEntry = NULL;	/* no symbol */
	      } /* esle */
	  DeclSpec->MemberList = new SIR_Members(
		new SIR_Member(TypeEntry, SymbolEntry, $2->BitFieldSize,
					$2->Line, $2->FileInfo));
	  delete $1; delete $2;
	  $$ = DeclSpec;
	 }
	| member_default_declaring_list ',' member_identifier_declarator
	{ sir_type	*TypeEntry;
	  sir_symbol	*SymbolEntry;

	  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
				$1->BaseType, $3->DeclTypes);
	  if (PRS_CheckTypeDefinition(TypeEntry, false,
				false, SIR_STORAGE_NONE,
				$3->Line, $3->FileInfo->Filename.chars()))
	     { delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  if ($3->BitFieldSize != SIR_BITFIELD_SIZE_NONE)
	     { if (  (TypeEntry->Type != SIR_TYPE_INT)
		   &&(TypeEntry->Type != SIR_TYPE_UINT))
		  { PRS_error("Type of bitfield must be integer",
				PRS_ERROR_ILLEGAL_BITFIELD_TYPE,
				$3->Line, $3->FileInfo->Filename.chars());
		    delete $1; delete $3;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  if ($3->SymbolName)
	     { if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol($1, $3, TypeEntry))))
		  { PRS_SemanticError(SIR_Error);
		    delete $1; delete $3;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  else
	     { SymbolEntry = NULL;	/* no symbol */
	      } /* esle */
	  $1->MemberList->Append(
		new SIR_Member(TypeEntry, SymbolEntry, $3->BitFieldSize,
					$3->Line, $3->FileInfo));
	  delete $3;
	  $$ = $1;
	 };

member_declaring_list:		/* MemberDeclSpec */
	type_specifier member_declarator
	{ sir_declspec	*DeclSpec;
	  sir_type	*TypeEntry;
	  sir_symbol	*SymbolEntry;

	  DeclSpec = new SIR_DeclSpec($1);
	  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
				DeclSpec->BaseType, $2->DeclTypes);
	  if (PRS_CheckTypeDefinition(TypeEntry, false,
				false, SIR_STORAGE_NONE,
				$2->Line, $2->FileInfo->Filename.chars()))
	     { delete $2;
	       delete DeclSpec;
	       YYABORT;
	      } /* fi */
	  if ($2->BitFieldSize != SIR_BITFIELD_SIZE_NONE)
	     { if (  (TypeEntry->Type != SIR_TYPE_INT)
		   &&(TypeEntry->Type != SIR_TYPE_UINT))
		  { PRS_error("Type of bitfield must be integer",
				PRS_ERROR_ILLEGAL_BITFIELD_TYPE,
				$2->Line, $2->FileInfo->Filename.chars());
		    delete $2;
		    delete DeclSpec;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  if ($2->SymbolName)
	     { if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $2, TypeEntry))))
		  { PRS_SemanticError(SIR_Error);
		    delete $2;
		    delete DeclSpec;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  else
	     { SymbolEntry = NULL;	/* no symbol */
	      } /* esle */
	  DeclSpec->MemberList = new SIR_Members(
		new SIR_Member(TypeEntry, SymbolEntry, $2->BitFieldSize,
					$2->Line, $2->FileInfo));
	  delete $2;
	  $$ = DeclSpec;
	 }
	| member_declaring_list ',' member_declarator
	{ sir_type	*TypeEntry;
	  sir_symbol	*SymbolEntry;

	  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
				$1->BaseType, $3->DeclTypes);
	  if (PRS_CheckTypeDefinition(TypeEntry, false,
				false, SIR_STORAGE_NONE,
				$3->Line, $3->FileInfo->Filename.chars()))
	     { delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  if ($3->BitFieldSize != SIR_BITFIELD_SIZE_NONE)
	     { if (  (TypeEntry->Type != SIR_TYPE_INT)
		   &&(TypeEntry->Type != SIR_TYPE_UINT))
		  { PRS_error("Type of bitfield must be integer",
				PRS_ERROR_ILLEGAL_BITFIELD_TYPE,
				$3->Line, $3->FileInfo->Filename.chars());
		    delete $1; delete $3;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  if ($3->SymbolName)
	     { if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol($1, $3, TypeEntry))))
		  { PRS_SemanticError(SIR_Error);
		    delete $1; delete $3;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  else
	     { SymbolEntry = NULL;	/* no symbol */
	      } /* esle */
	  $1->MemberList->Append(
		new SIR_Member(TypeEntry, SymbolEntry, $3->BitFieldSize,
					$3->Line, $3->FileInfo));
	  delete $3;
	  $$ = $1;
	 };

member_declarator:		/* MmbrDeclarator */
	declarator bit_field_size_opt
	{ unsigned int	BitFieldSize;

	  if ($2)
	     { BitFieldSize = $2->IntegerEval();
	       if (SIR_Error)
		  { PRS_SemanticError(SIR_Error, $2);
		    delete $1; delete $2;
		    YYABORT;
		   } /* fi */
	       if (  (BitFieldSize < SIR_BITFIELD_SIZE_MIN)
		   ||(BitFieldSize > SIR_BITFIELD_SIZE_MAX))
		  { TmpErrMsg.form(
				"Size of bit field (%u) out of range %d to %d",
				BitFieldSize,
				SIR_BITFIELD_SIZE_MIN, SIR_BITFIELD_SIZE_MAX);
		    PRS_error(TmpErrMsg.chars(),
				PRS_ERROR_ILLEGAL_BITFIELD_SIZE,
				PRS_lineno, PRS_CurrentFile.chars());
		    delete $1; delete $2;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  else
	     { BitFieldSize = SIR_BITFIELD_SIZE_NONE;
	      } /* esle */
	  $1->BitFieldSize = BitFieldSize;
	  $$ = $1;
	  delete $2;
	 }
	| bit_field_size
	{ unsigned int	BitFieldSize;

	  BitFieldSize = $1->IntegerEval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $1);
	       delete $1;
	       YYABORT;
	      } /* fi */
	  if (  (BitFieldSize < SIR_BITFIELD_SIZE_MIN)
	      ||(BitFieldSize > SIR_BITFIELD_SIZE_MAX))
	     { TmpErrMsg.form("Size of bit field (%u) out of range %d to %d",
				BitFieldSize,
				SIR_BITFIELD_SIZE_MIN, SIR_BITFIELD_SIZE_MAX);
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_ILLEGAL_BITFIELD_SIZE,
				PRS_lineno, PRS_CurrentFile.chars());
	       delete $1;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Declarator(new sir_type_list(), NULL,
				PRS_lineno, PRS_CurrFileInfo, BitFieldSize);
	  delete $1;
	 };

member_identifier_declarator:	/* MmbrDeclarator */
	identifier_declarator
		{ /* note: this mid-rule action (suggested by the grammar) */
		  /*       is not used because we don't have direct access */
		  /*       to the declaration specifier; therefore the     */
		  /*       symbol table is not updated ASAP (which is only */
		  /*       a minor problem; bit_field_size_opt expression  */
		  /*       cannot use the identifier_declarator)           */
		 }
		bit_field_size_opt
	{ unsigned int	BitFieldSize;

	  if ($3)
	     { BitFieldSize = $3->IntegerEval();
	       if (SIR_Error)
		  { PRS_SemanticError(SIR_Error, $3);
		    delete $1; delete $3;
		    YYABORT;
		   } /* fi */
	       if (  (BitFieldSize < SIR_BITFIELD_SIZE_MIN)
		   ||(BitFieldSize > SIR_BITFIELD_SIZE_MAX))
		  { TmpErrMsg.form(
				"Size of bit field (%u) out of range %d to %d",
				BitFieldSize,
				SIR_BITFIELD_SIZE_MIN, SIR_BITFIELD_SIZE_MAX);
		    PRS_error(TmpErrMsg.chars(),
				PRS_ERROR_ILLEGAL_BITFIELD_SIZE,
				PRS_lineno, PRS_CurrentFile.chars());
		    delete $1; delete $3;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  else
	     { BitFieldSize = SIR_BITFIELD_SIZE_NONE;
	      } /* esle */
	  $1->BitFieldSize = BitFieldSize;
	  $$ = $1;
	  delete $3;
	 }
	| bit_field_size
	{ unsigned int	BitFieldSize;

	  BitFieldSize = $1->IntegerEval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $1);
	       delete $1;
	       YYABORT;
	      } /* fi */
	  if (  (BitFieldSize < SIR_BITFIELD_SIZE_MIN)
	      ||(BitFieldSize > SIR_BITFIELD_SIZE_MAX))
	     { TmpErrMsg.form("Size of bit field (%u) out of range %d to %d",
				BitFieldSize,
				SIR_BITFIELD_SIZE_MIN, SIR_BITFIELD_SIZE_MAX);
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_ILLEGAL_BITFIELD_SIZE,
				PRS_lineno, PRS_CurrentFile.chars());
	       delete $1;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Declarator(new sir_type_list(), NULL,
				PRS_lineno, PRS_CurrFileInfo, BitFieldSize);
	  delete $1;
	 };

bit_field_size_opt:		/* Expression */
	/* nothing */
	{ $$ = NULL;
	 }
	| bit_field_size
	{ $$ = $1;
	 };

bit_field_size:			/* Expression */
	':' constant_expression
	{ $$ = $2;
	 }

/* note: although the grammar didn't suggest mid-rule actions here	*/
/*       we handle enum exactly like struct/union			*/
enum_name:			/* Type */
	TOK_ENUM
		{ /* !!! mid-rule action !!! */
		  sir_usertype	*NewUserType;

		  if (!(NewUserType = PRS_CurrScope->UserTypes->Declare(
				SIR_USERTYPE_ENUM,
				NULL,	/* unnamed */
				NULL,	/* enum has no local scope */
				PRS_TheDesign->Types)))
		     { PRS_SemanticError(SIR_Error);
		       YYABORT;
		      } /* fi */
		  $<UserType>$ = NewUserType;
		  CurrEnumType = NewUserType;	/* publish type for list */
		 }
		'{' enumerator_list '}'
	{ sir_usertype	*NewUserType;
	  ERROR		Error;

	  assert(CurrEnumType != NULL);	/* was used */
	  CurrEnumType = NULL;		/* not used anymore */
	  NewUserType = $<UserType>2;
	  if ((Error = PRS_CurrScope->UserTypes->Define(NewUserType, $4)))
	     { PRS_SemanticError(SIR_Error);
	       YYABORT;
	      } /* fi */
	  $$ = NewUserType->Type;
	 }
	| TOK_ENUM identifier_or_typedef_name
		{ /* !!! mid-rule action !!! */
		  sir_usertype	*NewUserType;

		  if (!(NewUserType = PRS_CurrScope->UserTypes->Declare(
				SIR_USERTYPE_ENUM,
				$2->Name.chars(),
				NULL,	/* enum has no local scope */
				PRS_TheDesign->Types)))
		     { PRS_SemanticError(SIR_Error);
		       delete $2;
		       YYABORT;
		      } /* fi */
		  $<UserType>$ = NewUserType;
		  CurrEnumType = NewUserType;	/* publish type for list */
		 }
		'{' enumerator_list '}'
	{ sir_usertype	*NewUserType;
	  ERROR		Error;

	  assert(CurrEnumType != NULL);	/* was used */
	  CurrEnumType = NULL;		/* not used anymore */
	  NewUserType = $<UserType>3;
	  if ((Error = PRS_CurrScope->UserTypes->Define(NewUserType, $5)))
	     { PRS_SemanticError(SIR_Error);
	       delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = NewUserType->Type;
	  delete $2;
	 }
	| TOK_ENUM identifier_or_typedef_name
	{ sir_usertype	*UserType;

	  if (  (UserType = PRS_CurrScope->UserTypes->Find(
					$2->Name.chars()))
	      &&(UserType->Class == SIR_USERTYPE_ENUM))
	     { $$ = UserType->Type;
	      } /* fi */
	  else
	     { if (!(UserType = PRS_CurrScope->UserTypes->Declare(
				SIR_USERTYPE_ENUM,
				$2->Name.chars(),
				NULL,
				PRS_TheDesign->Types)))
		  { PRS_SemanticError(SIR_Error);
		    delete $2;
		    YYABORT;
		   } /* fi */
	       $$ = UserType->Type;
	      } /* esle */
	  delete $2;
	 };

enumerator_list:		/* MemberList */
	identifier_or_typedef_name enumerator_value_opt
	{ sir_symbol	*SymbolEntry;

	  assert(CurrEnumType != NULL);	/* must have been set before */
	  if (!(SymbolEntry = PRS_CurrScope->Declare(
			new SIR_Symbol(SIR_SYMBOL_IDENTIFIER,
					$1->Name.chars(),
					CurrEnumType->Type,
					SIR_STORAGE_NONE, 0,
					NULL,
					$1->Line, $1->FileInfo))))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  if ($2)
	     { SymbolEntry->EnumValue = $2->IntegerEval();
	       if (SIR_Error)
		  { PRS_SemanticError(SIR_Error, $2);
		    delete $1; delete $2;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  else
	     { SymbolEntry->EnumValue = 0; /* default enum starts from 0 */
	      } /* esle */
	  $$ = new SIR_Members(
		new SIR_Member(CurrEnumType->Type, SymbolEntry, 0,
						$1->Line, $1->FileInfo));
	  delete $1; delete $2;
	 }
	| enumerator_list ',' identifier_or_typedef_name enumerator_value_opt
	{ sir_symbol	*SymbolEntry;

	  assert(CurrEnumType != NULL);	/* must have been set before */
	  if (!(SymbolEntry = PRS_CurrScope->Declare(
			new SIR_Symbol(SIR_SYMBOL_IDENTIFIER,
					$3->Name.chars(),
					CurrEnumType->Type,
					SIR_STORAGE_NONE, 0,
					NULL,
					$3->Line, $3->FileInfo))))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $3; delete $4;
	       YYABORT;
	      } /* fi */
	  if ($4)
	     { SymbolEntry->EnumValue = $4->IntegerEval();
	       if (SIR_Error)
		  { PRS_SemanticError(SIR_Error, $4);
		    delete $1; delete $3; delete $4;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  else
	     { SymbolEntry->EnumValue = $1->Last()->Symbol->EnumValue + 1;
	      } /* esle */
	  $1->Append(new SIR_Member(CurrEnumType->Type, SymbolEntry, 0,
						$3->Line, $3->FileInfo));
	  $$ = $1;
	  delete $3; delete $4;
	 };

enumerator_value_opt:		/* Expression */
	/* nothing */
	{ $$ = NULL;
	 }
	| '=' constant_expression
	{ $$ = $2;
	 };

parameter_type_list:		/* ParameterList */
	parameter_list
	{ $$ = $1;
	 }
	| parameter_list ',' TOK_ELLIPSIS
	{ $1->Append(new SIR_Parameter(
			PRS_TheDesign->Types->FindOrInsert(SIR_TYPE_ANY_TYPE)));
	  $$ = $1;
	 };

parameter_list:			/* ParameterList */
	parameter_declaration
	{ if (PRS_CheckTypeDefinition($1->Type, true))
	     { delete $1;
	       YYABORT;
	      } /* fi */
	  if (  ($1->Type->Type == SIR_TYPE_VOID)
	      &&($1->Symbol))
	     { PRS_error("Parameter cannot be of type 'void'");
	       delete $1;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameters($1);
	 }
	| parameter_list ',' parameter_declaration
	{ if ($1->First()->Type->Type == SIR_TYPE_VOID)
	     { PRS_error(
		"First parameter is 'void', no more parameters allowed");
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  if (PRS_CheckTypeDefinition($3->Type, false))
	     { delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  $1->Append($3);
	  $$ = $1;
	 }
	/*** SpecC-only: interface parameter ***/
	| interface_parameter
	{ $$ = new SIR_Parameters($1);
	 }
	| parameter_list ',' interface_parameter
	{ if ($1->First()->Type->Type == SIR_TYPE_VOID)
	     { PRS_error(
		"First parameter is 'void', no more parameters allowed");
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  $1->Append($3);
	  $$ = $1;
	 };

parameter_declaration:		/* Parameter */
	declaration_specifier
	{ if (  ($1->StorageClass != SIR_STORAGE_NONE)
	      &&($1->StorageClass != SIR_STORAGE_AUTO)
	      &&($1->StorageClass != SIR_STORAGE_REGISTER))
	     { PRS_error(
		"Parameter storage class must be empty, 'auto' or 'register'",
			PRS_ERROR_PARAMETER_STORAGE_CLASS,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete $1;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameter($1->BaseType);
	  delete $1;
	 }
	| declaration_specifier abstract_declarator
	{ if (  ($1->StorageClass != SIR_STORAGE_NONE)
	      &&($1->StorageClass != SIR_STORAGE_AUTO)
	      &&($1->StorageClass != SIR_STORAGE_REGISTER))
	     { PRS_error(
		"Parameter storage class must be empty, 'auto' or 'register'",
			PRS_ERROR_PARAMETER_STORAGE_CLASS,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameter(
		PRS_TheDesign->Types->FindOrInsertDeclaratorType(
						$1->BaseType, $2->DeclTypes));
	  delete $1; delete $2;
	 }
	| declaration_specifier identifier_declarator
	{ sir_type	*TypeEntry;
	  sir_symbol	*SymbolEntry;

	  /* StorageClass is checked in Declare() */
	  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
						$1->BaseType, $2->DeclTypes);
	  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol($1, $2, TypeEntry))))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameter(TypeEntry, SymbolEntry);
	  delete $1; delete $2;
	 }
	| declaration_specifier parameter_typedef_declarator
	{ sir_type	*TypeEntry;
	  sir_symbol	*SymbolEntry;

	  /* StorageClass is checked in Declare() */
	  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
						$1->BaseType, $2->DeclTypes);
	  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol($1, $2, TypeEntry))))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameter(TypeEntry, SymbolEntry);
	  delete $1; delete $2;
	 }
	| declaration_qualifier_list
	{ if (  ($1->StorageClass != SIR_STORAGE_NONE)
	      &&($1->StorageClass != SIR_STORAGE_AUTO)
	      &&($1->StorageClass != SIR_STORAGE_REGISTER))
	     { PRS_error(
		"Parameter storage class must be empty, 'auto' or 'register'",
			PRS_ERROR_PARAMETER_STORAGE_CLASS,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete $1;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameter(
			PRS_TheDesign->Types->FindOrInsert($1));
	  delete $1;
	 }
	| declaration_qualifier_list abstract_declarator
	{ if (  ($1->StorageClass != SIR_STORAGE_NONE)
	      &&($1->StorageClass != SIR_STORAGE_AUTO)
	      &&($1->StorageClass != SIR_STORAGE_REGISTER))
	     { PRS_error(
		"Parameter storage class must be empty, 'auto' or 'register'",
			PRS_ERROR_PARAMETER_STORAGE_CLASS,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameter(
		PRS_TheDesign->Types->FindOrInsertDeclaratorType(
				PRS_TheDesign->Types->FindOrInsert($1),
				$2->DeclTypes));
	  delete $1; delete $2;
	 }
	| declaration_qualifier_list identifier_declarator
	{ sir_declspec	*DeclSpec;
	  sir_type	*TypeEntry;
	  sir_symbol	*SymbolEntry;

	  /* StorageClass is checked in Declare() */
	  DeclSpec = new SIR_DeclSpec(PRS_TheDesign->Types, $1);
	  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
				DeclSpec->BaseType, $2->DeclTypes);
	  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $2, TypeEntry))))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       delete DeclSpec;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameter(TypeEntry, SymbolEntry);
	  delete $1; delete $2;
	  delete DeclSpec;
	 }
	| type_specifier
	{ $$ = new SIR_Parameter($1);
	 }
	| type_specifier abstract_declarator
	{ $$ = new SIR_Parameter(
		PRS_TheDesign->Types->FindOrInsertDeclaratorType(
						$1, $2->DeclTypes));
	  delete $2;
	 }
	| type_specifier identifier_declarator
	{ sir_declspec	*DeclSpec;
	  sir_type	*TypeEntry;
	  sir_symbol	*SymbolEntry;

	  /* StorageClass is checked in Declare() */
	  DeclSpec = new SIR_DeclSpec($1);
	  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
					$1, $2->DeclTypes);
	  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $2, TypeEntry))))
	     { PRS_SemanticError(SIR_Error);
	       delete $2;
	       delete DeclSpec;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameter(TypeEntry, SymbolEntry);
	  delete $2;
	  delete DeclSpec;
	 }
	| type_specifier parameter_typedef_declarator
	{ sir_declspec	*DeclSpec;
	  sir_type	*TypeEntry;
	  sir_symbol	*SymbolEntry;

	  /* StorageClass is checked in Declare() */
	  DeclSpec = new SIR_DeclSpec($1);
	  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
					$1, $2->DeclTypes);
	  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $2, TypeEntry))))
	     { PRS_SemanticError(SIR_Error);
	       delete $2;
	       delete DeclSpec;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameter(TypeEntry, SymbolEntry);
	  delete $2;
	  delete DeclSpec;
	 }
	| type_qualifier_list
	{ assert($1->StorageClass == SIR_STORAGE_NONE);
	  $$ = new SIR_Parameter(
			PRS_TheDesign->Types->FindOrInsert($1));
	  delete $1;
	 }
	| type_qualifier_list abstract_declarator
	{ assert($1->StorageClass == SIR_STORAGE_NONE);
	  $$ = new SIR_Parameter(
		PRS_TheDesign->Types->FindOrInsertDeclaratorType(
				PRS_TheDesign->Types->FindOrInsert($1),
				$2->DeclTypes));
	  delete $1; delete $2;
	 }
	| type_qualifier_list identifier_declarator
	{ sir_declspec	*DeclSpec;
	  sir_type	*TypeEntry;
	  sir_symbol	*SymbolEntry;

	  DeclSpec = new SIR_DeclSpec(PRS_TheDesign->Types, $1);
	  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
				DeclSpec->BaseType, $2->DeclTypes);
	  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $2, TypeEntry))))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       delete DeclSpec;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameter(TypeEntry, SymbolEntry);
	  delete $1; delete $2;
	  delete DeclSpec;
	 };

/* note: the following has been changed from the ANSI-C grammar:	*/
/*	- identifier_or_typedef_name is modified to include also	*/
/*	  class names							*/

identifier_or_typedef_name:	/* Name */
	identifier
	{ $$ = $1;
	 }
	| typedef_or_class_name
	{ $$ = $1;
	 };

type_name:			/* Type */
	type_specifier
	{ $$ = $1;
	 }
	| type_specifier abstract_declarator
	{ $$ = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
					$1, $2->DeclTypes);
	  delete $2;
	 }
	| type_qualifier_list
	{ $$ = PRS_TheDesign->Types->FindOrInsert($1);
	  delete $1;
	 }
	| type_qualifier_list abstract_declarator
	{ $$ = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
				PRS_TheDesign->Types->FindOrInsert($1),
				$2->DeclTypes);
	  delete $1; delete $2;
	 };

initializer_opt:		/* Initializer */
	/* nothing */
	{ $$ = NULL;
	 }
	| '=' initializer
	{ $$ = $2;
	 };

/* note: the following has been changed from the ANSI-C grammar:	*/
/*	- an initializer is not an assignment_expression,		*/
/*	  but a constant_expression					*/
/*	  (which probably is the case anyway for 99.9% of C programs)	*/

initializer:			/* Initializer */
	'{' initializer_list '}'
	{ $$ = new SIR_Initializer($2, NULL);
	 }
	| '{' initializer_list ',' '}'
	{ $$ = new SIR_Initializer($2, NULL);
	 }
	| constant_expression	/* was: assignment_expression */
	{ sir_constant	*Constant;

	  Constant = $1->Eval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $1);
	       delete Constant;
	       delete $1;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Initializer(NULL, Constant);
	  delete $1;
	 };

initializer_list:		/* InitializerList */
	initializer
	{ $$ = new SIR_Initials($1);
	 }
	| initializer_list ',' initializer
	{ $1->Append($3);
	  $$ = $1;
	 };


/*** Statements *********************************************************/


/* note: the following rule has been added to the ANSI-C grammar:	*/
/*	- line_info obtains and keeps the current line information	*/
/*	  so that statements get their correct location info		*/

line_info:			/* LineInfo */
	/* nothing */
	{ $$ = new PRS_ScannerToken();
	 };

statement:			/* Statement */
	labeled_statement
	{ $$ = $1;
	 }
	| compound_statement
	{ $$ = $1;
	 }
	| expression_statement
	{ $$ = $1;
	 }
	| selection_statement
	{ $$ = $1;
	 }
	| iteration_statement
	{ $$ = $1;
	 }
	| jump_statement
	{ $$ = $1;
	 }
	/*** SpecC-only: SpecC statements ***/
	| spec_c_statement
	{ $$ = $1;
	 };

labeled_statement:		/* Statement */
	identifier_or_typedef_name ':' statement
	{ sir_label	*Label;

	  assert(PRS_CurrScope->ParentSymbol != NULL);
	  assert(PRS_CurrScope->ParentSymbol->Labels != NULL);
	  if (!(Label = PRS_CurrScope->ParentSymbol->Labels->Define(
			$1->Name.chars(), NULL)))	/* no statement yet */
	     { PRS_SemanticError(SIR_Error, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Statement(SIR_STMNT_LABELED, PRS_CurrScope->ParentSymbol,
				Label, $3, $1->Line, $1->FileInfo);
	  Label->Statement = $$;	/* now we know the statement */
	  delete $1;
	 }
	| TOK_CASE line_info constant_expression ':' statement
	{ sir_constant	*Constant;

	  Constant = $3->Eval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $3);
	       delete Constant;
	       delete $2; delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	  if (Constant->Type == SIR_CONST_CHARSTRING)
	     { PRS_error("Constant expression not integer",
			PRS_ERROR_CONST_EXPR_NOT_INTEGER,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete Constant;
	       delete $2; delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	  Constant->Converted(SIR_CONST_INT);
	  $$ = new SIR_Statement(SIR_STMNT_CASE, PRS_CurrScope->ParentSymbol,
				Constant, $5, $2->Line, $2->FileInfo);
	  delete $2; delete $3;
	 }
	| TOK_DEFAULT line_info ':' statement
	{ $$ = new SIR_Statement(SIR_STMNT_DEFAULT, PRS_CurrScope->ParentSymbol,
				(sir_expression*)NULL,
				$4, NULL,
				$2->Line, $2->FileInfo);
	  delete $2;
	 };

/* note: the following has been changed from the ANSI-C grammar:	*/
/*	- rule compound_scope is used to prepare an inner scope for	*/
/*	  each compound_statement (and to obtain the line infos)	*/

compound_statement:		/* Statement */
	compound_scope '{' '}'
	{ $1->Statements = new SIR_Statements($1->Scope);
	  PRS_CurrScope = PRS_CurrScope->Parent;
	  $$ = $1;
	 }
	| compound_scope '{' declaration_list '}'
	{ $1->Statements = new SIR_Statements($1->Scope);
	  PRS_CurrScope = PRS_CurrScope->Parent;
	  $$ = $1;
	 }
	| compound_scope '{' statement_list '}'
	{ $1->Statements = $3;
	  PRS_CurrScope = PRS_CurrScope->Parent;
	  $$ = $1;
	 }
	| compound_scope '{' declaration_list statement_list '}'
	{ $1->Statements = $4;
	  PRS_CurrScope = PRS_CurrScope->Parent;
	  $$ = $1;
	 };

compound_scope:			/* Statement */
	/* nothing */
	{ assert(PRS_CurrScope->ParentSymbol != NULL);
	  PRS_CurrScope = new SIR_Symbols(
			PRS_CurrScope,			/* open new        */
			SIR_SCOPE_STATEMENT,		/* statement scope */
			PRS_CurrScope->ParentSymbol);	/* prior is parent */
	  $$ = new SIR_Statement(SIR_STMNT_COMPOUND,
			PRS_CurrScope->ParentSymbol,
			PRS_CurrScope,
			NULL,	/* no statements yet */
			PRS_lineno, PRS_CurrFileInfo);
	 };

declaration_list:		/* void */
	declaration
	{ /* nothing to do */
	 }
	| declaration_list declaration
	{ /* nothing to do */
	 }
	/*** SpecC-only: note definitions in compound statements ***/
	| note_definition
	{ /* nothing to do */
	 }
	| declaration_list note_definition
	{ /* nothing to do */
	 };

statement_list:			/* StatementList */
	statement
	{ $$ = new SIR_Statements(PRS_CurrScope, $1);
	 }
	| statement_list statement
	{ $1->Append($2);
	  $$ = $1;
	 }
	/*** SpecC-only: note definitions in compound statements ***/
	| statement_list note_definition
	{ $$ = $1;
	 };

expression_statement:		/* Statement */
	comma_expression_opt ';'
	{ sir_lineinfo	*LineInfo;

	  LineInfo = $1->GetFirstLineInfo();
	  if (LineInfo)
	     { $$ = new SIR_Statement(SIR_STMNT_EXPRESSION,
				PRS_CurrScope->ParentSymbol, $1,
				NULL, NULL,
				LineInfo->Line, LineInfo->File);
	      } /* fi */
	  else
	     { $$ = new SIR_Statement(SIR_STMNT_EXPRESSION,
				PRS_CurrScope->ParentSymbol, $1);
	      } /* esle */
	 };

selection_statement:		/* Statement */
	  TOK_IF line_info '(' comma_expression ')' statement
	{ $$ = new SIR_Statement(SIR_STMNT_IF, PRS_CurrScope->ParentSymbol,
				$4,
				$6, NULL,
				$2->Line, $2->FileInfo);
	  delete $2;
	 }
	| TOK_IF line_info '(' comma_expression ')' statement TOK_ELSE statement
	{ $$ = new SIR_Statement(SIR_STMNT_IF_ELSE, PRS_CurrScope->ParentSymbol,
				$4,
				$6, $8,
				$2->Line, $2->FileInfo);
	  delete $2;
	 }
	| TOK_SWITCH line_info '(' comma_expression ')' statement
	{ $$ = new SIR_Statement(SIR_STMNT_SWITCH, PRS_CurrScope->ParentSymbol,
				$4,
				$6, NULL,
				$2->Line, $2->FileInfo);
	  delete $2;
	 };

iteration_statement:		/* Statement */
	TOK_WHILE line_info '(' comma_expression_opt ')' statement
	{ $$ = new SIR_Statement(SIR_STMNT_WHILE, PRS_CurrScope->ParentSymbol,
				$4,
				$6, NULL,
				$2->Line, $2->FileInfo);
	  delete $2;
	 }
	| TOK_DO line_info statement TOK_WHILE '(' comma_expression ')' ';'
	{ $$ = new SIR_Statement(SIR_STMNT_DO_WHILE,
				PRS_CurrScope->ParentSymbol,
				$6,
				$3, NULL,
				$2->Line, $2->FileInfo);
	  delete $2;
	 }
	| TOK_FOR line_info '(' comma_expression_opt ';'
		comma_expression_opt ';' comma_expression_opt ')' statement
	{ $$ = new SIR_Statement(SIR_STMNT_FOR, PRS_CurrScope->ParentSymbol,
				$4,
				$10, NULL,
				$2->Line, $2->FileInfo,
				$6, $8);
	  delete $2;
	 };

jump_statement:			/* Statement */
	TOK_GOTO identifier_or_typedef_name ';'
	{ sir_label	*Label;

	  assert(PRS_CurrScope->ParentSymbol != NULL);
	  assert(PRS_CurrScope->ParentSymbol->Labels != NULL);
	  Label = PRS_CurrScope->ParentSymbol->Labels->FindOrInsert(
				$2->Name.chars());
	  $$ = new SIR_Statement(SIR_STMNT_GOTO, PRS_CurrScope->ParentSymbol,
				Label, NULL,
				$2->Line, $2->FileInfo);
	  delete $2;
	 }
	| TOK_CONTINUE ';'
	{ $$ = new SIR_Statement(SIR_STMNT_CONTINUE,
				PRS_CurrScope->ParentSymbol,
				(sir_expression*)NULL,
				NULL, NULL,
				PRS_lineno, PRS_CurrFileInfo);
	 }
	| TOK_BREAK ';'
	{ $$ = new SIR_Statement(SIR_STMNT_BREAK, PRS_CurrScope->ParentSymbol,
				(sir_expression*)NULL,
				NULL, NULL,
				PRS_lineno, PRS_CurrFileInfo);
	 }
	| TOK_RETURN line_info comma_expression_opt ';'
	{ sir_type	*ReturnType,
			*ExpectedType;

	  assert($3 != NULL);
	  ReturnType = $3->Type;
	  assert(PRS_CurrScope->ParentSymbol->IsFunction());
	  ExpectedType = PRS_CurrScope->ParentSymbol->Type->SubType;
	  if (!(ReturnType->IsConvertableTo(ExpectedType)))
	     { PRS_error("Type mismatch in return statement",
				PRS_ERROR_TYPE_MISMATCH_IN_RETURN,
				$2->Line, $2->FileInfo->Filename.chars());
	       delete $2; delete $3;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Statement(SIR_STMNT_RETURN, PRS_CurrScope->ParentSymbol,
				$3,
				NULL, NULL,
				$2->Line, $2->FileInfo);
	  delete $2;
	 };


/*** External Definitions ***********************************************/


/* note: the following has been changed from the ANSI-C grammar:	*/
/*	- translation unit is allowed to be empty!			*/

translation_unit:		/* void */
	/* nothing */
	{ /* nothing to do */
	 }
	| external_definition_list
	{ /* nothing to do */
	 };

external_definition_list:	/* void */
	external_definition
	{ /* nothing to do */
	 }
	| external_definition_list external_definition
	{ /* nothing to do */
	 };

external_definition:		/* void */
	function_definition
	{ /* nothing to do */
	 }
	| declaration
	{ /* nothing to do */
	 }
	/*** SpecC-only: SpecC specific definitions ***/
	| spec_c_definition
	{ /* nothing to do */
	 };

/* note: the following has been changed from the ANSI-C grammar:	*/
/*	- since in SpecC functions are not defined only in the global	*/
/*	  name space, they can have typedef_names (shadowing them)	*/
/*	- for the function definition rule, identifier_declarator	*/
/*	  was changed to declarator in those rules which don't have	*/
/*	  the default-int return type					*/

function_definition:		/* void */
	identifier_declarator
		{ /* !!! mid-rule action !!! */
		  sir_declspec	*DeclSpec;
		  sir_type	*TypeEntry;
		  sir_symbol	*SymbolEntry;

		  DeclSpec = new SIR_DeclSpec(	/* default return type 'int' */
			PRS_TheDesign->Types->FindOrInsert(SIR_TYPE_INT));
		  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
					DeclSpec->BaseType, $1->DeclTypes);
		  if (TypeEntry->Type != SIR_TYPE_FUNCTION)
		     { PRS_error("Function declarator expected",
				PRS_ERROR_FUNCTION_DECLARATOR_EXPECTED,
				$1->Line, $1->FileInfo->Filename.chars());
		       delete $1;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $1, TypeEntry,
					true)))) /* prepare function def. */
		     { PRS_SemanticError(SIR_Error);
		       delete $1;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		  PRS_CurrScope = SymbolEntry->ParamScope; /* use local scope */
		  $<Symbol>$ = SymbolEntry;
		  delete DeclSpec;
		 }
		compound_statement
	{ sir_symbol	*Symbol;
	  ERROR		Error;

	  assert(PRS_CurrScope->Parent != NULL);
	  PRS_CurrScope = PRS_CurrScope->Parent; /* back to outer scope */
	  Symbol = $<Symbol>2;
	  if ((Error = PRS_CurrScope->DefineFunction(Symbol, $3)))
	     { PRS_SemanticError(Error, Symbol);
	       delete $1;
	       YYABORT;
	      } /* fi */
	  delete $1;
	 }
	| declaration_specifier declarator
		{ /* !!! mid-rule action !!! */
		  sir_type	*TypeEntry;
		  sir_symbol	*SymbolEntry;

		  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
					$1->BaseType, $2->DeclTypes);
		  if (TypeEntry->Type != SIR_TYPE_FUNCTION)
		     { PRS_error("Function declarator expected",
				PRS_ERROR_FUNCTION_DECLARATOR_EXPECTED,
				$2->Line, $2->FileInfo->Filename.chars());
		       delete $1; delete $2;
		       YYABORT;
		      } /* fi */
		  if (  ($1->StorageClass != SIR_STORAGE_NONE)
		      &&($1->StorageClass != SIR_STORAGE_STATIC))
		     { PRS_error("Storage class of function definition must"
					" be empty or 'static'",
				PRS_ERROR_BAD_STORAGECLASS_FOR_FUNCTION,
				$2->Line, $2->FileInfo->Filename.chars());
		       delete $1; delete $2;
		       YYABORT;
		      } /* fi */
		  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol($1, $2, TypeEntry,
					true)))) /* prepare function def. */
		     { PRS_SemanticError(SIR_Error);
		       delete $1; delete $2;
		       YYABORT;
		      } /* fi */
		  PRS_CurrScope = SymbolEntry->ParamScope; /* use local scope */
		  $<Symbol>$ = SymbolEntry;
		 }
		compound_statement
	{ sir_symbol	*Symbol;
	  ERROR		Error;

	  assert(PRS_CurrScope->Parent != NULL);
	  PRS_CurrScope = PRS_CurrScope->Parent; /* back to outer scope */
	  Symbol = $<Symbol>3;
	  if ((Error = PRS_CurrScope->DefineFunction(Symbol, $4)))
	     { PRS_SemanticError(Error, Symbol);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $1; delete $2;
	 }
	| type_specifier declarator
		{ /* !!! mid-rule action !!! */
		  sir_declspec	*DeclSpec;
		  sir_type	*TypeEntry;
		  sir_symbol	*SymbolEntry;

		  DeclSpec = new SIR_DeclSpec($1);
		  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
						$1, $2->DeclTypes);
		  if (TypeEntry->Type != SIR_TYPE_FUNCTION)
		     { PRS_error("Function declarator expected",
				PRS_ERROR_FUNCTION_DECLARATOR_EXPECTED,
				$2->Line, $2->FileInfo->Filename.chars());
		       delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $2, TypeEntry,
					true)))) /* prepare function def. */
		     { PRS_SemanticError(SIR_Error);
		       delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		  PRS_CurrScope = SymbolEntry->ParamScope; /* use local scope */
		  $<Symbol>$ = SymbolEntry;
		  delete DeclSpec;
		 }
		compound_statement
	{ sir_symbol	*Symbol;
	  ERROR		Error;

	  assert(PRS_CurrScope->Parent != NULL);
	  PRS_CurrScope = PRS_CurrScope->Parent; /* back to outer scope */
	  Symbol = $<Symbol>3;
	  if ((Error = PRS_CurrScope->DefineFunction(Symbol, $4)))
	     { PRS_SemanticError(Error, Symbol);
	       delete $2;
	       YYABORT;
	      } /* fi */
	  delete $2;
	 }
	| declaration_qualifier_list identifier_declarator
		{ /* !!! mid-rule action !!! */
		  sir_declspec	*DeclSpec;
		  sir_type	*TypeEntry;
		  sir_symbol	*SymbolEntry;

		  DeclSpec = new SIR_DeclSpec(PRS_TheDesign->Types, $1);
		  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
					DeclSpec->BaseType, $2->DeclTypes);
		  if (TypeEntry->Type != SIR_TYPE_FUNCTION)
		     { PRS_error("Function declarator expected",
				PRS_ERROR_FUNCTION_DECLARATOR_EXPECTED,
				$2->Line, $2->FileInfo->Filename.chars());
		       delete $1; delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		  if (  ($1->StorageClass != SIR_STORAGE_NONE)
		      &&($1->StorageClass != SIR_STORAGE_STATIC))
		     { PRS_error("Storage class of function definition must"
					" be empty or 'static'",
				PRS_ERROR_BAD_STORAGECLASS_FOR_FUNCTION,
				$2->Line, $2->FileInfo->Filename.chars());
		       delete $1; delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $2, TypeEntry,
					true)))) /* prepare function def. */
		     { PRS_SemanticError(SIR_Error);
		       delete $1; delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		  PRS_CurrScope = SymbolEntry->ParamScope; /* use local scope */
		  $<Symbol>$ = SymbolEntry;
		  delete DeclSpec;
		 }
		compound_statement
	{ sir_symbol	*Symbol;
	  ERROR		Error;

	  assert(PRS_CurrScope->Parent != NULL);
	  PRS_CurrScope = PRS_CurrScope->Parent; /* back to outer scope */
	  Symbol = $<Symbol>3;
	  if ((Error = PRS_CurrScope->DefineFunction(Symbol, $4)))
	     { PRS_SemanticError(Error, Symbol);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $1; delete $2;
	 }
	| type_qualifier_list identifier_declarator
		{ /* !!! mid-rule action !!! */
		  sir_declspec	*DeclSpec;
		  sir_type	*TypeEntry;
		  sir_symbol	*SymbolEntry;

		  DeclSpec = new SIR_DeclSpec(PRS_TheDesign->Types, $1);
		  TypeEntry = PRS_TheDesign->Types->FindOrInsertDeclaratorType(
					DeclSpec->BaseType, $2->DeclTypes);
		  if (TypeEntry->Type != SIR_TYPE_FUNCTION)
		     { PRS_error("Function declarator expected",
				PRS_ERROR_FUNCTION_DECLARATOR_EXPECTED,
				$2->Line, $2->FileInfo->Filename.chars());
		       delete $1; delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		  assert($1->StorageClass == SIR_STORAGE_NONE);
		  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(DeclSpec, $2, TypeEntry,
					true)))) /* prepare function def. */
		     { PRS_SemanticError(SIR_Error);
		       delete $1; delete $2;
		       delete DeclSpec;
		       YYABORT;
		      } /* fi */
		  PRS_CurrScope = SymbolEntry->ParamScope; /* use local scope */
		  $<Symbol>$ = SymbolEntry;
		  delete DeclSpec;
		 }
		compound_statement
	{ sir_symbol	*Symbol;
	  ERROR		Error;

	  assert(PRS_CurrScope->Parent != NULL);
	  PRS_CurrScope = PRS_CurrScope->Parent; /* back to outer scope */
	  Symbol = $<Symbol>3;
	  if ((Error = PRS_CurrScope->DefineFunction(Symbol, $4)))
	     { PRS_SemanticError(Error, Symbol);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  delete $1; delete $2;
	 };

declarator:			/* Declarator */
	identifier_declarator
	{ $$ = $1;
	 }
	| typedef_declarator
	{ $$ = $1;
	 };

typedef_declarator:		/* Declarator */
	paren_typedef_declarator
	{ $$ = $1;
	 }
	| parameter_typedef_declarator
	{ $$ = $1;
	 };

parameter_typedef_declarator:	/* Declarator */
	typedef_or_class_name
	{ $$ = new SIR_Declarator(new sir_type_list(),
			$1->Name.chars(),
			$1->Line, $1->FileInfo);
	  delete $1;
	 }
	| typedef_or_class_name postfixing_abstract_declarator
	{ $2->SymbolName = new string($1->Name);
	  $2->Line = $1->Line;
	  $2->FileInfo = $1->FileInfo;
	  $$ = $2;
	  delete $1;
	 }
	| clean_typedef_declarator
	{ $$ = $1;
	 };

clean_typedef_declarator:	/* Declarator */
	clean_postfix_typedef_declarator
	{ $$ = $1;
	 }
	| '*' parameter_typedef_declarator
	{ $2->DeclTypes->Append(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_POINTER,
				NULL, 0, false, false));
	  $$ = $2;
	 }
	| '*' type_qualifier_list parameter_typedef_declarator
	{ $3->DeclTypes->Append(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_POINTER,
				NULL, 0, $2->IsConst, $2->IsVolatile));
	  $$ = $3;
	  delete $2;
	 };

clean_postfix_typedef_declarator:	/* Declarator */
	'(' clean_typedef_declarator ')'
	{ $$ = $2;
	 }
	| '(' clean_typedef_declarator ')' postfixing_abstract_declarator
	{	/* note: this is a pointer ($2) to a function ($4)! */
	  $2->DeclTypes->Concat($4->DeclTypes);
	  $$ = $2;
	  delete $4;
	 };

paren_typedef_declarator:	/* Declarator */
	paren_postfix_typedef_declarator
	{ $$ = $1;
	 }
	| '*' '(' simple_paren_typedef_declarator ')'
	{ $3->DeclTypes->Append(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_POINTER,
				NULL, 0, false, false));
	  $$ = $3;
	 }
	| '*' type_qualifier_list '(' simple_paren_typedef_declarator ')'
	{ $4->DeclTypes->Append(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_POINTER,
				NULL, 0, $2->IsConst, $2->IsVolatile));
	  $$ = $4;
	  delete $2;
	 }
	| '*' paren_typedef_declarator
	{ $2->DeclTypes->Append(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_POINTER,
				NULL, 0, false, false));
	  $$ = $2;
	 }
	| '*' type_qualifier_list paren_typedef_declarator
	{ $3->DeclTypes->Append(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_POINTER,
				NULL, 0, $2->IsConst, $2->IsVolatile));
	  $$ = $3;
	  delete $2;
	 };

paren_postfix_typedef_declarator:	/* Declarator */
	'(' paren_typedef_declarator ')'
	{ $$ = $2;
	 }
	| '(' simple_paren_typedef_declarator postfixing_abstract_declarator ')'
	{	/* note: this is a function ($3) with a typedef name ($2) */
	  $2->DeclTypes->Precat($3->DeclTypes);
	  assert($2->Parameters == NULL);
	  assert($2->ParamScope == NULL);
	  $2->Parameters = $3->Parameters;
	  $2->ParamScope = $3->ParamScope;
	  $3->Parameters = NULL;
	  $3->ParamScope = NULL;
	  $$ = $2;
	  delete $3;
	 }
	| '(' paren_typedef_declarator ')' postfixing_abstract_declarator
	{	/* note: this is a pointer ($2) to a function ($4)! */
	  $2->DeclTypes->Concat($4->DeclTypes);
	  $$ = $2;
	  delete $4;
	 };

simple_paren_typedef_declarator:	/* Declarator */
	typedef_or_class_name
	{ $$ = new SIR_Declarator(new sir_type_list(),
			$1->Name.chars(),
			$1->Line, $1->FileInfo);
	  delete $1;
	 }
	| '(' simple_paren_typedef_declarator ')'
	{ $$ = $2;
	 };

identifier_declarator:		/* Declarator */
	unary_identifier_declarator
	{ $$ = $1;
	 }
	| paren_identifier_declarator
	{ $$ = $1;
	 };

unary_identifier_declarator:	/* Declarator */
	postfix_identifier_declarator
	{ $$ = $1;
	 }
	| '*' identifier_declarator
	{ $2->DeclTypes->Append(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_POINTER,
				NULL, 0, false, false));
	  $$ = $2;
	 }
	| '*' type_qualifier_list identifier_declarator
	{ $3->DeclTypes->Append(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_POINTER,
				NULL, 0, $2->IsConst, $2->IsVolatile));
	  $$ = $3;
	  delete $2;
	 };

postfix_identifier_declarator:	/* Declarator */
	paren_identifier_declarator postfixing_abstract_declarator
	{	/* note: this is a function or array ($2) with name ($1) */
	  $1->DeclTypes->Precat($2->DeclTypes);
	  assert($1->Parameters == NULL);
	  assert($1->ParamScope == NULL);
	  $1->Parameters = $2->Parameters;
	  $1->ParamScope = $2->ParamScope;
	  $2->Parameters = NULL;
	  $2->ParamScope = NULL;
	  $$ = $1;
	  delete $2;
	 }
	| '(' unary_identifier_declarator ')'
	{ $$ = $2;
	 }
	| '(' unary_identifier_declarator ')' postfixing_abstract_declarator
	{	/* note: this is a pointer ($2) to a function ($4)! */
	  $2->DeclTypes->Concat($4->DeclTypes);
	  $$ = $2;
	  delete $4;
	 };

paren_identifier_declarator:	/* Declarator */
	identifier
	{ $$ = new SIR_Declarator(new sir_type_list(),
			$1->Name.chars(),
			$1->Line, $1->FileInfo);
	  delete $1;
	 }
	| '(' paren_identifier_declarator ')'
	{ $$ = $2;
	 };

abstract_declarator:		/* AbstrDeclarator */
	unary_abstract_declarator
	{ $$ = $1;
	 }
	| postfix_abstract_declarator
	{ $$ = $1;
	 }
	| postfixing_abstract_declarator
	{ $$ = $1;
	 };

postfixing_abstract_declarator:	/* AbstrDeclarator */
	array_abstract_declarator
	{ $$ = $1;
	 }
	| '('
		{ /* !!! mid-rule action !!! */
		  $<Scope>$ = PRS_CurrScope;	/* push old scope */
		  PRS_CurrScope = new SIR_Symbols(	/* create local scope */
					PRS_CurrScope,
					SIR_SCOPE_PARAMETER);
		 }
		')'
	{ $$ = new SIR_Declarator(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_FUNCTION,
					new SIR_TypePtrs()),
		new SIR_Parameters(),
		PRS_CurrScope);	/* temp. local scope */
	  PRS_CurrScope = $<Scope>2;	/* restore old scope */
	 }
	| '('
		{ /* !!! mid-rule action !!! */
		  $<Scope>$ = PRS_CurrScope;	/* push old scope */
		  PRS_CurrScope = new SIR_Symbols(	/* create local scope */
					PRS_CurrScope,
					SIR_SCOPE_PARAMETER);
		 }
		parameter_type_list ')'
	{ $$ = new SIR_Declarator(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_FUNCTION,
					new SIR_TypePtrs($3)),
		$3,
		PRS_CurrScope);	/* temp. local scope */
	  PRS_CurrScope = $<Scope>2;	/* restore old scope */
	 };

array_abstract_declarator:	/* AbstrDeclarator */
	'[' ']'
	{ $$ = new SIR_Declarator(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_ARRAY,
				NULL, SIR_UNKNOWN_ARRAY_SIZE));
	 }
	| '[' constant_expression ']'
	{ int		Const;

	  Const = $2->IntegerEval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $2);
	       delete $2;
	       YYABORT;
	      } /* fi */
	  if (Const < 0)
	     { TmpErrMsg.form("Negative array size (%d) specified", Const);
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_NEGATIVE_ARRAY_SIZE,
				PRS_lineno, PRS_CurrentFile.chars());
	       delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Declarator(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_ARRAY,
				NULL, Const));
	  delete $2;
	 }
	| array_abstract_declarator '[' constant_expression ']'
	{ int		Const;

	  Const = $3->IntegerEval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  if (Const < 0)
	     { TmpErrMsg.form("Negative array size (%d) specified", Const);
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_NEGATIVE_ARRAY_SIZE,
				PRS_lineno, PRS_CurrentFile.chars());
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  $1->DeclTypes->Append(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_ARRAY,
				NULL, Const));
	  $$ = $1;
	  delete $3;
	 };

unary_abstract_declarator:	/* AbstrDeclarator */
	'*'
	{ $$ = new SIR_Declarator(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_POINTER,
				NULL, 0, false, false));
	 }
	| '*' type_qualifier_list
	{ $$ = new SIR_Declarator(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_POINTER,
				NULL, 0, $2->IsConst, $2->IsVolatile));
	  delete $2;
	 }
	| '*' abstract_declarator
	{ $2->DeclTypes->Append(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_POINTER,
				NULL, 0, false, false));
	  $$ = $2;
	 }
	| '*' type_qualifier_list abstract_declarator
	{ $3->DeclTypes->Append(
		new SIR_Type(SIR_UNKNOWN_SUBTYPE, SIR_TYPE_POINTER,
				NULL, 0, $2->IsConst, $2->IsVolatile));
	  $$ = $3;
	  delete $2;
	 };

postfix_abstract_declarator:	/* AbstrDeclarator */
	'(' unary_abstract_declarator ')'
	{ $$ = $2;
	 }
	| '(' postfix_abstract_declarator ')'
	{ $$ = $2;
	 }
	| '(' postfixing_abstract_declarator ')'
	{ $$ = $2;
	 }
	| '(' unary_abstract_declarator ')' postfixing_abstract_declarator
	{	/* note: this is a pointer ($2) to a function ($4)! */
	  $2->DeclTypes->Concat($4->DeclTypes);
	  $$ = $2;
	  delete $4;
	 };


/*** Preprocessor *******************************************************/


/* nothing to do (is handled before either by cpp or by the scanner) */


/*** SpecC Constructs ***************************************************/


spec_c_definition:		/* void */
	import_definition
	{ /* nothing to do */
	 }
	| end_of_import
	{ /* nothing to do */
	 }
	| behavior_declaration
	{ /* nothing to do */
	 }
	| behavior_definition
	{ /* nothing to do */
	 }
	| channel_declaration
	{ /* nothing to do */
	 }
	| channel_definition
	{ /* nothing to do */
	 }
	| interface_declaration
	{ /* nothing to do */
	 }
	| interface_definition
	{ /* nothing to do */
	 }
	| note_definition
	{ /* nothing to do */
	 };

import_definition:		/* void */
	TOK_IMPORT string_literal_list ';'
	{ const char	*ImportFileName,
			*DesignName;
	  string	PreprocFileName;
	  FILE		*ImportFile;
	  sir_design	*ImportedDesign;
	  prs_stackitem	*StackEntry;
	  ERROR		Error;

	  if ($2->CharString.length() < 1)
	     { PRS_error("Empty filename", PRS_ERROR_EMPTY_FILENAME,
			$2->Line, $2->FileInfo->Filename.chars());
	       delete $2;
	       YYABORT;
	      } /* fi */
	  if (!(ImportFileName = GL_FindFileInDirList(
				$2->CharString.chars(),
				NULL,			/* try without suffix */
				PRS_ImportDirList)))
	     { if (!(ImportFileName = GL_FindFileInDirList(
				$2->CharString.chars(),
				PRS_IMPORT_SC_FILE_SUFFIX, /* try SC suffix */
				PRS_ImportDirList)))
		  { TmpErrMsg.form(
			"Import file \"%s\" not found" GL_ERROR_MSG_NEWLINE
			"in import search path",
				$2->CharString.chars());
		    PRS_error(TmpErrMsg.chars(),
				PRS_ERROR_IMPORT_FILE_NOT_FOUND,
				$2->Line, $2->FileInfo->Filename.chars());
		    delete $2;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  assert(ImportFileName != NULL);	/* file has been found */
	  /* first, determine the imported design name */
	  DesignName = GL_GetBaseName(ImportFileName);
	  if ((Error = GL_CheckIdentifier(DesignName)))
	     { PRS_SemanticError(Error);
	       delete $2;
	       YYABORT;
	      } /* fi */
	  StackEntry = PRS_FileStack;
	  while(StackEntry)
	     { if (0 == strcmp(StackEntry->Design->Name.chars(), DesignName))
		  { TmpErrMsg.form("Cyclic import of design '%s'", DesignName);
		    PRS_error(TmpErrMsg.chars(),
				PRS_ERROR_CYCLIC_IMPORT,
				$2->Line, $2->FileInfo->Filename.chars());
		    delete $2;
		    YYABORT;
		   } /* fi */
	       StackEntry = StackEntry->Next;
	      } /* elihw */
	  if (PRS_TheDesign->ImportList->Find(DesignName))
	     { GL_PrintWarningFmt(GL_WARN_INFORMATIVE,
			"In file \"%s\" in line %u," GL_WARN_MSG_NEWLINE
			"import of file \"%s\" ignored;" GL_WARN_MSG_NEWLINE
			"design '%s' is already imported",
				$2->FileInfo->Filename.chars(), $2->Line,
				ImportFileName, DesignName);
	       /* nothing to do, already there */
	      } /* fi */
	  else
	     { /* run the preprocessor on the source */
	       PreprocFileName.form("%s/%s%s",
					PRS_IMPORT_TMP_DIR,
					DesignName,
					PRS_IMPORT_SI_FILE_SUFFIX);
	       if ((Error = CCD_Preprocessor(ImportFileName,
						PreprocFileName.chars())))
		  { PRS_SemanticError(Error);
		    delete $2;
		    GL_DeleteFile(PreprocFileName.chars());
		    YYABORT;
		   } /* fi */
	       /* store the complete parser status on a stack */
	       /* and switch to the nested file;              */
	       if ((PRS_Error = PRS_Push(ImportFileName, DesignName,
						PreprocFileName.chars())))
		  { PRS_SemanticError(PRS_Error);
		    delete $2;
		    GL_DeleteFile(PreprocFileName.chars());
		    YYABORT;
		   } /* fi */
	       /* the scanner will notify us with TOK_SCANNER_EOF */
	       /* when the end of the imported file is reached    */
	       /* (see the next rule);                            */
	      } /* esle */
	  delete $2;
	 };

end_of_import:			/* void */
	TOK_SCANNER_EOF
	{ sir_design	*ImportedDesign;
	  string	ImportFileName;
	  ERROR		Error;

	  assert(PRS_FileStack != NULL);	/* must be in a nested file */
	  ImportedDesign = PRS_Pop(&ImportFileName);
	  ImportedDesign->GarbageCollection();
	  if ((Error = PRS_TheDesign->Integrate(
					ImportedDesign,
					ImportFileName.chars(),
					PRS_CurrentFile.chars(),
					PRS_lineno)))
	     { PRS_SemanticError(Error);
	       YYABORT;
	      } /* fi */
	 };

behavior_declaration:		/* void */
	behavior_specifier port_list_opt implements_interface_opt ';'
	{ sir_symbol	*SymbolEntry;
	  sir_type	*TypeEntry;

	  $1->Parameters = $2;
	  TypeEntry = PRS_TheDesign->Types->FindOrInsertClassType(
			SIR_TYPE_BEHAVIOR,
			$1->SymbolName->chars(),
			new SIR_TypePtrs($2));
	  if (!(SymbolEntry = PRS_CurrScope->Declare(
			new SIR_Symbol(SIR_SYMBOL_BEHAVIOR,
					$1, TypeEntry, $3, false))))
	     { PRS_SemanticError(SIR_Error);
	       delete $1;
	       YYABORT;
	      } /* fi */
	  delete $1;
	 };

behavior_definition:		/* void */
	behavior_specifier port_list_opt implements_interface_opt
		{ /* !!! mid-rule action !!! */
		  sir_symbol	*SymbolEntry;
		  sir_type	*TypeEntry;

		  $1->Parameters = $2;
		  TypeEntry = PRS_TheDesign->Types->FindOrInsertClassType(
				SIR_TYPE_BEHAVIOR,
				$1->SymbolName->chars(),
				new SIR_TypePtrs($2));
		  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(SIR_SYMBOL_BEHAVIOR,
						$1, TypeEntry, $3, true))))
		     { PRS_SemanticError(SIR_Error);
		       delete $1;
		       YYABORT;
		      } /* fi */
		  $<Symbol>$ = SymbolEntry;
		  PRS_CurrScope = SymbolEntry->ClassScope; /* enter class */
		 }
		'{' internal_definition_list_opt '}' ';'
	{ sir_symbol	*SymbolEntry;
	  ERROR		Error;

	  SymbolEntry = $<Symbol>4;
	  PRS_CurrScope = SymbolEntry->GetScope(); /* back to global scope */
	  if ((Error = PRS_CurrScope->DefineClass(SymbolEntry)))
	     { PRS_SemanticError(Error, SymbolEntry);
	       delete $1;
	       YYABORT;
	      } /* fi */
	  delete $1;
	 };

behavior_specifier:		/* Declarator */
	TOK_BEHAVIOR any_name
	{ $$ = new SIR_Declarator(NULL, $2->Name.chars(),
				$2->Line, $2->FileInfo);
	  PRS_CurrScope = new SIR_Symbols(	/* open scope for ports */
				PRS_CurrScope, SIR_SCOPE_CLASS);
	  $$->ParamScope = PRS_CurrScope;
	  delete $2;
	 };

channel_declaration:		/* void */
	channel_specifier port_list_opt implements_interface_opt ';'
	{ sir_symbol	*SymbolEntry;
	  sir_type	*TypeEntry;

	  $1->Parameters = $2;
	  TypeEntry = PRS_TheDesign->Types->FindOrInsertClassType(
			SIR_TYPE_CHANNEL,
			$1->SymbolName->chars(),
			new SIR_TypePtrs($2));
	  if (!(SymbolEntry = PRS_CurrScope->Declare(
			new SIR_Symbol(SIR_SYMBOL_CHANNEL,
					$1, TypeEntry, $3, false))))
	     { PRS_SemanticError(SIR_Error);
	       delete $1;
	       YYABORT;
	      } /* fi */
	  delete $1;
	 };

channel_definition:		/* void */
	channel_specifier port_list_opt implements_interface_opt
		{ /* !!! mid-rule action !!! */
		  sir_symbol	*SymbolEntry;
		  sir_type	*TypeEntry;

		  $1->Parameters = $2;
		  TypeEntry = PRS_TheDesign->Types->FindOrInsertClassType(
				SIR_TYPE_CHANNEL,
				$1->SymbolName->chars(),
				new SIR_TypePtrs($2));
		  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(SIR_SYMBOL_CHANNEL,
						$1, TypeEntry, $3, true))))
		     { PRS_SemanticError(SIR_Error);
		       delete $1;
		       YYABORT;
		      } /* fi */
		  $<Symbol>$ = SymbolEntry;
		  PRS_CurrScope = SymbolEntry->ClassScope; /* enter class */
		 }
		'{' internal_definition_list_opt '}' ';'
	{ sir_symbol	*SymbolEntry;
	  ERROR		Error;

	  SymbolEntry = $<Symbol>4;
	  PRS_CurrScope = SymbolEntry->GetScope(); /* back to global scope */
	  if ((Error = PRS_CurrScope->DefineClass(SymbolEntry)))
	     { PRS_SemanticError(Error, SymbolEntry);
	       delete $1;
	       YYABORT;
	      } /* fi */
	  delete $1;
	 };

channel_specifier:		/* Declarator */
	TOK_CHANNEL any_name
	{ $$ = new SIR_Declarator(NULL, $2->Name.chars(),
				$2->Line, $2->FileInfo);
	  PRS_CurrScope = new SIR_Symbols(	/* open scope for ports */
				PRS_CurrScope, SIR_SCOPE_CLASS);
	  $$->ParamScope = PRS_CurrScope;
	  delete $2;
	 };

port_list_opt:			/* ParameterList */
	/* nothing */
	{ $$ = new SIR_Parameters();
	  PRS_CurrScope = PRS_CurrScope->Parent; /* back to outer scope */
	 }
	| '(' ')'
	{ $$ = new SIR_Parameters();
	  PRS_CurrScope = PRS_CurrScope->Parent; /* back to outer scope */
	 }
	| '(' port_list ')'
	{ assert(PRS_CurrScope->ScopeInfo == SIR_SCOPE_CLASS);
	  if (! PRS_CurrScope->UserTypes->Empty())
	     { PRS_SemanticError(PRS_ERROR_USERTYPE_IN_PORT_LIST);
	       delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = $2;
	  PRS_CurrScope = PRS_CurrScope->Parent; /* back to outer scope */
	 };

port_list:			/* ParameterList */
	port_declaration
	{ if (PRS_CheckTypeDefinition($1->Type, true, true))
	     { delete $1;
	       YYABORT;
	      } /* fi */
	  if (  ($1->Type->Type == SIR_TYPE_VOID)
	      &&($1->Symbol))
	     { PRS_error("Port cannot be of type 'void'");
	       delete $1;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameters($1);
	 }
	| port_list ',' port_declaration
	{ if ($1->First()->Type->Type == SIR_TYPE_VOID)
	     { PRS_error(
		"First port is 'void', no more ports allowed");
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  if (PRS_CheckTypeDefinition($3->Type, false, true))
	     { delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  $1->Append($3);
	  $$ = $1;
	 };

port_declaration:		/* Parameter */
	port_direction parameter_declaration
	{ if (  ($1 == SIR_PORT_NONE)
	      &&($2->Type->Type != SIR_TYPE_VOID))
	     { $2->Type = PRS_TheDesign->Types->FindOrInsertModifiedType(
				$2->Type, $2->Type->Const, $2->Type->Volatile,
				SIR_PORT_INOUT);	/* default */
	      } /* fi */
	  else
	     { $2->Type = PRS_TheDesign->Types->FindOrInsertModifiedType(
				$2->Type, $2->Type->Const, $2->Type->Volatile,
				$1);
	      } /* esle */
	  if ($2->Symbol)
	     { $2->Symbol->Type = $2->Type;	/* update symbol type also! */
	      } /* fi */
	  $$ = $2;
	 }
	| interface_parameter
	{ $$ = $1;
	 };

port_direction:			/* Direction */
	/* nothing */
	{ $$ = SIR_PORT_NONE;
	 }
	| TOK_IN
	{ $$ = SIR_PORT_IN;
	 }
	| TOK_OUT
	{ $$ = SIR_PORT_OUT;
	 }
	| TOK_INOUT
	{ $$ = SIR_PORT_INOUT;
	 };

interface_parameter:		/* Parameter */
	interface_name
	{ assert($1->Symbol != NULL);	/* scanner found that interface */
	  $$ = new SIR_Parameter($1->Symbol->Type);
	  delete $1;
	 }
	| interface_name identifier
	{ sir_symbol	*SymbolEntry;

	  assert($1->Symbol != NULL);	/* scanner found that interface */
	  if (!(SymbolEntry = PRS_CurrScope->Declare(
			new SIR_Symbol(SIR_SYMBOL_IDENTIFIER,
					$2->Name.chars(),
					$1->Symbol->Type,
					SIR_STORAGE_NONE, 0,
					NULL,
					$2->Line, $2->FileInfo))))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Parameter($1->Symbol->Type, SymbolEntry);
	  delete $1; delete $2;
	 };

implements_interface_opt:	/* SymbolPtrList */
	/* nothing */
	{ $$ = new sir_symbol_ptrs();
	 }
	| TOK_IMPLEMENTS interface_list
	{ $$ = $2;
	 };

interface_list:			/* SymbolPtrList */
	interface_name
	{ $$ = new sir_symbol_ptrs(
		new SIR_SymbolPtr($1->Symbol));
	  delete $1;
	 }
	| interface_list ',' interface_name
	{ $1->Append(new SIR_SymbolPtr($3->Symbol));
	  $$ = $1;
	  delete $3;
	 };

internal_definition_list_opt:	/* void */
	/* nothing */
	{ /* nothing to do */
	 }
	| internal_definition_list
	{ /* nothing to do */
	 };

internal_definition_list:	/* void */
	internal_definition
	{ /* nothing to do */
	 }
	| internal_definition_list internal_definition
	{ /* nothing to do */
	 };

internal_definition:		/* void */
	function_definition
	{ /* nothing to do */
	 }
	| declaration
	{ /* nothing to do */
	 }
	| instantiation
	{ /* nothing to do */
	 }
	| note_definition
	{ /* nothing to do */
	 };

instantiation:			/* void */
	instance_declaring_list ';'
	{ delete $1;
	 };

instance_declaring_list:	/* DeclarationSpec */
	behavior_or_channel instance_declarator
	{ sir_symbol	*SymbolEntry;

	  SymbolEntry = new SIR_Symbol(
				$2->SymbolName->chars(),
				$1->BaseType,
				(sir_portmaps*) $2->Parameters,	/* see below */
				$2->Line, $2->FileInfo);
	  $2->Parameters = NULL;	/* not nice, but easy */
	  if (!(SymbolEntry = PRS_CurrScope->Declare(SymbolEntry)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = $1;
	  delete $2;
	 }
	| instance_declaring_list ',' instance_declarator
	{ sir_symbol	*SymbolEntry;

	  SymbolEntry = new SIR_Symbol(
				$3->SymbolName->chars(),
				$1->BaseType,
				(sir_portmaps*) $3->Parameters,	/* see below */
				$3->Line, $3->FileInfo);
	  $3->Parameters = NULL;	/* not nice, but easy */
	  if (!(SymbolEntry = PRS_CurrScope->Declare(SymbolEntry)))
	     { PRS_SemanticError(SIR_Error);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  $$ = $1;
	  delete $3;
	 };

instance_declarator:		/* Declarator */
	identifier port_mapping_list_opt
	{ $$ = new SIR_Declarator(NULL, $1->Name.chars(),
				$1->Line, $1->FileInfo);
	  $$->Parameters = (sir_parameters*) $2;	/* not nice, but easy */
	  delete $1;
	 }
	| typedef_or_class_name port_mapping_list_opt
	{ $$ = new SIR_Declarator(NULL, $1->Name.chars(),
				$1->Line, $1->FileInfo);
	  $$->Parameters = (sir_parameters*) $2;	/* not nice, but easy */
	  delete $1;
	 };

behavior_or_channel:		/* DeclarationSpec */
	behavior_name
	{ assert($1->Symbol != NULL);	/* scanner found the symbol */
	  $$ = new SIR_DeclSpec($1->Symbol->Type);
	  delete $1;
	 }
	| channel_name
	{ assert($1->Symbol != NULL);	/* scanner found the symbol */
	  $$ = new SIR_DeclSpec($1->Symbol->Type);
	  delete $1;
	 };

port_mapping_list_opt:		/* PortMapList */
	/* nothing */
	{ $$ = new SIR_PortMaps();	/* empty list required! */
	 }
	| '(' port_mapping_list ')'
	{ $$ = $2;
	 };

port_mapping_list:		/* PortMapList */
	port_mapping_opt
	{ $$ = new SIR_PortMaps(
			new SIR_PortMap($1));
	 }
	| port_mapping_list ',' port_mapping_opt
	{ $1->Append(new SIR_PortMap($3));
	  $$ = $1;
	 };

port_mapping_opt:		/* BitSliceList */
	/* nothing */
	{ $$ = new SIR_BitSlices(new SIR_BitSlice(NULL, 0, 0));
	 }
	| port_mapping
	{ $$ = $1;
	 };

port_mapping:			/* BitSliceList */
	bit_slice
	{ $$ = new SIR_BitSlices($1);
	 }
	| port_mapping '@' bit_slice
	{ $1->Append($3);
	  $$ = $1;
	 }
	| port_mapping '&' bit_slice
        { PRS_PrintWarning(GL_WARN_IMPORTANT,
		"Concatenation operator '&' deprecated. Use '@' instead");
	  $1->Append($3);
	  $$ = $1;
	 };

bit_slice:			/* BitSlice */
	constant
	{ $$ = new SIR_BitSlice($1);
	 }
	| '(' constant_expression ')'
	{ sir_constant	*Const;

	  Const = $2->Eval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $2);
	       delete $2;
	       delete Const;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_BitSlice(Const);
	  delete $2;
	 }
	| identifier
	{ if (!($1->Symbol))
	     { TmpErrMsg.form("Undeclared identifier '%s' in port mapping",
					$1->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_UNDECLARED_IDENTIFIER_2,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1;
	       YYABORT;
	      } /* fi */
	  if ($1->Symbol->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	     { $$ = new SIR_BitSlice($1->Symbol, $1->Symbol->Type->LeftBound,
					$1->Symbol->Type->RightBound);
	      } /* fi */
	  else
	     { $$ = new SIR_BitSlice($1->Symbol);
	      } /* esle */
	  delete $1;
	 }
	| identifier '[' constant_expression ':' constant_expression ']'
	{ int		Const1,
			Const2;
	  int		Left,
			Right;

	  if (!($1->Symbol))
	     { TmpErrMsg.form("Undeclared identifier '%s' in port mapping",
					$1->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_UNDECLARED_IDENTIFIER_2,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1; delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	  if ($1->Symbol->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)
	     { TmpErrMsg.form("Identifier '%s' is not of bitvector type",
					$1->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_PORT_MAPPING_1,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1; delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	  Left = $1->Symbol->Type->LeftBound;
	  Right = $1->Symbol->Type->RightBound;
	  Const1 = $3->IntegerEval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $3);
	       delete $1; delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	  Const2 = $5->IntegerEval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $5);
	       delete $1; delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	  if (  (  (Left > Right)
		 &&(  (Const1 > Left) || (Const1 < Right)
		    ||(Const2 > Left) || (Const2 < Right)))
	      ||(  (Left <= Right)
		 &&(  (Const1 < Left) || (Const1 > Right)
		    ||(Const2 < Left) || (Const2 > Right))))
	     { TmpErrMsg.form("Bitslice [%d:%d] out of range (%s[%d:%d])",
					Const1, Const2,
					$1->Name.chars(), Left, Right);
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_PORT_MAPPING_2,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1; delete $3; delete $5;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_BitSlice($1->Symbol, Const1, Const2);
	  delete $1; delete $3; delete $5;
	 }
	| identifier '[' constant_expression ']'
	{ int		Const;
	  int		Left,
			Right;

	  if (!($1->Symbol))
	     { TmpErrMsg.form("Undeclared identifier '%s' in port mapping",
					$1->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_UNDECLARED_IDENTIFIER_2,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  if ($1->Symbol->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)
	     { TmpErrMsg.form("Identifier '%s' is not of bitvector type",
					$1->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_PORT_MAPPING_1,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  Left = $1->Symbol->Type->LeftBound;
	  Right = $1->Symbol->Type->RightBound;
	  Const = $3->IntegerEval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $3);
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  if (  (  (Left > Right)
		 &&(  (Const > Left) || (Const < Right)))
	      ||(  (Left <= Right)
		 &&(  (Const < Left) || (Const > Right))))
	     { TmpErrMsg.form("Bitslice [%d:%d] out of range (%s[%d:%d])",
					Const, Const,
					$1->Name.chars(), Left, Right);
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_PORT_MAPPING_2,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_BitSlice($1->Symbol, Const, Const);
	  delete $1; delete $3;
	 };

interface_declaration:		/* void */
	interface_specifier ';'
	{ sir_symbol	*SymbolEntry;
	  sir_type	*TypeEntry;

	  TypeEntry = PRS_TheDesign->Types->FindOrInsertClassType(
			SIR_TYPE_INTERFACE,
			$1->SymbolName->chars(),
			NULL);
	  if (!(SymbolEntry = PRS_CurrScope->Declare(
			new SIR_Symbol(SIR_SYMBOL_INTERFACE,
					$1, TypeEntry, NULL, false))))
	     { PRS_SemanticError(SIR_Error);
	       delete $1;
	       YYABORT;
	      } /* fi */
	  delete $1;
	 };

interface_definition:		/* void */
	interface_specifier
		{ /* !!! mid-rule action !!! */
		  sir_symbol	*SymbolEntry;
		  sir_type	*TypeEntry;

		  $1->ParamScope = new SIR_Symbols(	/* open class scope */
				PRS_CurrScope, SIR_SCOPE_CLASS);
		  TypeEntry = PRS_TheDesign->Types->FindOrInsertClassType(
				SIR_TYPE_INTERFACE,
				$1->SymbolName->chars(),
				NULL);
		  if (!(SymbolEntry = PRS_CurrScope->Declare(
				new SIR_Symbol(SIR_SYMBOL_INTERFACE,
						$1, TypeEntry, NULL, true))))
		     { PRS_SemanticError(SIR_Error);
		       delete $1;
		       YYABORT;
		      } /* fi */
		  $<Symbol>$ = SymbolEntry;
		  PRS_CurrScope = SymbolEntry->ClassScope; /* enter class */
		 }
		'{' internal_declaration_list_opt '}' ';'
	{ sir_symbol	*SymbolEntry;
	  ERROR		Error;

	  SymbolEntry = $<Symbol>2;
	  PRS_CurrScope = SymbolEntry->GetScope(); /* back to global scope */
	  if ((Error = PRS_CurrScope->DefineClass(SymbolEntry)))
	     { PRS_SemanticError(Error, SymbolEntry);
	       delete $1;
	       YYABORT;
	      } /* fi */
	  delete $1;
	 };

interface_specifier:		/* Declarator */
	TOK_INTERFACE any_name
	{ $$ = new SIR_Declarator(NULL, $2->Name.chars(),
				$2->Line, $2->FileInfo);
	  delete $2;
	 };

internal_declaration_list_opt:	/* void */
	/* nothing */
	{ /* nothing to do */
	 }
	| internal_declaration_list
	{ /* nothing to do */
	 };

internal_declaration_list:	/* void */
	internal_declaration
	{ /* nothing to do */
	 }
	| internal_declaration_list internal_declaration
	{ /* nothing to do */
	 };

internal_declaration:		/* void */
	declaration
	{ /* nothing to do */
	 }
	| note_definition
	{ /* nothing to do */
	 };

note_definition:		/* void */
	TOK_NOTE any_name '=' note ';'
	{ ERROR		Error;

	  switch(PRS_CurrScope->ScopeInfo)
	     { case SIR_SCOPE_GLOBAL:
		  { if ((Error = PRS_TheDesign->Annotate(
				new SIR_Note($2->Name.chars(), $4))))
		       { PRS_SemanticError(SIR_Error);
			 delete $2;
			 YYABORT;
			} /* fi */
		    break;
		   }
	       case SIR_SCOPE_CLASS:
		  { assert(PRS_CurrScope->ParentSymbol != NULL);
		    if ((Error = PRS_CurrScope->ParentSymbol->Annotate(
					new SIR_Note($2->Name.chars(), $4))))
		       { PRS_SemanticError(SIR_Error);
			 delete $2;
			 YYABORT;
			} /* fi */
		    break;
		   }
	       case SIR_SCOPE_PARAMETER:
		  { assert(FALSE);	/* not possible due to grammar */
		   }
	       case SIR_SCOPE_STATEMENT:
		  { assert(PRS_CurrScope->ParentSymbol != NULL);
		    if ((Error = PRS_CurrScope->ParentSymbol->Annotate(
					new SIR_Note($2->Name.chars(), $4))))
		       { PRS_SemanticError(SIR_Error);
			 delete $2;
			 YYABORT;
			} /* fi */
		    break;
		   }
	       case SIR_SCOPE_USERTYPE:
		  { assert(PRS_CurrScope->ParentUType != NULL);
		    if ((Error = PRS_CurrScope->ParentUType->Annotate(
					new SIR_Note($2->Name.chars(), $4))))
		       { PRS_SemanticError(SIR_Error);
			 delete $2;
			 YYABORT;
			} /* fi */
		    break;
		   }
	       default:
		  { assert(FALSE);	/* bad ScopeInfo */
		   }
	      } /* hctiws */
	  delete $2;
	 }
	| TOK_NOTE any_name '.' any_name '=' note ';'
	{ ERROR		Error;
	  sir_label	*Label;
	  sir_symbol	*Symbol;
	  sir_usertype	*UserType;

	  /* search order:			*/
	  /* 1. Label in current function	*/
	  /* 2. Symbol in current scope		*/
	  /* 3. UserType in current scope	*/
	  /* 4. Symbol in any upper scope	*/
	  if (  (PRS_CurrScope->ParentSymbol)
	      &&(PRS_CurrScope->ParentSymbol->Labels)
	      &&(Label = PRS_CurrScope->ParentSymbol->Labels->Find(
						$2->Name.chars())))
	     { if ((Error = Label->Annotate(
				new SIR_Note($4->Name.chars(), $6))))
		  { PRS_SemanticError(SIR_Error);
		    delete $2; delete $4;
		    YYABORT;
		   } /* fi */
	      } /* fi */
	  else	/* no label found */
	     { if ((Symbol = PRS_CurrScope->FindLocally($2->Name.chars())))
		  { if ((Error = Symbol->Annotate(
				new SIR_Note($4->Name.chars(), $6))))
		       { PRS_SemanticError(SIR_Error);
			 delete $2; delete $4;
			 YYABORT;
			} /* fi */
		   } /* fi */
	       else /* no local symbol found */
		  { if ((UserType = PRS_CurrScope->UserTypes->FindLocally(
							$2->Name.chars())))
		       { if ((Error = UserType->Annotate(
					new SIR_Note($4->Name.chars(), $6))))
			    { PRS_SemanticError(SIR_Error);
			      delete $2; delete $4;
			      YYABORT;
			     } /* fi */
			} /* fi */
		    else /* no local usertype found */
		       { if (  (PRS_CurrScope->Parent)
			     &&(Symbol = PRS_CurrScope->Parent->Find(
							$2->Name.chars())))
			    { if ((Error = Symbol->Annotate(
					new SIR_Note($4->Name.chars(), $6))))
				 { PRS_SemanticError(SIR_Error);
				   delete $2; delete $4;
				   YYABORT;
				  } /* fi */
			     } /* fi */
			 else /* no symbol found */
			    { TmpErrMsg.form(
				"Undefined symbol '%s' to attach note '%s' to",
				$2->Name.chars(), $4->Name.chars());
			      PRS_error(TmpErrMsg.chars(),
					PRS_ERROR_UNDEFINED_SYMBOL_FOR_NOTE,
					$2->Line,
					$2->FileInfo->Filename.chars());
			      delete $2; delete $4; delete $6;
			      YYABORT;
			     } /* esle */
			} /* esle */
		   } /* esle */
	      } /* esle */
	  delete $2; delete $4;
	 };

note:				/* Constant */
	constant_expression
	{ $$ = $1->Eval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $1);
	       delete $1;
	       YYABORT;
	      } /* fi */
	  delete $1;
	 };

/* note: the following has been changed from the ANSI-C grammar:	*/
/*	- typedef_name is replaced by typedef_or_class_name (see below) */
/*	  whereever possible (everywhere except in the rules for	*/
/*	  typedef_type_specifier and typedef_declaration_specifier)	*/

typedef_or_class_name:		/* Name */
	typedef_name
	{ $$ = $1;
	 }
	| behavior_name
	{ $$ = $1;
	 }
	| channel_name
	{ $$ = $1;
	 }
	| interface_name
	{ $$ = $1;
	 };

any_name:			/* Name */
	identifier
	{ $$ = $1;
	 }
	| typedef_name
	{ $$ = $1;
	 }
	| behavior_name
	{ $$ = $1;
	 }
	| channel_name
	{ $$ = $1;
	 }
	| interface_name
	{ $$ = $1;
	 };


/*** SpecC Statements ***************************************************/


spec_c_statement:		/* Statement */
	concurrent_statement
	{ $$ = $1;
	 }
	| fsm_statement
	{ $$ = $1;
	 }
	| exception_statement
	{ $$ = $1;
	 }
	| timing_statement
	{ $$ = $1;
	 }
	| wait_statement
	{ $$ = $1;
	 }
	| waitfor_statement
	{ $$ = $1;
	 }
	| notify_statement
	{ $$ = $1;
	 };

concurrent_statement:		/* Statement */
	TOK_PAR line_info compound_statement
	{ $$ = new SIR_Statement(SIR_STMNT_PAR, PRS_CurrScope->ParentSymbol,
			(sir_expression*)NULL,
			$3, NULL,
			$2->Line, $2->FileInfo);
	  delete $2;
	 }
	| TOK_PIPE line_info compound_statement
	{ $$ = new SIR_Statement(SIR_STMNT_PIPE, PRS_CurrScope->ParentSymbol,
			(sir_expression*)NULL,
			$3, NULL,
			$2->Line, $2->FileInfo);
	  delete $2;
	 }
	| TOK_PIPE line_info '(' comma_expression_opt ';'
		comma_expression_opt ';' comma_expression_opt ')'
		compound_statement
	{ $$ = new SIR_Statement(SIR_STMNT_PIPE, PRS_CurrScope->ParentSymbol,
				$4,
				$10, NULL,
				$2->Line, $2->FileInfo,
				$6, $8);
	  delete $2;
	 };

fsm_statement:			/* Statement */
	TOK_FSM line_info '{' '}'
	{ $$ = new SIR_Statement(SIR_STMNT_FSM, PRS_CurrScope->ParentSymbol,
			new SIR_Transitions(),
			$2->Line, $2->FileInfo);
	  delete $2;
	 }
	| TOK_FSM line_info '{' transition_list '}'
	{ sir_transition *Trans1;

	  if ($4->Last()->Color == SIR_RED)	/* default next state? */
	     { $4->Last()->Color = SIR_WHITE;	/* break! */
	      } /* fi */
	  Trans1 = $4->First();
	  while(Trans1)
	     { assert(Trans1->Color == SIR_WHITE);
	       if (Trans1->NextState)
		  { if (!($4->FindCurrState(Trans1->NextState)))
		       { TmpErrMsg.form("State label '%s' not defined",
					Trans1->NextState->Name.chars());
			 PRS_error(TmpErrMsg.chars(),
				PRS_ERROR_STATE_LABEL_NOT_DEFINED,
				Trans1->LineInfo->Line,
				Trans1->LineInfo->File->Filename.chars());
			 delete $2; delete $4;
			 YYABORT;
			} /* fi */
		   } /* fi */
	       Trans1 = Trans1->Succ();
	      } /* elihw */
	  $$ = new SIR_Statement(SIR_STMNT_FSM, PRS_CurrScope->ParentSymbol,
				$4, $2->Line, $2->FileInfo);
	  delete $2;
	 };

transition_list:		/* TransitionList */
	transition
	{ $$ = $1;
	 }
	| transition_list transition
	{ sir_symbol	*NewLabel;
	  sir_transition *Trans;

	  NewLabel = $2->First()->CurrState;
	  if ((Trans = $1->FindCurrState(NewLabel)))
	     { TmpErrMsg.form("State label '%s' defined twice",
					NewLabel->Name.chars());
	       PRS_error(TmpErrMsg.chars(),
			PRS_ERROR_STATE_LABEL_DEFINED_TWICE,
			$2->First()->LineInfo->Line,
			$2->First()->LineInfo->File->Filename.chars());
	       delete $1; delete $2;
	       YYABORT;
	      } /* elihw */
	  if ($1->Last()->Color == SIR_RED)
	     { $1->Last()->NextState = NewLabel;
	       $1->Last()->Color = SIR_WHITE;
	      } /* fi */
	  $1->Concat($2);
	  $$ = $1;
	  delete $2;
	 };

transition:			/* TransitionList */
	identifier ':'
	{ if (!($1->Symbol))
	     { TmpErrMsg.form("Undeclared identifier '%s'", $1->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_UNDECLARED_IDENTIFIER,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1;
	       YYABORT;
	      } /* fi */
	  if ($1->Symbol->Type->Type != SIR_TYPE_BEHAVIOR)
	     { TmpErrMsg.form("Type mismatch: '%s' is not a behavior instance",
					$1->Name.chars());
	       PRS_error(TmpErrMsg.chars(),
			PRS_ERROR_NOT_BEHAVIOR_INSTANCE,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Transitions(
		new SIR_Transition(PRS_CurrScope->ParentSymbol,
					$1->Symbol, NULL, NULL,
					$1->Line, $1->FileInfo));
	  $$->First()->Color = SIR_RED;	/* fill in default next state */
	  delete $1;
	 }
	| identifier ':' cond_branch_list
	{ sir_transition	*Trans;

	  if (!($1->Symbol))
	     { TmpErrMsg.form("Undeclared identifier '%s'", $1->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_UNDECLARED_IDENTIFIER,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  if ($1->Symbol->Type->Type != SIR_TYPE_BEHAVIOR)
	     { TmpErrMsg.form("Type mismatch: '%s' is not a behavior instance",
					$1->Name.chars());
	       PRS_error(TmpErrMsg.chars(),
			PRS_ERROR_NOT_BEHAVIOR_INSTANCE,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1; delete $3;
	       YYABORT;
	      } /* fi */
	  Trans = $3->First();
	  while(Trans)
	     { assert(Trans->CurrState == NULL);
	       Trans->CurrState = $1->Symbol;
	       Trans = Trans->Succ();
	      } /* elihw */
	  $$ = $3;
	  delete $1;
	 }
	| identifier ':' '{' '}'
	{ if (!($1->Symbol))
	     { TmpErrMsg.form("Undeclared identifier '%s'", $1->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_UNDECLARED_IDENTIFIER,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1;
	       YYABORT;
	      } /* fi */
	  if ($1->Symbol->Type->Type != SIR_TYPE_BEHAVIOR)
	     { TmpErrMsg.form("Type mismatch: '%s' is not a behavior instance",
					$1->Name.chars());
	       PRS_error(TmpErrMsg.chars(),
			PRS_ERROR_NOT_BEHAVIOR_INSTANCE,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Transitions(
		new SIR_Transition(PRS_CurrScope->ParentSymbol,
					$1->Symbol, NULL, NULL,
					$1->Line, $1->FileInfo));
	  $$->First()->Color = SIR_RED;	/* fill in default next state */
	  delete $1;
	 }
	| identifier ':' '{' cond_branch_list '}'
	{ sir_transition	*Trans;

	  if (!($1->Symbol))
	     { TmpErrMsg.form("Undeclared identifier '%s'", $1->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_UNDECLARED_IDENTIFIER,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1; delete $4;
	       YYABORT;
	      } /* fi */
	  if ($1->Symbol->Type->Type != SIR_TYPE_BEHAVIOR)
	     { TmpErrMsg.form(
			"Type mismatch: '%s' is not a behavior instance",
					$1->Name.chars());
	       PRS_error(TmpErrMsg.chars(),
			PRS_ERROR_NOT_BEHAVIOR_INSTANCE,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1; delete $4;
	       YYABORT;
	      } /* fi */
	  Trans = $4->First();
	  while(Trans)
	     { assert(Trans->CurrState == NULL);
	       Trans->CurrState = $1->Symbol;
	       Trans = Trans->Succ();
	      } /* elihw */
	  $$ = $4;
	  delete $1;
	 };

cond_branch_list:		/* TransitionList */
	cond_branch
	{ $$ = new SIR_Transitions($1);
	 }
	| cond_branch_list cond_branch
	{ $1->Append($2);
	  $$ = $1;
	 };

cond_branch:			/* Transition */
	TOK_IF line_info '(' comma_expression ')' TOK_GOTO identifier ';'
	{ if (!($7->Symbol))
	     { TmpErrMsg.form("Undeclared identifier '%s'", $7->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_UNDECLARED_IDENTIFIER,
			$7->Line, $7->FileInfo->Filename.chars());
	       delete $2; delete $4; delete $7;
	       YYABORT;
	      } /* fi */
	  if ($7->Symbol->Type->Type != SIR_TYPE_BEHAVIOR)
	     { TmpErrMsg.form(
			"Type mismatch: '%s' is not a behavior instance",
					$7->Name.chars());
	       PRS_error(TmpErrMsg.chars(),
			PRS_ERROR_NOT_BEHAVIOR_INSTANCE,
			$7->Line, $7->FileInfo->Filename.chars());
	       delete $2; delete $4; delete $7;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Transition(PRS_CurrScope->ParentSymbol,
					(sir_symbol*)NULL, $4, $7->Symbol,
					$2->Line, $2->FileInfo);
	  delete $2; delete $7;
	 }
	| TOK_GOTO identifier ';'
	{ if (!($2->Symbol))
	     { TmpErrMsg.form("Undeclared identifier '%s'", $2->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_UNDECLARED_IDENTIFIER,
			$2->Line, $2->FileInfo->Filename.chars());
	       delete $2;
	       YYABORT;
	      } /* fi */
	  if ($2->Symbol->Type->Type != SIR_TYPE_BEHAVIOR)
	     { TmpErrMsg.form("Type mismatch: '%s' is not a behavior instance",
					$2->Name.chars());
	       PRS_error(TmpErrMsg.chars(),
			PRS_ERROR_NOT_BEHAVIOR_INSTANCE,
			$2->Line, $2->FileInfo->Filename.chars());
	       delete $2;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Transition(PRS_CurrScope->ParentSymbol,
					(sir_symbol*)NULL, NULL, $2->Symbol,
					$2->Line, $2->FileInfo);
	  delete $2;
	 }
	| TOK_IF line_info '(' comma_expression ')' TOK_BREAK ';'
	{ $$ = new SIR_Transition(PRS_CurrScope->ParentSymbol,
					(sir_symbol*)NULL, $4, NULL,
					$2->Line, $2->FileInfo);
	  delete $2;
	 }
	| TOK_BREAK ';'
	{ $$ = new SIR_Transition(PRS_CurrScope->ParentSymbol,
					(sir_symbol*)NULL, NULL, NULL,
					PRS_lineno, PRS_CurrFileInfo);
	 };

exception_statement:		/* Statement */
	TOK_TRY line_info compound_statement exception_list_opt
	{ $$ = new SIR_Statement(SIR_STMNT_EXCEPTION,
			PRS_CurrScope->ParentSymbol,
			$3, $4,
			$2->Line, $2->FileInfo);
	  delete $2;
	 };

exception_list_opt:		/* ExceptionList */
	/* nothing */
	{ $$ = new SIR_Exceptions();
	 }
	| exception_list
	{ $$ = $1;
	 };

exception_list:			/* ExceptionList */
	exception
	{ $$ = new SIR_Exceptions($1);
	 };
	| exception_list exception
	{ $1->Append($2);
	  $$ = $1;
	 };

exception:			/* Exception */
	TOK_TRAP paren_event_list compound_statement
	{ sir_symbol_ptr *Event;

	  Event = $2->First();
	  while(Event)
	     { if (Event->Symbol->Type->Direction == SIR_PORT_OUT)
		  { TmpErrMsg.form("Trap event '%s' must not be output port",
					Event->Symbol->Name.chars());
		    PRS_error(TmpErrMsg.chars(), PRS_ERROR_TRAP_OUTPUT_PORT,
					PRS_lineno, PRS_CurrentFile.chars());
		    delete $2;
		    YYABORT;
		   } /* fi */
	       Event = Event->Succ();
	      } /* elihw */
	  $$ = new SIR_Exception(SIR_EXCEPTION_TRAP, $2, $3,
				PRS_lineno, PRS_CurrFileInfo);
	 }
	| TOK_INTERRUPT paren_event_list compound_statement
	{ sir_symbol_ptr *Event;

	  Event = $2->First();
	  while(Event)
	     { if (Event->Symbol->Type->Direction == SIR_PORT_OUT)
		  { TmpErrMsg.form(
				"Interrupt event '%s' must not be output port",
					Event->Symbol->Name.chars());
		    PRS_error(TmpErrMsg.chars(),
					PRS_ERROR_INTERRUPT_OUTPUT_PORT,
					PRS_lineno, PRS_CurrentFile.chars());
		    delete $2;
		    YYABORT;
		   } /* fi */
	       Event = Event->Succ();
	      } /* elihw */
	  $$ = new SIR_Exception(SIR_EXCEPTION_INTERRUPT, $2, $3,
				PRS_lineno, PRS_CurrFileInfo);
	 };

paren_event_list:		/* SymbolPtrList */
	event_list
	{ $$ = $1;
	 }
	| '(' event_list ')'
	{ $$ = $2;
	 };

event_list:			/* SymbolPtrList */
	event
	{ $$ = new sir_symbol_ptrs($1);
	 }
	| event_list ',' event
	{ $1->Append($3);
	  $$ = $1;
	 };

event:				/* SymbolPtr */
	identifier
	{ if (!($1->Symbol))
	     { TmpErrMsg.form("Undeclared identifier '%s'", $1->Name.chars());
	       PRS_error(TmpErrMsg.chars(), PRS_ERROR_UNDECLARED_IDENTIFIER,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1;
	       YYABORT;
	      } /* fi */
	  if ($1->Symbol->Type->Type != SIR_TYPE_EVENT)
	     { TmpErrMsg.form("Type mismatch: '%s' is not an event",
					$1->Name.chars());
	       PRS_error(TmpErrMsg.chars(),
			PRS_ERROR_EVENT_TYPE_MISMATCH,
			$1->Line, $1->FileInfo->Filename.chars());
	       delete $1;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_SymbolPtr($1->Symbol);
	  delete $1;
	 };

timing_statement:		/* Statement */
	TOK_DO line_info compound_statement
		TOK_TIMING '{' constraint_list_opt '}'
	{ sir_constraint	*Constr;

	  Constr = $6->First();
	  while(Constr)
	     { if (!($3->Statements->FindLabeledStmnt(Constr->Label1)))
		  { TmpErrMsg.form("Constraint label '%s'" GL_ERROR_MSG_NEWLINE
					"not in do-statement list",
				Constr->Label1->LabelName.chars());
		    PRS_error(TmpErrMsg.chars(),
				PRS_ERROR_CONSTRAINT_LABEL_NOT_IN_DO,
				$2->Line,
				$2->FileInfo->Filename.chars());
		    delete $2;
		    delete $3;
		    delete $6;
		    YYABORT;
		   } /* fi */
	       if (!($3->Statements->FindLabeledStmnt(Constr->Label2)))
		  { TmpErrMsg.form("Constraint label '%s'" GL_ERROR_MSG_NEWLINE
					"not in do-statement list",
				Constr->Label2->LabelName.chars());
		    PRS_error(TmpErrMsg.chars(),
				PRS_ERROR_CONSTRAINT_LABEL_NOT_IN_DO,
				$2->Line,
				$2->FileInfo->Filename.chars());
		    delete $2;
		    delete $3;
		    delete $6;
		    YYABORT;
		   } /* fi */
	       Constr = Constr->Succ();
	      } /* elihw */
	  $$ = new SIR_Statement(SIR_STMNT_TIMING, PRS_CurrScope->ParentSymbol,
				$3, $6, $2->Line, $2->FileInfo);
	  delete $2;
	 };

constraint_list_opt:		/* ConstraintList */
	/* nothing */
	{ $$ = new SIR_Constraints();
	 }
	| constraint_list
	{ $$ = $1;
	 };

constraint_list:		/* ConstraintList */
	constraint
	{ $$ = new SIR_Constraints($1);
	 };
	| constraint_list constraint
	{ if ($1->Find($2->Label1, $2->Label2))
	     { TmpErrMsg.form("Redefined constraint between labels"
							GL_ERROR_MSG_NEWLINE
				"'%s' and '%s'",
				$2->Label1->LabelName.chars(),
				$2->Label2->LabelName.chars());
	       PRS_error(TmpErrMsg.chars(),
				PRS_ERROR_REDEFINED_CONSTRAINT,
				$2->LineInfo->Line,
				$2->LineInfo->File->Filename.chars());
	       delete $1;
	       delete $2;
	       YYABORT;
	      } /* fi */
	  $1->Insert($2);
	  $$ = $1;
	 };

constraint:			/* Constraint */
	TOK_RANGE '(' any_name ';' any_name ';' time_opt ';' time_opt ')' ';'
	{ sir_label	*Label1,
			*Label2;

	  assert(PRS_CurrScope->ParentSymbol != NULL);
	  assert(PRS_CurrScope->ParentSymbol->Labels != NULL);
	  if (!(Label1 = PRS_CurrScope->ParentSymbol->Labels->Find(
							$3->Name.chars())))
	     { TmpErrMsg.form("Undefined label '%s' in range statement",
							$3->Name.chars());
	       PRS_error(TmpErrMsg.chars(),
			PRS_ERROR_UNDEFINED_LABEL_IN_RANGE,
			$3->Line, $3->FileInfo->Filename.chars());
	       delete $3;
	       delete $5;
	       delete $7;
	       delete $9;
	       YYABORT;
	      } /* fi */
	  if (!(Label2 = PRS_CurrScope->ParentSymbol->Labels->Find(
							$5->Name.chars())))
	     { TmpErrMsg.form("Undefined label '%s' in range statement",
							$5->Name.chars());
	       PRS_error(TmpErrMsg.chars(),
			PRS_ERROR_UNDEFINED_LABEL_IN_RANGE,
			$5->Line, $5->FileInfo->Filename.chars());
	       delete $3;
	       delete $5;
	       delete $7;
	       delete $9;
	       YYABORT;
	      } /* fi */
	  $$ = new SIR_Constraint(Label1, Label2, $7, $9,
					$3->Line, $3->FileInfo);
	  delete $3; delete $5;
	 };

time_opt:			/* Constant */
	/* nothing */
	{ $$ = NULL;
	 }
	| time
	{ sir_constant	*Constant;

	  Constant = $1->Eval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $1);
	       delete Constant;
	       delete $1;
	       YYABORT;
	      } /* fi */
	  if (Constant->Type == SIR_CONST_CHARSTRING)
	     { PRS_error("Constant expression not integer",
			PRS_ERROR_CONST_EXPR_NOT_INTEGER,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete Constant;
	       delete $1;
	       YYABORT;
	      } /* fi */
	  $$ = Constant->Converted(SIR_CONST_TIME);
	  delete $1;
	 };

time:				/* Expression */
	constant_expression
	{ $$ = $1;
	 };

wait_statement:			/* Statement */
	TOK_WAIT paren_event_list ';'
	{ sir_symbol_ptr *Event;

	  Event = $2->First();
	  while(Event)
	     { if (Event->Symbol->Type->Direction == SIR_PORT_OUT)
		  { TmpErrMsg.form("Waiting event '%s' must not be output port",
					Event->Symbol->Name.chars());
		    PRS_error(TmpErrMsg.chars(), PRS_ERROR_WAIT_OUTPUT_PORT,
					PRS_lineno, PRS_CurrentFile.chars());
		    delete $2;
		    YYABORT;
		   } /* fi */
	       Event = Event->Succ();
	      } /* elihw */
	  $$ = new SIR_Statement(SIR_STMNT_WAIT, PRS_CurrScope->ParentSymbol,
				$2, PRS_lineno, PRS_CurrFileInfo);
	 };

waitfor_statement:		/* Statement */
	TOK_WAITFOR time ';'
	{ sir_expression	*Expr;

#ifdef PRS_EVALUATE_WAITFOR
	  sir_constant		*Constant;
	  /* constant must be evaluatable at compile-time */
	  Constant = $2->Eval();
	  if (SIR_Error)
	     { PRS_SemanticError(SIR_Error, $2);
	       delete Constant;
	       delete $2;
	       YYABORT;
	      } /* fi */
	  if (Constant->Type == SIR_CONST_CHARSTRING)
	     { PRS_error("Constant expression not integer",
			PRS_ERROR_CONST_EXPR_NOT_INTEGER,
			PRS_lineno, PRS_CurrentFile.chars());
	       delete Constant;
	       delete $2;
	       YYABORT;
	      } /* fi */
	  Expr = SIR_Expression::New(Constant, PRS_TheDesign->Types);
	  assert(Expr != NULL);	/* must be ok */
	  Expr->SetLineInfo($2);
	  delete $2;
#else /* don't PRS_EVALUATE_WAITFOR */
	  Expr = $2;
#endif /* PRS_EVALUATE_WAITFOR */

	  $$ = new SIR_Statement(SIR_STMNT_WAITFOR, PRS_CurrScope->ParentSymbol,
				Expr, NULL, NULL,
				PRS_lineno, PRS_CurrFileInfo);
	 };

notify_statement:		/* Statement */
	TOK_NOTIFY paren_event_list ';'
	{ sir_symbol_ptr *Event;

	  Event = $2->First();
	  while(Event)
	     { if (Event->Symbol->Type->Direction == SIR_PORT_IN)
		  { TmpErrMsg.form("Notified event '%s' must not be input port",
					Event->Symbol->Name.chars());
		    PRS_error(TmpErrMsg.chars(), PRS_ERROR_NOTIFY_INPUT_PORT_1,
					PRS_lineno, PRS_CurrentFile.chars());
		    delete $2;
		    YYABORT;
		   } /* fi */
	       Event = Event->Succ();
	      } /* elihw */
	  $$ = new SIR_Statement(SIR_STMNT_NOTIFY, PRS_CurrScope->ParentSymbol,
				$2, PRS_lineno, PRS_CurrFileInfo);
	 }
	| TOK_NOTIFYONE paren_event_list ';'
	{ sir_symbol_ptr *Event;

	  Event = $2->First();
	  while(Event)
	     { if (Event->Symbol->Type->Direction == SIR_PORT_IN)
		  { TmpErrMsg.form("Notified event '%s' must not be input port",
					Event->Symbol->Name.chars());
		    PRS_error(TmpErrMsg.chars(), PRS_ERROR_NOTIFY_INPUT_PORT_2,
					PRS_lineno, PRS_CurrentFile.chars());
		    delete $2;
		    YYABORT;
		   } /* fi */
	       Event = Event->Succ();
	      } /* elihw */
	  $$ = new SIR_Statement(SIR_STMNT_NOTIFYONE,
				PRS_CurrScope->ParentSymbol,
				$2, PRS_lineno, PRS_CurrFileInfo);
	 };

%%
/************************************************************************/
/*** implementation of exported functions *******************************/
/************************************************************************/


	/* none */


/* EOF PRS_Parser.y */
