/************************************************************************/
/* FileInfo.cc: SpecC Internal Representation, FileInfo Classes		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/24/97 */
/************************************************************************/

/* last update: 05/31/01 */

/* modifications: (most recent first)
 *
 * 05/30/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/30/01 RD	created this header (last change was 12/16/98)
 */

#include "IntRep/FileInfo.h"
#include "IntRep/Design.h"

#include <stdlib.h>
#include <assert.h>


/*** constants and macros ***********************************************/


#define SIR_MAX_EMPTY_LINES	5

//#define SIR_DEBUG_FILE_GARBAGE_COLLECTOR	/* debug garbage collector */


/*** internal type declarations *****************************************/


	/* (none) */


/*** class implementations **********************************************/


	/********************/
	/*** SIR_LineInfo ***/
	/********************/


BOOL		SIR_LineInfo::Enabled = TRUE;	/* init static variables */
unsigned int	SIR_LineInfo::LastLine = 0;
sir_fileinfo	*SIR_LineInfo::LastFile = NULL;		/* THIS File */
int		SIR_LineInfo::Tabulator = 0;
const char	*SIR_LineInfo::CurrentFile = NULL;	/* unknown Name */
unsigned int	SIR_LineInfo::CurrentLine = 1;


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_LineInfo::SIR_LineInfo(		/* constructor #1 */
	unsigned int	Line,
	sir_fileinfo	*File)
{

assert((Line != 0) && (File != NULL));

SIR_LineInfo::Line = Line;
SIR_LineInfo::File = File;

} /* end of SIR_LineInfo::SIR_LineInfo #1 */


SIR_LineInfo::~SIR_LineInfo(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_LineInfo::~SIR_LineInfo */


void SIR_LineInfo::InitWriteSC(	/* prepares for writing new SpecC line infos */
	bool		Enable,
	const char	*ThisFilename = NULL,	/* default location if */
	unsigned int	ThisLineNumber = 1)	/* line info unknown   */
{

LastLine	= 0;		/* reset */
LastFile	= NULL;
Enabled		= Enable;
Tabulator	= 0;
CurrentFile	= ThisFilename;
CurrentLine	= ThisLineNumber;

} /* end of SIR_LineInfo::InitWriteSC */


ERROR SIR_LineInfo::WriteSC(	/* writes line and file in SpecC source */
	FILE		*OutFile,
	bool		PutSpace = FALSE)
{
unsigned int	i;
int		n;

if (! Enabled)
   { if (PutSpace)
	{ fputc('\n', OutFile);
	  CurrentLine++;
	  for(n=0; n<Tabulator; n++)
	     { fputc('\t', OutFile);
	      } /* rof */
	 } /* fi */
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (LastFile == File)	/* are we still in the last file? */
   { if (  (Line < LastLine)
	 ||(Line > LastLine + SIR_MAX_EMPTY_LINES))
	{ fprintf(OutFile, "\n\n#line %d \"%s\"\n",
			Line, File->Filename.chars());
	  CurrentLine += 3;
	  for(n=0; n<Tabulator; n++)
	     { fputc('\t', OutFile);
	      } /* rof */
	 } /* fi */
     else
	{ if (Line == LastLine)
	     { if (PutSpace)
		  { fputc(' ', OutFile);	/* output separator */
		   } /* fi */
	      } /* fi */
	  else
	     { for(i = Line - LastLine; i>0; i--)
		  { fputc('\n', OutFile);	/* output newlines */
		    CurrentLine++;
		   } /* rof */
	       for(n=0; n<Tabulator; n++)
		  { fputc('\t', OutFile);
		   } /* rof */
	      } /* esle */
	 } /* esle */
    } /* fi */
else	/* it's another file */
   { fprintf(OutFile, "\n\n#line %d \"%s\"\n",
			Line, File->Filename.chars());
     CurrentLine += 3;
     for(n=0; n<Tabulator; n++)
	{ fputc('\t', OutFile);
	 } /* rof */
     LastFile = File;
    } /* esle */

if (ferror(OutFile))
   { SIR_ErrMsg.form("Writing to file failed%s",
			GL_SystemErrorMessage());
     return(SIR_ERROR_WRITING_SPECC_FILE);
    } /* fi */

LastLine = Line;

return(SIR_ERROR_NO_ERROR);	/* writing successful */

} /* end of SIR_LineInfo::WriteSC */


void SIR_LineInfo::WriteNL(		/* writes a newline in SpecC source */
	FILE		*OutFile)	/* (regardless of any line info) */
{
int	n;

fputc('\n', OutFile);
CurrentLine++;
for(n=0; n<Tabulator; n++)
   { fputc('\t', OutFile);
    } /* rof */

if (Enabled)
   { LastLine++;
    } /* fi */

} /* end of SIR_LineInfo::WriteNL */


void SIR_LineInfo::WriteSPACE(		/* writes a space in SpecC source */
	FILE		*OutFile)	/* (regardless of any line info) */
{
int	n;

if (Enabled)
   { fputc(' ', OutFile);
     return;
    } /* fi */

fputc('\n', OutFile);
CurrentLine++;
for(n=0; n<Tabulator; n++)
   { fputc('\t', OutFile);
    } /* rof */

} /* end of SIR_LineInfo::WriteSPACE */


void SIR_LineInfo::WriteNewSC(		/* writes unknown line info */
	FILE		*OutFile,	/* (probably newly created stuff) */
	bool		PutSpace = FALSE)
{
int		n;

if (  (! Enabled)		/* switched off? */
    ||(CurrentFile == NULL)	/* current file name unknown? */
    ||(LastFile == NULL))	/* still in THIS current file? */
   { if (PutSpace)
	{ fputc('\n', OutFile);
	  CurrentLine++;
	  for(n=0; n<Tabulator; n++)
	     { fputc('\t', OutFile);
	      } /* rof */
	 } /* fi */
     return;
    } /* fi */

/* switch to THIS file */

CurrentLine += 3;
fprintf(OutFile, "\n\n#line %d \"%s\"\n",
			CurrentLine, CurrentFile);
for(n=0; n<Tabulator; n++)
   { fputc('\t', OutFile);
    } /* rof */
LastFile = NULL;		/* THIS File */
LastLine = CurrentLine;		/* THIS Line */

} /* end of SIR_LineInfo::WriteNewSC */


void SIR_LineInfo::Add2Tabulator(	/* increments tabulator for indenting */
	int		Tabs)		/* (decrements for Tabs < 0) */
{

Tabulator += Tabs;
assert(Tabulator >= 0);	/* otherwise hierarchy is lost! */

} /* end of SIR_LineInfo::Add2Tabulator */


	/********************/
	/*** SIR_FileInfo ***/
	/********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_FileInfo::SIR_FileInfo(		/* constructor #1 */
	const char	*Filename)
{

assert(Filename != NULL);

SIR_FileInfo::Filename	= Filename;
SIR_FileInfo::Alias	= NULL;

} /* end of SIR_FileInfo::SIR_FileInfo #1 */


SIR_FileInfo::SIR_FileInfo(		/* constructor #3 (duplicator) */
	sir_fileinfo	*Original)
{

SIR_FileInfo::Filename	= Original->Filename;
SIR_FileInfo::Alias	= NULL;

} /* end of SIR_FileInfo::SIR_FileInfo #3 */


SIR_FileInfo::~SIR_FileInfo(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_FileInfo::~SIR_FileInfo */


	/********************/
	/*** SIR_FileList ***/
	/********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_FileList::SIR_FileList(		/* constructor #1 */
	sir_fileinfo	*FirstEntry = NULL) :
		SIR_List<SIR_FileInfo>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_FileList::SIR_FileList #1 */


SIR_FileList::SIR_FileList(		/* constructor #2 (duplicator) */
	sir_file_list	*Original)
{
sir_fileinfo	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_FileInfo(Curr));
     Curr = Curr->Succ();
    } /* elihw */

} /* end of SIR_FileList::SIR_FileList #2 */


SIR_FileList::~SIR_FileList(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_FileList::~SIR_FileList */


ERROR SIR_FileList::Integrate(		/* integrates imported filelist */
	sir_file_list	*Imported)
{
sir_fileinfo	*ImpCurr,
		*ImpSucc;

ImpCurr = Imported->First();
while(ImpCurr)
   { ImpSucc = ImpCurr->Succ();
     if (!(ImpCurr->Alias = Find(ImpCurr->Filename.chars())))
	{ if (Curr())
	     { InsertBefore(Imported->Remove(ImpCurr));
	      } /* fi */
	  else
	     { Append(Imported->Remove(ImpCurr));
	      } /* esle */
	 } /* fi */
     ImpCurr = ImpSucc;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_FileList::Integrate */


void SIR_FileList::GarbageCollection(		/* garbage collector */
	sir_design	*Design)
{
sir_fileinfo	*Curr,
		*Succ;

#ifdef SIR_DEBUG_FILE_GARBAGE_COLLECTOR
fprintf(stderr, "GARBAGE_COLLECTOR: Starting with %d file entries...\n",
			NumElements());
#endif /* SIR_DEBUG_FILE_GARBAGE_COLLECTOR */

#ifndef NDEBUG
Curr = First();		/* make sure every entry is unmarked */
while(Curr)
   { assert(Curr->Color == SIR_WHITE);
     Curr = Curr->Succ();
    } /* elihw */
#endif /* NDEBUG */

Design->DFS_ForAllNodes(&SIR_Node::MarkUsedFiles, NULL);

Curr = First();		/* remove every unused entry (unmarked) */
while(Curr)
   { Succ = Curr->Succ();
     if (Curr->Color != SIR_WHITE)
	{ assert(Curr->Color == SIR_RED);	/* marked = used */
	  Curr->Color = SIR_WHITE;
	 } /* fi */
     else					/* unmarked = not used */
	{
#ifdef SIR_DEBUG_FILE_GARBAGE_COLLECTOR
	  fprintf(stderr, "GARBAGE_COLLECTOR: Detected unused file: \"%s\"\n",
				Curr->Filename.chars());
#endif /* SIR_DEBUG_FILE_GARBAGE_COLLECTOR */
	  Remove(Curr);
	  delete Curr;
	 } /* esle */
     Curr = Succ;
    } /* elihw */

#ifdef SIR_DEBUG_FILE_GARBAGE_COLLECTOR
fprintf(stderr, "GARBAGE_COLLECTOR: Ending with %d file entries...\n",
			NumElements());
#endif /* SIR_DEBUG_FILE_GARBAGE_COLLECTOR */

} /* end of SIR_FileList::GarbageCollection */


sir_fileinfo *SIR_FileList::FindOrInsert(	/* find an entry or insert it */
	const char	*Filename)		/* if it not exists           */
{
sir_fileinfo	*Entry;

if ((Entry = Find(Filename)))
   { return(Entry);
    } /* fi */

if (Curr())
   { return(InsertBefore(new SIR_FileInfo(Filename)));
    } /* fi */
else
   { return(Append(new SIR_FileInfo(Filename)));
    } /* esle */

} /* end of SIR_FileList::FindOrInsert */


sir_fileinfo *SIR_FileList::Find(		/* find an entry */
	const char	*Filename)	/* (returns NULL if not found) */
{
int		CmpVal;

First();	/* search for the entry */
while(Curr())
   { CmpVal = strcmp(Filename, Curr()->Filename.chars());
     if (CmpVal == 0)
	{ return(Curr());	/* found the entry */
	 } /* fi */
     if (CmpVal < 0)
	{ break;	/* not found (insert here) */
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_FileList::Find */


/* EOF FileInfo.cc */
