/************************************************************************/
/* Design.cc: SpecC Internal Representation, Design Class		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/24/97 */
/************************************************************************/

/* last update: 05/31/01 */

/* modifications: (most recent first)
 *
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/26/01 RD	bug fix for source code 'import'
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	eliminated support for automatic IP wrapping (scc -ip option)
 * 05/04/01 RD	added support for import of SpecC source (.sc) files
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/16/01 RD	added inclusion of SIR_LLONG_HEADER_NAME into C++ header
 * 04/06/01 RD	added support for non-native 'long long' type
 * 04/06/01 RD	started this header (last update was 04/08/99)
 */

#include "IntRep/Design.h"
#include "IntRep/Extern.h"

#include <stdlib.h>
#include <string.h>
#include <time.h>


/*** constants and macros ***********************************************/


	/* options to output 'long long' types */

#ifdef HAVE_LLONG
//#define LONGLONG_USE_NATIVE_OUTPUT_ROUTINES
#define LONGLONG_USE_SPECC_OUTPUT_ROUTINES
#else /* HAVE_LLONG */
#define LONGLONG_USE_SPECC_OUTPUT_ROUTINES
#endif /* HAVE_LLONG */


/*** internal type declarations *****************************************/


	/* (none) */


/*** prototypes of internal functions ***********************************/


static BOOL NeedForRangeChecker(	/* checks if range checker is needed */
	sir_design	*Design);

static ERROR DeclareRangeChecker(	/* generate decl. of range checker */
	FILE		*File);

static ERROR DefineRangeChecker(	/* generate def. of range checker */
	FILE		*File);


/*** exported variables *************************************************/


	/* none */


/*** internal variables *************************************************/


	/* none */


/*** class implementations **********************************************/


	/******************/
	/*** SIR_Design ***/
	/******************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Design::SIR_Design(			/* constructor #1 */
	const char	*Name)
{

assert(Name != NULL);
assert(*Name != 0);

SIR_Design::Name		= Name;
SIR_Design::FileList		= new SIR_FileList();
SIR_Design::ImportList		= new SIR_ImportList();
SIR_Design::Types		= new SIR_Types(this);
SIR_Design::Symbols		= new SIR_Symbols(NULL, SIR_SCOPE_GLOBAL);
SIR_Design::Notes		= NULL;

} /* end of SIR_Design::SIR_Design #1 */


SIR_Design::SIR_Design(			/* constructor #3 (duplicator) */
	sir_design	*Original)
{

SIR_Design::Name		= Original->Name;
SIR_Design::FileList		= new SIR_FileList(Original->FileList);
SIR_Design::ImportList		= new SIR_ImportList(Original->ImportList);
SIR_Design::Types		= new SIR_Types(Original->Types, this);
SIR_Design::Symbols		= new SIR_Symbols(Original->Symbols);
SIR_Design::Notes		= Original->Notes ?
					new SIR_Notes(Original->Notes) :
					NULL;

} /* end of SIR_Design::SIR_Design #3 */


SIR_Design::~SIR_Design(void)		/* destructor */
{

delete FileList;
delete ImportList;
delete Types;
delete Symbols;
delete Notes;

} /* end of SIR_Design::~SIR_Design */


ERROR SIR_Design::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

assert(MemberFct != 0);

/* this is not a node */

/* there are no nodes in FileList */
/* there are no nodes in ImportList */
/* there are no nodes in Types */
if (Symbols)
   { if ((SIR_Error = Symbols->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::DFS_ForAllNodes */


ERROR SIR_Design::DFS_ForAllSymbols(	/* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{

assert(MemberFct != 0);

/* this is not a symbol */

/* there are no symbols in FileList */
/* there are no symbols in ImportList */
/* there are no symbols in Types */
if (Symbols)
   { if ((SIR_Error = Symbols->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no symbols in Notes */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::DFS_ForAllSymbols */


ERROR SIR_Design::DFS_ForAllUserTypes(	/* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{

assert(MemberFct != 0);

/* this is not a usertype */

/* there are no usertypes in FileList */
/* there are no usertypes in ImportList */
/* there are no usertypes in Types (only links) */
if (Symbols)
   { if ((SIR_Error = Symbols->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no usertypes in Notes */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::DFS_ForAllUserTypes */


ERROR SIR_Design::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

assert(MemberFct != 0);

/* this is not a note */

/* there are no notes in FileList */
/* there are no notes in ImportList */
/* there are no notes in Types */
if (Symbols)
   { if ((SIR_Error = Symbols->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::DFS_ForAllNotes */


ERROR SIR_Design::DFS_ForAllStatements(	/* iterator over all statements */
	sir_stmnt_mptr	MemberFct,	/* (depth first) */
	sir_stmnt_marg	MemberFctArg)
{
sir_symbol	*Symbol1,
		*Symbol2;

Symbol1 = Symbols->First();	/* find all global functions */
while(Symbol1)
   { if (Symbol1->FctBody)
	{ if ((SIR_Error = Symbol1->FctBody->DFS_ForAllStatements(
						MemberFct, MemberFctArg)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* fi */
     if (  (Symbol1->IsBehaviorDefinition())
	 ||(Symbol1->IsChannelDefinition()))
	{ assert(Symbol1->ClassScope != NULL);
	  Symbol2 = Symbol1->ClassScope->First();	/* find all methods */
	  while(Symbol2)
	     { if (Symbol2->FctBody)
		  { if ((SIR_Error = Symbol2->FctBody->DFS_ForAllStatements(
						MemberFct, MemberFctArg)))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       Symbol2 = Symbol2->Succ();
	      } /* elihw */
	 } /* fi */
     Symbol1 = Symbol1->Succ();
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::DFS_ForAllStatements */


ERROR SIR_Design::Annotate(	/* attach a (global) note to the design */
	sir_note	*NewNote)	/* (consumes NewNote) */
{

assert(NewNote != NULL);

NewNote->Symbol = NULL;		/* global Note (no back pointers) */
NewNote->UserType = NULL;
NewNote->Label = NULL;

if (Notes)
   { if (! Notes->Insert(NewNote))
	{ delete NewNote;
	  return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { Notes = new SIR_Notes(NewNote);
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::Annotate */


ERROR SIR_Design::Integrate(		/* integrates an imported design */
	sir_design	*Imported,	/* (consumes the imported design) */
	const char	*ImportFile = NULL,	/* (default: no warnings) */
	const char	*SourceFile = NULL,
	unsigned int	LineNumber = 0)
{
sir_import	*NewImport,
		*DoubleImport;
sir_symbol	*Symbol,
		*SymbolSucc;
sir_usertype	*UserType,
		*UserTypeSucc;

assert(Imported != NULL);

	/* step 0: check if we already know that design */

if (ImportList->Find(Imported->Name.chars()))
   { delete Imported;
     return(SIR_ERROR_NO_ERROR);	/* already imported before */
    } /* fi */

	/* step 1: create import entry and pointers to it */

NewImport = ImportList->Insert(Imported->Name.chars());
SIR_Error = Imported->DFS_ForAllNotes(
			&SIR_Note::SetImported, NewImport);
assert(SIR_Error == SIR_ERROR_NO_ERROR);
SIR_Error = Imported->DFS_ForAllSymbols(
			&SIR_Symbol::SetImported, NewImport);
assert(SIR_Error == SIR_ERROR_NO_ERROR);
SIR_Error = Imported->DFS_ForAllUserTypes(
			&SIR_UserType::SetImported, NewImport);
assert(SIR_Error == SIR_ERROR_NO_ERROR);

	/* step 2: integrate file list */

if ((SIR_Error = FileList->Integrate(		/* merge file lists */
			Imported->FileList)))	/* (aliasing doubles) */
   { delete Imported;
     return(SIR_Error);
    } /* fi */
if ((SIR_Error = Imported->DFS_ForAllNodes(
			&SIR_Node::UnaliasFilePointer, NULL)))
   { delete Imported;
     return(SIR_Error);
    } /* fi */

	/* step 3: integrate import list and delete double imports */

if ((SIR_Error = ImportList->Integrate(		/* merge import lists */
			Imported->ImportList)))	/* (marking doubles    */
   { delete Imported;				/* with alias pointer) */
     return(SIR_Error);
    } /* fi */

if (ImportFile)	/* issue helpful warnings */
   { assert(SourceFile != NULL);
     DoubleImport = Imported->ImportList->First();
     while(DoubleImport)
	{ GL_PrintWarningFmt(GL_WARN_PEDANTIC,
		"In file \"%s\" in line %u," GL_WARN_MSG_NEWLINE
		"when importing file \"%s\"," GL_WARN_MSG_NEWLINE
		"nested design '%s' is already imported",
		SourceFile, LineNumber,
		ImportFile,
		DoubleImport->ImportName.chars());
	  DoubleImport = DoubleImport->Succ();
	 } /* elihw */
    } /* fi */

if((SIR_Error = Imported->DFS_ForAllNotes(	/* remove double notes */
			&SIR_Note::RemoveDoubleImport, NULL)))
   { delete Imported;
     return(SIR_Error);
    } /* fi */

	/* mark double symbols and usertypes with alias */

Symbol = Imported->Symbols->First();	/* move notes still left at global */
while(Symbol)				/* imported symbols to our symbols */
   { assert(Symbol->Imported != NULL);	/* we just set this */
     if (Symbol->Imported->Alias)	/* is double-imported? */
	{ if (!(Symbol->Alias =		/* find its alias */
		Symbols->FindLocally(Symbol->Name.chars())))
	     { delete Imported;
	       return(SIR_ERROR_DOUBLE_IMP_SYMBOL_NOT_FOUND);
	      } /* fi */
	  Symbol->SetAlias(Symbol->Alias);	/* set alias recursively */
	  if ((SIR_Error = SIR_Notes::Merge(&Symbol->Alias->Notes,
						&Symbol->Notes)))
	     { delete Imported;
	       return(SIR_Error);
	      } /* fi */
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */
UserType = Imported->Symbols->UserTypes->First();
while(UserType)				/* the same for global usertypes... */
   { assert(UserType->Imported != NULL);
     if (UserType->Imported->Alias)
	{ if (UserType->Name)
	     { if (!(UserType->Alias =	/* find named user-type */
		Symbols->UserTypes->FindLocally(UserType->Name->chars())))
		  { delete Imported;
		    return(SIR_ERROR_DOUBLE_IMP_USRTP_NOT_FOUND);
		   } /* fi */
	      } /* fi */
	  else
	     { if (! UserType->TypeDef)	/* unnamed usertype without typedef */
		  { UserType = UserType->Succ();	/* skip it! */
		    continue;
		   } /* fi */
	       if (!(UserType->Alias =	/* find unnamed usertype via typedef */
		Symbols->UserTypes->FindLocallyViaTypeDef(
					UserType->TypeDef->Name.chars())))
		  { delete Imported;
		    return(SIR_ERROR_DOUBLE_IMP_USRTP_NOT_FOUND_2);
		   } /* fi */
	      } /* esle */
	  UserType->SetAlias(UserType->Alias);	/* set alias recursively */
	  if ((SIR_Error = SIR_Notes::Merge(&UserType->Alias->Notes,
						&UserType->Notes)))
	     { delete Imported;
	       return(SIR_Error);
	      } /* fi */
	 } /* fi */
     UserType = UserType->Succ();
    } /* elihw */

	/* unalias everything else */

Imported->Types->UnAlias();		/* unalias imported types */
Symbol = Imported->Symbols->First();
while(Symbol)				/* unalias imported, non-dbl. symbols */
   { assert(Symbol->Imported != NULL);
     if (! Symbol->Imported->Alias)
	{ Symbol->UnAlias(Symbols);
	 } /* fi */
     Symbol = Symbol->Succ();
    } /* elihw */
UserType = Imported->Symbols->UserTypes->First();
while(UserType)				/* unalias imported, non-dbl. usrtps. */
   { assert(UserType->Imported != NULL);
     if (! UserType->Imported->Alias)
	{ UserType->UnAlias(Symbols);
	 } /* fi */
     UserType = UserType->Succ();
    } /* elihw */

	/* delete double-imported symbols, double-imported usertypes	*/
	/* (and their types) and double-imported local types		*/

Symbol = Imported->Symbols->First();
while(Symbol)
   { SymbolSucc = Symbol->Succ();
     if (Symbol->Imported->Alias)
	{ if ((SIR_Error = Symbol->DFS_ForAllUserTypes(
				&SIR_UserType::DeleteYourType, NULL)))
	     { delete Imported;
	       return(SIR_Error);
	      } /* fi */
	  Symbol->Remove();
	  delete Symbol;
	 } /* fi */
     Symbol = SymbolSucc;
    } /* elihw */
UserType = Imported->Symbols->UserTypes->First();
while(UserType)
   { UserTypeSucc = UserType->Succ();
     if (UserType->Imported->Alias)
	{ if ((SIR_Error = UserType->DFS_ForAllUserTypes(
				&SIR_UserType::DeleteYourType, NULL)))
	     { delete Imported;
	       return(SIR_Error);
	      } /* fi */
	  UserType->Remove();
	  delete UserType;
	 } /* fi */
     UserType = UserTypeSucc;
    } /* elihw */

	/* step 4: integrate user-defined types */

if ((SIR_Error = Symbols->UserTypes->Integrate(	/* merge top-level UserTypes */
			Imported->Symbols->UserTypes)))
   { delete Imported;
     return(SIR_Error);
    } /* fi */
/* we postpone pointer-unaliasing after type integration */

	/* step 5: integrate types */

if ((SIR_Error = Types->Integrate(		/* merge type tables */
			Imported->Types)))
   { delete Imported;
     return(SIR_Error);
    } /* fi */
Symbols->UserTypes->UnAlias(Symbols);	/* merged usertypes need unaliasing */
Types->UnAlias();			/* merged types need unaliasing */
Imported->Symbols->UnAlias(Symbols);	/* to-be-integrated symbols too */

	/* step 6: integrate symbols */

if ((SIR_Error = Symbols->Integrate(		/* merge top-level symbols */
			Imported->Symbols)))
   { delete Imported;
     return(SIR_Error);
    } /* fi */
Types->UnAlias();		/* unalias the merged design */
Symbols->UnAlias(Symbols);
if ((SIR_Error = Symbols->CheckClassContainment()))
   { delete Imported;
     return(SIR_Error);
    } /* fi */

	/* step 7: integrate global notes */

if ((SIR_Error = SIR_Notes::Merge(		/* merge global notes */
			&Notes, &Imported->Notes)))
   { delete Imported;
     return(SIR_Error);
    } /* fi */

	/* step 8: delete the skeleton of the imported design */

/* note: everything, that is left in the imported design, is the	*/
/*	 raw skeleton (list headers, etc.) and some meat (class		*/
/*	 bodies) which was already there in the original design;	*/
/*	 the left-over meat is marked with alias-pointers		*/
/*	 (pointing into the original design); in the merged design,	*/
/*	 all alias pointers (at symbols, usertypes, types, files,	*/
/*	 and imports) are NULL;						*/

delete Imported;

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Design::Integrate */


void SIR_Design::GarbageCollection(	/* garbage collector */
	void)
{

FileList->GarbageCollection(this);	/* clean up list of source files */
ImportList->GarbageCollection(this);	/* clean up list of imported files */
Types->GarbageCollection();		/* clean up type table */

} /* end of SIR_Design::GarbageCollection */


ERROR SIR_Design::WriteSC(	/* (re-) generates SpecC source code */
	FILE		*File,
	const char	*Filename,	/* (full path) */
	BOOL		WriteNotes = TRUE,
	BOOL		WriteLines = TRUE)
{
time_t		SysTime;
unsigned int	CurrLineNo;

assert(File != NULL);
assert(Filename != NULL);

time(&SysTime);		/* read current time and date */
CurrLineNo = 1;

fprintf(File,
	"///////////////////////////////////////////////////////////"
								"///////////\n"
	"// SpecC source code generated by " GL_SPECC_NAME " V"
						GL_SPECC_VERSION_STRING "\n"
	"// Design: %s\n"
	"// File:   %s\n"
	"// Time:   %s"
	"///////////////////////////////////////////////////////////"
								"///////////\n",
	Name.chars(),
	GL_GetFileFromPath(Filename),
	ctime(&SysTime));
CurrLineNo += 6;

if (ferror(File))
   { SIR_ErrMsg.form("Writing to file failed%s",
			GL_SystemErrorMessage());
     return(SIR_ERROR_WRITING_SPECC_FILE);
    } /* fi */

if (! WriteNotes)
   { fputs("\n// Note: Notes are suppressed in this file.\n", File);
     CurrLineNo += 2;
    } /* fi */
if (! WriteLines)
   { fputs("\n// Note: Line infos are suppressed in this file.\n", File);
     CurrLineNo += 2;
    } /* fi */

SIR_LineInfo::InitWriteSC(WriteLines,	/* init for new file */
			Filename, CurrLineNo);

if (Notes)			/* write all global notes */
   { if ((SIR_Error = Notes->WriteSC(File, WriteNotes)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

if (Symbols)			/* write all global decls. and defs. */
   { if ((SIR_Error = Symbols->WriteSC(File, WriteNotes)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

fprintf(File,
	"\n\n"
	"///////////////////////////////////////////////////////////"
								"///////////\n"
	"// End of file %s\n"
	"///////////////////////////////////////////////////////////"
								"///////////\n",
	GL_GetFileFromPath(Filename));

if (ferror(File))
   { SIR_ErrMsg.form("Writing to file failed%s",
			GL_SystemErrorMessage());
     return(SIR_ERROR_WRITING_SPECC_FILE);
    } /* fi */

return(SIR_ERROR_NO_ERROR);	/* successful */

} /* end of SIR_Design::WriteSC */


ERROR SIR_Design::WriteH(	/* generates C++ source code: header file */
	FILE		*File,
	const char	*Filename,	/* (full path) */
	BOOL		WriteNotes = TRUE,
	BOOL		WriteLines = TRUE)
{
time_t		SysTime;
unsigned int	CurrLineNo;

assert(File != NULL);
assert(Filename != NULL);

time(&SysTime);		/* read current time and date */
CurrLineNo = 1;

fprintf(File,
	"///////////////////////////////////////////////////////////"
								"///////////\n"
	"// C++ header file generated by " GL_SPECC_NAME " V"
						GL_SPECC_VERSION_STRING "\n"
	"// Design: %s\n"
	"// File:   %s\n"
	"// Time:   %s"
	"///////////////////////////////////////////////////////////"
								"///////////\n",
	Name.chars(),
	GL_GetFileFromPath(Filename),
	ctime(&SysTime));
CurrLineNo += 6;

if (ferror(File))
   { SIR_ErrMsg.form("Writing to file failed%s",
			GL_SystemErrorMessage());
     return(SIR_ERROR_WRITING_SPECC_FILE);
    } /* fi */

if (! WriteNotes)
   { fputs("\n// Note: Notes are suppressed in this file.\n", File);
     CurrLineNo += 2;
    } /* fi */
if (! WriteLines)
   { fputs("\n// Note: Line infos are suppressed in this file.\n", File);
     CurrLineNo += 2;
    } /* fi */

fputs("\n#include <" SIR_SIM_HEADER_NAME ">", File);
CurrLineNo += 1;
fputs("\n#include <" SIR_LLONG_HEADER_NAME ">\n", File);
CurrLineNo += 2;
fputs("\n#include <" SIR_BIT_HEADER_NAME ">\n", File);
CurrLineNo += 2;

SIR_LineInfo::InitWriteSC(WriteLines,	/* init for new file */
			Filename, CurrLineNo);

if (Notes)			/* write all global notes */
   { if ((SIR_Error = Notes->WriteSC(File, WriteNotes, TRUE)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

if (Symbols)			/* translate all global declarations */
   { if ((SIR_Error = Symbols->WriteH(File, WriteNotes)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

if (NeedForRangeChecker(this))
   { if (Symbols->FindLocally(SIR_CXX_RANGE_NAME))
	{ GL_PrintWarning(GL_WARN_IMPORTANT,
		"Skipping generation of function '" SIR_CXX_RANGE_NAME "';"
							GL_WARN_MSG_NEWLINE
		"global symbol '" SIR_CXX_RANGE_NAME "' already exists");
	 } /* fi */
     else
	{ if ((SIR_Error = DeclareRangeChecker(File)))
	     { return(SIR_Error);
	      } /* fi */
	 } /* esle */
    } /* fi */

fprintf(File,
	"\n\n"
	"///////////////////////////////////////////////////////////"
								"///////////\n"
	"// End of file %s\n"
	"///////////////////////////////////////////////////////////"
								"///////////\n",
	GL_GetFileFromPath(Filename));

if (ferror(File))
   { SIR_ErrMsg.form("Writing to file failed%s",
			GL_SystemErrorMessage());
     return(SIR_ERROR_WRITING_SPECC_FILE);
    } /* fi */

return(SIR_ERROR_NO_ERROR);	/* successful */

} /* end of SIR_Design::WriteH */


ERROR SIR_Design::WriteCC(	/* generates C++ source code: main file */
	FILE		*File,
	const char	*Filename,	/* (full path) */
	const char	*Headername,	/* (full path) */
	BOOL		WriteNotes = TRUE,
	BOOL		WriteLines = TRUE)
{
time_t		SysTime;
unsigned int	CurrLineNo;

assert(File != NULL);
assert(Filename != NULL);
assert(Headername != NULL);

time(&SysTime);		/* read current time and date */
CurrLineNo = 1;

fprintf(File,
	"///////////////////////////////////////////////////////////"
								"///////////\n"
	"// C++ source file generated by " GL_SPECC_NAME " V"
						GL_SPECC_VERSION_STRING "\n"
	"// Design: %s\n"
	"// File:   %s\n"
	"// Time:   %s"
	"///////////////////////////////////////////////////////////"
								"///////////\n",
	Name.chars(),
	GL_GetFileFromPath(Filename),
	ctime(&SysTime));
CurrLineNo += 6;

if (ferror(File))
   { SIR_ErrMsg.form("Writing to file failed%s",
			GL_SystemErrorMessage());
     return(SIR_ERROR_WRITING_SPECC_FILE);
    } /* fi */

if (! WriteNotes)
   { fputs("\n// Note: Notes are suppressed in this file.\n", File);
     CurrLineNo += 2;
    } /* fi */
if (! WriteLines)
   { CurrLineNo += 2;
     fputs("\n// Note: Line infos are suppressed in this file.\n", File);
    } /* fi */

fprintf(File, "\n#include \"%s\"\n", GL_GetFileFromPath(Headername));
CurrLineNo += 2;

SIR_LineInfo::InitWriteSC(WriteLines,	/* init for new file */
			Filename, CurrLineNo);

if (Symbols)			/* translate all global definitions */
   { if ((SIR_Error = Symbols->WriteCC(File, WriteNotes)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

if (  (NeedForRangeChecker(this))
    &&(!(Symbols->FindLocally(SIR_CXX_RANGE_NAME))))
   { if ((SIR_Error = DefineRangeChecker(File)))
	{ return(SIR_Error);
	 } /* fi */
    } /* esle */

fprintf(File,
	"\n\n"
	"///////////////////////////////////////////////////////////"
								"///////////\n"
	"// End of file %s\n"
	"///////////////////////////////////////////////////////////"
								"///////////\n",
	GL_GetFileFromPath(Filename));

if (ferror(File))
   { SIR_ErrMsg.form("Writing to file failed%s",
			GL_SystemErrorMessage());
     return(SIR_ERROR_WRITING_SPECC_FILE);
    } /* fi */

return(SIR_ERROR_NO_ERROR);	/* successful */

} /* end of SIR_Design::WriteCC */


/*** exported functions *************************************************/


	/* none */


/*** internal functions *************************************************/


BOOL NeedForRangeChecker(	/* checks whether range checker is needed */
	sir_design	*Design)
{

assert(Design != NULL);
assert(SIR_Error == SIR_ERROR_NO_ERROR);	/* needed for recursion */

if (SIR_ERROR_FLAG_YES == Design->DFS_ForAllStatements(
					&SIR_Statement::NeedsRangeChecker,
					(sir_stmnt_marg) NULL))
   { SIR_Error = SIR_ERROR_NO_ERROR;	/* clear the fake error */
     return(TRUE);
    } /* fi */

assert(SIR_Error == SIR_ERROR_FLAG_NO);
assert(SIR_Error == SIR_ERROR_NO_ERROR);

return(FALSE);	/* don't need it */

} /* end of NeedForRangeChecker */


ERROR DeclareRangeChecker(	/* generate declaration of range checker */
	FILE		*File)
{

SIR_LineInfo::WriteNewSC(File, true);

fputs("void " SIR_CXX_RANGE_NAME "("
	SIR_CXX_TIME_TYPE ", "
	SIR_CXX_TIME_TYPE ", "
	SIR_CXX_TIME_TYPE ", "
	"bool, "
	SIR_CXX_TIME_TYPE ", "
	"bool, "
	"const char *, "
	"const char *, "
	"const char *, "
	"unsigned int);",
	File);

return(SIR_ERROR_NO_ERROR);

} /* end of DeclareRangeChecker */


#ifdef LONGLONG_USE_NATIVE_OUTPUT_ROUTINES

#ifdef SOLARIS
#define FMT_LLD	"%lld"
#endif /* SOLARIS */
#ifdef SUNOS4
#define FMT_LLD	"%ld"	/* probably looses precision! */
#endif /* SUNOS4 */
#ifdef NETBSD
#define FMT_LLD	"%qd"
#endif /* NETBSD */
#ifdef LINUX
#define FMT_LLD	"%qd"
#endif /* LINUX */
#ifdef GNUWIN32
#define FMT_LLD	"%qd"
#endif /* GNUWIN32 */
#ifndef SOLARIS
#   ifndef SUNOS4
#      ifndef NETBSD
#         ifndef LINUX
#            ifndef GNUWIN32
#error "Unknown system! Don't know how to print 'long long int' values!"
#            endif /* GNUWIN32 */
#         endif /* LINUX */
#      endif /* NETBSD */
#   endif /* SUNOS4 */
#endif /* SOLARIS */

#endif /* LONGLONG_USE_NATIVE_OUTPUT_ROUTINES */


#ifdef LONGLONG_USE_SPECC_OUTPUT_ROUTINES

#define FMT_LLD	"%s"

#endif /* LONGLONG_USE_SPECC_OUTPUT_ROUTINES */


ERROR DefineRangeChecker(	/* generate definition of range checker */
	FILE		*File)
{

SIR_LineInfo::WriteNewSC(File, true);

fputs(	"void " SIR_CXX_RANGE_NAME "(", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\t" SIR_CXX_TIME_TYPE " Time1, ", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\t" SIR_CXX_TIME_TYPE " Time2, ", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\t" SIR_CXX_TIME_TYPE " Min, ", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\tbool CheckMin, ", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\t" SIR_CXX_TIME_TYPE " Max, ", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\tbool CheckMax, ", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\tconst char *Label1, ", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\tconst char *Label2, ", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\tconst char *File, ", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\tunsigned int Line)", File);
SIR_LineInfo::Add2Tabulator(1);
SIR_LineInfo::WriteNL(File);
fputc(	'{', File);
SIR_LineInfo::WriteNL(File);

#ifdef LONGLONG_USE_SPECC_OUTPUT_ROUTINES
fputs(	"char BuffMin[40],BuffMax[40],BuffT2[40],BuffT1[40],BuffT[40];", File);
SIR_LineInfo::WriteNL(File);
#endif /* LONGLONG_USE_SPECC_OUTPUT_ROUTINES */

fputs(	"if (  ((CheckMin) && (Time2 - Time1 < Min))", File);
SIR_LineInfo::WriteNL(File);
fputs(	"    ||((CheckMax) && (Time2 - Time1 > Max)))", File);
SIR_LineInfo::WriteNL(File);
fputs(	"   { " SIR_SIM_ABORT "(", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\"ERROR:\\tRange check failed in line %d\\n\"", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\"\\tin file \\\"%s\\\":\\n\"", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\"\\trange(%s; %s; " FMT_LLD "; " FMT_LLD ") is "
		FMT_LLD " - " FMT_LLD " = " FMT_LLD ".\\n\"", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\t\"Simulation aborted.\\n\",", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\tLine, File,", File);
SIR_LineInfo::WriteNL(File);

#ifdef LONGLONG_USE_SPECC_OUTPUT_ROUTINES

				/* 'time' is 'unsigned long long' */
fputs(	"\tLabel1, Label2, ull2str(10,&BuffMin[39],Min),"
			" ull2str(10,&BuffMax[39],Max),", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\tull2str(10,&BuffT2[39],Time2), ull2str(10,&BuffT1[39],Time1),"
			" ull2str(10,&BuffT[39],Time2 - Time1)); }", File);
SIR_LineInfo::WriteNL(File);

#else /* not LONGLONG_USE_SPECC_OUTPUT_ROUTINES */

fputs(	"\tLabel1, Label2, Min, Max,", File);
SIR_LineInfo::WriteNL(File);
fputs(	"\tTime2, Time1, Time2 - Time1); }", File);
SIR_LineInfo::WriteNL(File);

#endif /* LONGLONG_USE_SPECC_OUTPUT_ROUTINES */

fputc(	'}', File);
SIR_LineInfo::Add2Tabulator(-1);

return(SIR_ERROR_NO_ERROR);

} /* end of DefineRangeChecker */


#undef FMT_LLD	/* needed only locally */


/* EOF Design.cc */
