/************************************************************************/
/* Constraint.cc: SpecC Internal Representation, Constraint Class	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/20/97 */
/************************************************************************/

/* last update: 05/30/01 */

/* modifications: (most recent first)
 *
 * 05/30/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/19/01 RD	eliminated T_Value as member of SIR_Constant
 * 04/06/01 RD	added support for non-native 'long long' type;
 *		changed use of 'long long int' to TIME type
 * 01/30/01 RD	fixed a potential FMR problem in iterator
 *		SIR_Constraints::DFS_ForAllNodes
 */

#include "IntRep/Constraint.h"
#include "IntRep/Design.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* (none) */


/*** internal type declarations *****************************************/


	/* (none) */


/*** class implementations **********************************************/


	/**********************/
	/*** SIR_Constraint ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Constraint::SIR_Constraint(		/* constructor #1 */
	sir_label	*Label1,
	sir_label	*Label2,
	sir_constant	*MinTime,
	sir_constant	*MaxTime,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(Label1 != NULL);
assert(Label2 != NULL);

SIR_Constraint::Label1		= Label1;
SIR_Constraint::Label2		= Label2;
SIR_Constraint::MinTime		= MinTime;
SIR_Constraint::MaxTime		= MaxTime;

} /* end of SIR_Constraint::SIR_Constraint #1 */


SIR_Constraint::SIR_Constraint(		/* constructor #3 (duplicator) */
	sir_constraint	*Original) :
		SIR_Node(Original)
{

SIR_Constraint::Label1	= Original->Label1;
SIR_Constraint::Label2	= Original->Label2;
SIR_Constraint::MinTime	= Original->MinTime ?
				new SIR_Constant(Original->MinTime) :
				NULL;
SIR_Constraint::MaxTime	= Original->MaxTime ?
				new SIR_Constant(Original->MaxTime) :
				NULL;

} /* end of SIR_Constraint::SIR_Constraint #3 */


SIR_Constraint::~SIR_Constraint(void)	/* destructor */
{

delete MinTime;
delete MaxTime;

} /* end of SIR_Constraint::~SIR_Constraint */


ERROR SIR_Constraint::DFS_ForAllNodes(	/* iterator for over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (MinTime)
   { if ((SIR_Error = MinTime->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (MaxTime)
   { if ((SIR_Error = MaxTime->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Constraint::DFS_ForAllNodes */


	/***********************/
	/*** SIR_Constraints ***/
	/***********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Constraints::SIR_Constraints(	/* constructor #1 */
	sir_constraint	*FirstEntry = NULL) :
		SIR_List<SIR_Constraint>(FirstEntry)
{

/* nothing to do */

} /* end of SIR_Constraints::SIR_Constraints #1 */


SIR_Constraints::SIR_Constraints(	/* constructor #2 */
	sir_constraints	*Original)
{
sir_constraint	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_Constraint(Curr));
     Curr = Curr->Succ();
    } /* elihw */

} /* end of SIR_Constraints::SIR_Constraints #2 */


SIR_Constraints::~SIR_Constraints(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_Constraints::~SIR_Constraints */


ERROR SIR_Constraints::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_constraint	*Constraint,
		*Succ;

/* this is not a node, but there are nodes below */

Constraint = First();
while(Constraint)
   { Succ = Constraint->Succ();
     if ((SIR_Error = Constraint->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Constraint = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Constraints::DFS_ForAllNodes */


sir_constraint *SIR_Constraints::Insert(	/* insert a new constraint */
	sir_constraint	*NewOne)	/* (must not exist) */
{
sir_constraint	*Before;

assert(NewOne != NULL);

Before = Find(NewOne->Label1);
while(Before)
   { assert(Before->Label2 != NewOne->Label2);
     Before = Before->Succ();
     if (  (Before)
	 &&(Before->Label1 != NewOne->Label1))
	{ break;
	 } /* fi */
    } /* elihw */

if (Before)
   { return(InsertBefore(NewOne, Before));
    } /* fi */
else
   { return(Append(NewOne));
    } /* esle */

} /* end of SIR_Constraints::Insert */


sir_constraint *SIR_Constraints::Find(	/* find the first constraint ... */
	sir_label	*Label1)	/* (returns NULL if not found) */
{
sir_constraint	*Curr;

Curr = First();
while(Curr)
   { if (Curr->Label1 == Label1)
	{ break;
	 } /* fi */
     Curr = Curr->Succ();
    } /* elihw */

return(Curr);

} /* end of SIR_Constraints::Find */


sir_constraint *SIR_Constraints::Find(	/* find a constraint */
	sir_label	*Label1,	/* (returns NULL if not found) */
	sir_label	*Label2)
{
sir_constraint	*Curr;

Curr = Find(Label1);
while(Curr)
   { if (Label2 == Curr->Label2)
	{ return(Curr);
	 } /* fi */
     if (Label1 != Curr->Label1)
	{ return(NULL);
	 } /* fi */
     Curr = Curr->Succ();
    } /* elihw */

return(Curr);

} /* end of SIR_Constraints::Find */


/*** exported functions *************************************************/


	/* none */


/* EOF Constraint.cc */
