/************************************************************************/
/* CcDriver.h: driver for external C compiler				*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 08/31/97 */
/************************************************************************/

/* last update: 05/25/01 */

/* modifications: (most recent first)
 *
 * 05/25/01 RD	eliminated support for automatic IP wrapping (scc -ip option)
 * 04/23/01 RD	added -D_NO_LONGLONG for Solaris, non-native longlong support
 * 04/16/01 RD	added liblonglong to set of required libraries
 * 04/05/01 RD	added support for settable THREAD_BASE (qt, pthread)
 * 01/17/01 RD	added "-D__extension__=" to GNUWIN32 gcc call
 */


#ifndef CCDRIVER_H
#define CCDRIVER_H


#include "Global.h"


/*** constants and macros ***********************************************/


	/* flag for DEBUG mode */
/* #define CCD_DEBUG */


	/* need a thread basis library */
#ifndef THREAD_BASE
#define THREAD_BASE "pthread"	/* default */
#endif /* THREAD_BASE */


	/* error codes for this module */
#define CCD_ERROR_NO_ERROR			(NO_ERROR)
#define CCD_ERROR_OUT_OF_MEMORY			(CCD_ERROR_BASE +  1)
#define CCD_ERROR_SUBSHELL_NOT_AVAILABLE	(CCD_ERROR_BASE +  2)
#define CCD_ERROR_CANNOT_EXECUTE_SUBSHELL	(CCD_ERROR_BASE +  3)
#define CCD_ERROR_COMMAND_RETURNED_ERROR	(CCD_ERROR_BASE +  4)
#define CCD_ERROR_ILLEGAL_MODE			(CCD_ERROR_BASE +  5)
#define CCD_ERROR_ILLEGAL_TARGET		(CCD_ERROR_BASE +  6)
#define CCD_ERROR_EXACTLY_ONE_I_REQUIRED	(CCD_ERROR_BASE +  7)
#define CCD_ERROR_EXACTLY_ONE_O_REQUIRED	(CCD_ERROR_BASE +  8)
#define CCD_ERROR_MIN_ONE_FLAG_REQUIRED		(CCD_ERROR_BASE +  9)
#define CCD_ERROR_UNKNOWN_ESCAPE_SEQUENCE	(CCD_ERROR_BASE + 10)
#define CCD_ERROR_PERCENT_CHAR_NOT_ALLOWED	(CCD_ERROR_BASE + 11)


	/* default compiler calls and parameters */

#define CCD_SPECC_HOME		"$SPECC"	/* substituted by sub-shell */
#define CCD_SPECC_STD_INCLUDE	CCD_SPECC_HOME "/inc"
#define CCD_SPECC_STD_LIBRARY	CCD_SPECC_HOME "/lib"

#define CCD_STD_DEFINES		"-D__SPECC__" CCD_OS_DEFINES
#define CCD_STD_UNDEFINES	CCD_OS_UNDEFINES

#define CCD_STD_INCLUDE_PATH	"-I" CCD_SPECC_STD_INCLUDE
#define CCD_STD_LIBRARY_PATH	"-L" CCD_SPECC_STD_LIBRARY CCD_DYN_LIB_PATH
#define CCD_STD_LIBRARIES	"-lbit -lsim -llonglong -l" THREAD_BASE

#define CCD_DEFAULT_PP_CALL	"g++ -E -x c %p %i -o %o"
#define CCD_DEFAULT_CC_CALL	"g++ -c %c %i -o %o"
#define CCD_DEFAULT_LD_CALL	"g++ %i -o %o %l"

#define CCD_DEFAULT_PP_OPTS	""
#define CCD_DEFAULT_CC_OPTS	""
#define CCD_DEFAULT_LD_OPTS	""

#define CCD_OPTIMIZE_OPT	"-O2"
#define CCD_DEBUG_OPT		"-g"
#define CCD_NO_WARN_OPT		"-w"
#define CCD_IMPORTANT_WARN_OPT	"-Wall -Wno-unused"
#define CCD_STANDARD_WARN_OPT	"-Wall"
#define CCD_PEDANTIC_WARN_OPT	"-Wall"
#define CCD_DEFINITION_OPT	"-D"
#define CCD_UNDEF_OPT		"-U"
#define CCD_INCLUDE_PATH_OPT	"-I"
#define CCD_LIBRARY_PATH_OPT	"-L"
#define CCD_LIBRARY_LINK_OPT	"-l"


	/* special cases */

#ifdef DYNLIB
#define CCD_DYN_LIB_PATH	" -Xlinker -R -Xlinker " CCD_SPECC_STD_LIBRARY
#else /* DYNLIB */
#define CCD_DYN_LIB_PATH	/* empty */
#endif /* DYNLIB */

#ifdef SOLARIS
#ifdef HAVE_LLONG
#define CCD_OS_DEFINES		" -D__attribute__\\(x\\)="
#else /* !HAVE_LLONG */
#define CCD_OS_DEFINES		" -D__attribute__\\(x\\)= -D_NO_LONGLONG"
#endif /* HAVE_LLONG */
#define CCD_USAGE_DEF		", etc."
#define CCD_USAGE_UNDEF		"none"
#endif /* SOLARIS */
#ifdef SUNOS4
#define CCD_OS_DEFINES		" -D__USE_FIXED_PROTOTYPES__"	\
				" -D__attribute__\\(x\\)="
#define CCD_USAGE_DEF		", etc."
#define CCD_USAGE_UNDEF		"none"
#endif /* SUNOS4 */
#ifdef NETBSD
#define CCD_OS_DEFINES		" -D__attribute__\\(x\\)= -D__inline="
#define CCD_USAGE_DEF		", etc."
#define CCD_USAGE_UNDEF		"none"
#endif /* NETBSD */
#ifdef LINUX
#define CCD_OS_DEFINES		" -D__attribute__\\(x\\)="
#define CCD_OS_UNDEFINES	"-U__GNUC__"
#define CCD_USAGE_DEF		", etc."
#define CCD_USAGE_UNDEF		"__GNUC__"
#endif /* LINUX */
#ifdef GNUWIN32
#define CCD_OS_DEFINES		" -D__attribute__\\(x\\)= -D__extension__="
#define CCD_OS_UNDEFINES	"-U__CYGWIN__ -U__CYGWIN32__"
#define CCD_USAGE_DEF		", etc."
#define CCD_USAGE_UNDEF		"__CYGWIN__ __CYGWIN32__"
#endif /* GNUWIN32 */

#ifndef CCD_OS_DEFINES
#define CCD_OS_DEFINES		""
#endif /* CCD_OS_DEFINES */

#ifndef CCD_OS_UNDEFINES
#define CCD_OS_UNDEFINES	""
#endif /* CCD_OS_UNDEFINES */


	/* shortened and polished versions for usage info */

#define CCD_USAGE_DEFINES	"__SPECC__" CCD_USAGE_DEF
#define CCD_USAGE_UNDEFINES	CCD_USAGE_UNDEF
#define CCD_USAGE_STD_INCLUDE	CCD_SPECC_STD_INCLUDE
#define CCD_USAGE_STD_LIBRARY	CCD_SPECC_STD_LIBRARY
#define CCD_USAGE_STD_LIBRARIES	"lbit, lsim, llonglong, l" THREAD_BASE
#define CCD_USAGE_PP_CALL	CCD_DEFAULT_PP_CALL " %p..."
#define CCD_USAGE_CC_CALL	CCD_DEFAULT_CC_CALL " %c..."
#define CCD_USAGE_LD_CALL	CCD_DEFAULT_LD_CALL "..."
#define CCD_USAGE_PP_OPTS	"none"
#define CCD_USAGE_CC_OPTS	"none"
#define CCD_USAGE_LD_OPTS	"none"


/*** type definitions ***************************************************/


	/* compiler mode (see CCD_SetCompilerMode()) */

enum CCD_Mode
{
CCD_MODE_STANDARD,	// standard compilation
CCD_MODE_DEBUG,		// compilation with debugging support
CCD_MODE_OPTIMIZE	// compilation with optimization
};

typedef enum CCD_Mode		CCD_MODE;


/*** exported variables *************************************************/


extern ERROR	CCD_Error;	/* last error of this module */


/************************************************************************/
/*** prototype definitions for exported functions		      ***/
/************************************************************************/


	/*** error handling ***/

	/* recover from error status */

extern void CCD_Reset(void);


	/* error code to error message conversion */

extern const char *CCD_ErrorText(ERROR ErrorNumber);


	/*** option settings ***/

extern void CCD_SetDefaultOptions(void);

extern ERROR CCD_SetCompilerMode(CCD_MODE Mode);

extern ERROR CCD_SetPreprocessorCall(const char *FormatString);

extern ERROR CCD_SetCompilerCall(const char *FormatString);

extern ERROR CCD_SetLinkerCall(const char *FormatString);

extern ERROR CCD_SetPreprocessorOption(const char *Option);

extern ERROR CCD_SetCompilerOption(const char *Option);

extern ERROR CCD_SetLinkerOption(const char *Option);

extern ERROR CCD_SetMacroDefs(GL_STRINGLIST *DList);

extern ERROR CCD_SetUnDefs(GL_STRINGLIST *UList);

extern ERROR CCD_SetIncludePath(GL_STRINGLIST *IList);

extern ERROR CCD_SetLibraryPath(GL_STRINGLIST *LList);

extern ERROR CCD_SetLibraries(GL_STRINGLIST *lList);

extern ERROR CCD_NoStdMacroDefs(void);

extern ERROR CCD_NoStdUnDefs(void);

extern ERROR CCD_NoStdIncludePath(void);

extern ERROR CCD_NoStdLibraryPath(void);

extern ERROR CCD_NoStdLibraries(void);


	/*** main functionality ***/


extern ERROR CCD_Preprocessor(
	const char	*InputFilename,
	const char	*OutputFilename);

extern ERROR CCD_Compiler(
	const char	*InputFilename,
	const char	*OutputFilename);

extern ERROR CCD_Linker(
	const char	*InputFilename,
	const char	*OutputFilename);


#endif /* CCDRIVER_H */

/* EOF CcDriver.h */
