/************************************************************************/
/* Global.cc: global definitions, error handling, memory management...	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 07/02/97 */
/************************************************************************/

/* last update: 05/26/01 */

/* modifications: (most recent first)
 *
 * 05/26/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/03/01 RD	replaced GL_OpenFileWithDirList() by GL_FindFileInDirList()
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 03/08/01 RD	created this modifications list (last change was 03/31/99)
 */

#include "Global.h"

#include <ctype.h>
#include <assert.h>
#include <errno.h>
#include <sys/stat.h>
#include <string.h>
#include <unistd.h>
#include <malloc.h>
#include <stdarg.h>


/*** constants and macros ***********************************************/


#ifdef DEBUG
#ifndef GL_DEBUG
#define GL_DEBUG
#endif /* GL_DEBUG */
#endif /* DEBUG */

/* #define MEM_DEBUG */			/* display memory management */

/* #define GL_TEST_OUT_OF_MEMORY */	/* test memory management */


#ifdef SOLARIS
#define READ_ACCESS_CODE	"r"
#define WRITE_ACCESS_CODE	"w"
#endif /* SOLARIS */

#ifdef SUNOS4
#define READ_ACCESS_CODE	"r"
#define WRITE_ACCESS_CODE	"w"
#endif /* SUNOS4 */

#ifdef NETBSD
#define READ_ACCESS_CODE	"r"
#define WRITE_ACCESS_CODE	"w"
#endif /* NETBSD */

#ifdef LINUX
#define READ_ACCESS_CODE	"r"
#define WRITE_ACCESS_CODE	"w"
#endif /* LINUX */

#ifdef GNUWIN32
#define READ_ACCESS_CODE	"rb"
#define WRITE_ACCESS_CODE	"wb"
#endif /* GNUWIN32 */

#ifndef READ_ACCESS_CODE
#error "Unknown system! READ_ACCESS_CODE definition missing!"
#endif /* READ_ACCESS_CODE */
#ifndef WRITE_ACCESS_CODE
#error "Unknown system! WRITE_ACCESS_CODE definition missing!"
#endif /* WRITE_ACCESS_CODE */


/*** internal variables *************************************************/


static string	ErrorMessage;

static void	*SpareMemory = NULL;
static int	SpareMemorySize = 0;


/*** exported variables *************************************************/


ERROR	GL_Error = GL_ERROR_NO_ERROR;	/* last error of this module */

int	GL_VerbosityLevel = 0;		/* (published by main module) */
int	GL_WarningLevel = 0;		/* (published by main module) */


/************************************************************************/
/*** implementation of exported functions			      ***/
/************************************************************************/


/*** error handling *****************************************************/


	/* recover from error status */

void GL_Reset(void)		/* recover from error status */
{

GL_Error = GL_ERROR_NO_ERROR;

} /* end of GL_Reset */


	/* error code to error message conversion */

const char *GL_ErrorText(	/* returns message of error code */
	ERROR ErrorNumber)
{
switch (ErrorNumber)
   { case GL_ERROR_NO_ERROR:
	  return("No error (internal)");
     case GL_ERROR_OUT_OF_MEMORY:
	  return("Out of memory");
     case GL_ERROR_FILE_R_OPEN_FAILED:
	  return(ErrorMessage.chars()
		/* "Cannot open file \"%s\" for read access"
			+ GL_SystemErrorMessage */);
     case GL_ERROR_FILE_W_OPEN_FAILED:
	  return(ErrorMessage.chars()
		/* "Cannot open file \"%s\" for write access"
			+ GL_SystemErrorMessage */);
     case GL_ERROR_PROBLEM_WHILE_CLOSING_FILE:
	  return("Problem while closing the file");
     case GL_ERROR_FILE_NOT_READABLE:
	  return(ErrorMessage.chars()
		/* "File \"%s\" not readable"
			+ GL_SystemErrorMessage */);
     case GL_ERROR_UNLINK_FAILED:
	  return(ErrorMessage.chars()
		/* "Unlinking existing file \"%s\" failed"
			+ GL_SystemErrorMessage */);
     case GL_ERROR_READ_BYTES_FAILED:
	  return(ErrorMessage.chars()
		/* "Reading %d bytes from file failed"
			+ GL_SystemErrorMessage */);
     case GL_ERROR_WRITE_BYTES_FAILED:
	  return(ErrorMessage.chars()
		/* "Writing %d bytes to file failed"
			+ GL_SystemErrorMessage */);
     case GL_ERROR_NOT_AT_END_OF_FILE:
	  return("Not at end of file: excess data found");
     case GL_ERROR_NOT_A_LEGAL_IDENTIFIER:
	  return(ErrorMessage.chars()
		/* "'%s' is not a legal identifier" */);
     case GL_ERROR_NOT_A_LEGAL_IDENTIFIER_2:
	  return(ErrorMessage.chars()
		/* "'%s' is not a legal identifier," GL_ERROR_MSG_NEWLINE
		"it is a SpecC keyword" */);
     case GL_ERROR_NOT_A_LEGAL_IDENTIFIER_3:
	  return(ErrorMessage.chars()
		/* "'%s' is not a legal identifier," GL_ERROR_MSG_NEWLINE
		"it is reserved for future extensions" */);
     case GL_ERROR_NOT_A_LEGAL_FILENAME:
	  return(ErrorMessage.chars()
		/* "'%s' is not a legal file name" */);
     case GL_ERROR_NOT_A_LEGAL_FILE_PATH:
	  return(ErrorMessage.chars()
		/* "'%s' is not a legal file path" */);
     case GL_ERROR_FILE_DELETE_FAILED:
	  return(ErrorMessage.chars()
		/* "Cannot delete file \"%s\""
			+ GL_SystemErrorMessage */);
    } /* hctiws */

return("Unknown error number (internal)");

} /* end of GL_ErrorText */


const char *GL_SystemErrorMessage(void)
{
static string	Message;	/* ('static' ensures valid return value) */

if (errno == 0)
   { Message = "";
     return(Message.chars());
    } /* fi */

#ifdef SOLARIS
if (strerror(errno))
   { Message.form(GL_ERROR_MSG_NEWLINE "(%s (system error %d))",
			strerror(errno), errno);
    } /* fi */
else
   { Message.form(GL_ERROR_MSG_NEWLINE "(system error %d)",
			errno);
    } /* esle */
#endif /* SOLARIS */

#ifdef SUNOS4
Message.form(GL_ERROR_MSG_NEWLINE "(system error %d)", errno);
#endif /* SUNOS4 */

#ifdef NETBSD
if (errno < sys_nerr)
   { Message.form(GL_ERROR_MSG_NEWLINE "(%s (system error %d))",
			sys_errlist[errno], errno);
    } /* fi */
else
   { Message.form(GL_ERROR_MSG_NEWLINE "(system error %d)",
			errno);
    } /* esle */
#endif /* NETBSD */

#ifdef LINUX
if (strerror(errno))
   { Message.form(GL_ERROR_MSG_NEWLINE "(%s (system error %d))",
			strerror(errno), errno);
    } /* fi */
else
   { Message.form(GL_ERROR_MSG_NEWLINE "(system error %d)",
			errno);
    } /* esle */
#endif /* LINUX */

#ifdef GNUWIN32
if (strerror(errno))
   { Message.form(GL_ERROR_MSG_NEWLINE "(%s (system error %d))",
			strerror(errno), errno);
    } /* fi */
else
   { Message.form(GL_ERROR_MSG_NEWLINE "(system error %d)",
			errno);
    } /* esle */
#endif /* GNUWIN32 */

return(Message.chars());

} /* end of GL_SystemErrorMessage */


void GL_PrintError(
	int		ErrorNumber,
	const char	*ErrorMsg)
{

fprintf(stderr, "\nERROR #%04d: ", ErrorNumber);
fputs(ErrorMsg, stderr);
fputs(".\n\n", stderr);

} /* end of GL_PrintError */


void GL_PrintErrorFmt(
	int		ErrorNumber,
	const char	*Format,
	...)
{
va_list		ArgPtr;

va_start(ArgPtr, Format);

fprintf(stderr, "\nERROR #%04d: ", ErrorNumber);
vfprintf(stderr, Format, ArgPtr);
fprintf(stderr, ".\n\n");

va_end(ArgPtr);

} /* end of GL_PrintErrorFmt */


void GL_PrintWarning(
	int		TestWarnLevel,
	const char	*WarnMsg)
{

if (GL_WarningLevel >= TestWarnLevel)
   { fprintf(stderr, "\nWARNING: ");
     fputs(WarnMsg, stderr);
     fputs("!\n\n", stderr);
    } /* fi */

} /* end of GL_PrintWarning */


void GL_PrintWarningFmt(
	int		TestWarnLevel,
	const char	*Format,
	...)
{
va_list		ArgPtr;

va_start(ArgPtr, Format);

if (GL_WarningLevel >= TestWarnLevel)
   { fprintf(stderr, "\nWARNING: ");
     vfprintf(stderr, Format, ArgPtr);
     fprintf(stderr, "!\n\n");
    } /* fi */

va_end(ArgPtr);

} /* end of GL_PrintWarningFmt */


/*** memory management **************************************************/


	/* overload C++ memory management */

void *operator new (size_t NumBytes)
{
void	*MemAddress;

#ifdef MEM_DEBUG
printf("MEM_DEBUG: new(%8d) ", NumBytes);
#endif /* MEM_DEBUG */

MemAddress = GL_malloc(NumBytes);

#ifdef MEM_DEBUG
printf("succeeds and returns 0x%08lx\n", (ULONG)MemAddress);
#endif /* MEM_DEBUG */

return(MemAddress);

} /* end of operator new */


void *operator new[] (size_t NumBytes)
{
void	*MemAddress;

#ifdef MEM_DEBUG
printf("MEM_DEBUG: new[](%8d) ", NumBytes);
#endif /* MEM_DEBUG */

MemAddress = GL_malloc(NumBytes);

#ifdef MEM_DEBUG
printf("succeeds and returns 0x%08lx\n", (ULONG)MemAddress);
#endif /* MEM_DEBUG */

return(MemAddress);

} /* end of operator new[] */


void operator delete (void *MemAddress)
{

#ifdef MEM_DEBUG
printf("MEM_DEBUG: delete(0x%08lx)\n", (ULONG)MemAddress);
#endif /* MEM_DEBUG */

if (MemAddress)		/* delete must tolerate NULL argument */
   { GL_free(MemAddress);
    } /* fi */

} /* end of operator delete */


void operator delete[] (void *MemAddress)
{

#ifdef MEM_DEBUG
printf("MEM_DEBUG: delete[](0x%08lx)\n", (ULONG)MemAddress);
#endif /* MEM_DEBUG */

if (MemAddress)		/* delete must tolerate NULL argument */
   { GL_free(MemAddress);
    } /* fi */

} /* end of operator delete[] */


	/* smarter versions of malloc() and free() */

void *GL_malloc(size_t NumBytes)	/* never returns NULL */
{
void		*MemAddress;
int		NumRetrys;
unsigned int	Time;

if ((MemAddress = malloc(NumBytes)))	/* initial try for malloc() */
   { return(MemAddress);
    } /* fi */

if (SpareMemory)	/* if we reserved some memory try using that */
   { GL_PrintWarningFmt(GL_WARN_IMPORTANT,
	"Low on memory!!!" GL_WARN_MSG_NEWLINE
	"Could not allocate %d bytes of memory." GL_WARN_MSG_NEWLINE
	"Retry after freeing %d bytes of spare memory...",
#ifdef SUNOS4
		(int)NumBytes,
#else /* SUNOS4 */
		NumBytes,
#endif /* SUNOS4 */
		SpareMemorySize);
     free(SpareMemory);
     SpareMemory = NULL;
     SpareMemorySize = 0;
     if ((MemAddress = malloc(NumBytes)))	/* second try for malloc() */
	{ return(MemAddress);
	 } /* fi */
    } /* fi */

NumRetrys = 0;		/* wait some time and try again! */
Time = 1;
while(NumRetrys < GL_MEM_NUMBER_RETRYS)
   { NumRetrys++;
     GL_PrintWarningFmt(GL_WARN_IMPORTANT,
	"Out of memory!!!" GL_WARN_MSG_NEWLINE
	"Could not allocate %d bytes of memory." GL_WARN_MSG_NEWLINE
	"Retry after %u seconds...",
#ifdef SUNOS4
		(int)NumBytes,
#else /* SUNOS4 */
		NumBytes,
#endif /* SUNOS4 */
		Time);
     sleep(Time);
     if ((MemAddress = malloc(NumBytes)))
	{ return(MemAddress);
	 } /* fi */
     Time <<= 1;
    } /* elihw */

GL_PrintError(GL_ERROR_OUT_OF_MEMORY, GL_ErrorText(GL_ERROR_OUT_OF_MEMORY));
printf("Aborted.\n");

abort();	/* abort the program */

return(NULL);	/* never executed (but makes the compiler quiet) */

} /* end of GL_malloc */


void GL_free(void *MemAddress)
{

free(MemAddress);

} /* end of GL_free */


	/* spare memory handling */

ERROR GL_ReserveSpareMemory(size_t NumBytes)
{

if (!(SpareMemory = malloc(NumBytes)))
   { return(GL_ERROR_OUT_OF_MEMORY);
    } /* fi */
SpareMemorySize = NumBytes;

return(GL_ERROR_NO_ERROR);

} /* end of GL_ReserveSpareMemory */


void GL_FreeSpareMemory(void)
{

if (SpareMemory)
   { free(SpareMemory);
     SpareMemory = NULL;
     SpareMemorySize = 0;
    } /* fi */

} /* end of GL_FreeSpareMemory */


/*** general support ****************************************************/


ERROR GL_CheckIdentifier(		/* checks for a legal identifier */
	const char	*Name)
{
const char	*CharPtr;

if (Name == NULL)
   { ErrorMessage = "'<NULL>' is not a legal identifier";
     return(GL_ERROR_NOT_A_LEGAL_IDENTIFIER);
    } /* fi */
if (  (!(isalpha(*Name)))
    &&(*Name != '_'))
   { ErrorMessage.form("'%s' is not a legal identifier", Name);
     return(GL_ERROR_NOT_A_LEGAL_IDENTIFIER);
    } /* fi */

CharPtr = Name + 1;
while(*CharPtr)
   { if (  (!(isalnum(*CharPtr)))
	 &&(*CharPtr != '_'))
	{ ErrorMessage.form("'%s' is not a legal identifier", Name);
	  return(GL_ERROR_NOT_A_LEGAL_IDENTIFIER);
	 } /* fi */
     CharPtr++;
    } /* elihw */

return(GL_ERROR_NO_ERROR);

} /* end of GL_CheckIdentifier */


ERROR GL_CheckSpecCIdentifier(		/* checks for a legal identifier */
	const char	*Name)		/* and not being a SpecC keyword */
{
static const char *Keyword[] = {
	"auto",
	"break",
	"case",
	"char",
	"const",
	"continue",
	"default",
	"do",
	"double",
	"else",
	"enum",
	"extern",
	"float",
	"for",
	"goto",
	"if",
	"int",
	"long",
	"register",
	"return",
	"short",
	"signed",
	"sizeof",
	"static",
	"struct",
	"switch",
	"typedef",
	"union",
	"unsigned",
	"void",
	"volatile",
	"while",

	"behavior",
	"bit",
	"bool",
	"channel",
	"event",
	"false",
	"fsm",
	"implements",
	"import",
	"in",
	"inout",
	"interface",
	"interrupt",
	"note",
	"notify",
	"notifyone",
	"out",
	"par",
	"pipe",
	"piped",
	"range",
	"this",
	"timing",
	"trap",
	"true",
	"try",
	"wait",
	"waitfor",
	NULL };
static const char *Reserved[] = {
	"asm",
	"catch",
	"class",
	"const_cast",
	"delete",
	"dynamic_cast",
	"explicit",
	"export",
	"friend",
	"inline",
	"mutable",
	"namespace",
	"new",
	"operator",
	"private",
	"protected",
	"public",
	"reinterpret_cast",
	"static_cast",
	"template",
	"throw",
	"typeid",
	"typename",
	"using",
	"virtual",
	NULL };
int		Index;

if ((GL_Error = GL_CheckIdentifier(Name)))
   { return(GL_Error);
    } /* fi */

Index = 0;		/* there's surely a more efficient way to do this, */
while(Keyword[Index])	/* but this function usually is not time-critical  */
   { if (0 == strcmp(Keyword[Index], Name))
	{ ErrorMessage.form(
		"'%s' is not a legal identifier," GL_ERROR_MSG_NEWLINE
		"it is a SpecC keyword", Name);
	  return(GL_ERROR_NOT_A_LEGAL_IDENTIFIER_2);
	 } /* fi */
     Index++;
    } /* elihw */

Index = 0;
while(Reserved[Index])
   { if (0 == strcmp(Reserved[Index], Name))
	{ ErrorMessage.form(
		"'%s' is not a legal identifier," GL_ERROR_MSG_NEWLINE
		"it is reserved for future extensions", Name);
	  return(GL_ERROR_NOT_A_LEGAL_IDENTIFIER_3);
	 } /* fi */
     Index++;
    } /* elihw */

return(GL_ERROR_NO_ERROR);

} /* end of GL_CheckSpecCIdentifier */


ERROR GL_CheckFilename(		/* checks for a legal filename */
	const char	*Filename)
{
const char	*CharPtr;

if (Filename == NULL)
   { ErrorMessage = "'<NULL>' is not a legal file name";
     return(GL_ERROR_NOT_A_LEGAL_FILENAME);
    } /* fi */

CharPtr = Filename;
while(*CharPtr)
   { if (  (isspace(*CharPtr))
	 ||(iscntrl(*CharPtr))
	 ||(*CharPtr == '/'))
	{ ErrorMessage.form("'%s' is not a legal file name", Filename);
	  return(GL_ERROR_NOT_A_LEGAL_FILENAME);
	 } /* fi */
     CharPtr++;
    } /* elihw */

return(GL_ERROR_NO_ERROR);

} /* end of GL_CheckFilename */


ERROR GL_CheckPathFilename(	/* checks for a legal filename */
	const char	*PathFile)	/* with local or absolute path */
{
const char	*CharPtr;

if (PathFile == NULL)
   { ErrorMessage = "'<NULL>' is not a legal file path";
     return(GL_ERROR_NOT_A_LEGAL_FILE_PATH);
    } /* fi */

CharPtr = PathFile;
while(*CharPtr)
   { if (  (isspace(*CharPtr))
	 ||(iscntrl(*CharPtr)))
	{ ErrorMessage.form("'%s' is not a legal file path", PathFile);
	  return(GL_ERROR_NOT_A_LEGAL_FILE_PATH);
	 } /* fi */
     CharPtr++;
    } /* elihw */

return(GL_ERROR_NO_ERROR);

} /* end of GL_CheckPathFilename */


const char *GL_GetFileFromPath(		/* searches file name in a path */
	const char	*Path)
{
const char	*Cursor;

assert(Path != NULL);

Cursor = Path + strlen(Path);
while (((unsigned long) Cursor > (unsigned long) Path)
		&& (*Cursor != '/'))
   { Cursor--;
    } /* elihw */

if (*Cursor == '/')
   { Cursor++;
    } /* fi */

return(Cursor);

} /* end of GL_GetFileFromPath */


const char *GL_GetBaseName(		/* obtains the base name from a path */
	const char	*Path)		/* (file name without suffix) */
{
const char	*File;
char		*Buffer,
		*BuffPtr;
static string	BaseName;

File = GL_GetFileFromPath(Path);

Buffer = new char[strlen(File) +1];
strcpy(Buffer, File);
if ((BuffPtr = strchr(Buffer, '.')))
   { *BuffPtr = 0;
    } /* fi */
BaseName = Buffer;
delete[] Buffer;

return(BaseName.chars());

} /* end of GL_GetBaseName */


/*** file I/O ***********************************************************/


	/* support functions for file I/O */

FILE *GL_OpenFileForReading(	/* open a file for read access */
	const char *Filename)	/* (supports "-" for stdin) */
{
FILE	*FileHandle;

assert(Filename != NULL);

if (0 == strcmp(Filename, "-"))
   { return(stdin);
    } /* fi */

if ((GL_Error = GL_FileIsReadable(Filename)))
   { return(NULL);
    } /* fi */

if (!(FileHandle = fopen(Filename, READ_ACCESS_CODE)))
   { ErrorMessage.form("Cannot open file \"%s\" for read access",
			Filename);
     ErrorMessage += GL_SystemErrorMessage();
     GL_Error = GL_ERROR_FILE_R_OPEN_FAILED;
     return(NULL);
    } /* fi */

return(FileHandle);

} /* end of GL_OpenFileForReading */


FILE *GL_OpenFileForWriting(	/* open a (new) file for write access */
	const char *Filename)	/* (supports "-" for stdout) */
{
FILE	*FileHandle;

if (0 == strcmp(Filename, "-"))
   { return(stdout);
    } /* fi */

if ((GL_Error = GL_FileIsCreatable(Filename)))
   { return(NULL);
    } /* fi */

if (!(FileHandle = fopen(Filename, WRITE_ACCESS_CODE)))
   { ErrorMessage.form("Cannot open file \"%s\" for write access",
			Filename);
     ErrorMessage += GL_SystemErrorMessage();
     GL_Error = GL_ERROR_FILE_W_OPEN_FAILED;
     return(NULL);
    } /* fi */

return(FileHandle);

} /* end of GL_OpenFileForWriting */


void GL_CloseFile(		/* close a file */
	FILE *FileHandle)
{

assert(FileHandle != NULL);

if (  (FileHandle != stdin)
    &&(FileHandle != stdout))
   { fclose(FileHandle);
    } /* fi */

} /* end of GL_CloseFile */


ERROR GL_CloseFileWithCheck(	/* close a file and check for errors */
	FILE *FileHandle)
{

assert(FileHandle != NULL);

if (ferror(FileHandle))
   { if (  (FileHandle != stdin)
	 &&(FileHandle != stdout))
	{ fclose(FileHandle);
	 } /* fi */
     return(GL_ERROR_PROBLEM_WHILE_CLOSING_FILE);
    } /* fi */

if (  (FileHandle != stdin)
    &&(FileHandle != stdout))
   { if (fclose(FileHandle))
	{ return(GL_ERROR_PROBLEM_WHILE_CLOSING_FILE);
	 } /* fi */
    } /* fi */

return(GL_ERROR_NO_ERROR);

} /* end of GL_CloseFileWithCheck */


ERROR GL_DeleteFile(		/* delete a file */
	const char	*Filename)
{
if (unlink(Filename))
   { ErrorMessage.form("Cannot delete file \"%s\"",
			Filename);
     ErrorMessage += GL_SystemErrorMessage();
     return(GL_ERROR_FILE_DELETE_FAILED);
    } /* fi */

return(GL_ERROR_NO_ERROR);

} /* end of GL_DeleteFile */


ERROR GL_FileIsReadable(	/* check if file is read-accessable */
	const char	*Filename)	/* (supports "-" for stdin) */
{
struct stat	StatBuf;

assert(Filename != NULL);

if (0 == strcmp(Filename, "-"))
   { return(GL_ERROR_NO_ERROR);
    } /* fi */

if (0 != access(Filename, R_OK))
   { ErrorMessage.form("File \"%s\" not readable%s",
			Filename, GL_SystemErrorMessage());
     return(GL_ERROR_FILE_NOT_READABLE);
    } /* fi */
if (0 != stat(Filename, &StatBuf))
   { ErrorMessage.form("File \"%s\" not readable%s",
			Filename, GL_SystemErrorMessage());
     return(GL_ERROR_FILE_NOT_READABLE);
    } /* fi */
if ((StatBuf.st_mode & S_IFMT) != S_IFREG)
   { ErrorMessage.form("File \"%s\" not readable;" GL_ERROR_MSG_NEWLINE
			"not an ordinary file",
			Filename);
     return(GL_ERROR_FILE_NOT_READABLE);
    } /* fi */

return(GL_ERROR_NO_ERROR);

} /* end of GL_FileIsReadable */


ERROR GL_FileIsCreatable(	/* check if file is write-accessable */
	const char	*Filename)	/* (supports "-" for stdout) */
{

assert(Filename != NULL);

if (0 == strcmp(Filename, "-"))
   { return(GL_ERROR_NO_ERROR);
    } /* fi */

if (0 == access(Filename, F_OK))	/* existing? */
   { if (0 == unlink(Filename))
	{ return(GL_ERROR_NO_ERROR);	/* removed! */
	 } /* fi */
     else
	{ ErrorMessage.form("Unlinking existing file \"%s\" failed%s",
			Filename, GL_SystemErrorMessage());
	  return(GL_ERROR_UNLINK_FAILED);
	 } /* fi */
    } /* fi */

return(GL_ERROR_NO_ERROR);

} /* end of GL_FileIsCreatable */


ERROR GL_ReadEOF(		/* read (and check) EOF from file */
	FILE		*FileHandle)
{
char	Trash;

fread(&Trash, 1, 1, FileHandle);	/* try to read one more byte */
if (!(feof(FileHandle)))
   { clearerr(FileHandle);
     return(GL_ERROR_NOT_AT_END_OF_FILE);
    } /* fi */

clearerr(FileHandle);

return(GL_ERROR_NO_ERROR);

} /* end of GL_ReadEOF */


const char *GL_FindFileInDirList(	/* find a file for read access */
	const char	*Filename,	/* (applying DirList and Suffix) */
	const char	*Suffix = NULL,	/* (returns NULL if not found) */
	GL_STRINGLIST	*DirList = NULL)
{
GL_STRINGLIST	*Dir;
static string	NewFilename;	/* return buffer! */

assert(Filename != NULL);

if (Filename[0] == '/')		/* absolute filename? */
   { NewFilename = Filename;
     if (Suffix)
	{ NewFilename += Suffix;
	 } /* fi */
     if ((GL_ERROR_NO_ERROR == GL_FileIsReadable(NewFilename.chars())))
	{ return(NewFilename.chars());
	 } /* fi */
     return(NULL);	/* absolute file not found */
    } /* fi */

Dir = DirList;		/* apply the list of directories */
while(Dir)
   { NewFilename = Dir->String;
     NewFilename += "/";
     NewFilename += Filename;
     if (Suffix)
	{ NewFilename += Suffix;
	 } /* fi */
     if ((GL_ERROR_NO_ERROR == GL_FileIsReadable(NewFilename.chars())))
	{ return(NewFilename.chars());
	 } /* fi */
     Dir = Dir->Succ;
    } /* elihw */

return(NULL);	/* no such file found in directory list */

} /* end of GL_FindFileInDirList */


/************************************************************************/
/*** main (for debugging only)					      ***/
/************************************************************************/


#ifdef DEBUG	/* module self test */


int main(int argc, char **argv)
{
ERROR	Error;
#ifdef GL_TEST_OUT_OF_MEMORY
char	*Mem;
int	i;
#endif /* GL_TEST_OUT_OF_MEMORY */

printf("DEBUGGING: Global ('%s')\n\n", argv[0]);
if (argc != 1)
   { puts("WARNING: Arguments will be ignored!\n");
    } /* fi */

puts("Reserving 1MB spare memory...");
if ((Error = GL_ReserveSpareMemory(1*1024*1024)))
   { GL_PrintError(Error, GL_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

puts("\nTesting GL_SystemErrorMessage()...");
printf("GL_SystemErrorMessage() returns: \"%s\"\n\n",
	GL_SystemErrorMessage());

puts("Testing GL_FileIsReadable()...");
printf("GL_FileIsReadable(\"%s\") returns: %s\n\n", argv[0],
	((GL_Error = GL_FileIsReadable(argv[0])) ?
					"ERROR!" : "success"));
if (GL_Error)
   { GL_PrintError(GL_Error, GL_ErrorText(GL_Error));
     exit(RESULT_ERROR);
    } /* fi */

puts("Testing GL_FileIsCreatable()...");
printf("GL_FileIsCreatable(\"NonExistingFile\") returns: %s\n\n",
	((GL_Error = GL_FileIsCreatable("NonExistingFile")) ?
					"ERROR!" : "success"));
if (GL_Error)
   { GL_PrintError(GL_Error, GL_ErrorText(GL_Error));
     exit(RESULT_ERROR);
    } /* fi */

#ifdef GL_TEST_OUT_OF_MEMORY
puts("Testing OUT_OF_MEMORY_CONDITION...");
for(i=0; i< 1000000000; i++)
   { Mem = new char[512*1024];	/* eat memory in 0.5MB pieces */
     if (! Mem)
	{ puts("ERROR: OUT OF MEMORY (INTERNAL)!!!");	/* will never happen */
	  return(RESULT_ERROR);
	 } /* fi */
    } /* rof */
#endif /* GL_TEST_OUT_OF_MEMORY */

puts("Freeing spare memory...");
GL_FreeSpareMemory();

puts("\nTesting GL_CheckIdentifier()...");
if ((Error = GL_CheckIdentifier("xxx_yyy")))
   { GL_PrintError(Error, GL_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

puts("Testing GL_CheckSpecCIdentifier()...");
if ((Error = GL_CheckSpecCIdentifier("xxx_yyy")))
   { GL_PrintError(Error, GL_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

puts("Testing GL_CheckFilename()...");
if ((Error = GL_CheckFilename("My_Filename.xyz")))
   { GL_PrintError(Error, GL_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

puts("Testing GL_CheckPathFilename()...");
if ((Error = GL_CheckPathFilename("/tmp/../My_Filename.xyz")))
   { GL_PrintError(Error, GL_ErrorText(Error));
     exit(RESULT_ERROR);
    } /* fi */

puts("Testing GL_GetBaseName()...");
if (0 != strcmp(GL_GetBaseName("/tmp/../My_Design.xyz"), "My_Design"))
   { fputs("Test of GL_GetBaseName() failed!\n", stderr);
     exit(RESULT_ERROR);
    } /* fi */

puts("\nDone.");
return(RESULT_OK);

} /* end of main */

#endif /* DEBUG */


/* EOF Global.cc */
