/************************************************************************/
/* specc.cc: SpecC run-time simulation library, compiler-level API	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 03/22/01 */
/************************************************************************/

/* last update: 06/01/01 */

/* modifications: (most recent first)
 *
 * 06/01/01 RD	two bug fixes for bugs showing up on linux
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/16/01 RD	resolved name space competition with user by moving more
 *		definitions into _specc name space and adding prefixes
 * 05/15/01 RD	cleaned up the source code
 * 05/14/01 RD	added _specc::exception to separate data on heap and stack
 * 05/12/01 RD	bug fixes for empty constructs
 * 05/11/01 RD	added support for exception handling
 * 04/16/01 RD	renamed "SIM_Time" to "sim_time"
 * 04/04/01 RD	added support for 'pipe' and 'piped'
 * 04/02/01 RD	added class _specc::event_ptr
 * 04/02/01 RD	added support for 'wait' and 'notify'
 * 04/02/01 RD	adjustments for use of generic queues
 * 03/30/01 RD	added code for 'waitfor'
 * 03/28/01 RD	completed setup, filled in initial functions
 * 03/22/01 RD	initial version
 */


#include "specc.h"
#include "thread.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>


/*** constants and macros ***********************************************/


	/* (none) */


/*** internal type declarations *****************************************/


	/* (none) */


/*** internal function prototypes ***************************************/


	/* (none) */


/*** internal variables *************************************************/


	/* (none) */


/*** external variables *************************************************/


	/* (none) */


/************************************************************************/
/*** implementation of internal functions			      ***/
/************************************************************************/


	/* (none) */


/*** class implementations **********************************************/


	/**************/
	/*** _specc ***/
	/**************/


	/*** wrapped variables ***/


sim_time	_specc::CurrentTime	/* current simulation time */
			= 0;
_specc::thread	*_specc::RootThread	/* the root thread */
			= NULL;
_specc::thread	*_specc::CurrentThread	/* the current thread */
			= NULL;
_specc::queue	_specc::ReadyQueue;	/* the ready queue */
					/* (threads in SIM_STATE_READY, */
					/* in planned execution order)  */

_specc::queue	_specc::WaitforQueue;	/* the event queue for 'waitfor' */
					/* (threads in SIM_STATE_WAITFOR,     */
					/* ordered by increasing time stamps) */

_specc::queue	_specc::WaitQueue;	/* the queue for 'wait'ing threads */
					/* (threads in SIM_STATE_WAIT, */
					/* in no particular order)     */

_specc::queue	_specc::NotifiedQueue;	/* the queue for 'notify'ed threads */
					/* (threads in SIM_STATE_NOTIFIED, */
					/* in no particular order)         */

_specc::queue	_specc::TryQueue;	/* the queue of 'try'ing threads */
					/* (threads in SIM_STATE_TRYING, */
					/* in order of creation)         */

_specc::queue	_specc::SuspendQueue;	/* the queue of suspended threads */
					/* (threads in SIM_STATE_SUSPENDED or */
					/* SIM_STATE_SUSPENDED2, any order) */

_specc::event	*_specc::NotifiedEvents	/* list of notified events */
			= NULL;		/* (for 'wait' and 'notify') */

_specc::event_list *_specc::Notify1List	/* list of 'notifyone'ed events */
			= NULL;


	/*** wrapped functions ***/


void _specc::start(void)	/* initialize the simulation engine */
{

_specc::thread::Start();		/* initialize the thread package */

_specc::thread::Create(NULL, NULL);	/* create the root thread */

} /* end of _specc::start */


void _specc::end(void)		/* clean up the simulation engine */
{

assert(RootThread != NULL);		/* root thread must exist */
assert(RootThread == CurrentThread);	/* and must be running    */

RootThread->Delete();			/* delete the root thread */

_specc::thread::End();			/* clean up the thread pkg. */

} /* end of _specc::end */


void _specc::abort(		/* cleanly abort with a message */
	const char	*Format,	/* (arguments as for printf) */
	...		)
{
va_list         Args;

va_start(Args, Format);
vfprintf(stderr, Format, Args);		/* print message */
va_end(Args);

::abort();				/* abort the program */

} /* end of _specc::abort */


void _specc::par(		/* SpecC 'par' replacement */
	fork		*First,		/* (NULL-terminated list of forks) */
	...		)
{
va_list		Args;
fork		*Fork;
thread		*ThisThread;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

if (! First)
   { ThisThread->Schedule();	/* empty 'par' construct is    */
     return;			/* just a possible task switch */
    } /* fi */

va_start(Args, First);
Fork = First;
while(Fork)
   { assert(Fork->Behavior != NULL);	/* must be initialized */
     _specc::thread::Create(Fork->Behavior,	/* create a new child */
				ThisThread);
     Fork = va_arg(Args, fork*);		/* for every argument */
    } /* elihw */
va_end(Args);

ThisThread->Status = SIM_STATE_JOIN;
ThisThread->Schedule();				/* call scheduler */

assert(ThisThread->Status == SIM_STATE_RUNNING);

ThisThread->Join(NULL);				/* join with all children */
ThisThread->DeleteChildren();			/* delete the children */

} /* end of _specc::par */


void _specc::pipe(		/* SpecC 'pipe' replacement (infinite, old) */
	fork		*First,		/* (NULL-terminated list of forks, */
	...		)		/* NULL-term. list of piped vars.) */
{
va_list		Args;
fork		*Fork;
thread		*ThisThread;
piped_base	*PipedVar;
unsigned int	PipeStages,
		ActiveStages,
		i;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

if (! First)
   { ThisThread->Status = SIM_STATE_JOIN;
     while(true)			/* empty 'pipe' construct is  */
	{ ThisThread->Schedule();	/* a dead end for this thread */
	 } /* elihw */			/* (i.e. should never happen) */
    } /* fi */

PipeStages = 0;
va_start(Args, First);		/* determine number of pipeline stages */
Fork = First;
while(Fork)
   { PipeStages++;
     Fork = va_arg(Args, fork*);
    } /* elihw */
va_end(Args);

ActiveStages = 1;
while(true)	/* endless loop */
   { va_start(Args, First);
     Fork = First;
     for(i=0; i<ActiveStages; i++)
	{ assert(Fork != NULL);			/* must exist */
	  assert(Fork->Behavior != NULL);	/* must be initialized */
	  _specc::thread::Create(Fork->Behavior,
				ThisThread);	/* create a new child     */
	  Fork = va_arg(Args, fork*);		/* for every active stage */
	 } /* elihw */
     while(Fork)
	{ Fork = va_arg(Args, fork*);	/* skip inactive stages */
	 } /* elihw */

     ThisThread->Status = SIM_STATE_JOIN;
     ThisThread->Schedule();			/* call scheduler */

     assert(ThisThread->Status == SIM_STATE_RUNNING);

     ThisThread->Join(NULL);			/* join with all children */
     ThisThread->DeleteChildren();		/* delete the children */

     PipedVar = va_arg(Args, piped_base*);
     while(PipedVar)
	{ PipedVar->update();			/* shift all piped data */
	  PipedVar = va_arg(Args, piped_base*);
	 } /* elihw */
     va_end(Args);

     if (ActiveStages < PipeStages)		/* repeat for next stage */
	{ ActiveStages++;
	 } /* fi */
    } /* elihw */

/* never reached */

} /* end of _specc::pipe #1 */


// note:
//	the following function only specifies the execution of
//	one pipeline iteration; in particular, only the specified
//	range of stages is actually executed, exactly once;
//	therefore, this function must be wrapped in a controlling
//	loop for the pipeline; the initialization, condition and
//	iterator of the pipeline loop are given with the SpecC
//	'pipe' construct and thus can be easily integrated into
//	the loop by the SpecC compiler; also, an appropriate pipeline
//	flushing phase should be specified;
//
//	for example, the SpecC construct
//
//		pipe(i=0; i<10; i++)
//		   { a.main(); b.main(); c.main(); }
//
//	should be translated as follows:
//
//		{
//		unsigned int _First, _Last;
//
//		_First = _Last = 1;
//		for(i=0; i<10; i++)
//		   { pipe(3, _First, _Last, a_fork, b_fork, c_fork, NULL);
//		     if (_Last < 3)
//			  _Last++;
//		    }
//		while (_First++ < _Last)
//		   { pipe(3, _First, _Last, a_fork, b_fork, c_fork, NULL);
//		     if (_Last < 3)
//			  _Last++;
//		    }
//		}


void _specc::pipe(		/* SpecC 'pipe' replacement (finite, new) */
	unsigned int	NumStages,	/* total number of pipeline stages */
	unsigned int	FirstStage,	/* first active stage */
	unsigned int	LastStage,	/* last active stage */
	...		)		/* (list of forks (length NumStages), */
{					/* NULL-termin. list of piped vars.)  */
va_list		Args;
fork		*Fork;
thread		*ThisThread;
piped_base	*PipedVar;
unsigned int	i;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

if (NumStages == 0)
   { ThisThread->Schedule();	/* empty 'pipe(...)' construct is */
     return;			/* just a possible task switch    */
    } /* fi */

assert(FirstStage <= LastStage);	/* don't allow stupid values */
assert(LastStage <= NumStages);

if (NumStages == 0)		/* empty 'pipe' construct is    */
   { ThisThread->Schedule();	/* just a possible task switch  */
     va_start(Args, LastStage);	/* plus one shift of piped data */
     PipedVar = va_arg(Args, piped_base*);
     while(PipedVar)
	{ PipedVar->update();
	  PipedVar = va_arg(Args, piped_base*);
	 } /* elihw */
     va_end(Args);
     return;
    } /* fi */

va_start(Args, LastStage);
for(i=1; i<=NumStages; i++)
   { Fork = va_arg(Args, fork*);
     assert(Fork != NULL);		/* must exist */
     assert(Fork->Behavior != NULL);	/* must be initialized */
     if ((i>=FirstStage) && (i<=LastStage))	 /* for active stages, */
	{ _specc::thread::Create(Fork->Behavior, /* create a new child */
				ThisThread);
	 } /* fi */
    } /* rof */

ThisThread->Status = SIM_STATE_JOIN;
ThisThread->Schedule();			/* call scheduler */

assert(ThisThread->Status == SIM_STATE_RUNNING);

ThisThread->Join(NULL);				/* join with all children */
ThisThread->DeleteChildren();			/* delete the children */

PipedVar = va_arg(Args, piped_base*);
while(PipedVar)
   { PipedVar->update();			/* shift all piped data */
     PipedVar = va_arg(Args, piped_base*);
    } /* elihw */
va_end(Args);

} /* end of _specc::pipe #2 */


void _specc::tryTrapInterrupt(	/* SpecC 'try-trap-interrupt' replacement */
	try_block	*TryBlock,
	exception_block	*First,
	...		)
{
va_list		Args;
exception_block	*ExceptionBlock;
exception	**ExceptionPtr;
thread		*ThisThread,
		*SuspendedChild;
bool		HasBeenInterrupted;
sim_time	InterruptDelay;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

if (  (! TryBlock)
    ||(! TryBlock->Behavior))
   { ThisThread->Schedule();	/* empty 'try' construct is    */
     return;			/* just a possible task switch */
    } /* fi */

_specc::thread::Create(TryBlock->Behavior,	/* create a new child */
				ThisThread);

va_start(Args, First);
ExceptionBlock = First;
ExceptionPtr = &ThisThread->Exceptions;
assert(ThisThread->Exceptions == NULL);		/* must be fresh */
while(ExceptionBlock)
   { *ExceptionPtr = new exception(ExceptionBlock);
     assert(*ExceptionPtr != NULL);
     ExceptionPtr = &(*ExceptionPtr)->Next;
     ExceptionBlock = va_arg(Args, exception_block*);
    } /* elihw */
va_end(Args);

TryQueue.Append(ThisThread);	/* append entry in the 'try' queue */

do { ThisThread->Status = SIM_STATE_TRYING;
     ThisThread->Schedule();			/* call scheduler */

     assert(ThisThread->Status == SIM_STATE_RUNNING);

     HasBeenInterrupted = (ThisThread->NumChildren == 2);
     if (HasBeenInterrupted)
	{ assert(ThisThread->ExceptionChild != NULL);
	  InterruptDelay = CurrentTime -
		ThisThread->ExceptionChild->CreationTime;
	  ThisThread->Join(		/* join with the completed interrupt */
		ThisThread->ExceptionChild);
	  ThisThread->ExceptionChild->Delete();
	  ThisThread->ExceptionChild = NULL;
	  TryQueue.Append(ThisThread);	/* re-enter in the 'try' queue */
	  assert(ThisThread->NumChildren == 1);
	  SuspendedChild = ThisThread->FirstChild;
	  assert(SuspendedChild != NULL);
	  SuspendedChild->Resume(	/* resume all interrupted children,   */
		InterruptDelay);	/* taking interrupt time into account */
	 } /* fi */
    } while(HasBeenInterrupted);

assert(ThisThread->NumChildren == 1);
ThisThread->Join(ThisThread->FirstChild); /* join and delete the child */
ThisThread->FirstChild->Delete();	  /* (a 'try' or 'abort' behavior) */

delete ThisThread->Exceptions;	/* remove the exception list */
ThisThread->Exceptions = NULL;
ThisThread->ExceptionChild = NULL;

} /* end of _specc::tryTrapInterrupt */


void _specc::waitfor(		/* SpecC 'waitfor' replacement */
	sim_time	Delay)
{
thread		*ThisThread;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

ThisThread->TimeStamp = CurrentTime + Delay;
WaitforQueue.Insert(ThisThread);	/* insert entry in the waitfor queue */

ThisThread->Status = SIM_STATE_WAITFOR;	/* change state to WAITFOR */
ThisThread->Schedule();			/* call the scheduler */

assert(ThisThread->Status == SIM_STATE_RUNNING);

} /* end of _specc::waitfor */


void _specc::wait(		/* SpecC 'wait' replacement */
	event		*First,		/* (NULL-terminated list of events) */
	...		)
{
va_list		Args;
thread		*ThisThread;
event		*Event;
event_ptr	**EventPtrPtr;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

assert(ThisThread->EventList == NULL);	/* event list not in use */

// note:
//	here, it is tempting to use the variable argument list directly
//	instead of copying it, but, on the other hand, it is really dirty
//	if we access data on this thread's stack space with va_arg() from
//	another thread (is that really portable?); so, to be on the safe
//	and clean side, the argument list is copied onto the heap;

va_start(Args, First);		/* copy argument list */
Event = First;
EventPtrPtr = &ThisThread->EventList;
while(Event)
   { if (!(*EventPtrPtr = new _specc::event_ptr(Event)))
	{ fatalError("Cannot build event list", ENOMEM);
	 } /* fi */
     EventPtrPtr = &(*EventPtrPtr)->Next;
     Event = va_arg(Args, event*);
    } /* elihw */
va_end(Args);

WaitQueue.Append(ThisThread);	/* create entry in the wait queue */

ThisThread->Status = SIM_STATE_WAIT;	/* change state to WAIT */
ThisThread->Schedule();			/* call the scheduler */

delete ThisThread->EventList;	/* delete the copied list again */
ThisThread->EventList = NULL;

assert(ThisThread->Status == SIM_STATE_RUNNING);

} /* end of _specc::wait */


void _specc::notify(		/* SpecC 'notify' replacement */
	event		*First,		/* (NULL-terminated list of events) */
	...		)
{
va_list		Args;
thread		*ThisThread;
event		*Event;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

va_start(Args, First);		/* include all arguments into the */
Event = First;			/* list of notified events	  */
while(Event)
   { if (! Event->Notified)	/* not notified yet? */
	{ Event->Notified = true;
	  Event->Next = NotifiedEvents;	/* insert at the beginning */
	  NotifiedEvents = Event;
	 } /* fi */
     Event = va_arg(Args, event*);
    } /* elihw */
va_end(Args);

/* no scheduler call here */

// note:
//	since we may be in a channel method here,
//	no task switching is allowed at this point!
//
// quote from rule 2.3.2(k), LRM v1.0:
//	"Methods encapsulated in channels are executed in non-preemptive
//	(atomic) manner. A thread executing a channel method is guaranteed
//	not to be interrupted by other threads, unless it is waiting for an
//	event (at a 'wait' statement) or waiting for simulation time increase
//	(at a 'waitfor' statement). [...]"

} /* end of _specc::notify */


void _specc::notifyone(		/* SpecC 'notifyone' replacement */
	event		*First,		/* (NULL-terminated list of events) */
	...		)
{
va_list		Args;
thread		*ThisThread;
event		*Event;
event_ptr	*EventList,
		**EventPtrPtr;
event_list	*NewEventList;

ThisThread = CurrentThread;
assert(ThisThread->Status == SIM_STATE_RUNNING);

va_start(Args, First);		/* store all arguments into the */
Event = First;			/* list of 'notifyone'ed events */
EventList = NULL;
EventPtrPtr = &EventList;
while(Event)
   { if (!(*EventPtrPtr = new _specc::event_ptr(Event)))
	{ fatalError("Cannot build event list", ENOMEM);
	 } /* fi */
     EventPtrPtr = &(*EventPtrPtr)->Next;
     Event = va_arg(Args, event*);
    } /* elihw */
va_end(Args);
if (!(NewEventList = new _specc::event_list(EventList)))
   { fatalError("Cannot build event list", ENOMEM);
    } /* fi */
NewEventList->Next = Notify1List;
Notify1List = NewEventList;

/* no scheduler call here */

// note: (see above)

} /* end of _specc::notifyone */


sim_time _specc::getCurrentTime(void)	/* obtain current simulation time */
{

return(CurrentTime);

} /* end of _specc::getCurrentTime */


void _specc::fatalError(		/* fatal error, abort simulation */
	const char	*Msg,
	int		ErrorNo)
{

fprintf(stderr,
	"\n"
	"libsim: FATAL ERROR %d: %s!\n"
	"libsim: Aborted.\n",
	ErrorNo, Msg);

exit(RESULT_FATAL);

} /* end of _specc::fatalError */


	/*********************/
	/*** _specc::event ***/
	/*********************/


_specc::event::event(void)	/* constructor #1 */
{

_specc::event::Notified		= false;
_specc::event::Triggered	= false;
_specc::event::Next		= NULL;

} /* end of _specc::event::event */


_specc::event::~event(void)	/* destructor */
{

/* nothing to do */

} /* end of _specc::event::~event */


void _specc::event::Reset(	/* reset the list of events */
	void)
{

if (Next)
   { Next->Reset();	/* use recursion */
    } /* fi */

Notified = false;
Next = NULL;

} /* end of _specc::event::Reset */


	/*************************/
	/*** _specc::event_ptr ***/
	/*************************/


_specc::event_ptr::event_ptr(	/* constructor #1 */
	event	*Event)
{

_specc::event_ptr::Next		= NULL;
_specc::event_ptr::Event	= Event;

} /* end of _specc::event_ptr::event_ptr */


_specc::event_ptr::~event_ptr(void)	/* destructor */
{

delete Next;

} /* end of _specc::event_ptr::~event_ptr */


	/**************************/
	/*** _specc::event_list ***/
	/**************************/


_specc::event_list::event_list(		/* constructor #1 */
	event_ptr	*EventList)
{

_specc::event_list::Next	= NULL;
_specc::event_list::EventList	= EventList;

} /* end of _specc::event_list::event_list */


_specc::event_list::~event_list(void)	/* destructor */
{

delete Next;
delete EventList;

} /* end of _specc::event_list::~event_list */


bool _specc::event_list::Triggers(	/* check whether any event matches */
	event_ptr	*EventList2)
{
event_ptr	*Event1,
		*Event2;
bool		Result;

if ((!this) || (!EventList2))
   { return(false);
    } /* fi */

Event1 = EventList;	/* set flags up */
while(Event1)
   { assert(Event1->Event->Triggered == false);	/* must be unused */
     Event1->Event->Triggered = true;
     Event1 = Event1->Next;
    } /* elihw */

Result = false;
Event2 = EventList2;	/* check for match */
while(Event2)
   { if (Event2->Event->Triggered)
	{ Result = true;
	  break;
	 } /* fi */
     Event2 = Event2->Next;
    } /* elihw */

Event1 = EventList;	/* reset the flags */
while(Event1)
   { Event1->Event->Triggered = false;
     Event1 = Event1->Next;
    } /* elihw */

return(Result);

} /* end of _specc::event_list::Triggers */


	/************************/
	/*** _specc::behavior ***/
	/************************/


_specc::behavior::behavior(	/* constructor #1 */
	void)
{

/* nothing to do */

} /* end of _specc::behavior::behavior */


_specc::behavior::~behavior(	/* destructor #1 */
	void)
{

/* nothing to do */

} /* end of _specc::behavior::~behavior */


	/***********************/
	/*** _specc::channel ***/
	/***********************/


_specc::channel::channel(	/* constructor #1 */
	void)
{

/* nothing to do */

} /* end of _specc::channel::channel */


_specc::channel::~channel(	/* destructor */
	void)
{

/* nothing to do */

} /* end of _specc::channel::~channel */


	/********************/
	/*** _specc::fork ***/
	/********************/


_specc::fork::fork(		/* constructor #1 */
	behavior	*Behavior)
{

assert(Behavior != NULL);

_specc::fork::Behavior	= Behavior;

} /* end of _specc::fork::fork */


_specc::fork::~fork(void)	/* destructor */
{

/* nothing to do */

} /* end of _specc::fork::~fork */


	/*************************/
	/*** _specc::try_block ***/
	/*************************/


_specc::try_block::try_block(	/* constructor #1 */
	behavior	*Behavior)
{

_specc::try_block::Behavior	= Behavior;

} /* end of _specc::try_block::try_block */


_specc::try_block::~try_block(		/* destructor */
	void)
{

/* nothing to do */

} /* end of _specc::try_block::~try_block */


	/*******************************/
	/*** _specc::exception_block ***/
	/*******************************/


_specc::exception_block::exception_block(	/* constructor #1 */
	bool		IsTrap,
	behavior	*Behavior,
	event		*First,
	...)
{
va_list		Args;
event		*Event;
event_ptr	**EventPtrPtr;

// note: Behavior may be NULL indicating an empty exception handler

_specc::exception_block::IsTrap		= IsTrap;
_specc::exception_block::Behavior	= Behavior;
_specc::exception_block::EventList	= NULL;

va_start(Args, First);		/* copy event list */
Event = First;
EventPtrPtr = &EventList;
while(Event)
   { if (!(*EventPtrPtr = new _specc::event_ptr(Event)))
	{ fatalError("Cannot build event list", ENOMEM);
	 } /* fi */
     EventPtrPtr = &(*EventPtrPtr)->Next;
     Event = va_arg(Args, event*);
    } /* elihw */
va_end(Args);

} /* end of _specc::exception_block::exception_block */


_specc::exception_block::~exception_block(	/* destructor */
	void)
{

delete EventList;

} /* end of _specc::exception_block::~exception_block */


	/*************************/
	/*** _specc::exception ***/
	/*************************/


_specc::exception::exception(		/* constructor #1 */
	exception_block	*ExceptionBlock)
{

assert(ExceptionBlock != NULL);

_specc::exception::Next		= NULL;
_specc::exception::IsTrap	= ExceptionBlock->IsTrap;
_specc::exception::Behavior	= ExceptionBlock->Behavior;
_specc::exception::EventList	= ExceptionBlock->EventList;

ExceptionBlock->EventList = NULL;	/* we take over this list */

} /* end of _specc::exception::exception */


_specc::exception::~exception(		/* destructor */
	void)
{

delete Next;
delete EventList;

} /* end of _specc::exception::~exception */


_specc::exception *_specc::exception::FirstMatch( /* obtain first match (#1) */
	event_ptr	*EventList2)		/* (or NULL if not found) */
{
event_ptr	*Event1,
		*Event2;
exception	*Exception,
		*Result;

if ((!this) || (!EventList2))
   { return(NULL);
    } /* fi */

Event2 = EventList2;	/* set flags up */
while(Event2)
   { assert(Event2->Event->Triggered == false);	/* must be unused */
     Event2->Event->Triggered = true;
     Event2 = Event2->Next;
    } /* elihw */

Result = NULL;		/* search for match */
Exception = this;
while(Exception)
   { Event1 = Exception->EventList;
     while(Event1)
	{ if (Event1->Event->Triggered)
	     { Result = Exception;
	       break;
	      } /* fi */
	  Event1 = Event1->Next;
	 } /* elihw */
     if (Result)
	{ break;
	 } /* fi */
     Exception = Exception->Next;
    } /* elihw */

Event2 = EventList2;	/* reset the flags */
while(Event2)
   { Event2->Event->Triggered = false;
     Event2 = Event2->Next;
    } /* elihw */

return(Result);

} /* end of _specc::exception::FirstMatch #1 */


_specc::exception *_specc::exception::FirstMatch( /* obtain first match (#2) */
	void)					/* (or NULL if not found) */
{
event_ptr	*Event;
exception	*Exception;

Exception = this;	/* search for match */
while(Exception)
   { Event = Exception->EventList;
     while(Event)
	{ if (Event->Event->Notified)
	     { return(Exception);
	      } /* fi */
	  Event = Event->Next;
	 } /* elihw */
     Exception = Exception->Next;
    } /* elihw */

return(NULL);

} /* end of _specc::exception_block::FirstMatch #2 */


	/**************************/
	/*** _specc::piped_base ***/
	/**************************/


_specc::piped_base::piped_base(void)	/* constructor #1 */
{

/* nothing to do */

} /* end of _specc::piped_base::piped_base */


_specc::piped_base::~piped_base(	/* destructor */
	void)
{

/* nothing to do */

} /* end of _specc::piped_base::~piped_base */


	/******************************/
	/*** _specc::piped template ***/
	/******************************/


/* (all inline functions, see header file "piped.h") */


/************************************************************************/
/*** implementation of exported functions			      ***/
/************************************************************************/


	/* (none) */


/* EOF specc.cc */
