/************************************************************************/
/* Note.cc: SpecC Internal Representation, Note Classes			*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/08/97 */
/************************************************************************/

/* last update: 05/31/01 */

/* modifications: (most recent first)
 *
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 01/30/01 RD	fixed a potential FMR problem in iterators
 *		SIR_Notes::DFS_ForAllXxxxx
 */

#include "IntRep/Note.h"
#include "IntRep/Symbol.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* (none) */


/*** internal type declarations *****************************************/


	/* (none) */


/*** class implementations **********************************************/


	/****************/
	/*** SIR_Note ***/
	/****************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Note::SIR_Note(			/* constructor #1 */
	const char	*Name,
	sir_constant	*Content,
	sir_symbol	*Symbol = NULL,		/* max. one */
	sir_usertype	*UserType = NULL,	/* of these */
	sir_label	*Label = NULL)		/* not NULL */
{

assert(Name != NULL);
assert(*Name != 0);
assert(Content != NULL);
assert((Symbol == NULL) || ((UserType == NULL) && (Label == NULL)));
assert((UserType == NULL) || ((Symbol == NULL) && (Label == NULL)));
assert((Label == NULL) || ((Symbol == NULL) && (UserType == NULL)));

SIR_Note::Name		= Name;
SIR_Note::Content	= Content;
SIR_Note::Symbol	= Symbol;
SIR_Note::UserType	= UserType;
SIR_Note::Label		= Label;
SIR_Note::Imported	= NULL;

} /* end of SIR_Note::SIR_Note #1 */


SIR_Note::SIR_Note(			/* constructor #3 (duplicator) */
	sir_note	*Original)
{

SIR_Note::Name		= Original->Name;
SIR_Note::Content	= Original->Content ?
				new SIR_Constant(Original->Content) :
				NULL;
SIR_Note::Symbol	= Original->Symbol;
SIR_Note::UserType	= Original->UserType;
SIR_Note::Label		= Original->Label;
SIR_Note::Imported	= Original->Imported;

} /* end of SIR_Note::SIR_Note #3 */


SIR_Note::~SIR_Note(void)		/* destructor */
{

delete Content;

} /* end of SIR_Note::~SIR_Note */


ERROR SIR_Note::DFS_ForAllNodes(	/* iterator for over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

/* this is not a node, but there are nodes below */

if (Content)
   { if ((SIR_Error = Content->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Note::DFS_ForAllNodes */


ERROR SIR_Note::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

return((this->*MemberFct)(MemberFctArg));

} /* end of SIR_Note::DFS_ForAllNotes */


ERROR SIR_Note::SetImported(	/* sets the Imported pointer (if unset) */
	sir_note_marg	ImportPtr)	/* or resets it if NULL */
{

if (ImportPtr == NULL)
   { Imported = NULL;
     return(SIR_ERROR_NO_ERROR);
    } /* fi */

if (! Imported)		/* import pointer still unset? */
   { Imported = (sir_import*) ImportPtr;
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Note::SetImported */


ERROR SIR_Note::TakeOverImport(		/* takes over an imported note */
	sir_note_marg	ImportPtr)
{

assert(ImportPtr != NULL);
if (Imported == (sir_import*) ImportPtr)
   { Imported = NULL;
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Note::TakeOverImport */


ERROR SIR_Note::MarkImportEntry(	/* marks its import entry being used */
	sir_note_marg	/* Unused */)
{

if (Imported)
   { Imported->Color = SIR_RED;	/* mark as being used */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Note::MarkImportEntry */


ERROR SIR_Note::RemoveDoubleImport(	/* deletes itself if double-imported */
	sir_note_marg	/* Unused */)
{

if (  (Imported)		/* is imported? */
    &&(Imported->Alias))	/* is double-imported? */
   { Remove();		/* remove myself from the list */
     delete this;	/* and commit suicide! */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Note::RemoveDoubleImport */


const char *SIR_Note::Print(		/* print ASCII definition */
	string		*Buffer,	/* (Buffer must be initialized) */
	BOOL		CplusplusMode = FALSE)
{

assert(Buffer != NULL);

if (CplusplusMode)
   { *Buffer += "/* ";
    } /* fi */

*Buffer += "note ";

if (Symbol)
   { *Buffer += Symbol->Name;
     *Buffer += '.';
    } /* fi */
if (  (UserType)
    &&(UserType->Name))
   { *Buffer += *UserType->Name;
     *Buffer += '.';
    } /* fi */
if (Label)
   { *Buffer += Label->LabelName;
     *Buffer += '.';
    } /* fi */

*Buffer += Name;
*Buffer += " = ";
*Buffer += Content->Print();
*Buffer += ';';

if (CplusplusMode)
   { *Buffer += " */";
    } /* fi */

return(Buffer->chars());

} /* end of SIR_Note::Print */


ERROR SIR_Note::WriteSC(		/* (re-) generates SpecC source code */
	FILE		*File,
	BOOL		CplusplusMode = FALSE)
{

assert(Content != NULL);

if (Content->LineInfo)
   { if ((SIR_Error = Content->LineInfo->WriteSC(File, true)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { SIR_LineInfo::WriteNewSC(File, true);
    } /* esle */

if (CplusplusMode)
   { fputs("/* ", File);	/* in C++ mode include notes as comments */
    } /* fi */
fputs("note ", File);

if (Symbol)
   { fputs(Symbol->Name.chars(), File);
     fputc('.', File);
    } /* fi */
if (  (UserType)
    &&(UserType->Name))
   { fputs(UserType->Name->chars(), File);
     fputc('.', File);
    } /* fi */
if (Label)
   { fputs(Label->LabelName.chars(), File);
     fputc('.', File);
    } /* fi */

fputs(Name.chars(), File);
fputs(" = ", File);
fputs(Content->Print(), File);
fputc(';', File);

if (CplusplusMode)
   { fputs(" */", File);
    } /* fi */

if (ferror(File))
   { SIR_ErrMsg.form("Writing to file failed%s",
			GL_SystemErrorMessage());
     return(SIR_ERROR_WRITING_SPECC_FILE);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Note::WriteSC */


void SIR_Note::Touch(		/* touch it (remove source information) */
	sir_design	*Design)
{

assert(Design != NULL);

if (Imported)		/* remove import information */
   { Imported->TakeOverAndRemove(Design);
    } /* fi */

if (Content)
   { Content->Strip();	/* remove source code info */
    } /* fi */

} /* end of SIR_Note::Touch */


	/*****************/
	/*** SIR_Notes ***/
	/*****************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Notes::SIR_Notes(			/* constructor #1 */
	sir_note	*FirstEntry = NULL) :
		SIR_List<SIR_Note>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_Notes::SIR_Notes #1 */


SIR_Notes::SIR_Notes(			/* constructor #2 (duplicator) */
	sir_notes	*Original)
{
sir_note	*Note;

Note = Original->First();
while(Note)
   { Append(new SIR_Note(Note));
     Note = Note->Succ();
    } /* elihw */

} /* end of SIR_Notes::SIR_Notes #2 */


SIR_Notes::~SIR_Notes(void)		/* destructor */
{

/* nothing to do */

} /* end of SIR_Notes::~SIR_Notes */


ERROR SIR_Notes::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_note	*Note,
		*Succ;

/* this is not a node, but there are nodes below */

Note = First();
while(Note)
   { Succ = Note->Succ();
     if ((SIR_Error = Note->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Note = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Notes::DFS_ForAllNodes */


ERROR SIR_Notes::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{
sir_note	*Note,
		*Succ;

Note = First();
while(Note)
   { Succ = Note->Succ();
     if ((SIR_Error = Note->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Note = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Notes::DFS_ForAllNotes */


sir_note *SIR_Notes::Find(	/* find an entry */
	const char	*Name)	/* (returns NULL if not found) */
{
int		CmpVal;

First();
while(Curr())
   { if (0 == (CmpVal = strcmp(Name, Curr()->Name.chars())))
	{ return(Curr());
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_Notes::Find */


sir_note *SIR_Notes::Insert(		/* inserts a new note */
	sir_note	*NewNote)	/* (may return NULL and SIR_Error) */
{
int		CmpVal;

First();
while(Curr())
   { if (0 == (CmpVal = strcmp(NewNote->Name.chars(), Curr()->Name.chars())))
	{ if (NewNote->Symbol)
	     { SIR_ErrMsg.form("Annotation '%s' at symbol '%s' specified twice",
				NewNote->Name.chars(),
				NewNote->Symbol->Name.chars());
	      } /* fi */
	  else
	     { if (NewNote->UserType)
		  { SIR_ErrMsg.form(
				"Annotation '%s' at %s '%s' specified twice",
				NewNote->Name.chars(),
				NewNote->UserType->ClassName(),
				NewNote->UserType->Name->chars());
		   } /* fi */
	       else
		  { if (NewNote->Label)
		       { SIR_ErrMsg.form(
				"Annotation '%s' at label '%s' specified twice",
				NewNote->Name.chars(),
				NewNote->Label->LabelName.chars());
			} /* fi */
		    else
		       { SIR_ErrMsg.form(
				"Annotation '%s' at top level specified twice",
				NewNote->Name.chars());
			} /* esle */
		   } /* esle */
	      } /* esle */
	  SIR_Error = SIR_ERROR_NOTE_SPECIFIED_TWICE;
	  return(NULL);
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

if (Curr())
   { return(InsertBefore(NewNote));
    } /* fi */
else
   { return(Append(NewNote));
    } /* else */

} /* end of SIR_Notes::Insert */


ERROR SIR_Notes::Merge(			/* merges two note-tables */
	sir_notes	**Notes1,	/* (merges 2 into 1) */
	sir_notes	**Notes2)	/* (supports automatic unaliasing) */
{
sir_note	*Curr,
		*Succ;

assert(Notes1 != NULL);
assert(Notes2 != NULL);

if (*Notes1)
   { if (*Notes2)
	{ Curr = (*Notes2)->First();
	  while(Curr)
	     { Succ = Curr->Succ();
	       if (!((*Notes1)->Insert((*Notes2)->Remove(Curr))))
		  { SIR_ErrMsg.prepend(
			"Error while importing:" GL_ERROR_MSG_NEWLINE);
		    delete Curr;
		    return(SIR_ERROR_NOTE_SPECIFIED_TWICE_2);
		   } /* fi */
	       if (  (Curr->Symbol)
		   &&(Curr->Symbol->Alias))
		  { Curr->Symbol = Curr->Symbol->Alias;
		   } /* fi */
	       if (  (Curr->UserType)
		   &&(Curr->UserType->Alias))
		  { Curr->UserType = Curr->UserType->Alias;
		   } /* fi */
	       Curr = Succ;
	      } /* elihw */
	  delete (*Notes2);
	  *Notes2 = NULL;
	 } /* fi */
     else
	{ /* nothing to do */
	 } /* esle */
    } /* fi */
else
   { if (*Notes2)
	{ *Notes1 = *Notes2;	/* just cut and paste */
	  *Notes2 = NULL;
	 } /* fi */
     else
	{ /* nothing to do */
	 } /* esle */
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Notes::Merge */


const char *SIR_Notes::Print(		/* print ASCII definitions */
	BOOL		DoIt,
	BOOL		CplusplusMode = FALSE)
{
static string	Buffer;
sir_note	*Note;

if (! DoIt)
   { Buffer = "";
     return(Buffer.chars());
    } /* fi */

Buffer = "";
Note = First();
while(Note)
   { Note->Print(&Buffer, CplusplusMode);
     Note = Note->Succ();
     if (Note)
	{ Buffer += ' ';
	 } /* fi */
    } /* elihw */

return(Buffer.chars());

} /* end of SIR_Notes::Print */


ERROR SIR_Notes::WriteSC(		/* (re-) generates SpecC source code */
	FILE		*File,
	BOOL		DoIt,
	BOOL		CplusplusMode = FALSE)
{
sir_note	*Note;

if (! DoIt)
   { return(SIR_ERROR_NO_ERROR);
    } /* fi */

Note = First();
while(Note)
   { if ((SIR_Error = Note->WriteSC(File, CplusplusMode)))
	{ return(SIR_Error);
	 } /* fi */
     Note = Note->Succ();
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Notes::WriteSC */


/* EOF Note.cc */
