/************************************************************************/
/* IntRep/Statement.h: SpecC Internal Representation, Statement Class	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/20/97 */
/************************************************************************/

/* last update: 05/31/01 */

/* modifications: (most recent first)
 *
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/05/01 RD	added support for new 'pipe' construct (with flushing)
 * 03/23/01 RD	added SIR_Statement::WrapInCompound() and
 *		SIR_Statement::CompoundSubBlocks()
 * 03/23/01 RD	started this header (last change was 04/13/99)
 */

#ifndef INTREP_STATEMENT_H
#define INTREP_STATEMENT_H


#include "Global.h"
#include "IntRep/Exception.h"
#include "IntRep/Constraint.h"
#include "IntRep/Transition.h"


/*** enumeration types **************************************************/


enum SIR_StatementType		/* statement types */
{				/* (see class SIR_Statement) */

SIR_STMNT_LABELED,		// Create() #3
SIR_STMNT_COMPOUND,		// Create() #1, no expression
SIR_STMNT_EXPRESSION,		// Create() #1, 1 expression
SIR_STMNT_IF,			// Create() #1, 1 expression
SIR_STMNT_IF_ELSE,		// Create() #1, 1 expression
SIR_STMNT_SWITCH,		// Create() #1, 1 expression
SIR_STMNT_CASE,			// Create() #2
SIR_STMNT_DEFAULT,		// Create() #1, no expression
SIR_STMNT_WHILE,		// Create() #1, 1 expression
SIR_STMNT_DO_WHILE,		// Create() #1, 1 expression
SIR_STMNT_FOR,			// Create() #1, 3 expressions
SIR_STMNT_GOTO,			// Create() #4
SIR_STMNT_CONTINUE,		// Create() #1, no expression
SIR_STMNT_BREAK,		// Create() #1, no expression
SIR_STMNT_RETURN,		// Create() #1, 1 expression
SIR_STMNT_PAR,			// Create() #1, no expression
SIR_STMNT_PIPE,			// Create() #1, no expression
SIR_STMNT_EXCEPTION,		// Create() #6
SIR_STMNT_TIMING,		// Create() #1, no expression
SIR_STMNT_FSM,			// Create() #1, no expression
SIR_STMNT_WAIT,			// Create() #5
SIR_STMNT_WAITFOR,		// Create() #1, 1 expression
SIR_STMNT_NOTIFY,		// Create() #5
SIR_STMNT_NOTIFYONE		// Create() #5
};

enum SIR_Dependency		/* type of dependency among symbols */
{
SIR_DEP_NONE		= 0,	/* independent */
SIR_DEP_INSTANTIATION	= 1,	/* instantiated by symbol */
SIR_DEP_IMPLEMENTATION	= 2,	/* implemented by symbol */
SIR_DEP_CONNECTION	= 3,	/* connected with (mapped to) symbol */
SIR_DEP_STATEMENT	= 4,	/* used in statement, symbol */
SIR_DEP_EXPRESSION	= 5	/* used in expression, statement, symbol */
};


/*** type definitions ***************************************************/


typedef enum SIR_StatementType		SIR_STMNT_TYPE;
typedef enum SIR_Dependency		SIR_DEPENDENCY;

typedef class SIR_Statement		sir_statement;
typedef SIR_List<sir_statement>		sir_stmnt_list;
typedef class SIR_Statements		sir_statements;
typedef void *				sir_stmnt_marg;
typedef ERROR (SIR_Statement::*sir_stmnt_mptr)(void*);

typedef class SIR_Symbol		sir_symbol;	/* cyclic link */
typedef class SIR_Symbols		sir_symbols;	/* cyclic link */
typedef class SIR_SymbolPtrs		sir_symbol_ptrs;/* cyclic link */
typedef class SIR_SymbolPtrs		sir_events;	/* cyclic link */
typedef void *				sir_symbol_marg;/* cyclic link */
typedef ERROR (SIR_Symbol::*sir_symbol_mptr)(void*);	/* cyclic link */

typedef class SIR_UserType		sir_usertype;	/* cyclic link */
typedef void *				sir_usertp_marg;/* cyclic link */
typedef ERROR (SIR_UserType::*sir_usertp_mptr)(void*);	/* cyclic link*/


/*** class declarations *************************************************/


	/*********************/
	/*** SIR_Statement ***/
	/*********************/


class SIR_Statement :			/* statement */
	public SIR_Node,			/* is a node */
	public SIR_ListElem<SIR_Statement>	/* and a list element */
{
public:
SIR_STMNT_TYPE	StmntType;	/* statement type */
sir_symbol	*Function;	/* link back to the function this is part of */
sir_expression	*Expression1;	/* first expression (or NULL) */
sir_expression	*Expression2;	/* second expression (or NULL) */
sir_expression	*Expression3;	/* third expression (or NULL) */
sir_constant	*Constant;	/* constant (or NULL) */
sir_statement	*Statement1;	/* first sub-statement (NULL if none) */
sir_statement	*Statement2;	/* second sub-statement (NULL if none) */
sir_label	*Label;		/* pointer to label (NULL if unlabeled) */
sir_symbols	*Scope;		/* local scope (or NULL) */
sir_statements	*Statements;	/* list of statements (or NULL) */
sir_symbol_ptrs	*Events;	/* list of events (or NULL) */
sir_exceptions	*Exceptions;	/* exceptions (or NULL) */
sir_constraints	*Constraints;	/* constraints (or NULL) */
sir_transitions	*Transitions;	/* transitions (or NULL) */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Statement(				/* constructor #1 */
	SIR_STMNT_TYPE	StmntType,	/* (for general statements) */
	sir_symbol	*Function,
	sir_expression	*Expression1 = NULL,
	sir_statement	*Statement1 = NULL,
	sir_statement	*Statement2 = NULL,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL,
	sir_expression	*Expression2 = NULL,
	sir_expression	*Expression3 = NULL);

SIR_Statement(				/* constructor #2 */
	SIR_STMNT_TYPE	StmntType,	/* (for compound statements) */
	sir_symbol	*Function,
	sir_symbols	*Scope,
	sir_statements	*Statements,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Statement(				/* constructor #3 */
	SIR_STMNT_TYPE	StmntType,	/* (for case statements) */
	sir_symbol	*Function,
	sir_constant	*Constant,
	sir_statement	*Statement1 = NULL,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Statement(				/* constructor #4 */
	SIR_STMNT_TYPE	StmntType,	/* (for statements with labels) */
	sir_symbol	*Function,
	sir_label	*Label,
	sir_statement	*Statement1 = NULL,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Statement(				/* constructor #5 */
	SIR_STMNT_TYPE	StmntType,	/* (for statements with events) */
	sir_symbol	*Function,
	sir_symbol_ptrs	*Events,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Statement(				/* constructor #6 */
	SIR_STMNT_TYPE	StmntType,	/* (for exception statements) */
	sir_symbol	*Function,
	sir_statement	*Statement1,
	sir_exceptions	*Exceptions,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Statement(				/* constructor #7 */
	SIR_STMNT_TYPE	StmntType,	/* (for timing statement) */
	sir_symbol	*Function,
	sir_statement	*Statement1,
	sir_constraints	*Constraints,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Statement(				/* constructor #8 */
	SIR_STMNT_TYPE	StmntType,	/* (for fsm statement) */
	sir_symbol	*Function,
	sir_transitions	*Transitions,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL);

SIR_Statement(				/* constructor #10 (duplicator) */
	sir_statement	*Original);

~SIR_Statement(void);			/* destructor */


BOOL IsCleanBehaviorCall(void);		/* checks for "clean" behavior call */
					/* (e.g. "b1.main();") */

BOOL IsCleanListOfBehaviorCalls(	/* checks for "clean" comp. behavior */
	BOOL	OneCallMax = FALSE,	/* (e.g. "{ b1.main(); b2.main(); }") */
	BOOL	OneCallMin = TRUE);

ERROR CheckCompoundBehavior(		/* checks for "clean" comp. behavior */
	BOOL	OneCallMax = FALSE,	/* (e.g. "{ b1.main(); b2.main(); }") */
	BOOL	OneCallMin = TRUE);	/* (returns error if not clean) */

BOOL IsDirectDependant(			/* checks if stmnt. depends on symbol */
	sir_symbol	*ThatSymbol);	/* (directly) */

BOOL IsIndirectDependant(		/* checks if stmnt. depends on symbol */
	sir_symbol	*ThatSymbol,	/* (indirectly) */
	sir_expression	**DepExpr = NULL);


sir_statements *GetList(void);	/* determines the list of this statement */
				/* (returns NULL if not in a list) */


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);

ERROR DFS_ForAllSymbols(	/* iterator over all symbols (depth first) */
	sir_symbol_mptr	MemberFct,
	sir_symbol_marg	MemberFctArg);

ERROR DFS_ForAllUserTypes(	/* iterator over all usertypes (depth first) */
	sir_usertp_mptr	MemberFct,
	sir_usertp_marg	MemberFctArg);

ERROR DFS_ForAllNotes(		/* iterator over all notes (depth first) */
	sir_note_mptr	MemberFct,
	sir_note_marg	MemberFctArg);

ERROR DFS_ForAllStatements(	/* iterator over all statements (depth first) */
	sir_stmnt_mptr	MemberFct,
	sir_stmnt_marg	MemberFctArg);

ERROR DFS_ForAllExpressions(	/* iterator over all expressions (dfs) */
	sir_expr_mptr	MemberFct,
	sir_expr_marg	MemberFctArg);

BOOL DFS_FindDependant(		/* searches for dependants (depth first) */
	sir_symbol	*ThatSymbol,
	sir_statement	**DepStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason);


ERROR MarkYourselfAndYourLabel(	/* iterator for color marking */
	sir_stmnt_marg	ColorArg);

ERROR CheckRequiredLabel(	/* iterator for label checking */
	sir_stmnt_marg	/* Unused */);

ERROR CheckLabeledStmnt(	/* iterator for label checking */
	sir_stmnt_marg	/* Unused */);

ERROR CheckReturnType(		/* iterator for return type checking */
	sir_stmnt_marg	TypeArg);

ERROR NeedsRangeChecker(	/* iterator for range checking */
	sir_stmnt_marg	/* Unused */);


void SetAlias(			/* sets all type, usertype, symbol alias' */
	sir_statement	*Alias);/* (iterates over symbols and usertypes) */

void UnAlias(			/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols);


ERROR WritePipeStmnt(		/* generates C++ code for 'pipe' construct */
	FILE		*File,
	BOOL		WriteNotes);

ERROR WriteSC(			/* (re-) generates SpecC source code */
	FILE		*File,
	BOOL		WriteNotes,
	BOOL		CplusplusMode = FALSE,
	BOOL		PutNewLine = TRUE,
	BOOL		PutArgNotes = TRUE);
};


	/**********************/
	/*** SIR_Statements ***/
	/**********************/


class SIR_Statements :		/* sequential statements */
	public SIR_List<SIR_Statement>	/* is simply a list of statements */
{					/* with additional member(function)s */
public:
sir_symbols	*CmpndScope;	/* link to compound statement scope */


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Statements(			/* constructor #1 */
	sir_symbols	*CmpndScope = NULL,
	sir_statement	*FirstEntry = NULL);

SIR_Statements(			/* constructor #3 (duplicator) */
	sir_statements	*Original);

~SIR_Statements(void);		/* destructor */


ERROR DFS_ForAllNodes(		/* iterator over all nodes (depth first) */
	sir_node_mptr	MemberFct,
	sir_node_marg	MemberFctArg);

ERROR DFS_ForAllSymbols(	/* iterator over all symbols (depth first) */
	sir_symbol_mptr	MemberFct,
	sir_symbol_marg	MemberFctArg);

ERROR DFS_ForAllUserTypes(	/* iterator over all usertypes (depth first) */
	sir_usertp_mptr	MemberFct,
	sir_usertp_marg	MemberFctArg);

ERROR DFS_ForAllNotes(		/* iterator over all notes (depth first) */
	sir_note_mptr	MemberFct,
	sir_note_marg	MemberFctArg);

ERROR DFS_ForAllStatements(	/* iterator over all statements (depth first) */
	sir_stmnt_mptr	MemberFct,
	sir_stmnt_marg	MemberFctArg);

ERROR DFS_ForAllExpressions(	/* iterator over all expressions (dfs) */
	sir_expr_mptr	MemberFct,
	sir_expr_marg	MemberFctArg);

BOOL DFS_FindDependant(		/* searches for dependants (depth first) */
	sir_symbol	*ThatSymbol,
	sir_statement	**DepStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason);


void SetAlias(			/* sets all type, usertype, symbol alias' */
	sir_statements	*Alias);/* (iterates over symbols and usertypes) */


sir_statement *FindLabeledStmnt(/* find a labeled statement in this list */
	sir_label	*Label);	/* (returns NULL if not found) */
};


/*** exported functions *************************************************/


	/* none */


#endif /* INTREP_STATEMENT_H */

/* EOF IntRep/Statement.h */
