/************************************************************************/
/* Expression.cc: SpecC Internal Representation, Expression Class	*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 09/24/97 */
/************************************************************************/

/* last update: 05/31/01 */

/* modifications: (most recent first)
 *
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/27/01 RD	added semantic checking for port accesses, etc.
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/25/01 RD	removed code not needed for the SCRC
 * 05/22/01 RD	added SIR_Expression::FixTypePtr()
 * 05/21/01 RD	added support for (native, non-native) array assignment
 * 05/05/01 RD	added explicit type casts for bitvectors and longlongs
 *		used with question mark operator
 * 04/30/01 RD	replaced sir_bitop::resultLen() and sir_bitop::resultUnsigned()
 *		with new macros _BITLEN_*() and _BITUSGN_*()
 * 04/30/01 RD	adjusted SIR_BIT_SLICE_NAME to changes in bitlib
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/26/01 RD	added explicit promotion to (long) double for class types
 * 04/25/01 RD	fixed a false assertion about promotion in IntegerEval()
 * 04/25/01 RD	added (renamed) CastNeedsExplicitTypeConv()
 * 04/24/01 RD	extended SIR_Expression::ExplicitTypeConv() for floats;
 *		added ExpectedLength argument to SIR_Expression::WriteSC[2];
 *		added non-native long long support for SIR_RELAXED() macro
 * 04/24/01 RD	fixed type cast with non-native long long
 * 04/23/01 RD	fixed the condition around use of ExplicitTypeConv()
 * 04/20/01 RD	added SIR_Expression::ExplicitTypeConv()
 * 04/17/01 RD	adjusted a couple of issues for non-native 'long long'
 * 04/17/01 RD	added missing length argument to Converted() functions
 * 04/16/01 RD	added support for non-native 'long long' type;
 *		changed use of 'long long int' to TIME type
 * 01/30/01 RD	fixed a potential FMR problem in iterators
 *		SIR_Expressions::DFS_ForAllXxxxx
 */

#include "IntRep/Expression.h"
#include "IntRep/Extern.h"
#include "IntRep/Symbol.h"

#include <stdlib.h>
#include <assert.h>


/*** constants and macros ***********************************************/


	/* constants for interface to bitvectors */

#define SIR_BITOP_UNARY_OP	'~'
#define SIR_BITOP_BINARY_OP	'|'
#define SIR_BITOP_CONCATENATION	'@'
#define SIR_BITOP_BITSLICE	'['


	/* macro for 'relaxed' type conversion (assignments) */

#ifdef HAVE_LLONG
#define SIR_RELAXED(type, from)					\
	( (  ((type) == SIR_TYPE_BIT)				\
	   ||((type) == SIR_TYPE_UBIT))				\
	 ? (SIR_TYPE_ANY_TYPE) : (type))
#else /* !HAVE_LLONG */
#define SIR_RELAXED(type, from)					\
	(  (  ((type) == SIR_TYPE_BIT)				\
	    ||((type) == SIR_TYPE_UBIT))			\
	 ||(  (  ((type) == SIR_TYPE_LONGLONG)			\
	       ||((type) == SIR_TYPE_ULONGLONG))		\
	    &&(  ((from) != SIR_TYPE_BIT)			\
	       &&((from) != SIR_TYPE_UBIT)))			\
	 ? (SIR_TYPE_ANY_TYPE) : (type))
#endif /* HAVE_LLONG */


	/* macro for explicit promotion to (long) double */

#ifdef HAVE_LLONG
#define DOUBLE_EXPECTED(expected, current)			\
	((  (  ((expected) == SIR_TYPE_DOUBLE)			\
	     ||((expected) == SIR_TYPE_LONGDOUBLE))		\
	  &&(  ((current) == SIR_TYPE_BIT)			\
	     ||((current) == SIR_TYPE_UBIT))) ?			\
	(expected) : SIR_TYPE_ANY_TYPE)
#else /* !HAVE_LLONG */
#define DOUBLE_EXPECTED(expected, current)			\
	((  (  ((expected) == SIR_TYPE_DOUBLE)			\
	     ||((expected) == SIR_TYPE_LONGDOUBLE))		\
	  &&(  ((current) == SIR_TYPE_LONGLONG)			\
	     ||((current) == SIR_TYPE_ULONGLONG)		\
	     ||((current) == SIR_TYPE_BIT)			\
	     ||((current) == SIR_TYPE_UBIT))) ?			\
	(expected) : SIR_TYPE_ANY_TYPE)
#endif /* HAVE_LLONG */


/*** internal type declarations *****************************************/


enum SIR_WhatToDo
{
SIR_DO_LL_COMPUTATION,
SIR_DO_ULL_COMPUTATION,
SIR_DO_LD_COMPUTATION,
SIR_DO_BIT_COMPUTATION,
SIR_DO_ERROR
};

enum SIR_WhichArg
{
SIR_LEFT_ARG,
SIR_RIGHT_ARG
};

typedef enum SIR_WhatToDo		SIR_WHAT_TO_DO;
typedef enum SIR_WhichArg		SIR_WHICH_ARG;


/*** internal function prototypes ***************************************/


static SIR_WHAT_TO_DO SIR_PrepareOperand( /* 'ultra-promote' the constant */
	sir_expression		*Arg,	  /* (fills in all possible values) */
	LONG_LONG		*TmpValueLL,
	UNSIGNED_LONG_LONG	*TmpValueULL,
	long double		*TmpValueLD,
	sir_bit			**TmpValueBIT,
	unsigned int		*Line = NULL,
	sir_fileinfo		**FileInfo = NULL);

static SIR_WHAT_TO_DO SIR_PrepareBinaryOperands(
	sir_expression		*Arg1,
	sir_expression		*Arg2,
	LONG_LONG		*TmpValueLL1,
	UNSIGNED_LONG_LONG	*TmpValueULL1,
	long double		*TmpValueLD1,
	sir_bit			**TmpValueBIT1,
	LONG_LONG		*TmpValueLL2,
	UNSIGNED_LONG_LONG	*TmpValueULL2,
	long double		*TmpValueLD2,
	sir_bit			**TmpValueBIT2,
	unsigned int		*Line = NULL,
	sir_fileinfo		**FileInfo = NULL);

static bool SIR_IsConvertable2Bool(	/* checks for legal bool conversion */
	SIR_TYPETYPE	TypeType);	/* (supports all types except '...') */

static bool SIR_IsConvertable2BitVector(/* virtually creates a bitvector */
	SIR_TYPETYPE	OrigType,	/* (supports all types except '...') */
	int		OrigLeft,
	int		OrigRight,
	SIR_TYPETYPE	*ResultType = NULL,
	int		*ResultLen = NULL,
	int		*ResultLeft = NULL,
	int		*ResultRight = NULL);

static sir_constant *SIR_Convert2BitVector( /* converts Const to BIT or UBIT */
	sir_constant	*Const);

static void SIR_PutParenthesis(		/* put parenthesis if needed */
	SIR_EXPRTYPE	ExprType,
	char		Parenthesis,
	sir_expression	*Arg,
	SIR_WHICH_ARG	WhichArg,
	FILE		*File);

static BOOL CastNeedsExplicitTypeConv(	/* explicit type conv. needed? */
	SIR_TYPETYPE	FromType,
	SIR_TYPETYPE	ToType);


/*** internal variables *************************************************/


int	SIR_ExprPrecedence[] =	/* as defined in B. Stroustrup, 3rd edition,  */
{				/* "The C++ programming language", p.120-121) */
0,	// SIR_EXPR_VOID
99,	// SIR_EXPR_CONSTANT
99,	// SIR_EXPR_IDENTIFIER
99,	// SIR_EXPR_PARENTHESES
99,	// SIR_EXPR_THIS
16,	// SIR_EXPR_ARRAY_ACCESS
16,	// SIR_EXPR_FUNCTION_CALL
16,	// SIR_EXPR_MEMBER_ACCESS
16,	// SIR_EXPR_MEMBER_POINTER
16,	// SIR_EXPR_POST_INCREMENT
16,	// SIR_EXPR_POST_DECREMENT
16,	// SIR_EXPR_BITSLICE
15,	// SIR_EXPR_PRE_INCREMENT
15,	// SIR_EXPR_PRE_DECREMENT
15,	// SIR_EXPR_ADDRESS_OF
15,	// SIR_EXPR_CONTENT_OF
15,	// SIR_EXPR_POSITIVE
15,	// SIR_EXPR_NEGATIVE
15,	// SIR_EXPR_NOT
15,	// SIR_EXPR_LOGICAL_NOT
15,	// SIR_EXPR_SIZEOF_EXPR
15,	// SIR_EXPR_SIZEOF_TYPE
15,	// SIR_EXPR_TYPE_CONVERSION
14,	// SIR_EXPR_CONCATENATION
13,	// SIR_EXPR_MULTIPLY
13,	// SIR_EXPR_DIVIDE
13,	// SIR_EXPR_MODULO
12,	// SIR_EXPR_ADD
12,	// SIR_EXPR_SUBTRACT
11,	// SIR_EXPR_SHIFT_LEFT
11,	// SIR_EXPR_SHIFT_RIGHT
10,	// SIR_EXPR_LESS
10,	// SIR_EXPR_GREATER
10,	// SIR_EXPR_LESS_EQUAL
10,	// SIR_EXPR_GREATER_EQUAL
9,	// SIR_EXPR_EQUAL
9,	// SIR_EXPR_NOT_EQUAL
8,	// SIR_EXPR_AND
7,	// SIR_EXPR_EOR
6,	// SIR_EXPR_OR
5,	// SIR_EXPR_LOGICAL_AND
4,	// SIR_EXPR_LOGICAL_OR
2,	// SIR_EXPR_CONDITION
3,	// SIR_EXPR_ASSIGNMENT
3,	// SIR_EXPR_MUL_ASSIGN
3,	// SIR_EXPR_DIV_ASSIGN
3,	// SIR_EXPR_MOD_ASSIGN
3,	// SIR_EXPR_ADD_ASSIGN
3,	// SIR_EXPR_SUB_ASSIGN
3,	// SIR_EXPR_SHL_ASSIGN
3,	// SIR_EXPR_SHR_ASSIGN
3,	// SIR_EXPR_AND_ASSIGN
3,	// SIR_EXPR_EOR_ASSIGN
3,	// SIR_EXPR_OR_ASSIGN
1	// SIR_EXPR_COMMA
};


/*** class implementations **********************************************/


	/**********************/
	/*** SIR_Expression ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Expression::SIR_Expression(		/* constructor #1 (for constants) */
	SIR_EXPRTYPE	ExprType,
	sir_constant	*Constant,
	sir_type	*Type,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(ExprType == SIR_EXPR_CONSTANT);
assert(Constant != NULL);
assert(Type != NULL);
assert(  (  (Constant->Type != SIR_CONST_CHARSTRING)
	  &&((enum SIR_ConstType)Type->Type == Constant->Type))
       ||(  (Constant->Type == SIR_CONST_CHARSTRING)
	  &&(Type->Type == SIR_TYPE_POINTER)));

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= Constant;
SIR_Expression::Symbol		= NULL;
SIR_Expression::Arg1		= NULL;
SIR_Expression::Arg2		= NULL;
SIR_Expression::Arg3		= NULL;
SIR_Expression::Args		= NULL;
SIR_Expression::TypeArg		= NULL;
SIR_Expression::Type		= Type;
SIR_Expression::LeftBound	= 0;
SIR_Expression::RightBound	= 0;

} /* end of SIR_Expression::SIR_Expression #1 */


SIR_Expression::SIR_Expression(		/* constructor #2 (for identifiers) */
	SIR_EXPRTYPE	ExprType,
	sir_symbol	*Symbol,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(ExprType == SIR_EXPR_IDENTIFIER);
assert(Symbol != NULL);

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= NULL;
SIR_Expression::Symbol		= Symbol;
SIR_Expression::Arg1		= NULL;
SIR_Expression::Arg2		= NULL;
SIR_Expression::Arg3		= NULL;
SIR_Expression::Args		= NULL;
SIR_Expression::TypeArg		= NULL;
SIR_Expression::Type		= Symbol->Type;
SIR_Expression::LeftBound	= 0;
SIR_Expression::RightBound	= 0;

} /* end of SIR_Expression::SIR_Expression #2 */


SIR_Expression::SIR_Expression(		/* constructor #3 */
	SIR_EXPRTYPE	ExprType,	/* (for std. expr. with 0-3 args.) */
	sir_type	*Type,
	sir_expression	*Arg1 = NULL,
	sir_expression	*Arg2 = NULL,
	sir_expression	*Arg3 = NULL,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(Type != NULL);

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= NULL;
SIR_Expression::Symbol		= NULL;
SIR_Expression::Arg1		= Arg1;
SIR_Expression::Arg2		= Arg2;
SIR_Expression::Arg3		= Arg3;
SIR_Expression::Args		= NULL;
SIR_Expression::TypeArg		= NULL;
SIR_Expression::Type		= Type;
SIR_Expression::LeftBound	= 0;
SIR_Expression::RightBound	= 0;

} /* end of SIR_Expression::SIR_Expression #3 */


SIR_Expression::SIR_Expression(		/* constructor #4 */
	SIR_EXPRTYPE	ExprType,	/* (for function calls) */
	sir_type	*Type,
	sir_expression	*Arg1,
	sir_expressions	*Args,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(ExprType == SIR_EXPR_FUNCTION_CALL);
assert(Type != NULL);
assert(Arg1 != NULL);
assert(Args != NULL);

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= NULL;
SIR_Expression::Symbol		= NULL;
SIR_Expression::Arg1		= Arg1;
SIR_Expression::Arg2		= NULL;
SIR_Expression::Arg3		= NULL;
SIR_Expression::Args		= Args;
SIR_Expression::TypeArg		= NULL;
SIR_Expression::Type		= Type;
SIR_Expression::LeftBound	= 0;
SIR_Expression::RightBound	= 0;

} /* end of SIR_Expression::SIR_Expression #4 */


SIR_Expression::SIR_Expression(		/* constructor #5 */
	SIR_EXPRTYPE	ExprType,	/* (for member access) */
	sir_type	*Type,
	sir_expression	*Arg1,
	sir_symbol	*Symbol,	/* (Symbol==NULL for bhvr. main()) */
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(  (ExprType == SIR_EXPR_MEMBER_ACCESS)
       ||(ExprType == SIR_EXPR_MEMBER_POINTER));
assert(Type != NULL);
assert(Arg1 != NULL);

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= NULL;
SIR_Expression::Symbol		= Symbol;
SIR_Expression::Arg1		= Arg1;
SIR_Expression::Arg2		= NULL;
SIR_Expression::Arg3		= NULL;
SIR_Expression::Args		= NULL;
SIR_Expression::TypeArg		= NULL;
SIR_Expression::Type		= Type;
SIR_Expression::LeftBound	= 0;
SIR_Expression::RightBound	= 0;

} /* end of SIR_Expression::SIR_Expression #5 */


SIR_Expression::SIR_Expression(		/* constructor #6 */
	SIR_EXPRTYPE	ExprType,	/* (for sizeof(type) or casting) */
	sir_type	*Type,
	sir_type	*TypeArg,
	sir_expression	*Arg1 = NULL,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(  (ExprType == SIR_EXPR_SIZEOF_TYPE)
       ||(ExprType == SIR_EXPR_TYPE_CONVERSION));
assert(Type != NULL);
assert(TypeArg != NULL);

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= NULL;
SIR_Expression::Symbol		= NULL;
SIR_Expression::Arg1		= Arg1;
SIR_Expression::Arg2		= NULL;
SIR_Expression::Arg3		= NULL;
SIR_Expression::Args		= NULL;
SIR_Expression::TypeArg		= TypeArg;
SIR_Expression::Type		= Type;
SIR_Expression::LeftBound	= 0;
SIR_Expression::RightBound	= 0;

} /* end of SIR_Expression::SIR_Expression #6 */


SIR_Expression::SIR_Expression(		/* constructor #7 */
	SIR_EXPRTYPE	ExprType,	/* (for bit slicing) */
	sir_type	*Type,
	sir_expression	*Arg1,
	int		LeftBound,
	int		RightBound,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(ExprType == SIR_EXPR_BITSLICE);
assert(Type != NULL);
assert(Arg1 != NULL);

SIR_Expression::ExprType	= ExprType;
SIR_Expression::Constant	= NULL;
SIR_Expression::Symbol		= NULL;
SIR_Expression::Arg1		= Arg1;
SIR_Expression::Arg2		= NULL;
SIR_Expression::Arg3		= NULL;
SIR_Expression::Args		= NULL;
SIR_Expression::TypeArg		= NULL;
SIR_Expression::Type		= Type;
SIR_Expression::LeftBound	= LeftBound;
SIR_Expression::RightBound	= RightBound;

} /* end of SIR_Expression::SIR_Expression #7 */


SIR_Expression::SIR_Expression(		/* constructor #9 (duplicator) */
	sir_expression	*Original) :	/* (recursive!) */
		SIR_Node(Original)
{

assert(Original != NULL);

SIR_Expression::ExprType	= Original->ExprType;
SIR_Expression::Constant	= (Original->Constant ?
					new SIR_Constant(Original->Constant) :
					NULL);
SIR_Expression::Symbol		= Original->Symbol;
SIR_Expression::Arg1		= (Original->Arg1 ?
					new SIR_Expression(Original->Arg1) :
					NULL);
SIR_Expression::Arg2		= (Original->Arg2 ?
					new SIR_Expression(Original->Arg2) :
					NULL);
SIR_Expression::Arg3		= (Original->Arg3 ?
					new SIR_Expression(Original->Arg3) :
					NULL);
SIR_Expression::Args		= (Original->Args ?
					new SIR_Expressions(Original->Args) :
					NULL);
SIR_Expression::TypeArg		= Original->TypeArg;
SIR_Expression::Type		= Original->Type;
SIR_Expression::LeftBound	= Original->LeftBound;
SIR_Expression::RightBound	= Original->RightBound;

} /* end of SIR_Expression::SIR_Expression #9 */


SIR_Expression::~SIR_Expression(void)	/* destructor */
{

delete Constant;
delete Arg1;
delete Arg2;
delete Arg3;
delete Args;

} /* end of SIR_Expression::~SIR_Expression */


sir_expression *SIR_Expression::New(	/* creates an identifier expr. (#2b) */
	sir_symbol	*Symbol)	/* (returns NULL if SIR_Error) */
{
sir_expression	*NewExpr;

assert(Symbol != NULL);

NewExpr = new SIR_Expression(SIR_EXPR_IDENTIFIER, Symbol);

NewExpr->Type = NewExpr->Type;

return(NewExpr);

} /* end of SIR_Expression::New #2b */


sir_expression *SIR_Expression::New(	/* creates a special expression (#8b) */
	SIR_EXPRTYPE	ExprType,	/* (supports VOID, THIS) */
	sir_types	*TypeTable,	/* (returns NULL if SIR_Error) */
	sir_symbol	*ClassSymbol = NULL)	/* class for THIS (or NULL) */
{
sir_expression	*NewExpr;
sir_type	*ResultType = NULL;	/* make compiler quiet */

assert(TypeTable != NULL);

switch(ExprType)
   { case SIR_EXPR_VOID:
	{ ResultType = TypeTable->FindOrInsert(SIR_TYPE_VOID);
	  break;
	 }
     case SIR_EXPR_THIS:
	{ assert(ClassSymbol != NULL);
	  assert(ClassSymbol->Type != NULL);
	  ResultType = ClassSymbol->Type;
	  break;
	 }
     default:
	{ assert(FALSE);	/* illegal expression type */
	 }
    } /* hctiws */

NewExpr = new SIR_Expression(ExprType, ResultType);

return(NewExpr);

} /* end of SIR_Expression::New #8b */


BOOL SIR_Expression::IsLvalue(	/* checks if this expr. is an lvalue */
	void)
{

switch (ExprType)
   { case SIR_EXPR_VOID:
	{ assert(FALSE);	/* should never appear as sub-expression */
	 }
     case SIR_EXPR_CONSTANT:
	{ return(Type->Type == SIR_TYPE_POINTER);
	 }
     case SIR_EXPR_IDENTIFIER:
	{ return(TRUE);
	 }
     case SIR_EXPR_PARENTHESES:
	{ return(Arg1->IsLvalue());
	 }
#ifdef SIR_DELTA_SUPPORT
     case SIR_EXPR_DELTA:
	{ return(FALSE);
	 }
#endif /* SIR_DELTA_SUPPORT */
     case SIR_EXPR_THIS:
	{ return(FALSE);
	 }
     case SIR_EXPR_ARRAY_ACCESS:
	{ return(TRUE);
	 }
     case SIR_EXPR_FUNCTION_CALL:
	{ return(FALSE);
	 }
     case SIR_EXPR_MEMBER_ACCESS:
     case SIR_EXPR_MEMBER_POINTER:
	{ return(TRUE);
	 }
     case SIR_EXPR_POST_INCREMENT:
     case SIR_EXPR_POST_DECREMENT:
	{ return(FALSE);
	 }
     case SIR_EXPR_BITSLICE:
	{ return(TRUE);
	 }
     case SIR_EXPR_PRE_INCREMENT:
     case SIR_EXPR_PRE_DECREMENT:
	{ return(TRUE);
	 }
     case SIR_EXPR_ADDRESS_OF:
	{ return(FALSE);
	 }
     case SIR_EXPR_CONTENT_OF:
	{ return(TRUE);
	 }
     case SIR_EXPR_POSITIVE:
     case SIR_EXPR_NEGATIVE:
	{ return(FALSE);
	 }
     case SIR_EXPR_NOT:
     case SIR_EXPR_LOGICAL_NOT:
	{ return(FALSE);
	 }
     case SIR_EXPR_SIZEOF_EXPR:
     case SIR_EXPR_SIZEOF_TYPE:
	{ return(FALSE);
	 }
     case SIR_EXPR_TYPE_CONVERSION:
	{ return(FALSE);
	 }
     case SIR_EXPR_CONCATENATION:
	{ return(FALSE);
	 }
     case SIR_EXPR_MULTIPLY:
     case SIR_EXPR_DIVIDE:
     case SIR_EXPR_MODULO:
     case SIR_EXPR_ADD:
     case SIR_EXPR_SUBTRACT:
     case SIR_EXPR_SHIFT_LEFT:
     case SIR_EXPR_SHIFT_RIGHT:
	{ return(FALSE);
	 }
     case SIR_EXPR_LESS:
     case SIR_EXPR_GREATER:
     case SIR_EXPR_LESS_EQUAL:
     case SIR_EXPR_GREATER_EQUAL:
     case SIR_EXPR_EQUAL:
     case SIR_EXPR_NOT_EQUAL:
	{ return(FALSE);
	 }
     case SIR_EXPR_AND:
     case SIR_EXPR_EOR:
     case SIR_EXPR_OR:
     case SIR_EXPR_LOGICAL_AND:
     case SIR_EXPR_LOGICAL_OR:
	{ return(FALSE);
	 }
     case SIR_EXPR_CONDITION:
	{ return(FALSE);
	 }
     case SIR_EXPR_ASSIGNMENT:
	{ return(TRUE);
	 }
     case SIR_EXPR_MUL_ASSIGN:
     case SIR_EXPR_DIV_ASSIGN:
     case SIR_EXPR_MOD_ASSIGN:
     case SIR_EXPR_ADD_ASSIGN:
     case SIR_EXPR_SUB_ASSIGN:
     case SIR_EXPR_SHL_ASSIGN:
     case SIR_EXPR_SHR_ASSIGN:
     case SIR_EXPR_AND_ASSIGN:
     case SIR_EXPR_EOR_ASSIGN:
     case SIR_EXPR_OR_ASSIGN:
	{ return(TRUE);
	 }
     case SIR_EXPR_COMMA:
	{ return(Arg2->IsLvalue());
	 }
     default:
	{ assert(FALSE);	/* bad expression type */
	 }
    } /* hctiws */

return(FALSE);	/* never reached */

} /* end of SIR_Expression::IsLvalue */


BOOL SIR_Expression::IsModifiableLvalue(	/* checks if modifiable lv. */
	void)
{

return(  (Type->Const == FALSE)
       &&(IsLvalue()));

} /* end of SIR_Expression::IsModifiableLvalue */


BOOL SIR_Expression::IsWritable(	/* checks if allows write access */
	void)
{

return(Type->Direction != SIR_PORT_IN);

} /* end of SIR_Expression::IsWritable */


BOOL SIR_Expression::IsReadable(	/* checks if allows read access */
	void)
{

return(Type->Direction != SIR_PORT_OUT);

} /* end of SIR_Expression::IsReadable */


ERROR SIR_Expression::CheckWriteAccess(	/* checks error for write access */
	void)
{

if (!(IsModifiableLvalue()))
   { if (ExprType == SIR_EXPR_IDENTIFIER)
	{ SIR_ErrMsg.form("Modifiable lvalue expected for '%s'",
				Symbol->Name.chars());
	 } /* fi */
     else
	{ SIR_ErrMsg = "Modifiable lvalue expected";
	 } /* esle */
     return(SIR_Error = SIR_ERROR_MODIFIABLE_LVALUE);
    } /* fi */

if (!(IsWritable()))
   { if (ExprType == SIR_EXPR_IDENTIFIER)
	{ SIR_ErrMsg.form("Write access to input port '%s'",
				Symbol->Name.chars());
	 } /* fi */
     else
	{ SIR_ErrMsg = "Write access to input port";
	 } /* esle */
     return(SIR_Error = SIR_ERROR_PORT_WRITE_ACCESS);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::CheckWriteAccess */


ERROR SIR_Expression::CheckReadAccess(	/* checks error for read access */
	void)
{

if (!(IsReadable()))
   { if (ExprType == SIR_EXPR_IDENTIFIER)
	{ SIR_ErrMsg.form("Read access to output port '%s'",
				Symbol->Name.chars());
	 } /* fi */
     else
	{ SIR_ErrMsg = "Read access to output port";
	 } /* esle */
     return(SIR_Error = SIR_ERROR_PORT_READ_ACCESS);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::CheckReadAccess */


BOOL SIR_Expression::IsDependant(	/* checks if expr. depends on symbol */
	sir_symbol	*ThatSymbol)
{

if (Symbol == ThatSymbol)
   { return(TRUE);	/* references that symbol */
    } /* fi */

return(FALSE);	/* independent! */

} /* end of SIR_Expression::IsDependant */


ERROR SIR_Expression::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (Constant)
   { if ((SIR_Error = Constant->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Arg1)
   { if ((SIR_Error = Arg1->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Arg2)
   { if ((SIR_Error = Arg2->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Arg3)
   { if ((SIR_Error = Arg3->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Args)
   { if ((SIR_Error = Args->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::DFS_ForAllNodes */


ERROR SIR_Expression::DFS_ForAllExpressions(	/* iterator over all exprs. */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

/* there are no expressions in Constant */
if (Arg1)
   { if ((SIR_Error = Arg1->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Arg2)
   { if ((SIR_Error = Arg2->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Arg3)
   { if ((SIR_Error = Arg3->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Args)
   { if ((SIR_Error = Args->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::DFS_ForAllExpressions */


BOOL SIR_Expression::DFS_FindDependant(	/* searches for dependants (DFS) */
	sir_symbol	*ThatSymbol,
	sir_expression	**DepExpr)
{
sir_expression	*Arg;

assert(ThatSymbol != NULL);

if (IsDependant(ThatSymbol))		/* check this expression */
   { if (DepExpr)
	{ *DepExpr = this;
	 } /* fi */
     return(TRUE);
    } /* fi */

/* there are no expressions in Constant */
if (Arg1)
   { if (Arg1->DFS_FindDependant(ThatSymbol, DepExpr))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Arg2)
   { if (Arg2->DFS_FindDependant(ThatSymbol, DepExpr))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Arg3)
   { if (Arg3->DFS_FindDependant(ThatSymbol, DepExpr))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Args)
   { Arg = Args->First();
     while(Arg)
	{ if (Arg->DFS_FindDependant(ThatSymbol, DepExpr))
	     { return(TRUE);
	      } /* fi */
	  Arg = Arg->Succ();
	 } /* elihw */
    } /* fi */

return(FALSE);

} /* end of SIR_Expression::DFS_FindDependant */


ERROR SIR_Expression::MarkUsedTypes(	/* marks the type entries used here */
	sir_expr_marg	/* Unused */)
{

Type->MarkUsed();

if (TypeArg)
   { TypeArg->MarkUsed();
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::MarkUsedTypes */


void SIR_Expression::UnAlias(void)/* unalias all type, usertype, symbol links */
{
sir_expression	*Arg;

/* there is nothing to unalias in Constant */
if (  (Symbol)
    &&(Symbol->Alias))
   { Symbol = Symbol->Alias;
    } /* fi */
if (Arg1)
   { Arg1->UnAlias();
    } /* fi */
if (Arg2)
   { Arg2->UnAlias();
    } /* fi */
if (Arg3)
   { Arg3->UnAlias();
    } /* fi */
if (Args)
   { Arg = Args->First();
     while(Arg)
	{ Arg->UnAlias();
	  Arg = Arg->Succ();
	 } /* elihw */
    } /* fi */
if (  (TypeArg)
    &&(TypeArg->Alias))
   { TypeArg = TypeArg->Alias;
    } /* fi */
if (Type->Alias)
   { Type = Type->Alias;
    } /* fi */

} /* end of SIR_Expression::UnAlias */


void SIR_Expression::ExplicitTypeConv(	/* perform explicit type conversion */
	SIR_CONSTTYPE	FromType,
	SIR_CONSTTYPE	ToType,
	const char	**Prefix,
	const char	**Suffix)	/* (caller must substitute length!) */
{

assert(Prefix != NULL);
assert(Suffix != NULL);

*Prefix = NULL;	/* default: no conversion, or implicit conversion */
*Suffix = NULL;

if (  (FromType == SIR_CONST_FLOAT)
    ||(FromType == SIR_CONST_DOUBLE)
    ||(FromType == SIR_CONST_LONGDOUBLE))
   { if (  (ToType == SIR_CONST_BIT)
	 ||(ToType == SIR_CONST_UBIT))
	{ *Prefix = SIR_BIT_BASE_CLASS_NAME "(";
	  if (ToType == SIR_CONST_BIT)
	     { *Suffix = ", %d, "	/* caller supplies length! */
				SIR_BIT_SIGNED_FLAG ")";
	      } /* fi */
	  else
	     { *Suffix = ", %d,	"	/* caller supplies length! */
				SIR_BIT_UNSIGNED_FLAG ")";
	      } /* esle */
	 } /* fi */
#ifndef HAVE_LLONG
     else if (  (ToType == SIR_CONST_LONGLONG)
	      ||(ToType == SIR_CONST_ULONGLONG))
	{ if (ToType == SIR_CONST_LONGLONG)
	     { *Prefix = SIR_LLONG_CLASS_NAME "<" SIR_LLONG_SIGNED_FLAG ">(";
	       *Suffix = ", " SIR_LLONG_SIGNED_FLAG ")";
	      } /* fi */
	  else
	     { *Prefix = SIR_LLONG_CLASS_NAME "<" SIR_LLONG_UNSIGNED_FLAG ">(";
	       *Suffix = ", " SIR_LLONG_UNSIGNED_FLAG ")";
	      } /* esle */
	 } /* fi else */
#endif /* HAVE_LLONG */
    } /* fi */
else
   { if (  (ToType == SIR_CONST_BIT)
	 ||(ToType == SIR_CONST_UBIT))
	{ if (  (FromType != SIR_CONST_BIT)
	      &&(FromType != SIR_CONST_UBIT))
	     { /* convert integral type to bitvector */
	       *Prefix = SIR_BIT_BASE_CLASS_NAME "(";
	       *Suffix = ")";
	      } /* fi */
	 } /* fi */
     else
	{ if (  (FromType == SIR_CONST_BIT)
	      ||(FromType == SIR_CONST_UBIT))
	     { /* convert bitvector to integral or floating type */
	       switch(ToType)
		  { case SIR_CONST_BOOL:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_BOOL_NAME "())";
			 break;
			}
		    case SIR_CONST_CHAR:
		       { *Prefix = "((char)(";
			 *Suffix = ")." SIR_BIT_2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_UCHAR:
		       { *Prefix = "((unsigned char)(";
			 *Suffix = ")." SIR_BIT_2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_SHORT:
		       { *Prefix = "((short)(";
			 *Suffix = ")." SIR_BIT_2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_USHORT:
		       { *Prefix = "((unsigned short)(";
			 *Suffix = ")." SIR_BIT_2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_INT:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_UINT:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_UINT_NAME "())";
			 break;
			}
		    case SIR_CONST_LONG:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_LONG_NAME "())";
			 break;
			}
		    case SIR_CONST_ULONG:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_ULONG_NAME "())";
			 break;
			}
		    case SIR_CONST_LONGLONG:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_LLONG_NAME "())";
			 break;
			}
		    case SIR_CONST_ULONGLONG:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_ULLONG_NAME "())";
			 break;
			}
		    case SIR_CONST_FLOAT:
		       { *Prefix = "((float)(";
			 *Suffix = ")." SIR_BIT_2_DOUBLE_NAME "())";
			 break;
			}
		    case SIR_CONST_DOUBLE:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_DOUBLE_NAME "())";
			 break;
			}
		    case SIR_CONST_LONGDOUBLE:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_BIT_2_LDOUBLE_NAME "())";
			 break;
			}
		    case SIR_CONST_BIT:
		    case SIR_CONST_UBIT:
		       { assert(FALSE);	/* handled earlier */
			}
		    default:
		       { assert(FALSE);	/* unsupported conversion */
			}
		   } /* hctiws */
	      } /* fi */
#ifndef HAVE_LLONG
	  if (  (FromType == SIR_CONST_LONGLONG)
	      ||(FromType == SIR_CONST_ULONGLONG))
	     { /* convert 'long long' to integral or floating type */
	       switch(ToType)
		  { case SIR_CONST_BOOL:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_BOOL_NAME "())";
			 break;
			}
		    case SIR_CONST_CHAR:
		       { *Prefix = "((char)(";
			 *Suffix = ")." SIR_LLONG2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_UCHAR:
		       { *Prefix = "((unsigned char)(";
			 *Suffix = ")." SIR_LLONG2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_SHORT:
		       { *Prefix = "((short)(";
			 *Suffix = ")." SIR_LLONG2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_USHORT:
		       { *Prefix = "((unsigned short)(";
			 *Suffix = ")." SIR_LLONG2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_INT:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_INT_NAME "())";
			 break;
			}
		    case SIR_CONST_UINT:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_UINT_NAME "())";
			 break;
			}
		    case SIR_CONST_LONG:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_LONG_NAME "())";
			 break;
			}
		    case SIR_CONST_ULONG:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_ULONG_NAME "())";
			 break;
			}
		    case SIR_CONST_LONGLONG:
		    case SIR_CONST_ULONGLONG:
		       { /* no conversion necessary */
			 break;
			}
		    case SIR_CONST_FLOAT:
		       { *Prefix = "((float)(";
			 *Suffix = ")." SIR_LLONG2_DOUBLE_NAME "())";
			 break;
			}
		    case SIR_CONST_DOUBLE:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_DOUBLE_NAME "())";
			 break;
			}
		    case SIR_CONST_LONGDOUBLE:
		       { *Prefix = "((";
			 *Suffix = ")." SIR_LLONG2_LDOUBLE_NAME "())";
			 break;
			}
		    case SIR_CONST_BIT:
		    case SIR_CONST_UBIT:
		       { assert(FALSE);	/* handled earlier */
			}
		    default:
		       { assert(FALSE);	/* unsupported conversion */
			}
		   } /* hctiws */
	      } /* fi */
#endif /* !HAVE_LLONG */
	 } /* esle */
    } /* esle */

} /* end of SIR_Expression::ExplicitTypeConv */


ERROR SIR_Expression::WriteSC(	/* (re-) generates SpecC source code */
	FILE		*File,
	BOOL		WriteNotes,
	BOOL		CplusplusMode = FALSE,
	SIR_TYPETYPE	ExpectedType = SIR_TYPE_ANY_TYPE,
	int		ExpectedLength = 0)	/* for expected bitvectors */
{
sir_expression	*Arg;
sir_constant	*TmpConst;
string		Buffer;
const char	*Prefix,
		*Suffix;

if (LineInfo)
   { if ((SIR_Error = LineInfo->WriteSC(File)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

Prefix = NULL;
Suffix = NULL;
if (  (CplusplusMode)		/* explicitly create automatic conversions */
    &&(ExpectedType < SIR_TYPE_VOID)	/* (exploit smart type enumeration) */
    &&(Type->Type < SIR_TYPE_VOID))
   { ExplicitTypeConv(SIR_Type::ConstType(Type->Type),
			SIR_Type::ConstType(ExpectedType),
			&Prefix, &Suffix);
     if (Prefix)
	{ fputs(Prefix, File);	/* start explicit conversion */
	 } /* fi */
    } /* fi */

switch (ExprType)
   { case SIR_EXPR_VOID:
	{ /* nothing */
	  break;
	 }
     case SIR_EXPR_CONSTANT:
	{ fputs(Constant->Print(CplusplusMode), File);
	  break;
	 }
     case SIR_EXPR_IDENTIFIER:
	{ fputs(Symbol->Name.chars(), File);
	  break;
	 }
     case SIR_EXPR_PARENTHESES:	/* we keep redundant parentheses */
	{ fputc('(', File);	/* for readability purposes	 */
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  fputc(')', File);
	  break;
	 }
     case SIR_EXPR_THIS:
	{ if (CplusplusMode)
	     { fputs("(*this)", File);	/* in SpecC 'this' is _not_ a pointer */
	      } /* fi */		/* (it is expected as a reference) */
	  else
	     { fputs("this", File);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_ARRAY_ACCESS:
	{ SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, File);
	  if (  (CplusplusMode)
	      &&(  (Arg1->Type->Type == SIR_TYPE_BIT)
		 ||(Arg1->Type->Type == SIR_TYPE_UBIT)))
	     { fprintf(File, SIR_BIT_SLICE_NAME "(%d,%d,(",
				Arg1->Type->LeftBound, Arg1->Type->RightBound);
	       if ((SIR_Error = Arg2->WriteSC(File, WriteNotes, CplusplusMode,
							SIR_TYPE_INT)))
		  { return(SIR_Error);
		   } /* fi */
	       fputs("))", File);
	      } /* fi */
	  else
	     { fputc('[', File);
	       if ((SIR_Error = Arg2->WriteSC(File, WriteNotes, CplusplusMode,
							SIR_TYPE_INT)))
		  { return(SIR_Error);
		   } /* fi */
	       fputc(']', File);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_FUNCTION_CALL:
	{ sir_type_ptr	*ExpectTypePtr;
	  SIR_TYPETYPE	ExpectType;

	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, File);
	  fputc('(', File);
	  if (Args)
	     { assert(Arg1->Type->Type == SIR_TYPE_FUNCTION);
	       ExpectTypePtr = NULL;
	       if (Arg1->Type->Parameters)
		  { ExpectTypePtr = Arg1->Type->Parameters->First();
		   } /* fi */
	       Arg = Args->First();
	       while(Arg)
		  { if (ExpectTypePtr)
		       { ExpectType = ExpectTypePtr->Type->Type;
			} /* fi */
		    else
		       { ExpectType = SIR_TYPE_ANY_TYPE;
			} /* else */
		    SIR_PutParenthesis(SIR_EXPR_COMMA,
					'(', Arg1, SIR_RIGHT_ARG, File);
		    if ((SIR_Error = Arg->WriteSC(File, WriteNotes,
						CplusplusMode,
						SIR_RELAXED(ExpectType,
							Arg->Type->Type))))
		       { return(SIR_Error);
			} /* fi */
		    SIR_PutParenthesis(SIR_EXPR_COMMA,
					')', Arg1, SIR_RIGHT_ARG, File);
		    if (ExpectTypePtr)
		       { ExpectTypePtr = ExpectTypePtr->Succ();
			} /* fi */
		    Arg = Arg->Succ();
		    if (Arg)
		       { fputs(", ", File);
			} /* fi */
		   } /* elihw */
	      } /* fi */
	  fputc(')', File);
	  break;
	 }
     case SIR_EXPR_MEMBER_ACCESS:
	{ SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, File);
	  fputc('.', File);
	  if (Symbol)
	     { fputs(Symbol->Name.chars(), File);
	      } /* fi */
	  else
	     { fputs(GL_MAIN_METHOD_NAME, File);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_MEMBER_POINTER:
	{ SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, File);
	  fputs("->", File);
	  if (Symbol)
	     { fputs(Symbol->Name.chars(), File);
	      } /* fi */
	  else
	     { fputs(GL_MAIN_METHOD_NAME, File);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_POST_INCREMENT:
	{ SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, File);
	  fputs("++ ", File);
	  break;
	 }
     case SIR_EXPR_POST_DECREMENT:
	{ SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, File);
	  fputs("-- ", File);
	  break;
	 }
     case SIR_EXPR_BITSLICE:
	{ if (CplusplusMode)
	     { fputs("(", File);
	       SIR_PutParenthesis(SIR_EXPR_MEMBER_ACCESS,
					'(', Arg1, SIR_LEFT_ARG, File);
	       if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode,
					SIR_TYPE_BIT,	/* not relaxed! */
					-1 /* unused in this case! */)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(SIR_EXPR_MEMBER_ACCESS,
					')', Arg1, SIR_LEFT_ARG, File);
	       if (  (Arg1->Type->Type == SIR_TYPE_BIT)
		   ||(Arg1->Type->Type == SIR_TYPE_UBIT))
		  { /* Arg1 is possibly a non-normalized bitvector */
		    fprintf(File, SIR_BIT_SLICE_NAME "(%d,%d, %d,%d))",
				Arg1->Type->LeftBound, Arg1->Type->RightBound,
				LeftBound, RightBound);
		   } /* fi */
	       else
		  { /* Arg1 has been converted to a normalized bitvector */
		    fprintf(File, SIR_BIT_SLICE_NAME "(%d,%d))",
				LeftBound, RightBound);
		   } /* esle */
	      } /* fi */
	  else
	     { SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, File);
	       if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode,
					SIR_TYPE_BIT,	/* not relaxed! */
					-1 /* unused in this case! */)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, File);
	       fprintf(File, "[%d:%d]", LeftBound, RightBound);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_PRE_INCREMENT:
	{ fputs(" ++", File);
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, File);
	  break;
	 }
     case SIR_EXPR_PRE_DECREMENT:
	{ fputs(" --", File);
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, File);
	  break;
	 }
     case SIR_EXPR_ADDRESS_OF:
	{ fputs(" &", File);
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, File);
	  break;
	 }
     case SIR_EXPR_CONTENT_OF:
	{ fputs(" *", File);
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, File);
	  break;
	 }
     case SIR_EXPR_POSITIVE:
	{ fputs(" +", File);
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, File);
	  break;
	 }
     case SIR_EXPR_NEGATIVE:
	{ fputs(" -", File);
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, File);
	  break;
	 }
     case SIR_EXPR_NOT:
	{ fputs(" ~", File);
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, File);
	  break;
	 }
     case SIR_EXPR_LOGICAL_NOT:
	{ fputs(" !", File);
	  SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode,
							SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, File);
	  break;
	 }
     case SIR_EXPR_SIZEOF_EXPR:
	{ if (  (CplusplusMode)
	      &&(Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR))
	     { /* sizeof(bit[:]) cannot be overloaded, so we compute it here */
	       TmpConst = new SIR_Constant(SIR_CONST_UINT,
				(UNSIGNED_LONG_LONG) Arg1->Type->SizeOf());
	       fputs(TmpConst->Print(CplusplusMode), File);
	       delete TmpConst;
	      } /* fi */
	  else
	     { fputs("sizeof ", File);
	       SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, File);
	       if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, File);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_SIZEOF_TYPE:
	{ if (  (CplusplusMode)
	      &&(TypeArg->TypeClass() == SIR_TYPECLASS_BITVECTOR))
	     { /* sizeof(bit[:]) cannot be overloaded, so we compute it here */
	       TmpConst = new SIR_Constant(SIR_CONST_UINT,
				(UNSIGNED_LONG_LONG) TypeArg->SizeOf());
	       fputs(TmpConst->Print(CplusplusMode), File);
	       delete TmpConst;
	      } /* fi */
	  else
	     { fputs("sizeof(", File);
	       Buffer = "";
	       fputs(TypeArg->PrettyString(&Buffer, WriteNotes, CplusplusMode),
				File);
	       fputc(')', File);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_TYPE_CONVERSION:
	{ if (CastNeedsExplicitTypeConv(Arg1->Type->Type, TypeArg->Type))
	     { if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode,
						TypeArg->Type,
			_BITLEN(TypeArg->LeftBound, TypeArg->RightBound))))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else	/* C++ compiler knows how to handle the rest */
	     { fputc('(', File);
	       Buffer = "";
	       fputs(TypeArg->PrettyString(&Buffer, WriteNotes, CplusplusMode),
			File);
	       fputc(')', File);
	       SIR_PutParenthesis(ExprType, '(', Arg1, SIR_RIGHT_ARG, File);
	       if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(ExprType, ')', Arg1, SIR_RIGHT_ARG, File);
	      } /* else */
	  break;
	 }
     case SIR_EXPR_CONCATENATION:
	{ if (CplusplusMode)
	     { fputs(SIR_BIT_CONCAT_NAME "(", File);
	       SIR_PutParenthesis(SIR_EXPR_COMMA,
					'(', Arg1, SIR_RIGHT_ARG, File);
	       if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, true,
					SIR_TYPE_BIT,	/* not relaxed! */
					-1 /* unused in this case! */)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(SIR_EXPR_COMMA,
					')', Arg1, SIR_RIGHT_ARG, File);
	       fputc(',', File);
	       SIR_PutParenthesis(SIR_EXPR_COMMA,
					'(', Arg1, SIR_RIGHT_ARG, File);
	       if ((SIR_Error = Arg2->WriteSC(File, WriteNotes, true,
					SIR_TYPE_BIT,	/* not relaxed! */
					-1 /* unused in this case! */)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(SIR_EXPR_COMMA,
					')', Arg1, SIR_RIGHT_ARG, File);
	       fputc(')', File);
	      } /* fi */
	  else
	     { if ((SIR_Error = WriteSC2("@", File, WriteNotes, false)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_MULTIPLY:
	{ if ((SIR_Error = WriteSC2("*", File, WriteNotes, CplusplusMode,
				DOUBLE_EXPECTED(Type->Type, Arg1->Type->Type),
				DOUBLE_EXPECTED(Type->Type, Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_DIVIDE:
	{ if ((SIR_Error = WriteSC2("/", File, WriteNotes, CplusplusMode,
				DOUBLE_EXPECTED(Type->Type, Arg1->Type->Type),
				DOUBLE_EXPECTED(Type->Type, Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_MODULO:
	{ if ((SIR_Error = WriteSC2("%", File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_ADD:
	{ if (  (Arg1->Type->Type == SIR_TYPE_POINTER)
	      ||(Arg1->Type->Type == SIR_TYPE_ARRAY))
	     { if ((SIR_Error = WriteSC2("+", File,
					WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (  (Arg2->Type->Type == SIR_TYPE_POINTER)
		   ||(Arg2->Type->Type == SIR_TYPE_ARRAY))
		  { if ((SIR_Error = WriteSC2("+", File,
					WriteNotes, CplusplusMode,
					SIR_TYPE_INT, SIR_TYPE_ANY_TYPE)))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       else
		  { if ((SIR_Error = WriteSC2("+", File,
					WriteNotes, CplusplusMode,
				DOUBLE_EXPECTED(Type->Type, Arg1->Type->Type),
				DOUBLE_EXPECTED(Type->Type, Arg2->Type->Type))))
		       { return(SIR_Error);
			} /* fi */
		   } /* esle */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_SUBTRACT:
	{ if (  (Arg1->Type->Type == SIR_TYPE_POINTER)
	      ||(Arg1->Type->Type == SIR_TYPE_ARRAY))
	     { if ((SIR_Error = WriteSC2("-", File,
					WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if (  (Arg2->Type->Type == SIR_TYPE_POINTER)
		   ||(Arg2->Type->Type == SIR_TYPE_ARRAY))
		  { if ((SIR_Error = WriteSC2("-", File,
					WriteNotes, CplusplusMode,
					SIR_TYPE_INT, SIR_TYPE_ANY_TYPE)))
		       { return(SIR_Error);
			} /* fi */
		   } /* fi */
	       else
		  { if ((SIR_Error = WriteSC2("-", File,
					WriteNotes, CplusplusMode,
				DOUBLE_EXPECTED(Type->Type, Arg1->Type->Type),
				DOUBLE_EXPECTED(Type->Type, Arg2->Type->Type))))
		       { return(SIR_Error);
			} /* fi */
		   } /* esle */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_SHIFT_LEFT:
	{ if ((SIR_Error = WriteSC2("<<", File, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_SHIFT_RIGHT:
	{ if ((SIR_Error = WriteSC2(">>", File, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_LESS:
	{ if ((SIR_Error = WriteSC2("<", File, WriteNotes, CplusplusMode,
			DOUBLE_EXPECTED(Arg2->Type->Type, Arg1->Type->Type),
			DOUBLE_EXPECTED(Arg1->Type->Type, Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_GREATER:
	{ if ((SIR_Error = WriteSC2(">", File, WriteNotes, CplusplusMode,
			DOUBLE_EXPECTED(Arg2->Type->Type, Arg1->Type->Type),
			DOUBLE_EXPECTED(Arg1->Type->Type, Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_LESS_EQUAL:
	{ if ((SIR_Error = WriteSC2("<=", File, WriteNotes, CplusplusMode,
			DOUBLE_EXPECTED(Arg2->Type->Type, Arg1->Type->Type),
			DOUBLE_EXPECTED(Arg1->Type->Type, Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_GREATER_EQUAL:
	{ if ((SIR_Error = WriteSC2(">=", File, WriteNotes, CplusplusMode,
			DOUBLE_EXPECTED(Arg2->Type->Type, Arg1->Type->Type),
			DOUBLE_EXPECTED(Arg1->Type->Type, Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_EQUAL:
	{ if ((SIR_Error = WriteSC2("==", File, WriteNotes, CplusplusMode,
			DOUBLE_EXPECTED(Arg2->Type->Type, Arg1->Type->Type),
			DOUBLE_EXPECTED(Arg1->Type->Type, Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_NOT_EQUAL:
	{ if ((SIR_Error = WriteSC2("!=", File, WriteNotes, CplusplusMode,
			DOUBLE_EXPECTED(Arg2->Type->Type, Arg1->Type->Type),
			DOUBLE_EXPECTED(Arg1->Type->Type, Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_AND:
	{ if ((SIR_Error = WriteSC2("&", File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_EOR:
	{ if ((SIR_Error = WriteSC2("^", File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_OR:
	{ if ((SIR_Error = WriteSC2("|", File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_LOGICAL_AND:
	{ if ((SIR_Error = WriteSC2("&&", File, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_LOGICAL_OR:
	{ if ((SIR_Error = WriteSC2("||", File, WriteNotes, CplusplusMode,
						SIR_TYPE_BOOL, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_CONDITION:
	{ SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, File);
	  if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode,
							SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, File);
	  fputs(" ? ", File);
	  if (  (CplusplusMode)
	      &&(  (Type->Type == SIR_TYPE_BIT)
		 ||(Type->Type == SIR_TYPE_UBIT)))
	     { fprintf(File, "(" SIR_BIT_CLASS_NAME "<%d,%s>(",
				_BITLEN(Type->LeftBound, Type->RightBound),
				Type->Type == SIR_TYPE_BIT ?
				SIR_BIT_SIGNED_FLAG : SIR_BIT_UNSIGNED_FLAG);
	       if ((SIR_Error = Arg2->WriteSC(File, WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       fputs("))", File);
	      } /* fi */
#ifndef HAVE_LLONG
	  else if (  (CplusplusMode)
	      &&(  (Type->Type == SIR_TYPE_LONGLONG)
		 ||(Type->Type == SIR_TYPE_ULONGLONG)))
	     { fprintf(File, "(" SIR_LLONG_CLASS_NAME "<%s>(",
				Type->Type == SIR_TYPE_LONGLONG ?
			SIR_LLONG_SIGNED_FLAG : SIR_LLONG_UNSIGNED_FLAG);
	       if ((SIR_Error = Arg2->WriteSC(File, WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       fputs("))", File);
	      } /* fi esle */
#endif /* HAVE_LLONG */
	  else
	     { SIR_PutParenthesis(ExprType, '(', Arg2, SIR_LEFT_ARG, File);
	       if ((SIR_Error = Arg2->WriteSC(File, WriteNotes, CplusplusMode,
				DOUBLE_EXPECTED(Type->Type, Arg2->Type->Type))))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(ExprType, ')', Arg2, SIR_LEFT_ARG, File);
	      } /* esle */
	  fputs(" : ", File);
	  if (  (CplusplusMode)
	      &&(  (Type->Type == SIR_TYPE_BIT)
		 ||(Type->Type == SIR_TYPE_UBIT)))
	     { fprintf(File, "(" SIR_BIT_CLASS_NAME "<%d,%s>(",
				_BITLEN(Type->LeftBound, Type->RightBound),
				Type->Type == SIR_TYPE_BIT ?
				SIR_BIT_SIGNED_FLAG : SIR_BIT_UNSIGNED_FLAG);
	       if ((SIR_Error = Arg3->WriteSC(File, WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       fputs("))", File);
	      } /* fi */
#ifndef HAVE_LLONG
	  else if (  (CplusplusMode)
	      &&(  (Type->Type == SIR_TYPE_LONGLONG)
		 ||(Type->Type == SIR_TYPE_ULONGLONG)))
	     { fprintf(File, "(" SIR_LLONG_CLASS_NAME "<%s>(",
				Type->Type == SIR_TYPE_LONGLONG ?
			SIR_LLONG_SIGNED_FLAG : SIR_LLONG_UNSIGNED_FLAG);
	       if ((SIR_Error = Arg3->WriteSC(File, WriteNotes, CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       fputs("))", File);
	      } /* fi esle */
#endif /* HAVE_LLONG */
	  else
	     { SIR_PutParenthesis(ExprType, '(', Arg3, SIR_LEFT_ARG, File);
	       if ((SIR_Error = Arg3->WriteSC(File, WriteNotes, CplusplusMode,
				DOUBLE_EXPECTED(Type->Type, Arg3->Type->Type))))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_PutParenthesis(ExprType, ')', Arg3, SIR_LEFT_ARG, File);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_ASSIGNMENT:
	{ if ((SIR_Error = WriteSC2("=", File, WriteNotes, CplusplusMode,
			SIR_TYPE_ANY_TYPE, SIR_RELAXED(Arg1->Type->Type,
							Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_MUL_ASSIGN:
	{ if ((SIR_Error = WriteSC2("*=", File, WriteNotes, CplusplusMode,
			SIR_TYPE_ANY_TYPE, SIR_RELAXED(Arg1->Type->Type,
							Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_DIV_ASSIGN:
	{ if ((SIR_Error = WriteSC2("/=", File, WriteNotes, CplusplusMode,
			SIR_TYPE_ANY_TYPE, SIR_RELAXED(Arg1->Type->Type,
							Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_MOD_ASSIGN:
	{ if ((SIR_Error = WriteSC2("%=", File, WriteNotes, CplusplusMode,
			SIR_TYPE_ANY_TYPE, SIR_RELAXED(Arg1->Type->Type,
							Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_ADD_ASSIGN:
	{ if (Arg1->Type->Type == SIR_TYPE_POINTER)
	     /* note: SIR_TYPE_ARRAY not allowed */
	     { if ((SIR_Error = WriteSC2("+=", File, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if ((SIR_Error = WriteSC2("+=", File, WriteNotes, CplusplusMode,
			SIR_TYPE_ANY_TYPE, SIR_RELAXED(Arg1->Type->Type,
							Arg2->Type->Type))))
		  { return(SIR_Error);
		   } /* fi */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_SUB_ASSIGN:
	{ if (Arg1->Type->Type == SIR_TYPE_POINTER)
	     /* note: SIR_TYPE_ARRAY not allowed */
	     { if ((SIR_Error = WriteSC2("-=", File, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if ((SIR_Error = WriteSC2("-=", File, WriteNotes, CplusplusMode,
			SIR_TYPE_ANY_TYPE, SIR_RELAXED(Arg1->Type->Type,
							Arg2->Type->Type))))
		  { return(SIR_Error);
		   } /* fi */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_SHL_ASSIGN:
	{ if ((SIR_Error = WriteSC2("<<=", File, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_SHR_ASSIGN:
	{ if ((SIR_Error = WriteSC2(">>=", File, WriteNotes, CplusplusMode,
					SIR_TYPE_ANY_TYPE, SIR_TYPE_INT)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_AND_ASSIGN:
	{ if ((SIR_Error = WriteSC2("&=", File, WriteNotes, CplusplusMode,
			SIR_TYPE_ANY_TYPE, SIR_RELAXED(Arg1->Type->Type,
							Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_EOR_ASSIGN:
	{ if ((SIR_Error = WriteSC2("^=", File, WriteNotes, CplusplusMode,
			SIR_TYPE_ANY_TYPE, SIR_RELAXED(Arg1->Type->Type,
							Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_OR_ASSIGN:
	{ if ((SIR_Error = WriteSC2("|=", File, WriteNotes, CplusplusMode,
			SIR_TYPE_ANY_TYPE, SIR_RELAXED(Arg1->Type->Type,
							Arg2->Type->Type))))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_EXPR_COMMA:
	{ if ((SIR_Error = WriteSC2(",", File, WriteNotes, CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad expression type */
	 }
    } /* hctiws */

if (Suffix)
   { fprintf(File, Suffix, ExpectedLength);	/* finish explicit conversion */
    } /* fi */

if (ferror(File))
   { SIR_ErrMsg.form("Writing to file failed%s",
			GL_SystemErrorMessage());
     return(SIR_ERROR_WRITING_SPECC_FILE);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::WriteSC */


ERROR SIR_Expression::WriteSC2(	/* (re-) generates SpecC source code */
	const char	*Op,	/* (for binary ops only) */
	FILE		*File,
	BOOL		WriteNotes,
	BOOL		CplusplusMode = FALSE,
	SIR_TYPETYPE	ExpectedType1 = SIR_TYPE_ANY_TYPE,
	SIR_TYPETYPE	ExpectedType2 = SIR_TYPE_ANY_TYPE,
	int		ExpectedLength1 = 0,	/* for expected bitvectors */
	int		ExpectedLength2 = 0)	/* for expected bitvectors */
{

SIR_PutParenthesis(ExprType, '(', Arg1, SIR_LEFT_ARG, File);
if ((SIR_Error = Arg1->WriteSC(File, WriteNotes, CplusplusMode,
					ExpectedType1, ExpectedLength1)))
   { return(SIR_Error);
    } /* fi */
SIR_PutParenthesis(ExprType, ')', Arg1, SIR_LEFT_ARG, File);

fputc(' ', File);
fputs(Op, File);
fputc(' ', File);

SIR_PutParenthesis(ExprType, '(', Arg2, SIR_RIGHT_ARG, File);
if ((SIR_Error = Arg2->WriteSC(File, WriteNotes, CplusplusMode,
					ExpectedType2, ExpectedLength2)))
   { return(SIR_Error);
    } /* fi */
SIR_PutParenthesis(ExprType, ')', Arg2, SIR_RIGHT_ARG, File);

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::WriteSC2 */


sir_constant *SIR_Expression::Eval(void)	/* evaluate a constant expr. */
{						/* (returns a new Constant */
sir_constant		*EvalArg1,		/*  or NULL and SIR_Error) */
			*EvalArg2,
			*TmpResult;
LONG_LONG		TmpValueLL1,
			TmpValueLL2;
UNSIGNED_LONG_LONG	TmpValueULL1,
			TmpValueULL2;
long double		TmpValueLD1,
			TmpValueLD2;
sir_bit			*TmpValueBIT1,
			*TmpValueBIT2;
unsigned int		Line;
sir_fileinfo		*File;

if (  (  (Type->TypeClass() == SIR_TYPECLASS_OTHER)
       ||(Type->TypeClass() == SIR_TYPECLASS_CLASS))
    &&(  (Type->Type != SIR_TYPE_POINTER)
       ||(Type->SubType->Type != SIR_TYPE_CHAR)))
   { SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
     return(NULL);
    } /* fi */

switch (ExprType)
   { case SIR_EXPR_VOID:
	{ assert(FALSE);	/* bad expression type */
	 }
     case SIR_EXPR_CONSTANT:
	{ TmpResult = new SIR_Constant(Constant);
	  return(TmpResult->Converted(Type->ConstType(),
				_BITLEN(Type->LeftBound, Type->RightBound)));
	 }
     case SIR_EXPR_IDENTIFIER:
	{ if (  (Symbol->Type->Const)		/* constant variable */
	      &&(Symbol->Initializer)
	      &&(Symbol->Initializer->Initializer))
	     { TmpResult = new SIR_Constant(Symbol->Initializer->Initializer);
	       TmpResult->SetLineInfo(this);
	       return(TmpResult->Converted(Type->ConstType(),
				_BITLEN(Type->LeftBound, Type->RightBound)));
	      } /* fi */
	  if (Symbol->IsEnumMember())		/* enum constant */
	     { TmpResult = new SIR_Constant(SIR_CONST_INT, Symbol->EnumValue);
	       TmpResult->SetLineInfo(this);
	       return(TmpResult->Converted(Type->ConstType(),
				_BITLEN(Type->LeftBound, Type->RightBound)));
	      } /* fi */
	  SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
	  return(NULL);
	 }
     case SIR_EXPR_PARENTHESES:
	{ return(Arg1->Eval());		/* simple recursion */
	 }
     case SIR_EXPR_THIS:
     case SIR_EXPR_ARRAY_ACCESS:
     case SIR_EXPR_FUNCTION_CALL:
     case SIR_EXPR_MEMBER_ACCESS:
     case SIR_EXPR_MEMBER_POINTER:
     case SIR_EXPR_POST_INCREMENT:
     case SIR_EXPR_POST_DECREMENT:
	{ SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
	  return(NULL);
	 }
     case SIR_EXPR_BITSLICE:
	{ if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  SIR_Convert2BitVector(EvalArg1);
	  TmpResult = new SIR_Constant((*(EvalArg1->BIT_Value))(LeftBound,
								RightBound));
	  TmpResult->SetLineInfo(EvalArg1);
	  delete EvalArg1;
	  assert(TmpResult->Type == Type->ConstType());	/* need no conversion */
	  return(TmpResult);
	 }
     case SIR_EXPR_PRE_INCREMENT:
     case SIR_EXPR_PRE_DECREMENT:
     case SIR_EXPR_ADDRESS_OF:
     case SIR_EXPR_CONTENT_OF:
	{ SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
	  return(NULL);
	 }
     case SIR_EXPR_POSITIVE:
	{ return(Arg1->Eval()			/* nothing to compute */
		->Converted(Type->ConstType(),	/* (but maybe to convert) */
			_BITLEN(Type->LeftBound, Type->RightBound)));
	 }
     case SIR_EXPR_NEGATIVE:
	{ switch (SIR_PrepareOperand(Arg1,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
						- TmpValueLL1, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
						- TmpValueULL1, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
						- TmpValueLD1, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(-(*TmpValueBIT1), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1;
	  return(TmpResult);
	 }
     case SIR_EXPR_NOT:
	{ switch (SIR_PrepareOperand(Arg1,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
						~ TmpValueLL1, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
						~ TmpValueULL1, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(false);	/* invalid operand */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(~(*TmpValueBIT1), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1;
	  return(TmpResult);
	 }
     case SIR_EXPR_LOGICAL_NOT:
	{ if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  EvalArg1->Converted(SIR_CONST_BOOL);
	  EvalArg1->B_Value = ! EvalArg1->B_Value;
	  assert(EvalArg1->Type == Type->ConstType());	/* already converted */
	  return(EvalArg1);
	 }
     case SIR_EXPR_SIZEOF_EXPR:
	{ TmpResult = new SIR_Constant(SIR_CONST_UINT,
				(UNSIGNED_LONG_LONG) Arg1->Type->SizeOf());
	  TmpResult->SetLineInfo(this);
	  assert(SIR_CONST_UINT == Type->ConstType());	/* need no conversion */
	  return(TmpResult);
	 }
     case SIR_EXPR_SIZEOF_TYPE:
	{ TmpResult = new SIR_Constant(SIR_CONST_UINT,
				(UNSIGNED_LONG_LONG) TypeArg->SizeOf());
	  TmpResult->SetLineInfo(this);
	  assert(SIR_CONST_UINT == Type->ConstType());	/* need no conversion */
	  return(TmpResult);
	 }
     case SIR_EXPR_TYPE_CONVERSION:
	{ if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  switch(TypeArg->Type)
	     { case SIR_TYPE_BOOL:
	       case SIR_TYPE_CHAR:
	       case SIR_TYPE_UCHAR:
	       case SIR_TYPE_SHORT:
	       case SIR_TYPE_USHORT:
	       case SIR_TYPE_INT:
	       case SIR_TYPE_UINT:
	       case SIR_TYPE_LONG:
	       case SIR_TYPE_ULONG:
	       case SIR_TYPE_LONGLONG:
	       case SIR_TYPE_ULONGLONG:
	       case SIR_TYPE_FLOAT:
	       case SIR_TYPE_DOUBLE:
	       case SIR_TYPE_LONGDOUBLE:
	       case SIR_TYPE_BIT:
	       case SIR_TYPE_UBIT:
		  { EvalArg1->Converted(	/* exploit smart enumerations */
			(SIR_CONSTTYPE) TypeArg->Type,
			_BITLEN(TypeArg->LeftBound, TypeArg->RightBound));
		    break;
		   }
	       case SIR_TYPE_VOID:
	       case SIR_TYPE_EVENT:
		  { assert(FALSE);	/* type checking has to catch this */
		   }
	       case SIR_TYPE_POINTER:
		  { SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
		    delete EvalArg1;
		    return(NULL);
		   }
	       case SIR_TYPE_STRUCT:
	       case SIR_TYPE_UNION:
		  { assert(FALSE);	/* type checking has to catch this */
		   }
	       case SIR_TYPE_ENUM:
		  { EvalArg1->Converted(SIR_CONST_INT);
		    break;
		   }
	       case SIR_TYPE_ARRAY:
	       case SIR_TYPE_FUNCTION:
		  { SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
		    delete EvalArg1;
		    return(NULL);
		   }
	       case SIR_TYPE_ANY_TYPE:
	       case SIR_TYPE_BEHAVIOR:
	       case SIR_TYPE_CHANNEL:
	       case SIR_TYPE_INTERFACE:
		  { assert(FALSE);	/* type checking has to catch this */
		   }
	       default:
		  { assert(FALSE);	/* bad Type */
		   }
	      } /* hctiws */
	  assert(EvalArg1->Type == Type->ConstType());	/* already converted */
	  return(EvalArg1);
	 }
     case SIR_EXPR_CONCATENATION:
	{ if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  if (!(EvalArg2 = Arg2->Eval()))
	     { return(NULL);
	      } /* fi */
	  SIR_Convert2BitVector(EvalArg1);
	  SIR_Convert2BitVector(EvalArg2);
	  TmpResult = new SIR_Constant(
				sir_bit::concat(*EvalArg1->BIT_Value,
						*EvalArg2->BIT_Value));
	  if (EvalArg1->LineInfo)
	     { TmpResult->SetLineInfo(EvalArg1);
	      } /* fi */
	  else
	     { TmpResult->SetLineInfo(EvalArg2);
	      } /* esle */
	  delete EvalArg1;
	  delete EvalArg2;
	  assert(TmpResult->Type == Type->ConstType());	/* need no conversion */
	  return(TmpResult);
	 }
     case SIR_EXPR_MULTIPLY:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 * TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 * TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 * TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) * (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_DIVIDE:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { if (TmpValueLL2 == 0)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 / TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { if (TmpValueULL2 == 0)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 / TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { if (TmpValueLD2 == 0.0l)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 / TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { if (*TmpValueBIT2 == 0)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(
				(*TmpValueBIT1) / (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_MODULO:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { if (TmpValueLL2 == 0)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 % TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { if (TmpValueULL2 == 0)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 % TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(FALSE);	/* type checking has to catch this */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { if (*TmpValueBIT2 == 0)
		       { SIR_Error = SIR_ERROR_DIVISION_BY_ZERO;
			 delete TmpValueBIT1; delete TmpValueBIT2;
			 return(NULL);
			} /* fi */
		    TmpResult = new SIR_Constant(
				(*TmpValueBIT1) % (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_ADD:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 + TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 + TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 + TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) + (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_SUBTRACT:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 - TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 - TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 - TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) - (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_SHIFT_LEFT:
	{ if (SIR_PrepareOperand(Arg2,
			&TmpValueLL2, &TmpValueULL2,
			&TmpValueLD2, &TmpValueBIT2,
					&Line, &File) == SIR_DO_ERROR)
	     { return(NULL);
	      } /* fi */
	  switch (SIR_PrepareOperand(Arg1,
			&TmpValueLL1, &TmpValueULL1,
			&TmpValueLD1, &TmpValueBIT1))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
					TmpValueLL1 << TmpValueBIT2->toInt(),
					Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
					TmpValueULL1 << TmpValueBIT2->toInt(),
					Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(FALSE);	/* type checking has to catch this */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) << TmpValueBIT2->toInt(),
					Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_SHIFT_RIGHT:
	{ if (SIR_PrepareOperand(Arg2,
			&TmpValueLL2, &TmpValueULL2,
			&TmpValueLD2, &TmpValueBIT2,
					&Line, &File) == SIR_DO_ERROR)
	     { return(NULL);
	      } /* fi */
	  switch (SIR_PrepareOperand(Arg1,
			&TmpValueLL1, &TmpValueULL1,
			&TmpValueLD1, &TmpValueBIT1))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
					TmpValueLL1 >> TmpValueBIT2->toInt(),
					Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
					TmpValueULL1 >> TmpValueBIT2->toInt(),
					Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(FALSE);	/* type checking has to catch this */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) >> TmpValueBIT2->toInt(),
					Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_LESS:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 < TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 < TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 < TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				(*TmpValueBIT1) < (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_GREATER:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 > TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 > TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 > TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				(*TmpValueBIT1) > (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_LESS_EQUAL:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 <= TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 <= TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 <= TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				(*TmpValueBIT1) <= (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_GREATER_EQUAL:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 >= TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 >= TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 >= TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				(*TmpValueBIT1) >= (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_EQUAL:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 == TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 == TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 == TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				(*TmpValueBIT1) == (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_NOT_EQUAL:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 != TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 != TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLD1 != TmpValueLD2, Line, File);
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				(*TmpValueBIT1) != (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_AND:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 & TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 & TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(FALSE);	/* type checking has to catch this */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) & (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_EOR:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 ^ TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 ^ TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(FALSE);	/* type checking has to catch this */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) ^ (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_OR:
	{ switch (SIR_PrepareBinaryOperands(Arg1, Arg2,
				&TmpValueLL1, &TmpValueULL1,
				&TmpValueLD1, &TmpValueBIT1,
				&TmpValueLL2, &TmpValueULL2,
				&TmpValueLD2, &TmpValueBIT2,
				&Line, &File))
	     { case SIR_DO_LL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueLL1 | TmpValueLL2, Line, File);
		    break;
		   }
	       case SIR_DO_ULL_COMPUTATION:
		  { TmpResult = new SIR_Constant(Type->ConstType(),
				TmpValueULL1 | TmpValueULL2, Line, File);
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { assert(FALSE);	/* type checking has to catch this */
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { TmpResult = new SIR_Constant(
				(*TmpValueBIT1) | (*TmpValueBIT2), Line, File);
		    break;
		   }
	       default:	/* SIR_DO_ERROR */
		  { return(NULL);
		   }
	      } /* hctiws */
	  delete TmpValueBIT1; delete TmpValueBIT2;
	  return(TmpResult);
	 }
     case SIR_EXPR_LOGICAL_AND:
	{ if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  assert(Type->ConstType() == SIR_CONST_BOOL);	/* assert bool result */
	  EvalArg1->Converted(SIR_CONST_BOOL);
	  assert(EvalArg1->Type == SIR_CONST_BOOL);
	  if (! EvalArg1->B_Value)	/* short-cut evaluation! */
	     { return(EvalArg1);
	      } /* fi */
	  delete EvalArg1;
	  if (!(EvalArg2 = Arg2->Eval()))
	     { return(NULL);
	      } /* fi */
	  return(EvalArg2->Converted(SIR_CONST_BOOL));
	 }
     case SIR_EXPR_LOGICAL_OR:
	{ if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  assert(Type->ConstType() == SIR_CONST_BOOL);	/* assert bool result */
	  EvalArg1->Converted(SIR_CONST_BOOL);
	  assert(EvalArg1->Type == SIR_CONST_BOOL);
	  if (EvalArg1->B_Value)	/* short-cut evaluation! */
	     { return(EvalArg1);
	      } /* fi */
	  delete EvalArg1;
	  if (!(EvalArg2 = Arg2->Eval()))
	     { return(NULL);
	      } /* fi */
	  return(EvalArg2->Converted(SIR_CONST_BOOL));
	 }
     case SIR_EXPR_CONDITION:
	{ bool		Test;

	  if (!(EvalArg1 = Arg1->Eval()))
	     { return(NULL);
	      } /* fi */
	  EvalArg1->Converted(SIR_CONST_BOOL);
	  assert(EvalArg1->Type == SIR_CONST_BOOL);
	  Test = EvalArg1->B_Value;
	  delete EvalArg1;
	  if (Test)	/* evaluate either Arg2 or Arg3 (only one of them) */
	     { if (!(EvalArg2 = Arg2->Eval()))
		  { return(NULL);
		   } /* fi */
	      } /* fi */
	  else
	     { if (!(EvalArg2 = Arg3->Eval()))
		  { return(NULL);
		   } /* fi */
	      } /* esle */
	  if ((int)Type->Type >= (int)SIR_CONST_CHARSTRING)
	     { SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
	       delete EvalArg2;
	       return(NULL);
	      } /* fi */
	  return(EvalArg2->Converted(Type->ConstType(),
			_BITLEN(Type->LeftBound, Type->RightBound)));
	 }
     case SIR_EXPR_ASSIGNMENT:
     case SIR_EXPR_MUL_ASSIGN:
     case SIR_EXPR_DIV_ASSIGN:
     case SIR_EXPR_MOD_ASSIGN:
     case SIR_EXPR_ADD_ASSIGN:
     case SIR_EXPR_SUB_ASSIGN:
     case SIR_EXPR_SHL_ASSIGN:
     case SIR_EXPR_SHR_ASSIGN:
     case SIR_EXPR_AND_ASSIGN:
     case SIR_EXPR_EOR_ASSIGN:
     case SIR_EXPR_OR_ASSIGN:
     case SIR_EXPR_COMMA:
	{ SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
	  return(NULL);
	 }
     default:
	{ assert(FALSE);	/* bad expression type */
	 }
    } /* hctiws */

return(NULL);	/* should never be executed */

} /* end of SIR_Expression::Eval */


int SIR_Expression::IntegerEval(void)	/* evaluate a constant expr. to int */
{					/* (may set SIR_Error!) */
sir_constant	*Result;
int		IntValue;

if (!(Result = this->Eval()))	/* evaluate myself */
   { return(0);	/* SIR_Error is set */
    } /* fi */

IntValue = 0;	/* default in case of errors */

switch(Result->Type)
   { case SIR_CONST_BOOL:
	{ IntValue = Result->B_Value;
	  break;
	 }
     case SIR_CONST_CHAR:
	{ IntValue = Result->C_Value;
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ IntValue = Result->UC_Value;
	  break;
	 }
     case SIR_CONST_SHORT:
	{ IntValue = Result->S_Value;
	  break;
	 }
     case SIR_CONST_USHORT:
	{ IntValue = Result->US_Value;
	  break;
	 }
     case SIR_CONST_INT:
	{ IntValue = Result->I_Value;
	  break;
	 }
     case SIR_CONST_UINT:
	{ IntValue = Result->UI_Value;
	  break;
	 }
     case SIR_CONST_LONG:
	{ IntValue = Result->L_Value;
	  break;
	 }
     case SIR_CONST_ULONG:
	{ IntValue = Result->UL_Value;
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ IntValue = SIR_LLONG2I(Result->LL_Value);
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ IntValue = SIR_LLONG2I(Result->ULL_Value);
	  break;
	 }
     case SIR_CONST_FLOAT:
	{ assert(FALSE);	/* should have been promoted */
	 }
     case SIR_CONST_DOUBLE:
	{ IntValue = (int)Result->D_Value;
	  break;
	 }
     case SIR_CONST_LONGDOUBLE:
	{ IntValue = (int)Result->LD_Value;
	  break;
	 }
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ IntValue = Result->BIT_Value->toInt();
	  break;
	 }
     case SIR_CONST_CHARSTRING:
	{ SIR_Error = SIR_ERROR_CONST_EXPR_NOT_INTEGER;
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad Type */
	 }
    } /* hctiws */

delete Result;	/* not needed any more */

return(IntValue);

} /* end of SIR_Expression::IntegerEval */


sir_lineinfo *SIR_Expression::GetFirstLineInfo(	/* obtain first line info */
	void)
{
sir_lineinfo	*Line;
sir_expression	*Arg;

if (LineInfo)
   { return(LineInfo);
    } /* fi */

if (  (Constant)
    &&(Constant->LineInfo))
   { return(Constant->LineInfo);
    } /* fi */
if (  (Arg1)
    &&(Line = Arg1->GetFirstLineInfo()))
   { return(Line);
    } /* fi */
if (  (Arg2)
    &&(Line = Arg2->GetFirstLineInfo()))
   { return(Line);
    } /* fi */
if (  (Arg3)
    &&(Line = Arg3->GetFirstLineInfo()))
   { return(Line);
    } /* fi */
if (Args)
   { Arg = Args->First();
     while(Arg)
	{ if ((Line = Arg->GetFirstLineInfo()))
	     { return(Line);
	      } /* fi */
	  Arg = Arg->Succ();
	 } /* elihw */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expression::GetFirstLineInfo */


sir_expression *SIR_Expression::New(	/* creates a constant expression (#1) */
	sir_constant	*NewConst,	/* (returns NULL if SIR_Error) */
	sir_types	*TypeTable)
{
sir_expression	*NewExpr;

assert(TypeTable != NULL);
assert(NewConst != NULL);

NewExpr = new SIR_Expression(SIR_EXPR_CONSTANT, NewConst,
			TypeTable->FindOrInsert(NewConst));

return(NewExpr);

} /* end of SIR_Expression::New #1 */


sir_expression *SIR_Expression::New(	/* creates expr. with 1-3 args. (#3) */
	SIR_EXPRTYPE	ExprType,	/* (returns NULL if SIR_Error) */
	sir_expression	* /*New*/ Arg1,
	sir_expression	* /*New*/ Arg2 = NULL,
	sir_expression	* /*New*/ Arg3 = NULL)
{
sir_types	*TypeTable;
sir_expression	*NewExpr;
sir_type	*ResultType = NULL;	/* keep the compiler quiet */

assert(Arg1 != NULL);
// Arg2 may be NULL
// Arg3 may be NULL

assert(Arg1->Type != NULL);
TypeTable = Arg1->Type->GetTable();
assert(TypeTable != NULL);

switch(ExprType)
   { case SIR_EXPR_VOID:
     case SIR_EXPR_CONSTANT:
     case SIR_EXPR_IDENTIFIER:
	{ assert(FALSE);	/* not supported/handled here */
	 }
     case SIR_EXPR_PARENTHESES:
	{ ResultType = Arg1->Type;	/* (nothing changes) */
	  break;
	 }
     case SIR_EXPR_THIS:
	{ assert(FALSE);	/* not supported/handled here */
	 }
     case SIR_EXPR_ARRAY_ACCESS:
	{ if (  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
	      &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	     { SIR_Error = SIR_ERROR_INVALID_ARRAY_INDEX;
	       return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ARRAY)
	      ||(Arg1->Type->Type == SIR_TYPE_POINTER))
	     { ResultType = Arg1->Type->SubType;
	      } /* fi */
	  else
	     { if (  (Arg1->Type->Type == SIR_TYPE_BIT)
		   ||(Arg1->Type->Type == SIR_TYPE_UBIT))
		  { ResultType = TypeTable->FindOrInsert(
				SIR_TYPE_UBIT,		/* [] operator on    */
				Arg1->Type->Const,	/* bitvectors yields */
				Arg1->Type->Volatile,	/* unsigned bit[0:0] */
				0, 0);
		   } /* fi */
	       else
		  { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_ARRAY_ACC;
		    return(NULL);
		   } /* esle */
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_FUNCTION_CALL:
	{ assert(FALSE);	/* handled in SIR_Expression::New() #4 */
	 }
     case SIR_EXPR_MEMBER_ACCESS:
     case SIR_EXPR_MEMBER_POINTER:
	{ assert(FALSE);	/* handled in SIR_Expression::New() #5 */
	 }
     case SIR_EXPR_POST_INCREMENT:
     case SIR_EXPR_POST_DECREMENT:
	{ if (  (Arg1->CheckReadAccess())
	      ||(Arg1->CheckWriteAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)
	      &&(Arg1->Type->Type != SIR_TYPE_POINTER))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_INCR_DECR;
	       return(NULL);
	      } /* fi */
	  ResultType = Arg1->Type;	/* (not promoted!) */
	  break;
	 }
     case SIR_EXPR_BITSLICE:
	{ assert(FALSE);	/* handled in SIR_Expression::New() #7 */
	 }
     case SIR_EXPR_PRE_INCREMENT:
     case SIR_EXPR_PRE_DECREMENT:
	{ if (  (Arg1->CheckReadAccess())
	      ||(Arg1->CheckWriteAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)
	      &&(Arg1->Type->Type != SIR_TYPE_POINTER))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_INCR_DECR;
	       return(NULL);
	      } /* fi */
	  ResultType = Arg1->Type;	/* (not promoted!) */
	  break;
	 }
     case SIR_EXPR_ADDRESS_OF:
	{ /* .... check if address operator & is legal for Arg1->Type */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_POINTER, Arg1->Type);
	  break;
	 }
     case SIR_EXPR_CONTENT_OF:
	{ if (Arg1->CheckReadAccess())
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_POINTER)
	      ||(Arg1->Type->Type == SIR_TYPE_ARRAY))
	     { ResultType = Arg1->Type->SubType;
	      } /* fi */
	  else
	     { SIR_Error = SIR_ERROR_OPERAND_NOT_POINTER_OR_ARRAY;
	       return(NULL);
	      } /* esle */
	  break;
	 }
     case SIR_EXPR_POSITIVE:
     case SIR_EXPR_NEGATIVE:
	{ SIR_TYPETYPE	Arg1Type,
			ResType;
	  int		Arg1Len,
			ResLen;

	  if (Arg1->CheckReadAccess())
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_ARITH_OP;
	       return(NULL);
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONGDOUBLE)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGDOUBLE);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_DOUBLE)
	      ||(Arg1->Type->Type == SIR_TYPE_FLOAT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_DOUBLE);
	       break;
	      } /* fi */
	  if (Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	     { if (! SIR_IsConvertable2BitVector(Arg1->Type->Type,
			Arg1->Type->LeftBound, Arg1->Type->RightBound,
			&Arg1Type, &Arg1Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_UNARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_UNARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_ULONGLONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONGLONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_ULONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_UINT)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_NOT:
	{ SIR_TYPETYPE	Arg1Type,
			ResType;
	  int		Arg1Len,
			ResLen;

	  if (Arg1->CheckReadAccess())
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
	      &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BIT_OP;
	       return(NULL);
	      } /* fi */
	  if (Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	     { if (! SIR_IsConvertable2BitVector(Arg1->Type->Type,
			Arg1->Type->LeftBound, Arg1->Type->RightBound,
			&Arg1Type, &Arg1Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_UNARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_UNARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_ULONGLONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONGLONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_ULONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_UINT)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_LOGICAL_NOT:
	{ if (Arg1->CheckReadAccess())
	     { return(NULL);
	      } /* fi */
	  if (! SIR_IsConvertable2Bool(Arg1->Type->Type))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BOOL_OP;
	       return(NULL);
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_BOOL);
	  break;
	 }
     case SIR_EXPR_SIZEOF_EXPR:
	{ switch(Arg1->Type->Type)
	     { case SIR_TYPE_VOID:
		  { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_VOID;
		    return(NULL);
		   }
	       case SIR_TYPE_EVENT:
		  { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_EVENT;
		    return(NULL);
		   }
	       case SIR_TYPE_FUNCTION:
		  { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_FUNCTION;
		    return(NULL);
		   }
	       case SIR_TYPE_ANY_TYPE:
		  { assert(FALSE);	/* cannot happen */
		   }
	       case SIR_TYPE_BEHAVIOR:
	       case SIR_TYPE_CHANNEL:
	       case SIR_TYPE_INTERFACE:
		  { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_CLASS;
		    return(NULL);
		   }
	       default:	/* any other type */
		  { /* type is legal for sizeof() operator */
		    break;
		   }
	      } /* hctiws */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	  break;
	 }
     case SIR_EXPR_SIZEOF_TYPE:
     case SIR_EXPR_TYPE_CONVERSION:
	{ assert(FALSE);	/* handled in SIR_Expression::New() #6 */
	 }
     case SIR_EXPR_CONCATENATION:
	{ SIR_TYPETYPE	Arg1Type,
			Arg2Type,
			ResType;
	  int		Arg1Len,
			Arg2Len,
			ResLen;

	  if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (! SIR_IsConvertable2BitVector(Arg1->Type->Type,
			Arg1->Type->LeftBound, Arg1->Type->RightBound,
			&Arg1Type, &Arg1Len))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_CONCAT;
	       return(NULL);
	      } /* fi */
	  if (! SIR_IsConvertable2BitVector(Arg2->Type->Type,
			Arg2->Type->LeftBound, Arg2->Type->RightBound,
			&Arg2Type, &Arg2Len))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_CONCAT;
	       return(NULL);
	      } /* fi */
	  ResType = (_BITUSGN_CONCAT(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  ResLen = _BITLEN_CONCAT(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT));
	  ResultType = TypeTable->FindOrInsert(ResType, false, false,
						ResLen-1, 0);
	  break;
	 }
     case SIR_EXPR_ADD:
     case SIR_EXPR_SUBTRACT:
	{ /* handle pointer arithmetic first */
	  if (  (Arg1->Type->Type == SIR_TYPE_POINTER)
	      ||(Arg1->Type->Type == SIR_TYPE_ARRAY))
	     { if (  (Arg1->CheckReadAccess())
		   ||(Arg2->CheckReadAccess()))
		  { return(NULL);
		   } /* fi */
	       if (  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		   &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
		  { SIR_Error = SIR_ERROR_INVALID_POINTER_ARITHMETIC;
		    return(NULL);
		   } /* fi */
	       ResultType = Arg1->Type;
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_POINTER)
	      ||(Arg2->Type->Type == SIR_TYPE_ARRAY))
	     { if (  (Arg1->CheckReadAccess())
		   ||(Arg2->CheckReadAccess()))
		  { return(NULL);
		   } /* fi */
	       if (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		   &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
		  { SIR_Error = SIR_ERROR_INVALID_POINTER_ARITHMETIC;
		    return(NULL);
		   } /* fi */
	       ResultType = Arg2->Type;
	       break;
	      } /* fi */
	  /* now go on as with multiplication, division, ... */
	 }
/*   case SIR_EXPR_ADD:		fall-through case! */
/*   case SIR_EXPR_SUBTRACT:	fall-through case! */
     case SIR_EXPR_MULTIPLY:
     case SIR_EXPR_DIVIDE:
	{ SIR_TYPETYPE	Arg1Type,
			Arg2Type,
			ResType;
	  int		Arg1Len,
			Arg2Len,
			ResLen;

	  if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_ARITH_OP;
	       return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONGDOUBLE)
	      ||(Arg2->Type->Type == SIR_TYPE_LONGDOUBLE))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGDOUBLE);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_DOUBLE)
	      ||(Arg2->Type->Type == SIR_TYPE_DOUBLE)
	      ||(Arg1->Type->Type == SIR_TYPE_FLOAT)
	      ||(Arg2->Type->Type == SIR_TYPE_FLOAT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_DOUBLE);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	      ||(Arg2->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR))
	     { if (! SIR_IsConvertable2BitVector(Arg1->Type->Type,
			Arg1->Type->LeftBound, Arg1->Type->RightBound,
			&Arg1Type, &Arg1Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       if (! SIR_IsConvertable2BitVector(Arg2->Type->Type,
			Arg2->Type->LeftBound, Arg2->Type->RightBound,
			&Arg2Type, &Arg2Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_BINARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_BINARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ULONGLONG)
	      ||(Arg2->Type->Type == SIR_TYPE_ULONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONGLONG)
	      ||(Arg2->Type->Type == SIR_TYPE_LONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ULONG)
	      ||(Arg2->Type->Type == SIR_TYPE_ULONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONG)
	      ||(Arg2->Type->Type == SIR_TYPE_LONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_UINT)
	      ||(Arg2->Type->Type == SIR_TYPE_UINT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_MODULO:
	{ SIR_TYPETYPE	Arg1Type,
			Arg2Type,
			ResType;
	  int		Arg1Len,
			Arg2Len,
			ResLen;

	  if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_MODULO_OP;
	       return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	      ||(Arg2->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR))
	     { if (! SIR_IsConvertable2BitVector(Arg1->Type->Type,
			Arg1->Type->LeftBound, Arg1->Type->RightBound,
			&Arg1Type, &Arg1Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       if (! SIR_IsConvertable2BitVector(Arg2->Type->Type,
			Arg2->Type->LeftBound, Arg2->Type->RightBound,
			&Arg2Type, &Arg2Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_BINARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_BINARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ULONGLONG)
	      ||(Arg2->Type->Type == SIR_TYPE_ULONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONGLONG)
	      ||(Arg2->Type->Type == SIR_TYPE_LONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ULONG)
	      ||(Arg2->Type->Type == SIR_TYPE_ULONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONG)
	      ||(Arg2->Type->Type == SIR_TYPE_LONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_UINT)
	      ||(Arg2->Type->Type == SIR_TYPE_UINT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_SHIFT_LEFT:
     case SIR_EXPR_SHIFT_RIGHT:
	{ SIR_TYPETYPE	Arg1Type,
			ResType;
	  int		Arg1Len,
			ResLen;

	  if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_SHIFT_OP;
	       return(NULL);
	      } /* fi */
	  /* result type is promoted type of left argument */
	  if (Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	     { if (! SIR_IsConvertable2BitVector(Arg1->Type->Type,
			Arg1->Type->LeftBound, Arg1->Type->RightBound,
			&Arg1Type, &Arg1Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_UNARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_UNARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_ULONGLONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONGLONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_ULONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_LONG)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (Arg1->Type->Type == SIR_TYPE_UINT)
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_LESS:
     case SIR_EXPR_GREATER:
     case SIR_EXPR_LESS_EQUAL:
     case SIR_EXPR_GREATER_EQUAL:
     case SIR_EXPR_EQUAL:
     case SIR_EXPR_NOT_EQUAL:
	{ if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  /* .... check if Arg1 and Arg2 are comparable */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_BOOL);
	  break;
	 }
     case SIR_EXPR_AND:
     case SIR_EXPR_EOR:
     case SIR_EXPR_OR:
	{ SIR_TYPETYPE	Arg1Type,
			Arg2Type,
			ResType;
	  int		Arg1Len,
			Arg2Len,
			ResLen;

	  if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (  (Arg1->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg1->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BIT_OP;
	       return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	      ||(Arg2->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR))
	     { if (! SIR_IsConvertable2BitVector(Arg1->Type->Type,
			Arg1->Type->LeftBound, Arg1->Type->RightBound,
			&Arg1Type, &Arg1Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       if (! SIR_IsConvertable2BitVector(Arg2->Type->Type,
			Arg2->Type->LeftBound, Arg2->Type->RightBound,
			&Arg2Type, &Arg2Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_BINARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_BINARY(Arg1Len, (Arg1Type == SIR_TYPE_UBIT),
					Arg2Len, (Arg2Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ULONGLONG)
	      ||(Arg2->Type->Type == SIR_TYPE_ULONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONGLONG)
	      ||(Arg2->Type->Type == SIR_TYPE_LONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ULONG)
	      ||(Arg2->Type->Type == SIR_TYPE_ULONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_LONG)
	      ||(Arg2->Type->Type == SIR_TYPE_LONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_UINT)
	      ||(Arg2->Type->Type == SIR_TYPE_UINT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_LOGICAL_AND:
     case SIR_EXPR_LOGICAL_OR:
	{ if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (! SIR_IsConvertable2Bool(Arg1->Type->Type))
	      ||(! SIR_IsConvertable2Bool(Arg2->Type->Type)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BOOL_OP;
	       return(NULL);
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_BOOL);
	  break;
	 }
     case SIR_EXPR_CONDITION:
	{ SIR_TYPETYPE	Arg2Type,
			Arg3Type,
			ResType;
	  int		Arg2Len,
			Arg3Len,
			ResLen;

	  if (  (Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess())
	      ||(Arg3->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (! SIR_IsConvertable2Bool(Arg1->Type->Type))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BOOL_OP;
	       return(NULL);
	      } /* fi */
	  /* check for short-cut, if Arg2 and Arg3 are of the same type */
	  if (Arg2->Type == Arg3->Type)	/* types identical? */
	     { ResultType = Arg2->Type;
	       break;
	      } /* fi */
	  /* otherwise, we apply the usual conversion rules */
	  if (  (  (Arg2->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		 &&(Arg2->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR))
	      ||(  (Arg3->Type->TypeClass() != SIR_TYPECLASS_INTEGRAL)
		 &&(Arg3->Type->TypeClass() != SIR_TYPECLASS_FLOATING)
		 &&(Arg3->Type->TypeClass() != SIR_TYPECLASS_BITVECTOR)))
	     { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_COND_EXPR;
	       return(NULL);
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_LONGDOUBLE)
	      ||(Arg3->Type->Type == SIR_TYPE_LONGDOUBLE))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGDOUBLE);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_DOUBLE)
	      ||(Arg3->Type->Type == SIR_TYPE_DOUBLE)
	      ||(Arg2->Type->Type == SIR_TYPE_FLOAT)
	      ||(Arg3->Type->Type == SIR_TYPE_FLOAT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_DOUBLE);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR)
	      ||(Arg3->Type->TypeClass() == SIR_TYPECLASS_BITVECTOR))
	     { if (! SIR_IsConvertable2BitVector(Arg2->Type->Type,
			Arg2->Type->LeftBound, Arg2->Type->RightBound,
			&Arg2Type, &Arg2Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       if (! SIR_IsConvertable2BitVector(Arg3->Type->Type,
			Arg3->Type->LeftBound, Arg3->Type->RightBound,
			&Arg3Type, &Arg3Len))
		  { assert(false);	/* must be convertable */
		   } /* fi */
	       ResType = (_BITUSGN_BINARY(Arg2Len, (Arg2Type == SIR_TYPE_UBIT),
					Arg3Len, (Arg3Type == SIR_TYPE_UBIT))
					? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	       ResLen = _BITLEN_BINARY(Arg2Len, (Arg2Type == SIR_TYPE_UBIT),
					Arg3Len, (Arg3Type == SIR_TYPE_UBIT));
	       ResultType = TypeTable->FindOrInsert(ResType, false, false,
							ResLen-1, 0);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_ULONGLONG)
	      ||(Arg3->Type->Type == SIR_TYPE_ULONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_LONGLONG)
	      ||(Arg3->Type->Type == SIR_TYPE_LONGLONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONGLONG);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_ULONG)
	      ||(Arg3->Type->Type == SIR_TYPE_ULONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_ULONG);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_LONG)
	      ||(Arg3->Type->Type == SIR_TYPE_LONG))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_LONG);
	       break;
	      } /* fi */
	  if (  (Arg2->Type->Type == SIR_TYPE_UINT)
	      ||(Arg3->Type->Type == SIR_TYPE_UINT))
	     { ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
	       break;
	      } /* fi */
	  ResultType = TypeTable->FindOrInsert(SIR_TYPE_INT);
	  break;
	 }
     case SIR_EXPR_ASSIGNMENT:
	{ if (  (Arg1->CheckWriteAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  if (  (Arg1->Type->Type == SIR_TYPE_ARRAY)	/* array assignment? */
	      &&(Arg2->Type->Type == SIR_TYPE_ARRAY))
	     { if (  (Arg1->Type->Size <= 0)
		   ||(Arg2->Type->Size <= 0)
		   ||(Arg1->Type->Size != Arg2->Type->Size)
		   ||(Arg1->Type->SubType->Modified() !=
					Arg2->Type->SubType->Modified()))
		  { SIR_Error = SIR_ERROR_INVALID_ARRAY_ASSIGNMENT;
		    return(NULL);
		   } /* fi */
	       ResultType = TypeTable->FindOrInsert(SIR_TYPE_VOID);
	       break;
	      } /* fi */
	  /* .... check if type of RHS can be assigned to LHS */
	  ResultType = Arg1->Type;	/* (not promoted!) */
	  break;
	 }
     case SIR_EXPR_MUL_ASSIGN:
     case SIR_EXPR_DIV_ASSIGN:
     case SIR_EXPR_MOD_ASSIGN:
     case SIR_EXPR_ADD_ASSIGN:
     case SIR_EXPR_SUB_ASSIGN:
     case SIR_EXPR_SHL_ASSIGN:
     case SIR_EXPR_SHR_ASSIGN:
     case SIR_EXPR_AND_ASSIGN:
     case SIR_EXPR_EOR_ASSIGN:
     case SIR_EXPR_OR_ASSIGN:
	{ if (  (Arg1->CheckWriteAccess())
	      ||(Arg1->CheckReadAccess())
	      ||(Arg2->CheckReadAccess()))
	     { return(NULL);
	      } /* fi */
	  /* .... check if type of RHS can be op.-assigned to LHS */
	  ResultType = Arg1->Type;	/* (not promoted!) */
	  break;
	 }
     case SIR_EXPR_COMMA:
	{ ResultType = Arg2->Type;	/* (not promoted!) */
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad expression type */
	 }
    } /* hctiws */

assert(ResultType != NULL);	/* must have been determined */

NewExpr = new SIR_Expression(ExprType, ResultType, Arg1, Arg2, Arg3);

return(NewExpr);

} /* end of SIR_Expression::New #3 */


sir_expression *SIR_Expression::New(	/* creates a function call expr. (#4) */
	sir_expression	* /*New*/ Arg1,	/* (returns NULL if SIR_Error) */
	sir_expressions	* /*New*/ Args)
{
sir_types	*TypeTable;
sir_expression	*NewExpr;
sir_type	*ResultType;
sir_type_ptr	*ExpectedArg;
sir_expression	*SuppliedArg;

assert(Arg1 != NULL);
assert(Args != NULL);
assert(Arg1->Type != NULL);
TypeTable = Arg1->Type->GetTable();
assert(TypeTable != NULL);

if (Arg1->Type->Type != SIR_TYPE_FUNCTION)
   { SIR_Error = SIR_ERROR_CALL_OF_NON_FUNCTION;
     return(NULL);
    } /* fi */

if (Arg1->Type->VoidParameters())
   { if (Args->NumElements() > 0)
	{ SIR_Error = SIR_ERROR_TOO_MANY_ARGUMENTS;
	  return(NULL);
	 } /* fi */
    } /* fi */
else
   { ExpectedArg = Arg1->Type->Parameters->First();
     SuppliedArg = Args->First();
     while(ExpectedArg)
	{ if (ExpectedArg->Type->Type == SIR_TYPE_ANY_TYPE)
	     { while(SuppliedArg)
		  { if (SuppliedArg->CheckReadAccess())
		       { return(NULL);
			} /* fi */
		    SuppliedArg = SuppliedArg->Succ();
		   } /* elihw */
	       break;	/* stop checking */
	      } /* fi */
	  if (!(SuppliedArg))
	     { SIR_Error = SIR_ERROR_TOO_FEW_ARGUMENTS;
	       return(NULL);
	      } /* fi */
	  if (SuppliedArg->CheckReadAccess())
	     { return(NULL);
	      } /* fi */
	  if (!(SuppliedArg->Type->IsConvertableTo(
					ExpectedArg->Type)))
	     { SIR_Error = SIR_ERROR_ARGUMENT_TYPE_MISMATCH;
	       return(NULL);
	      } /* fi */
	  ExpectedArg = ExpectedArg->Succ();
	  SuppliedArg = SuppliedArg->Succ();
	 } /* elihw */
     if (SuppliedArg)
	{ SIR_Error = SIR_ERROR_TOO_MANY_ARGUMENTS;
	  return(NULL);
	 } /* fi */
    } /* esle */

ResultType = Arg1->Type->SubType;

NewExpr = new SIR_Expression(SIR_EXPR_FUNCTION_CALL, ResultType, Arg1, Args);

return(NewExpr);

} /* end of SIR_Expression::New #4 */


sir_expression *SIR_Expression::New(	/* creates a member access expr. (#5) */
	SIR_EXPRTYPE	ExprType,	/* (returns NULL if SIR_Error) */
	sir_expression	* /*New*/ Arg1,
	const char	*MemberName)
{
sir_types	*TypeTable;
sir_expression	*NewExpr;
sir_type	*TypeEntry,
		*ResultType,
		*VoidType;
sir_symbol	*Member;
sir_symbol_ptr	*Interface;

assert(Arg1 != NULL);
assert(Arg1->Type != NULL);
TypeTable = Arg1->Type->GetTable();
assert(TypeTable != NULL);

TypeEntry = NULL;	/* keep compiler quiet */
switch(ExprType)
   { case SIR_EXPR_MEMBER_ACCESS:
	{ TypeEntry = Arg1->Type;
	  break;
	 } /* esac */
     case SIR_EXPR_MEMBER_POINTER:
	{ if (Arg1->Type->Type != SIR_TYPE_POINTER)
	     { SIR_Error = SIR_ERROR_LEFT_ARG_NOT_A_POINTER;
	       return(NULL);
	      } /* fi */
	  TypeEntry = Arg1->Type->SubType;
	  break;
	 } /* esac */
     default:
	{ assert(FALSE);	/* wrong expression type */
	 } /* tluafed */
    } /* hctiws */

assert(TypeEntry != NULL);
switch(TypeEntry->Type)
   { case SIR_TYPE_STRUCT:	/* structs/unions: every member is public */
     case SIR_TYPE_UNION:
	{ assert(TypeEntry->UserType != NULL);	/* must exist */
	  if (! TypeEntry->UserType->Scope)
	     { SIR_Error = SIR_ERROR_ACCESS_TO_INCOMPLETE_TYPE;
	       return(NULL);
	      } /* fi */
	  if (!(Member = TypeEntry->UserType->Scope->FindLocally(MemberName)))
	     { SIR_ErrMsg.form(
			"There is no member '%s' in this aggregate type",
			MemberName);
	       SIR_Error = SIR_ERROR_NO_SUCH_MEMBER_IN_USERTYPE;
	       return(NULL);
	      } /* fi */
	  ResultType = Member->Type;
	  break;
	 } /* esac */
     case SIR_TYPE_BEHAVIOR:	/* main method and impl. interf. are public */
	{ if (0 == strcmp(MemberName, GL_MAIN_METHOD_NAME))
	     { Member = NULL;	/* means: "main" */
	       VoidType = TypeTable->FindOrInsert(SIR_TYPE_VOID);
	       ResultType = TypeTable->FindOrInsert(
				SIR_TYPE_FUNCTION,
				VoidType,
				new SIR_TypePtrs(
					new SIR_TypePtr(VoidType)));
	       break;
	      } /* fi */
	/* note: while in the following code the list of implemented	*/
	/* interfaces is searched in left-to-right order, the order	*/
	/* actually doesn't matter; the only thing that counts is	*/
	/* that we find one member (which must be suitable);		*/
	/* later, during modifications to implemented interfaces or	*/
	/* their members, the same search could yield a different	*/
	/* (but compatible) member (R.D., Dec 12, 98)			*/
	  assert(TypeEntry->ClassSymbol != NULL);	/* must exist */
	  assert(TypeEntry->ClassSymbol->Interfaces != NULL);
	  Interface = TypeEntry->ClassSymbol->Interfaces->First();
	  Member = NULL;
	  while(Interface)
	     { assert(Interface->Symbol != NULL);
	       if (! Interface->Symbol->IsInterfaceDefinition())
		  { SIR_ErrMsg.form(
			"Access to incomplete interface '%s'"
							GL_ERROR_MSG_NEWLINE
			"of behavior '%s'",
				Interface->Symbol->Name.chars(),
				TypeEntry->ClassSymbol->Name.chars());
		    SIR_Error = SIR_ERROR_ACCESS_TO_INCOMPLETE_IF_1;
		    return(NULL);
		   } /* fi */
	       if ((Member = Interface->Symbol->ClassScope->FindLocally(
								MemberName)))
		  { break;
		   } /* fi */
	       Interface = Interface->Succ();
	      } /* elihw */
	  if (Member == NULL)
	     { SIR_ErrMsg.form(
		"There is no method '%s' in the list" GL_ERROR_MSG_NEWLINE
		"of implemented interfaces for behavior '%s'",
			MemberName, TypeEntry->ClassSymbol->Name.chars());
	       SIR_Error = SIR_ERROR_NO_SUCH_METHOD_IMPLEMENTED_1;
	       return(NULL);
	      } /* fi */
	  ResultType = Member->Type;
	  break;
	 } /* esac */
     case SIR_TYPE_CHANNEL:	/* members of impl. interfaces are public */
	{ assert(TypeEntry->ClassSymbol != NULL);	/* must exist */
	  assert(TypeEntry->ClassSymbol->Interfaces != NULL);
	/* note: (see above) */
	  Interface = TypeEntry->ClassSymbol->Interfaces->First();
	  Member = NULL;
	  while(Interface)
	     { assert(Interface->Symbol != NULL);
	       if (! Interface->Symbol->IsInterfaceDefinition())
		  { SIR_ErrMsg.form(
			"Access to incomplete interface '%s'"
							GL_ERROR_MSG_NEWLINE
			"of channel '%s'",
				Interface->Symbol->Name.chars(),
				TypeEntry->ClassSymbol->Name.chars());
		    SIR_Error = SIR_ERROR_ACCESS_TO_INCOMPLETE_IF_2;
		    return(NULL);
		   } /* fi */
	       if ((Member = Interface->Symbol->ClassScope->FindLocally(
								MemberName)))
		  { break;
		   } /* fi */
	       Interface = Interface->Succ();
	      } /* elihw */
	  if (Member == NULL)
	     { SIR_ErrMsg.form(
		"There is no method '%s' in the list" GL_ERROR_MSG_NEWLINE
		"of implemented interfaces for channel '%s'",
			MemberName, TypeEntry->ClassSymbol->Name.chars());
	       SIR_Error = SIR_ERROR_NO_SUCH_METHOD_IMPLEMENTED_2;
	       return(NULL);
	      } /* fi */
	  ResultType = Member->Type;
	  break;
	 } /* esac */
     case SIR_TYPE_INTERFACE:	/* interface: every method decl. is public */
	{ assert(TypeEntry->ClassSymbol != NULL);	/* must exist */
	  if (! TypeEntry->ClassSymbol->ClassScope)
	     { SIR_ErrMsg.form("Access to incomplete interface '%s'",
				TypeEntry->ClassSymbol->Name.chars());
	       SIR_Error = SIR_ERROR_ACCESS_TO_INCOMPLETE_INTERFC;
	       return(NULL);
	      } /* fi */
	  if (!(Member = TypeEntry->ClassSymbol->ClassScope->FindLocally(
								MemberName)))
	     { SIR_ErrMsg.form(
		"There is no method '%s' in interface '%s'",
			MemberName, TypeEntry->ClassSymbol->Name.chars());
	       SIR_Error = SIR_ERROR_NO_SUCH_METHOD_IN_INTERFACE;
	       return(NULL);
	      } /* fi */
	  ResultType = Member->Type;
	  break;
	 } /* esac */
     default:
	{ SIR_Error = SIR_ERROR_NOT_AN_AGGREGATE_OR_CLASS;
	  return(NULL);
	 } /* tluafed */
    } /* hctiws */

assert(ResultType != NULL);	/* must have been found */

NewExpr = new SIR_Expression(ExprType, ResultType, Arg1, Member);

return(NewExpr);

} /* end of SIR_Expression::New #5 */


sir_expression *SIR_Expression::New(	/* creates sizeof(type)/cast expr.(#6)*/
	SIR_EXPRTYPE	ExprType,	/* (returns NULL if SIR_Error) */
	sir_type	*TypeArg,
	sir_expression	* /*New*/ Arg1 = NULL)
{
sir_types	*TypeTable;
sir_expression	*NewExpr;
sir_type	*ResultType;

assert(  (ExprType == SIR_EXPR_SIZEOF_TYPE)
       ||(ExprType == SIR_EXPR_TYPE_CONVERSION));
assert(TypeArg != NULL);
TypeTable = TypeArg->GetTable();
assert(TypeTable != NULL);

if (ExprType == SIR_EXPR_SIZEOF_TYPE)
   { switch(TypeArg->Type)
	{ case SIR_TYPE_VOID:
	     { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_VOID;
	       return(NULL);
	      }
	  case SIR_TYPE_EVENT:
	     { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_EVENT;
	       return(NULL);
	      }
	  case SIR_TYPE_FUNCTION:
	     { SIR_Error = SIR_ERROR_ILLEGAL_SIZEOF_FUNCTION;
	       return(NULL);
	      }
	  case SIR_TYPE_ANY_TYPE:
	  case SIR_TYPE_BEHAVIOR:
	  case SIR_TYPE_CHANNEL:
	  case SIR_TYPE_INTERFACE:
	     { assert(FALSE);	/* forbidden by grammar */
	      }
	  default:	/* any other type */
	     { /* type is legal for sizeof() operator */
	       break;
	      }
	 } /* hctiws */
     ResultType = TypeTable->FindOrInsert(SIR_TYPE_UINT);
    } /* fi */
else	/* ExprType == SIR_EXPR_TYPE_CONVERSION */
   { /* .... check if type casting is legal */
     ResultType = TypeArg;
    } /* esle */

NewExpr = new SIR_Expression(ExprType, ResultType, TypeArg, Arg1);

return(NewExpr);

} /* end of SIR_Expression::New #6 */


sir_expression *SIR_Expression::New(	/* creates a bit slice expr. (#7) */
	sir_expression	* /*New*/ Arg1,	/* (returns NULL if SIR_Error) */
	int		LeftBound,
	int		RightBound)
{
SIR_TYPETYPE	ArgTypeType,
		ResultTypeType;
int		ArgLeftBound,
		ArgRightBound;
int		ArgBitlen,
		ResultBitlen;
sir_types	*TypeTable;
sir_expression	*NewExpr;
sir_type	*ResultType;

assert(Arg1 != NULL);
assert(Arg1->Type != NULL);
TypeTable = Arg1->Type->GetTable();
assert(TypeTable != NULL);

if (! SIR_IsConvertable2BitVector(Arg1->Type->Type,
		Arg1->Type->LeftBound, Arg1->Type->RightBound,
		&ArgTypeType, &ArgBitlen, &ArgLeftBound, &ArgRightBound))
   { SIR_Error = SIR_ERROR_INVALID_OPERAND_FOR_BITSLICE;
     return(NULL);
    } /* fi */

if (  (  (ArgLeftBound >= ArgRightBound)
       &&(  (LeftBound > ArgLeftBound)  || (RightBound > ArgLeftBound)
	  ||(LeftBound < ArgRightBound) || (RightBound < ArgRightBound)))
    ||(  (ArgLeftBound <= ArgRightBound)
       &&(  (LeftBound < ArgLeftBound)  || (RightBound < ArgLeftBound)
	  ||(LeftBound > ArgRightBound) || (RightBound > ArgRightBound))))
   { SIR_Error = SIR_ERROR_INVALID_SLICE_FOR_BITSLICE;
     SIR_ErrMsg.form("Invalid slice [%d:%d] for argument type bit[%d:%d]",
			LeftBound, RightBound, ArgLeftBound, ArgRightBound);
     return(NULL);
    } /* fi */

ResultTypeType = (_BITUSGN_SLICE(ArgBitlen, (ArgTypeType == SIR_TYPE_UBIT),
					LeftBound, RightBound) ?
					SIR_TYPE_UBIT : SIR_TYPE_BIT);
ResultBitlen = _BITLEN_SLICE(ArgBitlen, (ArgTypeType == SIR_TYPE_UBIT),
					LeftBound, RightBound);
ResultType = TypeTable->FindOrInsert(ResultTypeType,
					Arg1->Type->Const,
					Arg1->Type->Volatile,
					ResultBitlen-1, 0);

NewExpr = new SIR_Expression(SIR_EXPR_BITSLICE, ResultType,
				Arg1, LeftBound, RightBound);

return(NewExpr);

} /* end of SIR_Expression::New #7 */


	/***********************/
	/*** SIR_Expressions ***/
	/***********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Expressions::SIR_Expressions(	/* constructor #1 */
	sir_expression	*FirstEntry = NULL) :
		SIR_List<SIR_Expression>(FirstEntry)
{

/* nothing else to do */

} /* end of SIR_Expressions::SIR_Expressions #1 */


SIR_Expressions::SIR_Expressions(	/* constructor #2 (duplicator) */
	sir_expressions	*Original)	/* (recursive!) */
{
sir_expression	*Expr;

assert(Original != NULL);

Expr = Original->First();
while(Expr)
   { Append(new SIR_Expression(Expr));
     Expr = Expr->Succ();
    } /* elihw */

} /* end of SIR_Expressions::SIR_Expressions #2 */


SIR_Expressions::~SIR_Expressions(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_Expressions::~SIR_Expressions */


ERROR SIR_Expressions::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_expression	*Expr,
		*Succ;

Expr = First();
while (Expr)
   { Succ = Expr->Succ();
     if ((SIR_Error = Expr->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Expr = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expressions::DFS_ForAllNodes */


ERROR SIR_Expressions::DFS_ForAllExpressions(	/* iterator over all exprs. */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{
sir_expression	*Expr,
		*Succ;

Expr = First();
while (Expr)
   { Succ = Expr->Succ();
     if ((SIR_Error = Expr->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Expr = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Expressions::DFS_ForAllExpressions */


/*** exported functions *************************************************/


	/* none */


/*** internal functions *************************************************/


SIR_WHAT_TO_DO SIR_PrepareOperand(	/* 'ultra-promote' the constant */
	sir_expression		*Arg,	/* (fills in all possible values) */
	LONG_LONG		*TmpValueLL,
	UNSIGNED_LONG_LONG	*TmpValueULL,
	long double		*TmpValueLD,
	sir_bit			**TmpValueBIT,
	unsigned int		*Line = NULL,
	sir_fileinfo		**FileInfo = NULL)
{
SIR_WHAT_TO_DO	DoWhat;
sir_constant	*EvalArg;

if (!(EvalArg = Arg->Eval()))
   { return(SIR_DO_ERROR);
    } /* fi */

DoWhat = SIR_DO_ERROR;	/* to keep the compiler quiet */
switch(EvalArg->Type)
   { case SIR_CONST_BOOL:
	{ *TmpValueLL  = EvalArg->B_Value;
	  *TmpValueULL = EvalArg->B_Value;
	  *TmpValueLD  = EvalArg->B_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->B_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_CHAR:
	{ *TmpValueLL  = EvalArg->C_Value;
	  *TmpValueULL = EvalArg->C_Value;
	  *TmpValueLD  = EvalArg->C_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->C_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ *TmpValueLL  = EvalArg->UC_Value;
	  *TmpValueULL = EvalArg->UC_Value;
	  *TmpValueLD  = EvalArg->UC_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->UC_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_SHORT:
	{ *TmpValueLL  = EvalArg->S_Value;
	  *TmpValueULL = EvalArg->S_Value;
	  *TmpValueLD  = EvalArg->S_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->S_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_USHORT:
	{ *TmpValueLL  = EvalArg->US_Value;
	  *TmpValueULL = EvalArg->US_Value;
	  *TmpValueLD  = EvalArg->US_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->US_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_INT:
	{ *TmpValueLL  = EvalArg->I_Value;
	  *TmpValueULL = EvalArg->I_Value;
	  *TmpValueLD  = EvalArg->I_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->I_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_UINT:
	{ *TmpValueLL  = EvalArg->UI_Value;
	  *TmpValueULL = EvalArg->UI_Value;
	  *TmpValueLD  = EvalArg->UI_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->UI_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_LONG:
	{ *TmpValueLL  = EvalArg->L_Value;
	  *TmpValueULL = EvalArg->L_Value;
	  *TmpValueLD  = EvalArg->L_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->L_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_ULONG:
	{ *TmpValueLL  = EvalArg->UL_Value;
	  *TmpValueULL = EvalArg->UL_Value;
	  *TmpValueLD  = EvalArg->UL_Value;
	  *TmpValueBIT = new sir_bit(EvalArg->UL_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ *TmpValueLL  = EvalArg->LL_Value;
	  *TmpValueULL = EvalArg->LL_Value;
	  *TmpValueLD  = SIR_LLONG2LD(EvalArg->LL_Value);
	  *TmpValueBIT = new sir_bit(EvalArg->LL_Value);
	  DoWhat = SIR_DO_LL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ *TmpValueLL  = 0;
	  *TmpValueULL = EvalArg->ULL_Value;
	  *TmpValueLD  = SIR_LLONG2LD(EvalArg->ULL_Value);
	  *TmpValueBIT = new sir_bit(EvalArg->ULL_Value);
	  DoWhat = SIR_DO_ULL_COMPUTATION;
	  break;
	 }
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ *TmpValueLL  = 0;
	  *TmpValueULL = 0;
	  *TmpValueLD  = EvalArg->BIT_Value->toLDouble();
	  *TmpValueBIT = new sir_bit(*EvalArg->BIT_Value);
	  DoWhat = SIR_DO_BIT_COMPUTATION;
	  break;
	 }
     case SIR_CONST_FLOAT:
	{ *TmpValueLL  = 0;
	  *TmpValueULL = 0;
	  *TmpValueLD  = EvalArg->F_Value;
	  *TmpValueBIT = NULL;
	  DoWhat = SIR_DO_LD_COMPUTATION;
	  break;
	 }
     case SIR_CONST_DOUBLE:
	{ *TmpValueLL  = 0;
	  *TmpValueULL = 0;
	  *TmpValueLD  = EvalArg->D_Value;
	  *TmpValueBIT = NULL;
	  DoWhat = SIR_DO_LD_COMPUTATION;
	  break;
	 }
     case SIR_CONST_LONGDOUBLE:
	{ *TmpValueLL  = 0;
	  *TmpValueULL = 0;
	  *TmpValueLD  = EvalArg->LD_Value;
	  *TmpValueBIT = NULL;
	  DoWhat = SIR_DO_LD_COMPUTATION;
	  break;
	 }
     case SIR_CONST_CHARSTRING:
	{ SIR_Error = SIR_ERROR_EXPRESSION_NOT_CONSTANT;
	  DoWhat = SIR_DO_ERROR;
	  break;
	 }
     default:
	{ assert(false);	/* bad type */
	 }
    } /* hctiws */

if (Line)	/* if requested, pass line infos (if any) */
   { assert(FileInfo != NULL);
     if (EvalArg->LineInfo)
	{ *Line = EvalArg->LineInfo->Line;
	  *FileInfo = EvalArg->LineInfo->File;
	 } /* fi */
     else
	{ *Line = 0;
	  *FileInfo = NULL;
	 } /* fi */
    } /* fi */

delete EvalArg;

return(DoWhat);

} /* end of SIR_PrepareOperand */


SIR_WHAT_TO_DO SIR_PrepareBinaryOperands(
	sir_expression		*Arg1,
	sir_expression		*Arg2,
	LONG_LONG		*TmpValueLL1,
	UNSIGNED_LONG_LONG	*TmpValueULL1,
	long double		*TmpValueLD1,
	sir_bit			**TmpValueBIT1,
	LONG_LONG		*TmpValueLL2,
	UNSIGNED_LONG_LONG	*TmpValueULL2,
	long double		*TmpValueLD2,
	sir_bit			**TmpValueBIT2,
	unsigned int		*Line = NULL,
	sir_fileinfo		**FileInfo = NULL)
{
SIR_WHAT_TO_DO	DoWhat1,
		DoWhat2,
		DoWhat;

if (Line)
   { assert(FileInfo != NULL);
     if ((DoWhat1 = SIR_PrepareOperand(Arg1,
		TmpValueLL1, TmpValueULL1, TmpValueLD1, TmpValueBIT1,
						Line, FileInfo))
			== SIR_DO_ERROR)
	{ return(SIR_DO_ERROR);
	 } /* fi */
     if (*Line == 0)
	{ if ((DoWhat2 = SIR_PrepareOperand(Arg2,
		TmpValueLL2, TmpValueULL2, TmpValueLD2, TmpValueBIT2,
						Line, FileInfo))
			== SIR_DO_ERROR)
	     { delete *TmpValueBIT1;
	       return(SIR_DO_ERROR);
	      } /* fi */
	 } /* fi */
     else
	{ if ((DoWhat2 = SIR_PrepareOperand(Arg2,
		TmpValueLL2, TmpValueULL2, TmpValueLD2, TmpValueBIT2))
			== SIR_DO_ERROR)
	     { delete *TmpValueBIT1;
	       return(SIR_DO_ERROR);
	      } /* fi */
	 } /* esle */
    } /* fi */
else
   { if ((DoWhat1 = SIR_PrepareOperand(Arg1,
		TmpValueLL1, TmpValueULL1, TmpValueLD1, TmpValueBIT1))
			== SIR_DO_ERROR)
	{ return(SIR_DO_ERROR);
	 } /* fi */
     if ((DoWhat2 = SIR_PrepareOperand(Arg2,
		TmpValueLL2, TmpValueULL2, TmpValueLD2, TmpValueBIT2))
			== SIR_DO_ERROR)
	{ delete *TmpValueBIT1;
	  return(SIR_DO_ERROR);
	 } /* fi */
    } /* esle */

DoWhat = SIR_DO_ERROR;	/* keep the compiler quiet */
switch(DoWhat1)
   { case SIR_DO_LL_COMPUTATION:
	{ DoWhat = DoWhat2;
	  break;
	 }
     case SIR_DO_ULL_COMPUTATION:
	{ switch(DoWhat2)
	     { case SIR_DO_LL_COMPUTATION:
	       case SIR_DO_ULL_COMPUTATION:
		  { DoWhat = SIR_DO_ULL_COMPUTATION;
		    break;
		   }
	       case SIR_DO_BIT_COMPUTATION:
		  { DoWhat = SIR_DO_BIT_COMPUTATION;
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { DoWhat = SIR_DO_LD_COMPUTATION;
		    break;
		   }
	       default:
		  { assert(FALSE);	/* bad DoWhat2 */
		   }
	      } /* hctiws */
	  break;
	 }
     case SIR_DO_BIT_COMPUTATION:
	{ switch(DoWhat2)
	     { case SIR_DO_LL_COMPUTATION:
	       case SIR_DO_ULL_COMPUTATION:
	       case SIR_DO_BIT_COMPUTATION:
		  { DoWhat = SIR_DO_BIT_COMPUTATION;
		    break;
		   }
	       case SIR_DO_LD_COMPUTATION:
		  { DoWhat = SIR_DO_LD_COMPUTATION;
		    break;
		   }
	       default:
		  { assert(FALSE);	/* bad DoWhat2 */
		   }
	      } /* hctiws */
	  break;
	 }
     case SIR_DO_LD_COMPUTATION:
	{ DoWhat = SIR_DO_LD_COMPUTATION;
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad DoWhat1 */
	 }
    } /* hctiws */

return(DoWhat);

} /* end of SIR_PrepareBinaryOperands */


bool SIR_IsConvertable2Bool(		/* checks for legal bool conversion */
	SIR_TYPETYPE	TypeType)	/* (supports all types except '...') */
{
switch(TypeType)
   { case SIR_TYPE_BOOL:
     case SIR_TYPE_CHAR:
     case SIR_TYPE_UCHAR:
     case SIR_TYPE_SHORT:
     case SIR_TYPE_USHORT:
     case SIR_TYPE_INT:
     case SIR_TYPE_UINT:
     case SIR_TYPE_LONG:
     case SIR_TYPE_ULONG:
     case SIR_TYPE_LONGLONG:
     case SIR_TYPE_ULONGLONG:
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
	{ return(true);
	 }
     case SIR_TYPE_VOID:
     case SIR_TYPE_EVENT:
	{ return(false);
	 }
     case SIR_TYPE_POINTER:
	{ return(true);
	 }
     case SIR_TYPE_STRUCT:
     case SIR_TYPE_UNION:
	{ return(false);
	 }
     case SIR_TYPE_ENUM:
     case SIR_TYPE_ARRAY:
     case SIR_TYPE_FUNCTION:
	{ return(true);
	 }
     case SIR_TYPE_ANY_TYPE:
	{ assert(FALSE);	/* not allowed here */
	 }
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
     case SIR_TYPE_INTERFACE:
	{ return(false);
	 }
     default:
	{ assert(FALSE);	/* bad type */
	 }
    } /* hctiws */

return(false);	/* to keep the compiler quiet */

} /* end of SIR_IsConvertable2Bool */


bool SIR_IsConvertable2BitVector(	/* virtually creates a bitvector */
	SIR_TYPETYPE	OrigType,	/* (supports all types except '...') */
	int		OrigLeft,
	int		OrigRight,
	SIR_TYPETYPE	*ResultType = NULL,
	int		*ResultLen = NULL,
	int		*ResultLeft = NULL,
	int		*ResultRight = NULL)
{
SIR_TYPETYPE	TypeType	= SIR_TYPE_ANY_TYPE;
int		BitLength	= 0,	/* to keep the compiler quiet */
		LeftBound	= 0,
		RightBound	= 0;

switch(OrigType)
   { case SIR_TYPE_BOOL:
	{ TypeType = (_BITUSGN_BOOL ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_BOOL;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_CHAR:
	{ TypeType = (_BITUSGN_CHAR ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_CHAR;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_UCHAR:
	{ TypeType = (_BITUSGN_UCHAR ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_UCHAR;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_SHORT:
	{ TypeType = (_BITUSGN_SHORT ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_SHORT;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_USHORT:
	{ TypeType = (_BITUSGN_USHORT ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_USHORT;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_INT:
	{ TypeType = (_BITUSGN_INT ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_INT;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_UINT:
	{ TypeType = (_BITUSGN_UINT ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_UINT;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_LONG:
	{ TypeType = (_BITUSGN_LONG ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_LONG;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_ULONG:
	{ TypeType = (_BITUSGN_ULONG ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_ULONG;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_LONGLONG:
	{ TypeType = (_BITUSGN_LLONG ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_LLONG;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_ULONGLONG:
	{ TypeType = (_BITUSGN_ULLONG ? SIR_TYPE_UBIT : SIR_TYPE_BIT);
	  BitLength = _BITLEN_ULLONG;
	  LeftBound = BitLength - 1;
	  RightBound = 0;
	  break;
	 }
     case SIR_TYPE_FLOAT:
     case SIR_TYPE_DOUBLE:
     case SIR_TYPE_LONGDOUBLE:
	{ return(false);	/* not convertable without specified bounds */
	 }
     case SIR_TYPE_BIT:
     case SIR_TYPE_UBIT:
	{ TypeType = OrigType;
	  LeftBound = OrigLeft;
	  RightBound = OrigRight;
	  BitLength = _BITLEN(LeftBound, RightBound);
	  break;
	 }
     case SIR_TYPE_VOID:
     case SIR_TYPE_EVENT:
     case SIR_TYPE_POINTER:
     case SIR_TYPE_STRUCT:
     case SIR_TYPE_UNION:
     case SIR_TYPE_ENUM:
     case SIR_TYPE_ARRAY:
     case SIR_TYPE_FUNCTION:
	{ return(false);	/* not convertable */
	 }
     case SIR_TYPE_ANY_TYPE:
	{ assert(FALSE);	/* type '...' must not appear here */
	 }
     case SIR_TYPE_BEHAVIOR:
     case SIR_TYPE_CHANNEL:
     case SIR_TYPE_INTERFACE:
	{ return(false);	/* not convertable */
	 }
     default:
	{ assert(FALSE);	/* bad type */
	 }
    } /* hctiws */

if (ResultType)
   { *ResultType = TypeType;
    } /* fi */
if (ResultLen)
   { *ResultLen = BitLength;
    } /* fi */
if (ResultLeft)
   { *ResultLeft = LeftBound;
    } /* fi */
if (ResultRight)
   { *ResultRight = RightBound;
    } /* fi */

return(true);

} /* end of SIR_IsConvertable2BitVector */


sir_constant *SIR_Convert2BitVector(	/* converts Const to BIT or UBIT */
	sir_constant	*Const)
{
bool	WillBeUnsigned = false;	/* keep compiler quiet */

switch(Const->Type)
   { case SIR_CONST_BOOL:
	{ WillBeUnsigned = _BITUSGN_BOOL;
	  break;
	 }
     case SIR_CONST_CHAR:
	{ WillBeUnsigned = _BITUSGN_CHAR;
	  break;
	 }
     case SIR_CONST_UCHAR:
	{ WillBeUnsigned = _BITUSGN_UCHAR;
	  break;
	 }
     case SIR_CONST_SHORT:
	{ WillBeUnsigned = _BITUSGN_SHORT;
	  break;
	 }
     case SIR_CONST_USHORT:
	{ WillBeUnsigned = _BITUSGN_USHORT;
	  break;
	 }
     case SIR_CONST_INT:
	{ WillBeUnsigned = _BITUSGN_INT;
	  break;
	 }
     case SIR_CONST_UINT:
	{ WillBeUnsigned = _BITUSGN_UINT;
	  break;
	 }
     case SIR_CONST_LONG:
	{ WillBeUnsigned = _BITUSGN_LONG;
	  break;
	 }
     case SIR_CONST_ULONG:
	{ WillBeUnsigned = _BITUSGN_ULONG;
	  break;
	 }
     case SIR_CONST_LONGLONG:
	{ WillBeUnsigned = _BITUSGN_LLONG;
	  break;
	 }
     case SIR_CONST_ULONGLONG:
	{ WillBeUnsigned = _BITUSGN_ULLONG;
	  break;
	 }
     case SIR_CONST_FLOAT:
     case SIR_CONST_DOUBLE:
     case SIR_CONST_LONGDOUBLE:
	{ assert(FALSE);	/* not convertable to bitvector */
	 }			/* without specified bounds     */
     case SIR_CONST_BIT:
     case SIR_CONST_UBIT:
	{ /* nothing to do */
	  return(Const);
	 }
     case SIR_CONST_CHARSTRING:
	{ assert(FALSE);	/* not convertable to bitvector */
	 }
     default:
	{ assert(FALSE);	/* bad constant type */
	 }
    } /* hctiws */

return(Const->Converted(WillBeUnsigned ? SIR_CONST_UBIT : SIR_CONST_BIT));

} /* end of SIR_Convert2BitVector */


static void SIR_PutParenthesis(		/* put parenthesis if needed */
	SIR_EXPRTYPE	ExprType,
	char		Parenthesis,
	sir_expression	*Arg,
	SIR_WHICH_ARG	WhichArg,
	FILE		*File)
{
int	ThisPrec,
	ArgPrec;

ThisPrec = SIR_ExprPrecedence[ExprType];
ArgPrec = SIR_ExprPrecedence[Arg->ExprType];

if (ArgPrec < ThisPrec)
  { fputc(Parenthesis, File);	/* needed because of lower precedence */
    return;
   } /* fi */
if (ArgPrec == ThisPrec)
   { if (  (  ((ThisPrec == 3) || (ThisPrec == 15)) /* right-associative? */
	    &&(WhichArg == SIR_LEFT_ARG))
	 ||(  ((ThisPrec != 3) && (ThisPrec != 15)) /* left-associative? */
	    &&(WhichArg == SIR_RIGHT_ARG)))
	{ fputc(Parenthesis, File);	/* needed because of ordering */
	  return;
	 } /* fi */
    } /* fi */

/* no parenthesis needed */

} /* end of SIR_PutParenthesis */


static BOOL CastNeedsExplicitTypeConv(	/* explicit type conv. needed? */
	SIR_TYPETYPE	FromType,
	SIR_TYPETYPE	ToType)
{

if (  (FromType == SIR_TYPE_FLOAT)
    ||(FromType == SIR_TYPE_DOUBLE)
    ||(FromType == SIR_TYPE_LONGDOUBLE))
   { if (  (ToType == SIR_TYPE_BIT)
	 ||(ToType == SIR_TYPE_UBIT))
	{ return(TRUE);
	 } /* fi */
#ifndef HAVE_LLONG
     if (  (ToType == SIR_TYPE_LONGLONG)
	 ||(ToType == SIR_TYPE_ULONGLONG))
	{ return(TRUE);
	 } /* fi */
#endif /* !HAVE_LLONG */
    } /* fi */
else
   { if (  (ToType != SIR_TYPE_BIT)
	 &&(ToType != SIR_TYPE_UBIT))
	{ if (  (FromType == SIR_TYPE_BIT)
	      ||(FromType == SIR_TYPE_UBIT))
	     { return(TRUE);
	      } /* fi */
#ifndef HAVE_LLONG
	  if (  (FromType == SIR_TYPE_LONGLONG)
	      ||(FromType == SIR_TYPE_ULONGLONG))
	     { return(TRUE);
	      } /* fi */
#endif /* !HAVE_LLONG */
	 } /* fi */
    } /* esle */

return(FALSE);	/* implicit, automatic, or forbidden conversion */

} /* end of CastNeedsExplicitTypeConv */


/* EOF Expression.cc */
