/************************************************************************/
/* Statement.cc: SpecC Internal Representation, Statement Class		*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/20/97 */
/************************************************************************/

/* last update: 05/31/01 */

/* modifications: (most recent first)
 *
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/27/01 RD	added semantic checking for event port accesses
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 05/21/01 RD	added support for (native, non-native) array assignment
 * 05/12/01 RD	bug fix: generate correct C++ code for empty 'par' and 'pipe'
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 04/18/01 RD	adjusted 'long long' constants for non-native longlong
 * 04/05/01 RD	added support for new 'pipe' construct (with flushing)
 * 03/23/01 RD	added SIR_Statement::WrapInCompound() and
 *		SIR_Statement::CompoundSubBlocks()
 * 03/20/01 RD	extended pipe construct to support termination
 * 01/30/01 RD	fixed a potential FMR problem in iterators
 *		SIR_Statements::DFS_ForAllXxxxx
 */


#include "IntRep/Statement.h"
#include "IntRep/Design.h"
#include "IntRep/Extern.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* macro for behavior main calls */

#define SIR_CALLED_BEHAVIOR(Stmnt)	\
	((Stmnt)->Expression1->Arg1->Arg1->Symbol)


/*** internal type declarations *****************************************/


	/* (none) */


/*** class implementations **********************************************/


	/*********************/
	/*** SIR_Statement ***/
	/*********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Statement::SIR_Statement(		/* constructor #1 */
	SIR_STMNT_TYPE	StmntType,
	sir_symbol	*Function,
	sir_expression	*Expression1 = NULL,
	sir_statement	*Statement1 = NULL,
	sir_statement	*Statement2 = NULL,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL,
	sir_expression	*Expression2 = NULL,
	sir_expression	*Expression3 = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(Function != NULL);

switch(StmntType)
   { case SIR_STMNT_LABELED:
     case SIR_STMNT_COMPOUND:
	{ assert(FALSE);	/* wrong constructor */
	 }
     case SIR_STMNT_EXPRESSION:
	{ assert(Expression1 != NULL);
	  break;
	 }
     case SIR_STMNT_IF:
	{ assert(Expression1 != NULL);
	  assert(Statement1 != NULL);
	  break;
	 }
     case SIR_STMNT_IF_ELSE:
	{ assert(Expression1 != NULL);
	  assert(Statement1 != NULL);
	  assert(Statement2 != NULL);
	  break;
	 }
     case SIR_STMNT_SWITCH:
	{ assert(Expression1 != NULL);
	  assert(Statement1 != NULL);
	  break;
	 }
     case SIR_STMNT_CASE:
	{ assert(FALSE);	/* wrong constructor */
	 }
     case SIR_STMNT_DEFAULT:
	{ assert(Statement1 != NULL);
	  break;
	 }
     case SIR_STMNT_WHILE:
     case SIR_STMNT_DO_WHILE:
	{ assert(Expression1 != NULL);
	  assert(Statement1 != NULL);
	  break;
	 }
     case SIR_STMNT_FOR:
	{ assert(Statement1 != NULL);
	  assert(Expression1 != NULL);
	  assert(Expression2 != NULL);
	  assert(Expression3 != NULL);
	  break;
	 }
     case SIR_STMNT_GOTO:
	{ assert(FALSE);	/* wrong constructor */
	 }
     case SIR_STMNT_CONTINUE:
     case SIR_STMNT_BREAK:
	{ break;
	 }
     case SIR_STMNT_RETURN:
	{ assert(Expression1 != NULL);
	  break;
	 }
     case SIR_STMNT_PAR:
     case SIR_STMNT_PIPE:
	{ assert(Statement1 != NULL);
	  assert(  (  (Expression1 == NULL)
		    &&(Expression2 == NULL)
		    &&(Expression3 == NULL))
		 ||(  (Expression1 != NULL)
		    &&(Expression2 != NULL)
		    &&(Expression3 != NULL)));
	  break;
	 }
     case SIR_STMNT_EXCEPTION:
     case SIR_STMNT_TIMING:
     case SIR_STMNT_FSM:
     case SIR_STMNT_WAIT:
	{ assert(FALSE);	/* wrong constructor */
	 }
     case SIR_STMNT_WAITFOR:
	{ assert(Expression1 != NULL);
	  break;
	 }
     case SIR_STMNT_NOTIFY:
     case SIR_STMNT_NOTIFYONE:
	{ assert(FALSE);	/* wrong constructor */
	 }
     default:
	{ assert(FALSE);	/* bad StmntType */
	 }
    } /* hctiws */

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= Expression1;
SIR_Statement::Expression2	= Expression2;
SIR_Statement::Expression3	= Expression3;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= Statement2;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

} /* end of SIR_Statement::SIR_Statement #1 */


SIR_Statement::SIR_Statement(		/* constructor #2 */
	SIR_STMNT_TYPE	StmntType,	/* (for compound statements) */
	sir_symbol	*Function,
	sir_symbols	*Scope,
	sir_statements	*Statements,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_COMPOUND);
assert(Function != NULL);
assert(Scope != NULL);
assert(Scope->ParentSymbol != NULL);	/* we always want a parent function */
/* Statements is temorary NULL in Parser */

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= NULL;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= Scope;
SIR_Statement::Statements	= Statements;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

Scope->ParentStmnt = this;	/* automatically fill in the back pointer */

if (Statements)
   { Statements->CmpndScope = Scope;	/* fill in compound scope link */
    } /* fi */

} /* end of SIR_Statement::SIR_Statement #2 */


SIR_Statement::SIR_Statement(		/* constructor #3 */
	SIR_STMNT_TYPE	StmntType,	/* (for case statements) */
	sir_symbol	*Function,
	sir_constant	*Constant,
	sir_statement	*Statement1 = NULL,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_CASE);
assert(Function != NULL);
assert(Constant != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= Constant;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

} /* end of SIR_Statement::SIR_Statement #3 */


SIR_Statement::SIR_Statement(		/* constructor #4 */
	SIR_STMNT_TYPE	StmntType,	/* (for statements with labels) */
	sir_symbol	*Function,
	sir_label	*Label,
	sir_statement	*Statement1 = NULL,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(  (StmntType == SIR_STMNT_GOTO)
       ||(StmntType == SIR_STMNT_LABELED));
assert(Function != NULL);
assert(Label != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= Label;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

} /* end of SIR_Statement::SIR_Statement #4 */


SIR_Statement::SIR_Statement(		/* constructor #5 */
	SIR_STMNT_TYPE	StmntType,	/* (for statements with events) */
	sir_symbol	*Function,
	sir_symbol_ptrs	*Events,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(  (StmntType == SIR_STMNT_WAIT)
       ||(StmntType == SIR_STMNT_NOTIFY)
       ||(StmntType == SIR_STMNT_NOTIFYONE));
assert(Function != NULL);
assert(Events != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= NULL;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= Events;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

} /* end of SIR_Statement::SIR_Statement #5 */


SIR_Statement::SIR_Statement(		/* constructor #6 */
	SIR_STMNT_TYPE	StmntType,	/* (for exception statements) */
	sir_symbol	*Function,
	sir_statement	*Statement1,
	sir_exceptions	*Exceptions,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_EXCEPTION);
assert(Function != NULL);
assert(Statement1 != NULL);
assert(Exceptions != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= Exceptions;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= NULL;

} /* end of SIR_Statement::SIR_Statement #6 */


SIR_Statement::SIR_Statement(		/* constructor #7 */
	SIR_STMNT_TYPE	StmntType,	/* (for timing statement) */
	sir_symbol	*Function,
	sir_statement	*Statement1,
	sir_constraints	*Constraints,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_TIMING);
assert(Function != NULL);
assert(Statement1 != NULL);
assert(Constraints != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= Statement1;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= Constraints;
SIR_Statement::Transitions	= NULL;

} /* end of SIR_Statement::SIR_Statement #7 */


SIR_Statement::SIR_Statement(		/* constructor #8 */
	SIR_STMNT_TYPE	StmntType,	/* (for fsm statement) */
	sir_symbol	*Function,
	sir_transitions	*Transitions,
	unsigned int	Line = 0,
	sir_fileinfo	*FileInfo = NULL) :
		SIR_Node(Line, FileInfo)
{

assert(StmntType == SIR_STMNT_FSM);
assert(Function != NULL);
assert(Transitions != NULL);

SIR_Statement::StmntType	= StmntType;
SIR_Statement::Function		= Function;
SIR_Statement::Expression1	= NULL;
SIR_Statement::Expression2	= NULL;
SIR_Statement::Expression3	= NULL;
SIR_Statement::Constant		= NULL;
SIR_Statement::Statement1	= NULL;
SIR_Statement::Statement2	= NULL;
SIR_Statement::Label		= NULL;
SIR_Statement::Scope		= NULL;
SIR_Statement::Statements	= NULL;
SIR_Statement::Events		= NULL;
SIR_Statement::Exceptions	= NULL;
SIR_Statement::Constraints	= NULL;
SIR_Statement::Transitions	= Transitions;

} /* end of SIR_Statement::SIR_Statement #8 */


SIR_Statement::SIR_Statement(		/* constructor #10 (duplicator) */
	sir_statement	*Original) :
		SIR_Node(Original)
{

SIR_Statement::StmntType	= Original->StmntType;
SIR_Statement::Function		= Original->Function;
SIR_Statement::Expression1	= Original->Expression1 ?
					new SIR_Expression(
						Original->Expression1) :
					NULL;
SIR_Statement::Expression2	= Original->Expression2 ?
					new SIR_Expression(
						Original->Expression2) :
					NULL;
SIR_Statement::Expression3	= Original->Expression3 ?
					new SIR_Expression(
						Original->Expression3) :
					NULL;
SIR_Statement::Constant		= Original->Constant ?
					new SIR_Constant(
						Original->Constant) :
					NULL;
SIR_Statement::Statement1	= Original->Statement1 ?
					new SIR_Statement(
						Original->Statement1) :
					NULL;
SIR_Statement::Statement2	= Original->Statement2 ?
					new SIR_Statement(
						Original->Statement2) :
					NULL;
SIR_Statement::Label		= Original->Label;
SIR_Statement::Scope		= Original->Scope ?
					new SIR_Symbols(
						Original->Scope) :
					NULL;
SIR_Statement::Statements	= Original->Statements ?
					new SIR_Statements(
						Original->Statements) :
					NULL;
SIR_Statement::Events		= Original->Events ?
					new SIR_SymbolPtrs(
						Original->Events) :
					NULL;
SIR_Statement::Exceptions	= Original->Exceptions ?
					new SIR_Exceptions(
						Original->Exceptions) :
					NULL;
SIR_Statement::Constraints	= Original->Constraints ?
					new SIR_Constraints(
						Original->Constraints) :
					NULL;
SIR_Statement::Transitions	= Original->Transitions ?
					new SIR_Transitions(
						Original->Transitions) :
					NULL;

} /* end of SIR_Statement::SIR_Statement #10 */


SIR_Statement::~SIR_Statement(void)	/* destructor */
{

delete Expression1;
delete Expression2;
delete Expression3;
delete Constant;
delete Statement1;
delete Statement2;
delete Scope;
delete Statements;
delete Events;
delete Exceptions;
delete Constraints;
delete Transitions;

} /* end of SIR_Statement::~SIR_Statement */


BOOL SIR_Statement::IsCleanBehaviorCall( /* checks for "clean" behavior call */
	void)					/* (e.g. "b1.main();") */
{
BOOL	Test;

Test = (  (StmntType == SIR_STMNT_EXPRESSION)
	&&(Expression1->ExprType == SIR_EXPR_FUNCTION_CALL)
	&&(Expression1->Arg1->ExprType == SIR_EXPR_MEMBER_ACCESS)
	&&(Expression1->Arg1->Arg1->Type->Type == SIR_TYPE_BEHAVIOR)
	&&(Expression1->Arg1->Arg1->ExprType == SIR_EXPR_IDENTIFIER)
	&&(Expression1->Arg1->Symbol == NULL)	/* means: "main"! */
	&&(Expression1->Args->Empty()));

if (Test)
   { assert(SIR_CALLED_BEHAVIOR(this) != NULL);
    } /* fi */

return(Test);

} /* end of SIR_Statement::IsCleanBehaviorCall */


BOOL SIR_Statement::IsCleanListOfBehaviorCalls(	/* check "clean" comp. bhvr. */
	BOOL	OneCallMax = FALSE,	/* (e.g. "{ b1.main(); b2.main(); }") */
	BOOL	OneCallMin = TRUE)
{
sir_statement	*Stmnt;

if (StmntType != SIR_STMNT_COMPOUND)
   { return(FALSE);
    } /* fi */

assert(Scope != NULL);
assert(Statements != NULL);

if (  (! Scope->Empty())
    ||(! Scope->UserTypes->Empty()))
   { return(FALSE);
    } /* fi */

if (  (OneCallMin)
    &&(Statements->NumElements() < 1))
   { return(FALSE);
    } /* fi */
if (  (OneCallMax)
    &&(Statements->NumElements() > 1))
   { return(FALSE);
    } /* fi */

Stmnt = Statements->First();
while(Stmnt)
   { if (! Stmnt->IsCleanBehaviorCall())
	{ return(FALSE);
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

return(TRUE);	/* looks clean */

} /* end of SIR_Statement::IsCleanListOfBehaviorCalls */


ERROR SIR_Statement::CheckCompoundBehavior( /* checks for "clean" beh. calls */
	BOOL	OneCallMax = FALSE,	/* (e.g. "{ b1.main(); b2.main(); }") */
	BOOL	OneCallMin = TRUE)	/* (returns error if not clean) */
{
string TmpString;

assert(StmntType == SIR_STMNT_COMPOUND);

if (! IsCleanListOfBehaviorCalls(OneCallMax, OneCallMin))
   { SIR_ErrMsg = "Behavioral hierarchy not clean in compound statement;"
							GL_ERROR_MSG_NEWLINE
			"unable to generate simulation model";
     if (LineInfo)
	{ TmpString.form(GL_ERROR_MSG_NEWLINE
			"(line %d, file \"%s\")",
			LineInfo->Line, LineInfo->File->Filename.chars());
	  SIR_ErrMsg += TmpString;
	 } /* fi */
     return(SIR_ERROR_COMPOUND_STMNT_NOT_CLEAN);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckCompoundBehavior */


BOOL SIR_Statement::IsDirectDependant(	/* checks if stmnt. depends on symbol */
	sir_symbol	*ThatSymbol)	/* (directly) */
{
sir_symbol_ptr	*Event;
sir_exception	*Except;
sir_transition	*Trans;

/* Function is a _back_ link, so ignore it */

if (Events)
   { Event = Events->First();
     while(Event)
	{ if (Event->Symbol == ThatSymbol)
	     { return(TRUE);		/* references that event */
	      } /* fi */
	  Event = Event->Succ();
	 } /* elihw */
    } /* fi */

if (Exceptions)
   { Except = Exceptions->First();
     while(Except)
	{ assert(Except->Events != NULL);
	  Event = Except->Events->First();
	  while(Event)
	     { if (Event->Symbol == ThatSymbol)
		  { return(TRUE);	/* references that event */
		   } /* fi */
	       Event = Event->Succ();
	      } /* elihw */
	  Except = Except->Succ();
	 } /* elihw */
    } /* fi */

if (Transitions)
   { Trans = Transitions->First();
     while(Trans)
	{ if (  (Trans->CurrState == ThatSymbol)
	      ||(Trans->NextState == ThatSymbol))
	     { return(TRUE);		/* references that state */
	      } /* fi */
	  Trans = Trans->Succ();
	 } /* elihw */
    } /* fi */

return(FALSE);	/* independent! */

} /* end of SIR_Statement::IsDirectDependant */


BOOL SIR_Statement::IsIndirectDependant(/* checks if stmnt. depends on symbol */
	sir_symbol	*ThatSymbol,	/* (indirectly) */
	sir_expression	**DepExpr = NULL)
{
sir_transition	*Trans;

if (Expression1)
   { if (Expression1->DFS_FindDependant(ThatSymbol, DepExpr))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Expression2)
   { if (Expression2->DFS_FindDependant(ThatSymbol, DepExpr))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Expression3)
   { if (Expression3->DFS_FindDependant(ThatSymbol, DepExpr))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Transitions)
   { Trans = Transitions->First();
     while(Trans)
	{ if (Trans->Condition)
	     { if (Trans->Condition->DFS_FindDependant(ThatSymbol, DepExpr))
		  { return(TRUE);
		   } /* fi */
	      } /* fi */
	  Trans = Trans->Succ();
	 } /* elihw */
    } /* fi */

return(FALSE);	/* independent! */

} /* end of SIR_Statement::IsIndirectDependant */


sir_statements *SIR_Statement::GetList(	/* determines the list of this stmnt. */
	void)
{

return((sir_statements*) Head());

} /* end of SIR_Statement::GetList */


ERROR SIR_Statement::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

if (Expression1)
   { if ((SIR_Error = Expression1->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Expression2)
   { if ((SIR_Error = Expression2->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Expression3)
   { if ((SIR_Error = Expression3->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Constant)
   { if ((SIR_Error = Constant->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Constraints)
   { if ((SIR_Error = Constraints->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Transitions)
   { if ((SIR_Error = Transitions->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllNodes */


ERROR SIR_Statement::DFS_ForAllSymbols(	/* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{

/* this is not a symbol */

/* there are no symbols in Expression1 */
/* there are no symbols in Expression2 */
/* there are no symbols in Expression3 */
/* there are no symbols in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no symbols in Constraints */
/* there are no symbols in Transitions */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllSymbols */


ERROR SIR_Statement::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{

/* this is not a usertype */

/* there are no usertypes in Expression1 */
/* there are no usertypes in Expression2 */
/* there are no usertypes in Expression3 */
/* there are no usertypes in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no usertypes in Constraints */
/* there are no usertypes in Transitions */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllUserTypes */


ERROR SIR_Statement::DFS_ForAllNotes( /* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

/* this is not a note */

/* there are no notes in Expression1 */
/* there are no notes in Expression2 */
/* there are no notes in Expression3 */
/* there are no notes in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Scope)
   { if ((SIR_Error = Scope->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no notes in Constraints */
/* there are no notes in Transitions */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllNotes */


ERROR SIR_Statement::DFS_ForAllStatements( /* iterator over all statements */
	sir_stmnt_mptr	MemberFct,	/* (depth first) */
	sir_stmnt_marg	MemberFctArg)
{

if ((SIR_Error = (this->*MemberFct)(MemberFctArg)))	/* process this node */
   { return(SIR_Error);
    } /* fi */

/* there are no statements in Expression1 */
/* there are no statements in Expression2 */
/* there are no statements in Expression3 */
/* there are no statements in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllStatements(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllStatements(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no statements in Scope */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllStatements(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllStatements(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no statements in Constraints */
/* there are no statements in Transitions */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllStatements */


ERROR SIR_Statement::DFS_ForAllExpressions( /* iterator over all expressions */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{

/* this is not an expression */

if (Expression1)
   { if ((SIR_Error = Expression1->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Expression2)
   { if ((SIR_Error = Expression2->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Expression3)
   { if ((SIR_Error = Expression3->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Constant */
if (Statement1)
   { if ((SIR_Error = Statement1->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if ((SIR_Error = Statement2->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Scope */
if (Statements)
   { if ((SIR_Error = Statements->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { if ((SIR_Error = Exceptions->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
/* there are no expressions in Constraints */
if (Transitions)
   { if ((SIR_Error = Transitions->DFS_ForAllExpressions(MemberFct,
							MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::DFS_ForAllExpressions */


BOOL SIR_Statement::DFS_FindDependant(	/* searches for dependants (DFS) */
	sir_symbol	*ThatSymbol,
	sir_statement	**DepStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason)
{
sir_exception	*Except;

assert(ThatSymbol != NULL);

if (IsDirectDependant(ThatSymbol))	/* check this statement */
   { if (DepStmnt)			/* (for direct dep.) */
	{ *DepStmnt = this;
	 } /* fi */
     if (DepExpr)
	{ *DepExpr = NULL;
	 } /* fi */
     if (Reason)
	{ *Reason = SIR_DEP_STATEMENT;
	 } /* fi */
     return(TRUE);
    } /* fi */

if (IsIndirectDependant(ThatSymbol, DepExpr))	/* check this statement */
   { if (DepStmnt)				/* (for indirect dep.) */
	{ *DepStmnt = this;
	 } /* fi */
     if (Reason)
	{ *Reason = SIR_DEP_EXPRESSION;
	 } /* fi */
     return(TRUE);
    } /* fi */

/* there are no statements in Expression1 */
/* there are no statements in Expression2 */
/* there are no statements in Expression3 */
/* there are no statements in Constant */
if (Statement1)
   { if (Statement1->DFS_FindDependant(ThatSymbol, DepStmnt, DepExpr, Reason))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Statement2)
   { if (Statement2->DFS_FindDependant(ThatSymbol, DepStmnt, DepExpr, Reason))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
/* there are no statements in Scope */
if (Statements)
   { if (Statements->DFS_FindDependant(ThatSymbol, DepStmnt, DepExpr, Reason))
	{ return(TRUE);
	 } /* fi */
    } /* fi */
if (Exceptions)
   { Except = Exceptions->First();
     while(Except)
	{ assert(Except->Handler != NULL);
	  if (Except->Handler->DFS_FindDependant(ThatSymbol, DepStmnt,
							DepExpr, Reason))
	     { return(TRUE);
	      } /* fi */
	  Except = Except->Succ();
	 } /* elihw */
    } /* fi */
/* there are no statements in Constraints */
/* there are no statements in Transitions */

return(FALSE);

} /* end of SIR_Statement::DFS_FindDependant */


ERROR SIR_Statement::MarkYourselfAndYourLabel(	/* iterator for color marking */
	sir_stmnt_marg	ColorArg)
{
SIR_COLOR	MyColor;

MyColor = *((SIR_COLOR*) ColorArg);

Color = MyColor;		/* mark yourself */
if (StmntType == SIR_STMNT_LABELED)
   { assert(Label != NULL);
     Label->Color = MyColor;	/* mark the label */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::MarkYourselfAndYourLabel */


ERROR SIR_Statement::CheckRequiredLabel(	/* iterator for label check */
	sir_stmnt_marg	/* Unused */)
{
sir_constraint	*Constr;

if (StmntType == SIR_STMNT_GOTO)
   { assert(Label != NULL);
     if (Label->Color == SIR_RED)
	{ SIR_ErrMsg.form("Label '%s' is required",
				Label->LabelName.chars());
	  return(SIR_ERROR_LABEL_REQUIRED);
	 } /* fi */
    } /* fi */
if (StmntType == SIR_STMNT_TIMING)
   { Constr = Constraints->First();
     while(Constr)
	{ assert(Constr->Label1 != NULL);
	  if (Constr->Label1->Color == SIR_RED)
	     { SIR_ErrMsg.form("Label '%s' is required",
					Constr->Label1->LabelName.chars());
	       return(SIR_ERROR_LABEL_REQUIRED);
	      } /* fi */
	  assert(Constr->Label2 != NULL);
	  if (Constr->Label2->Color == SIR_RED)
	     { SIR_ErrMsg.form("Label '%s' is required",
					Constr->Label2->LabelName.chars());
	       return(SIR_ERROR_LABEL_REQUIRED);
	      } /* fi */
	  Constr = Constr->Succ();
	 } /* elihw */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckRequiredLabel */


ERROR SIR_Statement::CheckLabeledStmnt(	/* iterator for label check */
	sir_stmnt_marg	/* Unused */)
{

if (StmntType == SIR_STMNT_LABELED)
   { assert(Label != NULL);
     SIR_ErrMsg.form("Double definition of label '%s'",
				Label->LabelName.chars());
     return(SIR_ERROR_DOUBLE_DEFINITION_OF_LABEL);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckLabeledStmnt */


ERROR SIR_Statement::CheckReturnType(	/* iterator for return type checking */
	sir_stmnt_marg	TypeArg)
{
sir_type	*ReturnType;

ReturnType = (sir_type*) TypeArg;

if (StmntType == SIR_STMNT_RETURN)
   { if (!(Expression1->Type->IsConvertableTo(ReturnType)))
	{ return(SIR_ERROR_RETURN_TYPE_MISMATCH);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::CheckReturnType */


ERROR SIR_Statement::NeedsRangeChecker(	/* iterator for range checking */
	sir_stmnt_marg	/* Unused */)
{

if (  (StmntType == SIR_STMNT_TIMING)
    &&(!(Constraints->Empty())))
   { return(SIR_ERROR_FLAG_YES);
    } /* fi */

return(SIR_ERROR_FLAG_NO);

} /* end of SIR_Statement::NeedsRangeChecker */


void SIR_Statement::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_statement	*Alias)	/* (iterates over symbols and usertypes) */
{

assert(Alias != NULL);

/* no need to process this node */

/* there are no symbols or usertypes in Expression1 */
/* there are no symbols or usertypes in Expression2 */
/* there are no symbols or usertypes in Expression3 */
/* there are no symbols or usertypes in Constant */
if (Statement1)
   { Statement1->SetAlias(Alias->Statement1);
    } /* fi */
if (Statement2)
   { Statement2->SetAlias(Alias->Statement2);
    } /* fi */
if (Scope)
   { Scope->SetAlias(Alias->Scope);
    } /* fi */
if (Statements)
   { Statements->SetAlias(Alias->Statements);
    } /* fi */
if (Exceptions)
   { Exceptions->SetAlias(Alias->Exceptions);
    } /* fi */
/* there are no symbols or usertypes in Constraints */
/* there are no symbols or usertypes in Transitions */

} /* end of SIR_Statement::SetAlias */


void SIR_Statement::UnAlias(	/* unalias all type, usertype, symbol links */
	sir_symbols	*GlobalSymbols)
{
sir_statement	*Stmnt;

if (  (Function)
    &&(Function->Alias))
   { Function = Function->Alias;
    } /* fi */
if (Expression1)
   { Expression1->UnAlias();
    } /* fi */
if (Expression2)
   { Expression2->UnAlias();
    } /* fi */
if (Expression3)
   { Expression3->UnAlias();
    } /* fi */
/* there is nothing to unalias in Constant */
if (Statement1)
   { Statement1->UnAlias(GlobalSymbols);
    } /* fi */
if (Statement2)
   { Statement2->UnAlias(GlobalSymbols);
    } /* fi */
if (Scope)
   { Scope->UnAlias(GlobalSymbols);
    } /* fi */
if (Statements)
   { Stmnt = Statements->First();
     while(Stmnt)
	{ Stmnt->UnAlias(GlobalSymbols);
	  Stmnt = Stmnt->Succ();
	 } /* elihw */
    } /* fi */
if (Events)
   { Events->UnAlias();
    } /* fi */
if (Exceptions)
   { Exceptions->UnAlias(GlobalSymbols);
    } /* fi */
/* there is nothing to unalias in Constraints */
if (Transitions)
   { Transitions->UnAlias();
    } /* fi */

} /* end of SIR_Statement::UnAlias */


ERROR SIR_Statement::WritePipeStmnt(	/* generates C++ code for 'pipe' */
	FILE		*File,
	BOOL		WriteNotes)
{
sir_statement	*Stmnt;
sir_symbol	*PipedSymbol;
sir_symbols	*PipedScope;

if (Expression1)	/* terminating (new) version */
   { assert(Expression2 != NULL);
     assert(Expression3 != NULL);
     fputs("unsigned int " SIR_CXX_FIRST_NAME ", " SIR_CXX_LAST_NAME ";",
		File);
     SIR_LineInfo::WriteNL(File);
     fputs("" SIR_CXX_FIRST_NAME " = " SIR_CXX_LAST_NAME " = 1;", File);
     SIR_LineInfo::WriteNL(File);
     fputs("for(", File);
     if ((SIR_Error = Expression1->WriteSC(File, WriteNotes, true)))
	{ return(SIR_Error);
	 } /* fi */
     fputs("; ", File);
     if ((SIR_Error = Expression2->WriteSC(File, WriteNotes, true,
							SIR_TYPE_BOOL)))
	{ return(SIR_Error);
	 } /* fi */
     fputs("; ", File);
     if ((SIR_Error = Expression3->WriteSC(File, WriteNotes, true)))
	{ return(SIR_Error);
	 } /* fi */
     fputc(')', File);
     SIR_LineInfo::WriteNL(File);
     fprintf(File, "   { " SIR_SIM_PIPE_NAME "(%u, "
			SIR_CXX_FIRST_NAME ", " SIR_CXX_LAST_NAME ",",
		Statement1->Statements->NumElements());
     SIR_LineInfo::WriteNL(File);
     Stmnt = Statement1->Statements->First();	/* threads */
     while(Stmnt)
	{ assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
	  fputs("\t&" SIR_CXX_FORK_VARIABLE, File);
	  fputs(SIR_CALLED_BEHAVIOR(Stmnt)->Name.chars(), File);
	  fputc(',', File);
	  SIR_LineInfo::WriteNL(File);
	  Stmnt = Stmnt->Succ();
	 } /* elihw */
     PipedScope = Statement1->Scope->Parent;	/* list of piped symbols */
     while(  (PipedScope)			/* in scope		 */
	   &&(PipedScope->ScopeInfo != SIR_SCOPE_GLOBAL))
	{ PipedSymbol = PipedScope->First();
	  while(PipedSymbol)
	     { if (PipedSymbol->StorageClass == SIR_STORAGE_PIPED)
		  { fputs("\t&", File);
		    fputs(PipedSymbol->Name.chars(), File);
		    fputc(',', File);
		    SIR_LineInfo::WriteNL(File);
		   } /* fi */
	       PipedSymbol = PipedSymbol->Succ();
	      } /* elihw */
	  PipedScope = PipedScope->Parent;
	 } /* elihw */
     fputs("\t0);", File);
     SIR_LineInfo::WriteNL(File);
     fprintf(File, "     if (" SIR_CXX_LAST_NAME " < %u) "
				SIR_CXX_LAST_NAME "++;",
		Statement1->Statements->NumElements());
     SIR_LineInfo::WriteNL(File);
     fputs("    }", File);
     SIR_LineInfo::WriteNL(File);
     fputs("while (" SIR_CXX_FIRST_NAME "++ < " SIR_CXX_LAST_NAME ")", File);
     SIR_LineInfo::WriteNL(File);
     fprintf(File, "   { " SIR_SIM_PIPE_NAME "(%u, " SIR_CXX_FIRST_NAME
				", " SIR_CXX_LAST_NAME ",",
		Statement1->Statements->NumElements());
     SIR_LineInfo::WriteNL(File);
     Stmnt = Statement1->Statements->First();	/* threads */
     while(Stmnt)
	{ assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
	  fputs("\t&" SIR_CXX_FORK_VARIABLE, File);
	  fputs(SIR_CALLED_BEHAVIOR(Stmnt)->Name.chars(), File);
	  fputc(',', File);
	  SIR_LineInfo::WriteNL(File);
	  Stmnt = Stmnt->Succ();
	 } /* elihw */
     PipedScope = Statement1->Scope->Parent;	/* list of piped symbols */
     while(  (PipedScope)			/* in scope		 */
	   &&(PipedScope->ScopeInfo != SIR_SCOPE_GLOBAL))
	{ PipedSymbol = PipedScope->First();
	  while(PipedSymbol)
	     { if (PipedSymbol->StorageClass == SIR_STORAGE_PIPED)
		  { fputs("\t&", File);
		    fputs(PipedSymbol->Name.chars(), File);
		    fputc(',', File);
		    SIR_LineInfo::WriteNL(File);
		   } /* fi */
	       PipedSymbol = PipedSymbol->Succ();
	      } /* elihw */
	  PipedScope = PipedScope->Parent;
	 } /* elihw */
     fputs("\t0);", File);
     SIR_LineInfo::WriteNL(File);
     fprintf(File, "     if (" SIR_CXX_LAST_NAME " < %u) "
				SIR_CXX_LAST_NAME "++;",
		Statement1->Statements->NumElements());
     SIR_LineInfo::WriteNL(File);
     fputs("    }", File);
    } /* fi */
else			/* non-terminating (old) version */
   { assert(Expression2 == NULL);
     assert(Expression3 == NULL);
     fputs(SIR_SIM_PIPE_NAME "(", File);
     Stmnt = Statement1->Statements->First();
     while(Stmnt)
	{ assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
	  fputs("&" SIR_CXX_FORK_VARIABLE, File);
	  fputs(SIR_CALLED_BEHAVIOR(Stmnt)->Name.chars(), File);
	  fputc(',', File);
	  SIR_LineInfo::WriteNL(File);
	  fputc('\t', File);
	  Stmnt = Stmnt->Succ();
	 } /* elihw */
     fputs("0,", File);		/* list of piped symbols */
     SIR_LineInfo::WriteNL(File);	/* in scope		 */
     PipedScope = Statement1->Scope->Parent;
     while(  (PipedScope)
	   &&(PipedScope->ScopeInfo != SIR_SCOPE_GLOBAL))
	{ PipedSymbol = PipedScope->First();
	  while(PipedSymbol)
	     { if (PipedSymbol->StorageClass == SIR_STORAGE_PIPED)
		  { fputs("\t&", File);
		    fputs(PipedSymbol->Name.chars(), File);
		    fputc(',', File);
		    SIR_LineInfo::WriteNL(File);
		   } /* fi */
	       PipedSymbol = PipedSymbol->Succ();
	      } /* elihw */
	  PipedScope = PipedScope->Parent;
	 } /* elihw */
     fputs("\t0);", File);
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of WritePipeStmnt */


ERROR SIR_Statement::WriteSC(	/* (re-) generates SpecC source code */
	FILE		*File,
	BOOL		WriteNotes,
	BOOL		CplusplusMode = FALSE,
	BOOL		PutNewLine = TRUE,
	BOOL		PutArgNotes = TRUE)
{
sir_statement	*Stmnt;
sir_symbol_ptr	*Event;
sir_exception	*Except;
sir_constraint	*Constr;
sir_transition	*Trans;
sir_symbol	*ParamSymbol;
sir_constant	*TmpConst;
int		ExceptNum;
int		LabelNum;
string		TmpString;

if (LineInfo)
   { if ((SIR_Error = LineInfo->WriteSC(File, true)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { if (PutNewLine)
	{ SIR_LineInfo::WriteNewSC(File, true);
	 } /* fi */
     else
	{ SIR_LineInfo::WriteSPACE(File);
	 } /* fi */
    } /* esle */

switch(StmntType)
   { case SIR_STMNT_LABELED:
	{ fputs(Label->LabelName.chars(), File);
	  fputc(':', File);
	  if (Label->Color == SIR_RED)	/* is it a timing label? */
	     { SIR_LineInfo::WriteNewSC(File, true);
	       fputs(SIR_CXX_TIMESTAMP_PREFX, File);
	       fputs(Label->LabelName.chars(), File);
	       fputs(" = " SIR_SIM_CURR_TIME_NAME ";", File);
	      } /* fi */
	  if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
						CplusplusMode, FALSE)))
	     { return(SIR_Error);
	      } /* fi */
	  if (Label->Notes)
	     { if ((SIR_Error = Label->Notes->WriteSC(File, WriteNotes,
							CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  break;
	 }
     case SIR_STMNT_COMPOUND:
	{ fputc('{', File);
	  assert(Scope != NULL);	/* must have local scope */
	  if (  (PutArgNotes)
	      &&(Scope->Parent)	/* at highest level in function body? */
	      &&(Scope->Parent->ScopeInfo == SIR_SCOPE_PARAMETER))
	     { ParamSymbol = Scope->Parent->First();
	       while(ParamSymbol)
		  { if (ParamSymbol->Notes)	/* put notes at arguments */
		       { if ((SIR_Error = ParamSymbol->Notes->WriteSC(File,
						WriteNotes, CplusplusMode)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    ParamSymbol = ParamSymbol->Succ();
		   } /* elihw */
	      } /* fi */
	  if (CplusplusMode)
	     { if ((SIR_Error = Scope->WriteH(File, WriteNotes)))
		  { return(SIR_Error);
		   } /* fi */
	       if ((SIR_Error = Scope->WriteCC(File, WriteNotes)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* fi */
	  else
	     { if ((SIR_Error = Scope->WriteSC(File, WriteNotes)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* esle */
	  if (Statements)
	     { Stmnt = Statements->First();
	       while(Stmnt)
		  { if ((SIR_Error = Stmnt->WriteSC(File, WriteNotes,
							CplusplusMode, TRUE)))
		       { return(SIR_Error);
			} /* fi */
		    Stmnt = Stmnt->Succ();
		   } /* elihw */
	      } /* fi */
	  fputc('}', File);
	  break;
	 }
     case SIR_STMNT_EXPRESSION:
	{
#ifndef HAVE_ARYASGN
	  if (  (CplusplusMode)		/* array assignment handling */
	      &&(Expression1->ExprType == SIR_EXPR_ASSIGNMENT)
	      &&(Expression1->Arg1->Type->Type == SIR_TYPE_ARRAY)
	      &&(Expression1->Arg2->Type->Type == SIR_TYPE_ARRAY))
	     { sir_type		*ArrayType;
	       unsigned int	Depth,
				Index;
	       Depth = 0;
	       ArrayType = Expression1->Arg1->Type;
	       while(ArrayType->Type == SIR_TYPE_ARRAY)
		  { assert(ArrayType->Size > 0);	/* type checked */
		    fprintf(File, "{ unsigned int " SIR_CXX_INDEX_NAME "%u; "
					"for(" SIR_CXX_INDEX_NAME "%u=0;"
					SIR_CXX_INDEX_NAME "%u<%d;"
					SIR_CXX_INDEX_NAME "%u++) ",
				Depth, Depth, Depth, ArrayType->Size, Depth);
		    Depth++;
		    ArrayType = ArrayType->SubType;
		   } /* elihw */
	       if ((SIR_Error = Expression1->Arg1->WriteSC(File, WriteNotes,
							CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       for(Index=0; Index<Depth; Index++)
		  { fprintf(File, "[" SIR_CXX_INDEX_NAME "%u]", Index);
		   } /* rof */
	       fputs(" = ", File);
	       if ((SIR_Error = Expression1->Arg2->WriteSC(File, WriteNotes,
							CplusplusMode)))
		  { return(SIR_Error);
		   } /* fi */
	       for(Index=0; Index<Depth; Index++)
		  { fprintf(File, "[" SIR_CXX_INDEX_NAME "%u]", Index);
		   } /* rof */
	       fputc(';', File);
	       for(Index=0; Index<Depth; Index++)
		  { fputs(" }", File);
		   } /* rof */
	       break;
	      } /* fi */
#endif /* HAVE_ARYASGN */
	  if ((SIR_Error = Expression1->WriteSC(File, WriteNotes,
						CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  fputc(';', File);
	  break;
	 }
     case SIR_STMNT_IF:
	{ fputs("if (", File);
	  if ((SIR_Error = Expression1->WriteSC(File, WriteNotes,
						CplusplusMode, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  fputc(')', File);
	  if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_STMNT_IF_ELSE:
	{ fputs("if (", File);
	  if ((SIR_Error = Expression1->WriteSC(File, WriteNotes,
						CplusplusMode, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  fputc(')', File);
	  if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_LineInfo::WriteNL(File);
	  fputs("else ", File);
	  if ((SIR_Error = Statement2->WriteSC(File, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_STMNT_SWITCH:
	{ fputs("switch(", File);
	  if ((SIR_Error = Expression1->WriteSC(File, WriteNotes,
						CplusplusMode, SIR_TYPE_INT)))
	     { return(SIR_Error);
	      } /* fi */
	  fputc(')', File);
	  if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_STMNT_CASE:
	{ fputs("case ", File);
	  fputs(Constant->Print(), File);
	  fputc(':', File);
	  if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
						CplusplusMode, FALSE)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_STMNT_DEFAULT:
	{ fputs("default:", File);
	  if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
						CplusplusMode, FALSE)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_STMNT_WHILE:
	{ fputs("while(", File);
	  if ((SIR_Error = Expression1->WriteSC(File, WriteNotes,
						CplusplusMode, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  fputc(')', File);
	  if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_STMNT_DO_WHILE:
	{ fputs("do", File);
	  if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  SIR_LineInfo::WriteNL(File);
	  fputs("while(", File);
	  if ((SIR_Error = Expression1->WriteSC(File, WriteNotes,
						CplusplusMode, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  fputs(");", File);
	  break;
	 }
     case SIR_STMNT_FOR:
	{ fputs("for(", File);
	  if ((SIR_Error = Expression1->WriteSC(File, WriteNotes,
						CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  fputs("; ", File);
	  if ((SIR_Error = Expression2->WriteSC(File, WriteNotes,
						CplusplusMode, SIR_TYPE_BOOL)))
	     { return(SIR_Error);
	      } /* fi */
	  fputs("; ", File);
	  if ((SIR_Error = Expression3->WriteSC(File, WriteNotes,
						CplusplusMode)))
	     { return(SIR_Error);
	      } /* fi */
	  fputc(')', File);
	  if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
						CplusplusMode, TRUE)))
	     { return(SIR_Error);
	      } /* fi */
	  break;
	 }
     case SIR_STMNT_GOTO:
	{ fputs("goto ", File);
	  fputs(Label->LabelName.chars(), File);
	  fputc(';', File);
	  break;
	 }
     case SIR_STMNT_CONTINUE:
	{ fputs("continue;", File);
	  break;
	 }
     case SIR_STMNT_BREAK:
	{ fputs("break;", File);
	  break;
	 }
     case SIR_STMNT_RETURN:
	{ SIR_TYPETYPE	ExpectType;

	  fputs("return ", File);
	  assert(Function->Type->Type == SIR_TYPE_FUNCTION);
	  ExpectType = Function->Type->SubType->Type;
	  if (  (ExpectType == SIR_TYPE_BIT)
	      ||(ExpectType == SIR_TYPE_UBIT))
	     { ExpectType = SIR_TYPE_ANY_TYPE;	/* 'relaxed' expected type */
	      } /* fi */
	  if ((SIR_Error = Expression1->WriteSC(File,
					WriteNotes, CplusplusMode,
					ExpectType)))
	     { return(SIR_Error);
	      } /* fi */
	  fputc(';', File);
	  break;
	 }
     case SIR_STMNT_PAR:
	{ if (CplusplusMode)
	     { if ((SIR_Error = Statement1->CheckCompoundBehavior(
							FALSE, FALSE)))
		  { return(SIR_Error);
		   } /* fi */
	       fputc('{', File);
	       SIR_LineInfo::WriteNewSC(File, true);
	       if (! Statement1->Statements->Empty())
		  { fputs(SIR_SIM_FORK_NAME " ", File);
		    Stmnt = Statement1->Statements->First();
		    while(Stmnt)
		       { assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
			 fputs(SIR_CXX_FORK_VARIABLE, File);
			 fputs(SIR_CALLED_BEHAVIOR(Stmnt)->Name.chars(), File);
			 fputs("(&", File);
			 fputs(SIR_CALLED_BEHAVIOR(Stmnt)->Name.chars(), File);
			 fputc(')', File);
			 Stmnt = Stmnt->Succ();
			 if (Stmnt)
			    { fputc(',', File);
			      SIR_LineInfo::WriteNL(File);
			      fputc('\t', File);
			     } /* fi */
			} /* elihw */
		    fputc(';', File);
		    SIR_LineInfo::WriteNL(File);
		   } /* fi */
	       fputs(SIR_SIM_PAR_NAME "(", File);
	       Stmnt = Statement1->Statements->First();
	       while(Stmnt)
		  { assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
		    fputs("&" SIR_CXX_FORK_VARIABLE, File);
		    fputs(SIR_CALLED_BEHAVIOR(Stmnt)->Name.chars(), File);
		    fputc(',', File);
		    SIR_LineInfo::WriteNL(File);
		    fputc('\t', File);
		    Stmnt = Stmnt->Succ();
		   } /* elihw */
	       fputs("0); }", File);
	      } /* fi */
	  else
	     { fputs("par", File);
	       if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
							CplusplusMode, FALSE)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_PIPE:
	{ if (CplusplusMode)
	     { if ((SIR_Error = Statement1->CheckCompoundBehavior(
							FALSE, FALSE)))
		  { return(SIR_Error);
		   } /* fi */
	       fputc('{', File);
	       SIR_LineInfo::WriteNewSC(File, true);
	       if (! Statement1->Statements->Empty())
		  { fputs(SIR_SIM_FORK_NAME " ", File);
		    Stmnt = Statement1->Statements->First();
		    while(Stmnt)
		       { assert(SIR_CALLED_BEHAVIOR(Stmnt) != NULL);
			 fputs(SIR_CXX_FORK_VARIABLE, File);
			 fputs(SIR_CALLED_BEHAVIOR(Stmnt)->Name.chars(), File);
			 fputs("(&", File);
			 fputs(SIR_CALLED_BEHAVIOR(Stmnt)->Name.chars(), File);
			 fputc(')', File);
			 Stmnt = Stmnt->Succ();
			 if (Stmnt)
			    { fputc(',', File);
			      SIR_LineInfo::WriteNL(File);
			      fputc('\t', File);
			     } /* fi */
			} /* elihw */
		    fputc(';', File);
		    SIR_LineInfo::WriteNL(File);
		   } /* fi */
	       if ((SIR_Error = WritePipeStmnt(File, WriteNotes)))
		  { return(SIR_Error);
		   } /* fi */
	       fputc('}', File);
	      } /* fi */
	  else
	     { fputs("pipe", File);
	       if (Expression1)
		  { assert(Expression2 != NULL);
		    assert(Expression3 != NULL);
		    fputc('(', File);
		    if ((SIR_Error = Expression1->WriteSC(File, WriteNotes,
							CplusplusMode)))
		       { return(SIR_Error);
			} /* fi */
		    fputs("; ", File);
		    if ((SIR_Error = Expression2->WriteSC(File, WriteNotes,
						CplusplusMode, SIR_TYPE_BOOL)))
		       { return(SIR_Error);
			} /* fi */
		    fputs("; ", File);
		    if ((SIR_Error = Expression3->WriteSC(File, WriteNotes,
							CplusplusMode)))
		       { return(SIR_Error);
			} /* fi */
		    fputc(')', File);
		   } /* fi */
	       else
		  { assert(Expression2 == NULL);
		    assert(Expression3 == NULL);
		   } /* esle */
	       if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
							CplusplusMode, FALSE)))
		  { return(SIR_Error);
		   } /* fi */
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_EXCEPTION:
	{ if (CplusplusMode)
	     { if ((SIR_Error = Statement1->CheckCompoundBehavior(TRUE, FALSE)))
		  { return(SIR_Error);
		   } /* fi */
	       fputc('{', File);
	       SIR_LineInfo::WriteNewSC(File, true);
	       fputs(SIR_SIM_TRY_BLOCK " " SIR_CXX_TRY_VARIABLE, File);
	       if (Statement1->Statements->Empty())
		  { fputs("0(0);", File);	/* empty try block */
		   } /* fi */
	       else
		  { fputs(SIR_CALLED_BEHAVIOR(Statement1->Statements->First())
							->Name.chars(), File);
		    fputs("(&", File);
		    fputs(SIR_CALLED_BEHAVIOR(Statement1->Statements->First())
							->Name.chars(), File);
		    fputs(");", File);
		   } /* esle */
	       SIR_LineInfo::WriteNL(File);
	       Except = Exceptions->First();
	       ExceptNum = 0;
	       while(Except)
		  { if ((SIR_Error =
			Except->Handler->CheckCompoundBehavior(TRUE, FALSE)))
		       { return(SIR_Error);
			} /* fi */
		    fputs(SIR_SIM_EXCEPT_BLOCK " " SIR_CXX_EXCEPT_VARIABLE,
				File);
		    if (Except->Handler->Statements->Empty())
		       { fprintf(File, "%d", ExceptNum);
			} /* fi */
		    else
		       { fputs(SIR_CALLED_BEHAVIOR(
				Except->Handler->Statements->First())
						->Name.chars(), File);
			} /* esle */
		    if (Except->ExceptionType == SIR_EXCEPTION_TRAP)
		       { fputs("(" SIR_SIM_TRAP_FLAG, File);
			} /* fi */
		    else /* SIR_EXCEPTION_INTERRUPT */
		       { fputs("(" SIR_SIM_INTERRUPT_FLAG, File);
			} /* esle */
		    if (Except->Handler->Statements->Empty())
		       { fputs(", 0", File);
			} /* fi */
		    else
		       { fputs(", &", File);
			 fputs(SIR_CALLED_BEHAVIOR(
				Except->Handler->Statements->First())
						->Name.chars(), File);
			} /* esle */
		    Event = Except->Events->First();
		    while(Event)
		       { fputs(", &", File);
			 fputs(Event->Symbol->Name.chars(), File);
			 Event = Event->Succ();
			} /* elihw */
		    fputs(", 0);", File);
		    SIR_LineInfo::WriteNL(File);
		    Except = Except->Succ();
		    ExceptNum++;
		   } /* elihw */
	       fputs(SIR_SIM_EXCEPTION_NAME "(", File);
	       fputs("&" SIR_CXX_TRY_VARIABLE, File);
	       if (Statement1->Statements->Empty())
		  { fputs("0", File);
		   } /* fi */
	       else
		  { fputs(SIR_CALLED_BEHAVIOR(Statement1->Statements->First())
							->Name.chars(), File);
		   } /* esle */
	       Except = Exceptions->First();
	       ExceptNum = 0;
	       while(Except)
		  { fputc(',', File);
		    SIR_LineInfo::WriteNL(File);
		    fputs("\t&" SIR_CXX_EXCEPT_VARIABLE, File);
		    if (Except->Handler->Statements->Empty())
		       { fprintf(File, "%d", ExceptNum);
			} /* fi */
		    else
		       { fputs(SIR_CALLED_BEHAVIOR(
				Except->Handler->Statements->First())
						->Name.chars(), File);
			} /* esle */
		    Except = Except->Succ();
		    ExceptNum++;
		   } /* elihw */
	       fputs(", 0); }", File);
	      } /* fi */
	  else	/* not CplusplusMode */
	     { fputs("try", File);
	       if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
							CplusplusMode, FALSE)))
		  { return(SIR_Error);
		   } /* fi */
	       Except = Exceptions->First();
	       while(Except)
		  { if (Except->LineInfo)
		       { if ((SIR_Error =
				Except->LineInfo->WriteSC(File, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(File, true);
			} /* esle */
		    switch(Except->ExceptionType)
		       { case SIR_EXCEPTION_TRAP:
			    { fputs("trap(", File);
			      break;
			     }
			 case SIR_EXCEPTION_INTERRUPT:
			    { fputs("interrupt(", File);
			      break;
			     }
			 default:
			    { assert(FALSE);	/* bad exception type */
			     }
			} /* hctiws */
		    Event = Except->Events->First();
		    while(Event)
		       { fputs(Event->Symbol->Name.chars(), File);
			 Event = Event->Succ();
			 if (Event)
			    { fputs(", ", File);
			     } /* fi */
			} /* elihw */
		    fputc(')', File);
		    if ((SIR_Error = Except->Handler->WriteSC(File, WriteNotes,
							CplusplusMode, FALSE)))
		       { return(SIR_Error);
			} /* fi */
		    Except = Except->Succ();
		   } /* elihw */
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_TIMING:
	{ if (CplusplusMode)
	     { fputc('{', File);
	       assert(Statement1->Statements != NULL);
	       Stmnt = Statement1->Statements->First();
	       LabelNum = 0;
	       while(Stmnt)
		  { if (Stmnt->StmntType == SIR_STMNT_LABELED)
		       { LabelNum++;
			 if (LabelNum == 1)
			    { SIR_LineInfo::WriteNewSC(File, true);
			      fputs(" " SIR_CXX_TIME_TYPE " ", File);
			     } /* fi */
			 else
			    { fputc(',', File);
			      SIR_LineInfo::WriteNL(File);
			      fputc('\t', File);
			     } /* esle */
			 fputs(SIR_CXX_TIMESTAMP_PREFX, File);
			 fputs(Stmnt->Label->LabelName.chars(), File);
			 assert(Stmnt->Label->Color == SIR_WHITE);
			 Stmnt->Label->Color = SIR_RED;	/* mark timing labels */
			} /* fi */
		    Stmnt = Stmnt->Succ();
		   } /* elihw */
	       if (LabelNum > 0)
		  { fputc(';', File);
		   } /* fi */
	       if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
						CplusplusMode, TRUE)))
		  { return(SIR_Error);
		   } /* fi */
	       Constr = Constraints->First();
	       while(Constr)
		  { if (Constr->LineInfo)
		       { if ((SIR_Error = Constr->LineInfo->WriteSC(
								File, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(File, true);
			} /* esle */
		    fputs(SIR_CXX_RANGE_NAME "(" SIR_CXX_TIMESTAMP_PREFX, File);
		    fputs(Constr->Label1->LabelName.chars(), File);
		    fputs(", " SIR_CXX_TIMESTAMP_PREFX, File);
		    fputs(Constr->Label2->LabelName.chars(), File);
		    fputs(", ", File);
		    if (Constr->MinTime)
		       { TmpConst = (new SIR_Constant(Constr->MinTime))
					->Converted(SIR_CONST_LONGLONG);
			 fputs(TmpConst->Print(TRUE), File);
			 delete TmpConst;
			 fputs(", true, ", File);
			} /* fi */
		    else
		       { fputs("0, false, ", File);
			} /* esle */
		    if (Constr->MaxTime)
		       { TmpConst = (new SIR_Constant(Constr->MaxTime))
					->Converted(SIR_CONST_LONGLONG);
			 fputs(TmpConst->Print(TRUE), File);
			 delete TmpConst;
			 fputs(", true, ", File);
			} /* fi */
		    else
		       { fputs("0, false, ", File);
			} /* esle */
		    fputc('"', File);
		    fputs(Constr->Label1->LabelName.chars(), File);
		    fputs("\", \"", File);
		    fputs(Constr->Label2->LabelName.chars(), File);
		    fputs("\", ", File);
		    if (Constr->LineInfo)
		       { SIR_Constant::PrettyString(&TmpString,
				Constr->LineInfo->File->Filename.chars());
			 fputs(TmpString.chars(), File);
			 fprintf(File, ", %d", Constr->LineInfo->Line);
			} /* fi */
		    else
		       { if (SIR_LineInfo::CurrentFile)
			    { SIR_Constant::PrettyString(&TmpString,
						SIR_LineInfo::CurrentFile);
			      fputs(TmpString.chars(), File);
			      fprintf(File, ", %d", SIR_LineInfo::CurrentLine);
			     } /* fi */
			 else
			    { fputs("\"<unknown>\", 0", File);
			     } /* esle */
			} /* esle */
		    fputs(");", File);
		    Constr = Constr->Succ();
		   } /* elihw */
	       Stmnt = Statement1->Statements->First();
	       while(Stmnt)
		  { if (Stmnt->StmntType == SIR_STMNT_LABELED)
		       { assert(Stmnt->Label->Color == SIR_RED);
			 Stmnt->Label->Color = SIR_WHITE;	/* unmark */
			} /* fi */
		    Stmnt = Stmnt->Succ();
		   } /* elihw */
	       fputc('}', File);
	      } /* fi */
	  else
	     { fputs("do", File);
	       if ((SIR_Error = Statement1->WriteSC(File, WriteNotes,
						CplusplusMode, TRUE)))
		  { return(SIR_Error);
		   } /* fi */
	       SIR_LineInfo::WriteNL(File);
	       fputs("timing {", File);
	       Constr = Constraints->First();
	       while(Constr)
		  { if (Constr->LineInfo)
		       { if ((SIR_Error = Constr->LineInfo->WriteSC(
								File, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(File, true);
			} /* esle */
		    fputs("range(", File);
		    fputs(Constr->Label1->LabelName.chars(), File);
		    fputs("; ", File);
		    fputs(Constr->Label2->LabelName.chars(), File);
		    fputs("; ", File);
		    if (Constr->MinTime)
		       { fputs(Constr->MinTime->Print(CplusplusMode), File);
			} /* fi */
		    fputs("; ", File);
		    if (Constr->MaxTime)
		       { fputs(Constr->MaxTime->Print(CplusplusMode), File);
			} /* fi */
		    fputs(");", File);
		    Constr = Constr->Succ();
		   } /* elihw */
	       fputc('}', File);
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_FSM:
	{ if (CplusplusMode)
	     { fputc('{', File);
	       SIR_LineInfo::WriteNewSC(File, true);
	       fputs("enum { " SIR_CXX_STATE_PREFIX "0", File);
	       Trans = Transitions->First();
	       while(Trans)
		  { if (  (Trans->Pred() == NULL)
			||(Trans->Pred()->CurrState != Trans->CurrState))
		       { fputc(',', File);
			 SIR_LineInfo::WriteNL(File);
			 fputs(SIR_CXX_STATE_PREFIX, File);
			 fputs(Trans->CurrState->Name.chars(), File);
			} /* fi */
		    Trans = Trans->Succ();
		   } /* elihw */
	       fputs(" }", File);
	       SIR_LineInfo::WriteNL(File);
	       fputs("\t" SIR_CXX_STATE_VARIABLE
				" = " SIR_CXX_STATE_PREFIX, File);
	       if (Transitions->First())
		  { fputs(Transitions->First()->CurrState->Name.chars(), File);
		   } /* fi */
	       else
		  { fputs("0", File);
		   } /* esle */
	       fputc(';', File);
	       SIR_LineInfo::WriteNL(File);
	       fputs("do switch(" SIR_CXX_STATE_VARIABLE ") {", File);
	       Trans = Transitions->First();
	       while(Trans)
		  { if (Trans->LineInfo)
		       { if ((SIR_Error = Trans->LineInfo->WriteSC(File, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNL(File);
			} /* esle */
		    fputs("case " SIR_CXX_STATE_PREFIX, File);
		    fputs(Trans->CurrState->Name.chars(), File);
		    fputs(": { ", File);
		    fputs(Trans->CurrState->Name.chars(), File);
		    fputs("." GL_MAIN_METHOD_NAME "(); ", File);
		    if (Trans->Condition)
		       { fputs("if (", File);
			 if ((SIR_Error = Trans->Condition->WriteSC(File,
					WriteNotes, TRUE, SIR_TYPE_BOOL)))
			    { return(SIR_Error);
			     } /* fi */
			 fputs(") ", File);
			} /* fi */
		    fputs("{ " SIR_CXX_STATE_VARIABLE " = "
					SIR_CXX_STATE_PREFIX, File);
		    if (Trans->NextState)
		       { fputs(Trans->NextState->Name.chars(), File);
			} /* fi */
		    else
		       { fputc('0', File);
			} /* else */
		    fputs("; break; }", File);
		    while(  (Trans->Succ())
			  &&(Trans->Succ()->CurrState == Trans->CurrState))
		       { Trans = Trans->Succ();	/* combine same CurrStates */
			 if (Trans->LineInfo)
			    { if ((SIR_Error = Trans->LineInfo->WriteSC(File,
									true)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { SIR_LineInfo::WriteNL(File);
			     } /* esle */
			 if (Trans->Condition)
			    { fputs("if (", File);
			      if ((SIR_Error = Trans->Condition->WriteSC(File,
					WriteNotes, TRUE, SIR_TYPE_BOOL)))
				 { return(SIR_Error);
				  } /* fi */
			      fputs(") ", File);
			     } /* fi */
			 fputs("{ " SIR_CXX_STATE_VARIABLE " = "
					SIR_CXX_STATE_PREFIX, File);
			 if (Trans->NextState)
			    { fputs(Trans->NextState->Name.chars(), File);
			     } /* fi */
			 else
			    { fputc('0', File);
			     } /* esle */
			 fputs("; break; }", File);
			} /* elihw */
		    fputs(" }", File);
		    Trans = Trans->Succ();
		   } /* elihw */
	       SIR_LineInfo::WriteNewSC(File, true);
	       fputs("case " SIR_CXX_STATE_PREFIX "0: { "
			SIR_CXX_STATE_VARIABLE " = " SIR_CXX_STATE_PREFIX "0; "
			"break; } }", File);
	       SIR_LineInfo::WriteNL(File);
	       fputs("while(" SIR_CXX_STATE_VARIABLE " != "
					SIR_CXX_STATE_PREFIX "0); }", File);
	      } /* fi */
	  else
	     { fputs("fsm {", File);
	       Trans = Transitions->First();
	       while(Trans)
		  { if (Trans->LineInfo)
		       { if ((SIR_Error = Trans->LineInfo->WriteSC(File, true)))
			    { return(SIR_Error);
			     } /* fi */
			} /* fi */
		    else
		       { SIR_LineInfo::WriteNewSC(File, true);
			} /* esle */
		    fputs(Trans->CurrState->Name.chars(), File);
		    fputs(": { ", File);
		    if (Trans->Condition)
		       { fputs("if (", File);
			 if ((SIR_Error = Trans->Condition->WriteSC(File,
								WriteNotes)))
			    { return(SIR_Error);
			     } /* fi */
			 fputs(") ", File);
			} /* fi */
		    if (Trans->NextState)
		       { fputs("goto ", File);
			 fputs(Trans->NextState->Name.chars(), File);
			 fputc(';', File);
			} /* fi */
		    else
		       { fputs("break;", File);
			} /* else */
		    while(  (Trans->Succ())
			  &&(Trans->Succ()->CurrState == Trans->CurrState))
		       { Trans = Trans->Succ();	/* combine same CurrStates */
			 if (Trans->LineInfo)
			    { if ((SIR_Error = Trans->LineInfo->WriteSC(File,
									true)))
				 { return(SIR_Error);
				  } /* fi */
			     } /* fi */
			 else
			    { SIR_LineInfo::WriteNewSC(File, true);
			     } /* esle */
			 if (Trans->Condition)
			    { fputs("if (", File);
			      if ((SIR_Error = Trans->Condition->WriteSC(File,
								WriteNotes)))
				 { return(SIR_Error);
				  } /* fi */
			      fputs(") ", File);
			     } /* fi */
			 if (Trans->NextState)
			    { fputs("goto ", File);
			      fputs(Trans->NextState->Name.chars(), File);
			      fputc(';', File);
			     } /* fi */
			 else
			    { fputs("break;", File);
			     } /* esle */
			} /* elihw */
		    fputs(" }", File);
		    Trans = Trans->Succ();
		   } /* elihw */
	       fputs(" }", File);
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_WAIT:
	{ if (CplusplusMode)
	     { fputs(SIR_SIM_WAIT_NAME "(", File);
	       Event = Events->First();
	       while(Event)
		  { fputc('&', File);
		    fputs(Event->Symbol->Name.chars(), File);
		    Event = Event->Succ();
		    fputs(", ", File);
		   } /* elihw */
	       fputs("0);", File);
	      } /* fi */
	  else
	     { fputs("wait(", File);
	       Event = Events->First();
	       while(Event)
		  { fputs(Event->Symbol->Name.chars(), File);
		    Event = Event->Succ();
		    if (Event)
		       { fputs(", ", File);
			} /* fi */
		   } /* elihw */
	       fputs(");", File);
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_WAITFOR:
	{ if (CplusplusMode)
	     { fputs(SIR_SIM_WAITFOR_NAME "(", File);
	      } /* fi */
	  else
	     { fputs("waitfor ", File);
	      } /* esle */
	  if ((SIR_Error = Expression1->WriteSC(File, WriteNotes,
						CplusplusMode, SIR_TYPE_TIME)))
	     { return(SIR_Error);
	      } /* fi */
	  if (CplusplusMode)
	     { fputc(')', File);
	      } /* fi */
	  fputc(';', File);
	  break;
	 }
     case SIR_STMNT_NOTIFY:
	{ if (CplusplusMode)
	     { fputs(SIR_SIM_NOTIFY_NAME "(", File);
	       Event = Events->First();
	       while(Event)
		  { fputc('&', File);
		    fputs(Event->Symbol->Name.chars(), File);
		    Event = Event->Succ();
		    fputs(", ", File);
		   } /* elihw */
	       fputs("0);", File);
	      } /* fi */
	  else
	     { fputs("notify(", File);
	       Event = Events->First();
	       while(Event)
		  { fputs(Event->Symbol->Name.chars(), File);
		    Event = Event->Succ();
		    if (Event)
		       { fputs(", ", File);
			} /* fi */
		   } /* elihw */
	       fputs(");", File);
	      } /* esle */
	  break;
	 }
     case SIR_STMNT_NOTIFYONE:
	{ if (CplusplusMode)
	     { fputs(SIR_SIM_NOTIFYONE_NAME "(", File);
	       Event = Events->First();
	       while(Event)
		  { fputc('&', File);
		    fputs(Event->Symbol->Name.chars(), File);
		    Event = Event->Succ();
		    fputs(", ", File);
		   } /* elihw */
	       fputs("0);", File);
	      } /* fi */
	  else
	     { fputs("notifyone(", File);
	       Event = Events->First();
	       while(Event)
		  { fputs(Event->Symbol->Name.chars(), File);
		    Event = Event->Succ();
		    if (Event)
		       { fputs(", ", File);
			} /* fi */
		   } /* elihw */
	       fputs(");", File);
	      } /* esle */
	  break;
	 }
     default:
	{ assert(FALSE);	/* bad StmntType */
	 }
    } /* hctiws */

if (ferror(File))
   { SIR_ErrMsg.form("Writing to file failed%s",
			GL_SystemErrorMessage());
     return(SIR_ERROR_WRITING_SPECC_FILE);
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statement::WriteSC */


	/**********************/
	/*** SIR_Statements ***/
	/**********************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Statements::SIR_Statements(		/* constructor #1 */
	sir_symbols	*CmpndScope = NULL,
	sir_statement	*FirstEntry = NULL) :
		SIR_List<SIR_Statement>(FirstEntry)
{

SIR_Statements::CmpndScope = CmpndScope;

} /* end of SIR_Statements::SIR_Statements #1 */


SIR_Statements::SIR_Statements(		/* constructor #3 (duplicator) */
	sir_statements	*Original)
{
sir_statement	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_Statement(Curr));
     Curr = Curr->Succ();
    } /* elihw */

SIR_Statements::CmpndScope	= Original->CmpndScope;

} /* end of SIR_Statements::SIR_Statements #3 */


SIR_Statements::~SIR_Statements(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_Statements::~SIR_Statements */


ERROR SIR_Statements::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a node, but there are nodes below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllNodes */


ERROR SIR_Statements::DFS_ForAllSymbols( /* iterator over all symbols */
	sir_symbol_mptr	MemberFct,	/* (depth first) */
	sir_symbol_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a symbol, but there are symbols below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllSymbols(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllSymbols */


ERROR SIR_Statements::DFS_ForAllUserTypes( /* iterator over all usertypes */
	sir_usertp_mptr	MemberFct,	/* (depth first) */
	sir_usertp_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a usertype, but there are usertypes below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllUserTypes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllUserTypes */


ERROR SIR_Statements::DFS_ForAllNotes(	/* iterator over all notes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a note, but there are notes below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllNotes */


ERROR SIR_Statements::DFS_ForAllStatements(	/* iterator over all stmnts. */
	sir_stmnt_mptr	MemberFct,		/* (depth first) */
	sir_stmnt_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not a statement, but there are statements below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllStatements(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllStatements */


ERROR SIR_Statements::DFS_ForAllExpressions( /* iterator over all expressions */
	sir_expr_mptr	MemberFct,		/* (depth first) */
	sir_expr_marg	MemberFctArg)
{
sir_statement	*Stmnt,
		*Succ;

/* this is not an expression, but there are expressions below */

Stmnt = First();
while(Stmnt)
   { Succ = Stmnt->Succ();
     if ((SIR_Error = Stmnt->DFS_ForAllExpressions(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Stmnt = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Statements::DFS_ForAllExpressions */


BOOL SIR_Statements::DFS_FindDependant(	/* searches for dependants (DFS) */
	sir_symbol	*ThatSymbol,
	sir_statement	**DepStmnt,
	sir_expression	**DepExpr,
	SIR_DEPENDENCY	*Reason)
{
sir_statement	*Stmnt;

assert(ThatSymbol != NULL);

/* this is not a statement, but there are statements below */

Stmnt = First();
while(Stmnt)
   { if (Stmnt->DFS_FindDependant(ThatSymbol, DepStmnt, DepExpr, Reason))
	{ return(TRUE);
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

return(FALSE);

} /* end of SIR_Statements::DFS_FindDependant */


void SIR_Statements::SetAlias(	/* sets all type, usertype, symbol alias' */
	sir_statements	*Alias)	/* (iterates over symbols and usertypes) */
{
sir_statement	*Statement,
		*StatementAlias;

assert(Alias != NULL);

/* no need to process this node */

Statement = First();
StatementAlias = Alias->First();
while(Statement)
   { Statement->SetAlias(StatementAlias);
     Statement = Statement->Succ();
     StatementAlias = StatementAlias->Succ();
    } /* elihw */

} /* end of SIR_Statements::SetAlias */


sir_statement *SIR_Statements::FindLabeledStmnt(/* find a labeled statement */
	sir_label	*Label)		/* (returns NULL if not found) */
{
sir_statement	*Stmnt;

Stmnt = First();
while(Stmnt)
   { if (  (Stmnt->StmntType == SIR_STMNT_LABELED)
	 &&(Stmnt->Label == Label))
	{ break;
	 } /* fi */
     Stmnt = Stmnt->Succ();
    } /* elihw */

return(Stmnt);

} /* end of SIR_Statements::FindLabeledStmnt */


/*** exported functions *************************************************/


	/* none */


/* EOF Statement.cc */
