/************************************************************************/
/* Label.cc: SpecC Internal Representation, Label Class			*/
/************************************************************************/
/* Author: Rainer Doemer			first version: 10/20/97 */
/************************************************************************/

/* last update: 05/31/01 */

/* modifications: (most recent first)
 *
 * 05/31/01 RD	eliminated level 2 of SIR API
 * 05/25/01 RD	eliminated support for binary SIR files (import/export)
 * 04/30/01 RD	replaced use of obsolete form() from "stream.h" with own one
 * 01/30/01 RD	fixed a potential FMR problem in iteratorx
 *		SIR_Labels::DFS_ForAllXxxxx
 */

#include "IntRep/Label.h"
#include "IntRep/Symbol.h"

#include <assert.h>


/*** constants and macros ***********************************************/


	/* (none) */


/*** internal type declarations *****************************************/


	/* (none) */


/*** class implementations **********************************************/


	/*****************/
	/*** SIR_Label ***/
	/*****************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Label::SIR_Label(			/* constructor #1 */
	const char	*LabelName,
	sir_statement	*Statement = NULL,
	sir_notes	*Notes = NULL)
{

assert(LabelName != NULL);
assert(*LabelName != 0);

SIR_Label::LabelName	= LabelName;
SIR_Label::Statement	= Statement;
SIR_Label::Notes	= Notes;

} /* end of SIR_Label::SIR_Label #1 */


SIR_Label::SIR_Label(			/* constructor #3 (duplicator) */
	sir_label	*Original)
{

SIR_Label::LabelName	= Original->LabelName;
SIR_Label::Statement	= Original->Statement;
SIR_Label::Notes	= Original->Notes ?
				new SIR_Notes(Original->Notes) :
				NULL;

} /* end of SIR_Label::SIR_Label #3 */


SIR_Label::~SIR_Label(void)		/* destructor */
{

delete Notes;

} /* end of SIR_Label::~SIR_Label */


ERROR SIR_Label::DFS_ForAllNodes(	/* iterator for over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{

/* this is not a node, but there are nodes below */

if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Label::DFS_ForAllNodes */


ERROR SIR_Label::DFS_ForAllNotes(	/* iterator for over all nodes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{

/* this is not a note, but there are notes below */

if (Notes)
   { if ((SIR_Error = Notes->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
    } /* fi */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Label::DFS_ForAllNotes */


ERROR SIR_Label::Annotate(	/* attach a note to the label */
	sir_note	*NewNote)	/* (consumes NewNote) */
{

assert(NewNote != NULL);

NewNote->Symbol = NULL;		/* label note */
NewNote->UserType = NULL;
NewNote->Label = this;

if (Notes)
   { if (! Notes->Insert(NewNote))
	{ delete NewNote;
	  return(SIR_Error);
	 } /* fi */
    } /* fi */
else
   { Notes = new SIR_Notes(NewNote);
    } /* esle */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Label::Annotate */


	/******************/
	/*** SIR_Labels ***/
	/******************/


//++++++++++++++++++++++++++++ API Layer 1 +++++++++++++++++++++++++++++//


SIR_Labels::SIR_Labels(void)	/* constructor #1 (empty list) */
{

/* nothing to do */

} /* end of SIR_Labels::SIR_Labels #1 */


SIR_Labels::SIR_Labels(		/* constructor #2 (duplicator) */
	sir_labels	*Original)
{
sir_label	*Curr;

Curr = Original->First();
while(Curr)
   { Append(new SIR_Label(Curr));
     Curr = Curr->Succ();
    } /* elihw */

} /* end of SIR_Labels::SIR_Labels #2 */


SIR_Labels::~SIR_Labels(void)	/* destructor */
{

/* nothing to do */

} /* end of SIR_Labels::~SIR_Labels */


ERROR SIR_Labels::DFS_ForAllNodes(	/* iterator over all nodes */
	sir_node_mptr	MemberFct,	/* (depth first) */
	sir_node_marg	MemberFctArg)
{
sir_label	*Label,
		*Succ;

/* this is not a node, but there are nodes below */

Label = First();
while(Label)
   { Succ = Label->Succ();
     if ((SIR_Error = Label->DFS_ForAllNodes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Label = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Labels::DFS_ForAllNodes */


ERROR SIR_Labels::DFS_ForAllNotes(	/* iterator over all nodes */
	sir_note_mptr	MemberFct,	/* (depth first) */
	sir_note_marg	MemberFctArg)
{
sir_label	*Label,
		*Succ;

/* this is not a note, but there are notes below */

Label = First();
while(Label)
   { Succ = Label->Succ();
     if ((SIR_Error = Label->DFS_ForAllNotes(MemberFct, MemberFctArg)))
	{ return(SIR_Error);
	 } /* fi */
     Label = Succ;
    } /* elihw */

return(SIR_ERROR_NO_ERROR);

} /* end of SIR_Labels::DFS_ForAllNotes */


sir_label *SIR_Labels::Define(		/* define a label */
	const char	*LabelName,	/* (might return NULL and SIR_Error) */
	sir_statement	*Statement)
{
sir_label	*Label;

Label = FindOrInsert(LabelName);
assert(Label != NULL);

if (Label->Statement)	/* already defined? */
   { SIR_ErrMsg.form("Redefinition of label '%s'", LabelName);
     SIR_Error = SIR_ERROR_REDEFINITION_OF_LABEL;
     return(NULL);
    } /* fi */

Label->Statement = Statement;	/* set the statement */

return(Label);

} /* end of SIR_Labels::Define */


ERROR SIR_Labels::CheckDefStatements(	/* check if all labels are defined */
	void)
{
sir_label	*Label;

Label = First();
while(Label)
   { if (!(Label->Statement))
	{ SIR_ErrMsg.form("Undefined label '%s'", Label->LabelName.chars());
	  return(SIR_ERROR_UNDEFINED_LABEL);
	 } /* fi */
     Label = Label->Succ();
    } /* elihw */

return(SIR_ERROR_NO_ERROR);	/* everything ok */

} /* end of SIR_Labels::CheckDefStatements */


void SIR_Labels::DeleteMarkedLabels(	/* deletes the marked labels */
	void)
{
sir_label	*Label,
		*Succ;

Label = First();
while(Label)
   { Succ = Label->Succ();
     if (Label->Color == SIR_RED)
	{ Remove(Label);
	  delete Label;
	 } /* fi */
     Label = Succ;
    } /* elihw */

} /* end of SIR_Labels::DeleteMarkedLabels */


sir_label *SIR_Labels::FindOrInsert(	/* find or insert a label */
	const char	*LabelName)
{
int		CmpVal;
sir_label	*NewLabel;

First();
while(Curr())
   { if (0 == (CmpVal = strcmp(LabelName, Curr()->LabelName.chars())))
	{ return(Curr());	/* found */
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

NewLabel = new SIR_Label(LabelName);	/* not found, insert new one */

if (Curr())
   { return(InsertBefore(NewLabel));
    } /* fi */
else
   { return(Append(NewLabel));
    } /* esle */

} /* end of SIR_Labels::FindOrInsert */


sir_label *SIR_Labels::Insert(		/* insert a new label */
	sir_label	*Label)
{
int		CmpVal;

First();
while(Curr())
   { if (0 == (CmpVal = strcmp(Label->LabelName.chars(),
				Curr()->LabelName.chars())))
	{ assert(FALSE);	/* already exists */
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

if (Curr())
   { return(InsertBefore(Label));
    } /* fi */
else
   { return(Append(Label));
    } /* esle */

} /* end of SIR_Labels::Insert */


sir_label *SIR_Labels::Find(		/* find a label */
	const char	*LabelName)	/* (returns NULL if not found) */
{
int		CmpVal;

First();
while(Curr())
   { if (0 == (CmpVal = strcmp(LabelName, Curr()->LabelName.chars())))
	{ return(Curr());	/* found */
	 } /* fi */
     if (CmpVal < 0)
	{ break;
	 } /* fi */
     Next();
    } /* elihw */

return(NULL);	/* not found */

} /* end of SIR_Labels::Find */


/*** exported functions *************************************************/


	/* none */


/* EOF Label.cc */
